#define _CFFI_

/* We try to define Py_LIMITED_API before including Python.h.

   Mess: we can only define it if Py_DEBUG, Py_TRACE_REFS and
   Py_REF_DEBUG are not defined.  This is a best-effort approximation:
   we can learn about Py_DEBUG from pyconfig.h, but it is unclear if
   the same works for the other two macros.  Py_DEBUG implies them,
   but not the other way around.

   The implementation is messy (issue #350): on Windows, with _MSC_VER,
   we have to define Py_LIMITED_API even before including pyconfig.h.
   In that case, we guess what pyconfig.h will do to the macros above,
   and check our guess after the #include.

   Note that on Windows, with CPython 3.x, you need >= 3.5 and virtualenv
   version >= 16.0.0.  With older versions of either, you don't get a
   copy of PYTHON3.DLL in the virtualenv.  We can't check the version of
   CPython *before* we even include pyconfig.h.  ffi.set_source() puts
   a ``#define _CFFI_NO_LIMITED_API'' at the start of this file if it is
   running on Windows < 3.5, as an attempt at fixing it, but that's
   arguably wrong because it may not be the target version of Python.
   Still better than nothing I guess.  As another workaround, you can
   remove the definition of Py_LIMITED_API here.

   See also 'py_limited_api' in cffi/setuptools_ext.py.
*/
#if !defined(_CFFI_USE_EMBEDDING) && !defined(Py_LIMITED_API)
#  ifdef _MSC_VER
#    if !defined(_DEBUG) && !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#    include <pyconfig.h>
     /* sanity-check: Py_LIMITED_API will cause crashes if any of these
        are also defined.  Normally, the Python file PC/pyconfig.h does not
        cause any of these to be defined, with the exception that _DEBUG
        causes Py_DEBUG.  Double-check that. */
#    ifdef Py_LIMITED_API
#      if defined(Py_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_DEBUG, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_TRACE_REFS)
#        error "pyconfig.h unexpectedly defines Py_TRACE_REFS, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_REF_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_REF_DEBUG, but Py_LIMITED_API is set"
#      endif
#    endif
#  else
#    include <pyconfig.h>
#    if !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#  endif
#endif

#include <Python.h>
#ifdef __cplusplus
extern "C" {
#endif
#include <stddef.h>

/* This part is from file 'cffi/parse_c_type.h'.  It is copied at the
   beginning of C sources generated by CFFI's ffi.set_source(). */

typedef void *_cffi_opcode_t;

#define _CFFI_OP(opcode, arg)   (_cffi_opcode_t)(opcode | (((uintptr_t)(arg)) << 8))
#define _CFFI_GETOP(cffi_opcode)    ((unsigned char)(uintptr_t)cffi_opcode)
#define _CFFI_GETARG(cffi_opcode)   (((intptr_t)cffi_opcode) >> 8)

#define _CFFI_OP_PRIMITIVE       1
#define _CFFI_OP_POINTER         3
#define _CFFI_OP_ARRAY           5
#define _CFFI_OP_OPEN_ARRAY      7
#define _CFFI_OP_STRUCT_UNION    9
#define _CFFI_OP_ENUM           11
#define _CFFI_OP_FUNCTION       13
#define _CFFI_OP_FUNCTION_END   15
#define _CFFI_OP_NOOP           17
#define _CFFI_OP_BITFIELD       19
#define _CFFI_OP_TYPENAME       21
#define _CFFI_OP_CPYTHON_BLTN_V 23   // varargs
#define _CFFI_OP_CPYTHON_BLTN_N 25   // noargs
#define _CFFI_OP_CPYTHON_BLTN_O 27   // O  (i.e. a single arg)
#define _CFFI_OP_CONSTANT       29
#define _CFFI_OP_CONSTANT_INT   31
#define _CFFI_OP_GLOBAL_VAR     33
#define _CFFI_OP_DLOPEN_FUNC    35
#define _CFFI_OP_DLOPEN_CONST   37
#define _CFFI_OP_GLOBAL_VAR_F   39
#define _CFFI_OP_EXTERN_PYTHON  41

#define _CFFI_PRIM_VOID          0
#define _CFFI_PRIM_BOOL          1
#define _CFFI_PRIM_CHAR          2
#define _CFFI_PRIM_SCHAR         3
#define _CFFI_PRIM_UCHAR         4
#define _CFFI_PRIM_SHORT         5
#define _CFFI_PRIM_USHORT        6
#define _CFFI_PRIM_INT           7
#define _CFFI_PRIM_UINT          8
#define _CFFI_PRIM_LONG          9
#define _CFFI_PRIM_ULONG        10
#define _CFFI_PRIM_LONGLONG     11
#define _CFFI_PRIM_ULONGLONG    12
#define _CFFI_PRIM_FLOAT        13
#define _CFFI_PRIM_DOUBLE       14
#define _CFFI_PRIM_LONGDOUBLE   15

#define _CFFI_PRIM_WCHAR        16
#define _CFFI_PRIM_INT8         17
#define _CFFI_PRIM_UINT8        18
#define _CFFI_PRIM_INT16        19
#define _CFFI_PRIM_UINT16       20
#define _CFFI_PRIM_INT32        21
#define _CFFI_PRIM_UINT32       22
#define _CFFI_PRIM_INT64        23
#define _CFFI_PRIM_UINT64       24
#define _CFFI_PRIM_INTPTR       25
#define _CFFI_PRIM_UINTPTR      26
#define _CFFI_PRIM_PTRDIFF      27
#define _CFFI_PRIM_SIZE         28
#define _CFFI_PRIM_SSIZE        29
#define _CFFI_PRIM_INT_LEAST8   30
#define _CFFI_PRIM_UINT_LEAST8  31
#define _CFFI_PRIM_INT_LEAST16  32
#define _CFFI_PRIM_UINT_LEAST16 33
#define _CFFI_PRIM_INT_LEAST32  34
#define _CFFI_PRIM_UINT_LEAST32 35
#define _CFFI_PRIM_INT_LEAST64  36
#define _CFFI_PRIM_UINT_LEAST64 37
#define _CFFI_PRIM_INT_FAST8    38
#define _CFFI_PRIM_UINT_FAST8   39
#define _CFFI_PRIM_INT_FAST16   40
#define _CFFI_PRIM_UINT_FAST16  41
#define _CFFI_PRIM_INT_FAST32   42
#define _CFFI_PRIM_UINT_FAST32  43
#define _CFFI_PRIM_INT_FAST64   44
#define _CFFI_PRIM_UINT_FAST64  45
#define _CFFI_PRIM_INTMAX       46
#define _CFFI_PRIM_UINTMAX      47
#define _CFFI_PRIM_FLOATCOMPLEX 48
#define _CFFI_PRIM_DOUBLECOMPLEX 49
#define _CFFI_PRIM_CHAR16       50
#define _CFFI_PRIM_CHAR32       51

#define _CFFI__NUM_PRIM         52
#define _CFFI__UNKNOWN_PRIM           (-1)
#define _CFFI__UNKNOWN_FLOAT_PRIM     (-2)
#define _CFFI__UNKNOWN_LONG_DOUBLE    (-3)

#define _CFFI__IO_FILE_STRUCT         (-1)


struct _cffi_global_s {
    const char *name;
    void *address;
    _cffi_opcode_t type_op;
    void *size_or_direct_fn;  // OP_GLOBAL_VAR: size, or 0 if unknown
                              // OP_CPYTHON_BLTN_*: addr of direct function
};

struct _cffi_getconst_s {
    unsigned long long value;
    const struct _cffi_type_context_s *ctx;
    int gindex;
};

struct _cffi_struct_union_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_STRUCT_UNION
    int flags;               // _CFFI_F_* flags below
    size_t size;
    int alignment;
    int first_field_index;   // -> _cffi_fields array
    int num_fields;
};
#define _CFFI_F_UNION         0x01   // is a union, not a struct
#define _CFFI_F_CHECK_FIELDS  0x02   // complain if fields are not in the
                                     // "standard layout" or if some are missing
#define _CFFI_F_PACKED        0x04   // for CHECK_FIELDS, assume a packed struct
#define _CFFI_F_EXTERNAL      0x08   // in some other ffi.include()
#define _CFFI_F_OPAQUE        0x10   // opaque

struct _cffi_field_s {
    const char *name;
    size_t field_offset;
    size_t field_size;
    _cffi_opcode_t field_type_op;
};

struct _cffi_enum_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_ENUM
    int type_prim;           // _CFFI_PRIM_xxx
    const char *enumerators; // comma-delimited string
};

struct _cffi_typename_s {
    const char *name;
    int type_index;   /* if opaque, points to a possibly artificial
                         OP_STRUCT which is itself opaque */
};

struct _cffi_type_context_s {
    _cffi_opcode_t *types;
    const struct _cffi_global_s *globals;
    const struct _cffi_field_s *fields;
    const struct _cffi_struct_union_s *struct_unions;
    const struct _cffi_enum_s *enums;
    const struct _cffi_typename_s *typenames;
    int num_globals;
    int num_struct_unions;
    int num_enums;
    int num_typenames;
    const char *const *includes;
    int num_types;
    int flags;      /* future extension */
};

struct _cffi_parse_info_s {
    const struct _cffi_type_context_s *ctx;
    _cffi_opcode_t *output;
    unsigned int output_size;
    size_t error_location;
    const char *error_message;
};

struct _cffi_externpy_s {
    const char *name;
    size_t size_of_result;
    void *reserved1, *reserved2;
};

#ifdef _CFFI_INTERNAL
static int parse_c_type(struct _cffi_parse_info_s *info, const char *input);
static int search_in_globals(const struct _cffi_type_context_s *ctx,
                             const char *search, size_t search_len);
static int search_in_struct_unions(const struct _cffi_type_context_s *ctx,
                                   const char *search, size_t search_len);
#endif

/* this block of #ifs should be kept exactly identical between
   c/_cffi_backend.c, cffi/vengine_cpy.py, cffi/vengine_gen.py
   and cffi/_cffi_include.h */
#if defined(_MSC_VER)
# include <malloc.h>   /* for alloca() */
# if _MSC_VER < 1600   /* MSVC < 2010 */
   typedef __int8 int8_t;
   typedef __int16 int16_t;
   typedef __int32 int32_t;
   typedef __int64 int64_t;
   typedef unsigned __int8 uint8_t;
   typedef unsigned __int16 uint16_t;
   typedef unsigned __int32 uint32_t;
   typedef unsigned __int64 uint64_t;
   typedef __int8 int_least8_t;
   typedef __int16 int_least16_t;
   typedef __int32 int_least32_t;
   typedef __int64 int_least64_t;
   typedef unsigned __int8 uint_least8_t;
   typedef unsigned __int16 uint_least16_t;
   typedef unsigned __int32 uint_least32_t;
   typedef unsigned __int64 uint_least64_t;
   typedef __int8 int_fast8_t;
   typedef __int16 int_fast16_t;
   typedef __int32 int_fast32_t;
   typedef __int64 int_fast64_t;
   typedef unsigned __int8 uint_fast8_t;
   typedef unsigned __int16 uint_fast16_t;
   typedef unsigned __int32 uint_fast32_t;
   typedef unsigned __int64 uint_fast64_t;
   typedef __int64 intmax_t;
   typedef unsigned __int64 uintmax_t;
# else
#  include <stdint.h>
# endif
# if _MSC_VER < 1800   /* MSVC < 2013 */
#  ifndef __cplusplus
    typedef unsigned char _Bool;
#  endif
# endif
#else
# include <stdint.h>
# if (defined (__SVR4) && defined (__sun)) || defined(_AIX) || defined(__hpux)
#  include <alloca.h>
# endif
#endif

#ifdef __GNUC__
# define _CFFI_UNUSED_FN  __attribute__((unused))
#else
# define _CFFI_UNUSED_FN  /* nothing */
#endif

#ifdef __cplusplus
# ifndef _Bool
   typedef bool _Bool;   /* semi-hackish: C++ has no _Bool; bool is builtin */
# endif
#endif

/**********  CPython-specific section  **********/
#ifndef PYPY_VERSION


#if PY_MAJOR_VERSION >= 3
# define PyInt_FromLong PyLong_FromLong
#endif

#define _cffi_from_c_double PyFloat_FromDouble
#define _cffi_from_c_float PyFloat_FromDouble
#define _cffi_from_c_long PyInt_FromLong
#define _cffi_from_c_ulong PyLong_FromUnsignedLong
#define _cffi_from_c_longlong PyLong_FromLongLong
#define _cffi_from_c_ulonglong PyLong_FromUnsignedLongLong
#define _cffi_from_c__Bool PyBool_FromLong

#define _cffi_to_c_double PyFloat_AsDouble
#define _cffi_to_c_float PyFloat_AsDouble

#define _cffi_from_c_int(x, type)                                        \
    (((type)-1) > 0 ? /* unsigned */                                     \
        (sizeof(type) < sizeof(long) ?                                   \
            PyInt_FromLong((long)x) :                                    \
         sizeof(type) == sizeof(long) ?                                  \
            PyLong_FromUnsignedLong((unsigned long)x) :                  \
            PyLong_FromUnsignedLongLong((unsigned long long)x)) :        \
        (sizeof(type) <= sizeof(long) ?                                  \
            PyInt_FromLong((long)x) :                                    \
            PyLong_FromLongLong((long long)x)))

#define _cffi_to_c_int(o, type)                                          \
    ((type)(                                                             \
     sizeof(type) == 1 ? (((type)-1) > 0 ? (type)_cffi_to_c_u8(o)        \
                                         : (type)_cffi_to_c_i8(o)) :     \
     sizeof(type) == 2 ? (((type)-1) > 0 ? (type)_cffi_to_c_u16(o)       \
                                         : (type)_cffi_to_c_i16(o)) :    \
     sizeof(type) == 4 ? (((type)-1) > 0 ? (type)_cffi_to_c_u32(o)       \
                                         : (type)_cffi_to_c_i32(o)) :    \
     sizeof(type) == 8 ? (((type)-1) > 0 ? (type)_cffi_to_c_u64(o)       \
                                         : (type)_cffi_to_c_i64(o)) :    \
     (Py_FatalError("unsupported size for type " #type), (type)0)))

#define _cffi_to_c_i8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[1])
#define _cffi_to_c_u8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[2])
#define _cffi_to_c_i16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[3])
#define _cffi_to_c_u16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[4])
#define _cffi_to_c_i32                                                   \
                 ((int(*)(PyObject *))_cffi_exports[5])
#define _cffi_to_c_u32                                                   \
                 ((unsigned int(*)(PyObject *))_cffi_exports[6])
#define _cffi_to_c_i64                                                   \
                 ((long long(*)(PyObject *))_cffi_exports[7])
#define _cffi_to_c_u64                                                   \
                 ((unsigned long long(*)(PyObject *))_cffi_exports[8])
#define _cffi_to_c_char                                                  \
                 ((int(*)(PyObject *))_cffi_exports[9])
#define _cffi_from_c_pointer                                             \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[10])
#define _cffi_to_c_pointer                                               \
    ((char *(*)(PyObject *, struct _cffi_ctypedescr *))_cffi_exports[11])
#define _cffi_get_struct_layout                                          \
    not used any more
#define _cffi_restore_errno                                              \
    ((void(*)(void))_cffi_exports[13])
#define _cffi_save_errno                                                 \
    ((void(*)(void))_cffi_exports[14])
#define _cffi_from_c_char                                                \
    ((PyObject *(*)(char))_cffi_exports[15])
#define _cffi_from_c_deref                                               \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[16])
#define _cffi_to_c                                                       \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[17])
#define _cffi_from_c_struct                                              \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[18])
#define _cffi_to_c_wchar_t                                               \
    ((_cffi_wchar_t(*)(PyObject *))_cffi_exports[19])
#define _cffi_from_c_wchar_t                                             \
    ((PyObject *(*)(_cffi_wchar_t))_cffi_exports[20])
#define _cffi_to_c_long_double                                           \
    ((long double(*)(PyObject *))_cffi_exports[21])
#define _cffi_to_c__Bool                                                 \
    ((_Bool(*)(PyObject *))_cffi_exports[22])
#define _cffi_prepare_pointer_call_argument                              \
    ((Py_ssize_t(*)(struct _cffi_ctypedescr *,                           \
                    PyObject *, char **))_cffi_exports[23])
#define _cffi_convert_array_from_object                                  \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[24])
#define _CFFI_CPIDX  25
#define _cffi_call_python                                                \
    ((void(*)(struct _cffi_externpy_s *, char *))_cffi_exports[_CFFI_CPIDX])
#define _cffi_to_c_wchar3216_t                                           \
    ((int(*)(PyObject *))_cffi_exports[26])
#define _cffi_from_c_wchar3216_t                                         \
    ((PyObject *(*)(int))_cffi_exports[27])
#define _CFFI_NUM_EXPORTS 28

struct _cffi_ctypedescr;

static void *_cffi_exports[_CFFI_NUM_EXPORTS];

#define _cffi_type(index)   (                           \
    assert((((uintptr_t)_cffi_types[index]) & 1) == 0), \
    (struct _cffi_ctypedescr *)_cffi_types[index])

static PyObject *_cffi_init(const char *module_name, Py_ssize_t version,
                            const struct _cffi_type_context_s *ctx)
{
    PyObject *module, *o_arg, *new_module;
    void *raw[] = {
        (void *)module_name,
        (void *)version,
        (void *)_cffi_exports,
        (void *)ctx,
    };

    module = PyImport_ImportModule("_cffi_backend");
    if (module == NULL)
        goto failure;

    o_arg = PyLong_FromVoidPtr((void *)raw);
    if (o_arg == NULL)
        goto failure;

    new_module = PyObject_CallMethod(
        module, (char *)"_init_cffi_1_0_external_module", (char *)"O", o_arg);

    Py_DECREF(o_arg);
    Py_DECREF(module);
    return new_module;

  failure:
    Py_XDECREF(module);
    return NULL;
}


#ifdef HAVE_WCHAR_H
typedef wchar_t _cffi_wchar_t;
#else
typedef uint16_t _cffi_wchar_t;   /* same random pick as _cffi_backend.c */
#endif

_CFFI_UNUSED_FN static uint16_t _cffi_to_c_char16_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return (uint16_t)_cffi_to_c_wchar_t(o);
    else
        return (uint16_t)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char16_t(uint16_t x)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

_CFFI_UNUSED_FN static int _cffi_to_c_char32_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return (int)_cffi_to_c_wchar_t(o);
    else
        return (int)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char32_t(unsigned int x)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

union _cffi_union_alignment_u {
    unsigned char m_char;
    unsigned short m_short;
    unsigned int m_int;
    unsigned long m_long;
    unsigned long long m_longlong;
    float m_float;
    double m_double;
    long double m_longdouble;
};

struct _cffi_freeme_s {
    struct _cffi_freeme_s *next;
    union _cffi_union_alignment_u alignment;
};

_CFFI_UNUSED_FN static int
_cffi_convert_array_argument(struct _cffi_ctypedescr *ctptr, PyObject *arg,
                             char **output_data, Py_ssize_t datasize,
                             struct _cffi_freeme_s **freeme)
{
    char *p;
    if (datasize < 0)
        return -1;

    p = *output_data;
    if (p == NULL) {
        struct _cffi_freeme_s *fp = (struct _cffi_freeme_s *)PyObject_Malloc(
            offsetof(struct _cffi_freeme_s, alignment) + (size_t)datasize);
        if (fp == NULL)
            return -1;
        fp->next = *freeme;
        *freeme = fp;
        p = *output_data = (char *)&fp->alignment;
    }
    memset((void *)p, 0, (size_t)datasize);
    return _cffi_convert_array_from_object(p, ctptr, arg);
}

_CFFI_UNUSED_FN static void
_cffi_free_array_arguments(struct _cffi_freeme_s *freeme)
{
    do {
        void *p = (void *)freeme;
        freeme = freeme->next;
        PyObject_Free(p);
    } while (freeme != NULL);
}

/**********  end CPython-specific section  **********/
#else
_CFFI_UNUSED_FN
static void (*_cffi_call_python_org)(struct _cffi_externpy_s *, char *);
# define _cffi_call_python  _cffi_call_python_org
#endif


#define _cffi_array_len(array)   (sizeof(array) / sizeof((array)[0]))

#define _cffi_prim_int(size, sign)                                      \
    ((size) == 1 ? ((sign) ? _CFFI_PRIM_INT8  : _CFFI_PRIM_UINT8)  :    \
     (size) == 2 ? ((sign) ? _CFFI_PRIM_INT16 : _CFFI_PRIM_UINT16) :    \
     (size) == 4 ? ((sign) ? _CFFI_PRIM_INT32 : _CFFI_PRIM_UINT32) :    \
     (size) == 8 ? ((sign) ? _CFFI_PRIM_INT64 : _CFFI_PRIM_UINT64) :    \
     _CFFI__UNKNOWN_PRIM)

#define _cffi_prim_float(size)                                          \
    ((size) == sizeof(float) ? _CFFI_PRIM_FLOAT :                       \
     (size) == sizeof(double) ? _CFFI_PRIM_DOUBLE :                     \
     (size) == sizeof(long double) ? _CFFI__UNKNOWN_LONG_DOUBLE :       \
     _CFFI__UNKNOWN_FLOAT_PRIM)

#define _cffi_check_int(got, got_nonpos, expected)      \
    ((got_nonpos) == (expected <= 0) &&                 \
     (got) == (unsigned long long)expected)

#ifdef MS_WIN32
# define _cffi_stdcall  __stdcall
#else
# define _cffi_stdcall  /* nothing */
#endif

#ifdef __cplusplus
}
#endif

/************************************************************/


/* define our OpenSSL API compatibility level to 1.0.1. Any symbols older than
   that will raise an error during compilation. We can raise this number again
   after we drop 1.0.2 support in the distant future.  */
#define OPENSSL_API_COMPAT 0x10001000L

#include <openssl/opensslv.h>


#if defined(LIBRESSL_VERSION_NUMBER)
#define CRYPTOGRAPHY_IS_LIBRESSL 1
#else
#define CRYPTOGRAPHY_IS_LIBRESSL 0
#endif

/*
    LibreSSL removed e_os2.h from the public headers so we'll only include it
    if we're using vanilla OpenSSL.
*/
#if !CRYPTOGRAPHY_IS_LIBRESSL
#include <openssl/e_os2.h>
#endif
#if defined(_WIN32)
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <Wincrypt.h>
#include <Winsock2.h>
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
#define CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER     (LIBRESSL_VERSION_NUMBER >= 0x2070000f)
#define CRYPTOGRAPHY_LIBRESSL_28_OR_GREATER     (LIBRESSL_VERSION_NUMBER >= 0x2080000f)
#define CRYPTOGRAPHY_LIBRESSL_291_OR_GREATER     (LIBRESSL_VERSION_NUMBER >= 0x2090100f)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_332     (LIBRESSL_VERSION_NUMBER < 0x3030200f)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340     (LIBRESSL_VERSION_NUMBER < 0x3040000f)
#else
#define CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER (0)
#define CRYPTOGRAPHY_LIBRESSL_28_OR_GREATER (0)
#define CRYPTOGRAPHY_LIBRESSL_291_OR_GREATER (0)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_332 (0)
#define CRYPTOGRAPHY_LIBRESSL_LESS_THAN_340 (0)
#endif

#define CRYPTOGRAPHY_OPENSSL_102_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x10002000 && !CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_102L_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x100020cf && !CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_110_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x10100000 && !CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x1010006f && !CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x10101040 && !CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_300_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x30000000 && !CRYPTOGRAPHY_IS_LIBRESSL)

#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_102     (OPENSSL_VERSION_NUMBER < 0x10002000 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_102I     (OPENSSL_VERSION_NUMBER < 0x1000209f || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_110     (OPENSSL_VERSION_NUMBER < 0x10100000 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_110J     (OPENSSL_VERSION_NUMBER < 0x101000af || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111     (OPENSSL_VERSION_NUMBER < 0x10101000 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B     (OPENSSL_VERSION_NUMBER < 0x10101020 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_300     (OPENSSL_VERSION_NUMBER < 0x30000000 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111D     (OPENSSL_VERSION_NUMBER < 0x10101040 || CRYPTOGRAPHY_IS_LIBRESSL)
#if (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111D && !CRYPTOGRAPHY_IS_LIBRESSL &&     !defined(OPENSSL_NO_ENGINE)) || defined(USE_OSRANDOM_RNG_FOR_TESTING)
#define CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE 1
#else
#define CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE 0
#endif


#if CRYPTOGRAPHY_OPENSSL_300_OR_GREATER
#include <openssl/provider.h>
#include <openssl/proverr.h>
#endif


#include <openssl/aes.h>


#include <openssl/asn1.h>


#include <openssl/bn.h>


#include <openssl/bio.h>


#if !defined(OPENSSL_NO_CMAC)
#include <openssl/cmac.h>
#endif


#include <openssl/conf.h>


#include <openssl/crypto.h>


#if CRYPTOGRAPHY_OPENSSL_110_OR_GREATER
#include <openssl/ct.h>

typedef STACK_OF(SCT) Cryptography_STACK_OF_SCT;
#endif


#include <openssl/dh.h>


#include <openssl/dsa.h>


#include <openssl/ec.h>
#include <openssl/obj_mac.h>


#include <openssl/ecdh.h>


#include <openssl/ecdsa.h>


#include <openssl/engine.h>


#include <openssl/err.h>


#include <openssl/evp.h>




#include <openssl/hmac.h>


#include <openssl/obj_mac.h>


#include <openssl/objects.h>


#include <openssl/ocsp.h>


#include <openssl/opensslv.h>

#ifndef OPENSSL_NO_ENGINE
/* OpenSSL has ENGINE support so include all of this. */
#ifdef _WIN32
  #include <Wincrypt.h>
#else
  #include <fcntl.h>
  #include <unistd.h>
   /* for defined(BSD) */
  #include <sys/param.h>

  #ifdef BSD
    /* for SYS_getentropy */
    #include <sys/syscall.h>
  #endif

  #ifdef __APPLE__
    #include <sys/random.h>
    /* To support weak linking we need to declare this as a weak import even if
     * it's not present in sys/random (e.g. macOS < 10.12). */
    extern int getentropy(void *buffer, size_t size) __attribute((weak_import));
  #endif

  #ifdef __linux__
    /* for SYS_getrandom */
    #include <sys/syscall.h>
    #ifndef GRND_NONBLOCK
      #define GRND_NONBLOCK 0x0001
    #endif /* GRND_NONBLOCK */

    #ifndef SYS_getrandom
      /* We only bother to define the constants for platforms where we ship
       * wheels, since that's the predominant way you get a situation where
       * you don't have SYS_getrandom at compile time but do have the syscall
       * at runtime */
      #if defined __x86_64__
        #define SYS_getrandom 318
      #elif defined(__i386__)
        #define SYS_getrandom 355
      #endif
    #endif
  #endif /* __linux__ */
#endif /* _WIN32 */

#define CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM 1
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY 2
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM 3
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM 4

#ifndef CRYPTOGRAPHY_OSRANDOM_ENGINE
  #if defined(_WIN32)
    /* Windows */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM
  #elif defined(BSD) && defined(SYS_getentropy)
    /* OpenBSD 5.6+ & macOS with SYS_getentropy defined, although < 10.12 will fallback
     * to urandom */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY
  #elif defined(__linux__) && defined(SYS_getrandom)
    /* Linux 3.17+ */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM
  #else
    /* Keep this as last entry, fall back to /dev/urandom */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM
  #endif
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE */

/* Fallbacks need /dev/urandom helper functions. */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM || \
     CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM || \
     (CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY && \
     defined(__APPLE__))
  #define CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM 1
#endif

enum {
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED = -2,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS
};

enum {
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_NOT_INIT,
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK,
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS
};

/* engine ctrl */
#define CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION ENGINE_CMD_BASE

/* error reporting */
static void ERR_load_Cryptography_OSRandom_strings(void);
static void ERR_Cryptography_OSRandom_error(int function, int reason,
                                            char *file, int line);

#define CRYPTOGRAPHY_OSRANDOM_F_INIT 100
#define CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES 101
#define CRYPTOGRAPHY_OSRANDOM_F_FINISH 102
#define CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD 300
#define CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ 301

#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT 100
#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM 101
#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT 102

#define CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED 200

#define CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED 300
#define CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED 301

#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED 400
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED 402
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED 403
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT 404
#endif


#include <openssl/pem.h>


#include <openssl/pkcs12.h>


#include <openssl/rand.h>


#include <openssl/rsa.h>


#include <openssl/ssl.h>

typedef STACK_OF(SSL_CIPHER) Cryptography_STACK_OF_SSL_CIPHER;


#include <openssl/ssl.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `STACK_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef STACK_OF(X509) Cryptography_STACK_OF_X509;
typedef STACK_OF(X509_CRL) Cryptography_STACK_OF_X509_CRL;
typedef STACK_OF(X509_REVOKED) Cryptography_STACK_OF_X509_REVOKED;


#include <openssl/x509.h>

/*
 * See the comment above Cryptography_STACK_OF_X509 in x509.py
 */
typedef STACK_OF(X509_NAME) Cryptography_STACK_OF_X509_NAME;
typedef STACK_OF(X509_NAME_ENTRY) Cryptography_STACK_OF_X509_NAME_ENTRY;


#include <openssl/x509v3.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `LHASH_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef LHASH_OF(CONF_VALUE) Cryptography_LHASH_OF_CONF_VALUE;

typedef STACK_OF(ACCESS_DESCRIPTION) Cryptography_STACK_OF_ACCESS_DESCRIPTION;
typedef STACK_OF(DIST_POINT) Cryptography_STACK_OF_DIST_POINT;
typedef STACK_OF(POLICYQUALINFO) Cryptography_STACK_OF_POLICYQUALINFO;
typedef STACK_OF(POLICYINFO) Cryptography_STACK_OF_POLICYINFO;
typedef STACK_OF(ASN1_INTEGER) Cryptography_STACK_OF_ASN1_INTEGER;
typedef STACK_OF(GENERAL_SUBTREE) Cryptography_STACK_OF_GENERAL_SUBTREE;


#include <openssl/x509_vfy.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `STACK_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef STACK_OF(ASN1_OBJECT) Cryptography_STACK_OF_ASN1_OBJECT;
typedef STACK_OF(X509_OBJECT) Cryptography_STACK_OF_X509_OBJECT;


#include <openssl/pkcs7.h>


#include <openssl/ssl.h>
#include <openssl/x509.h>
#include <openssl/x509_vfy.h>
#include <openssl/crypto.h>

#ifdef _WIN32
#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#include <Wincrypt.h>
#include <Winsock2.h>
#else
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#endif




#if CRYPTOGRAPHY_OPENSSL_300_OR_GREATER
static const long Cryptography_HAS_PROVIDERS = 1;
#else
static const long Cryptography_HAS_PROVIDERS = 0;
typedef void OSSL_PROVIDER;
typedef void OSSL_LIB_CTX;
static const long PROV_R_BAD_DECRYPT = 0;
static const long PROV_R_XTS_DUPLICATED_KEYS = 0;
static const long PROV_R_WRONG_FINAL_BLOCK_LENGTH = 0;
#endif








#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
int BIO_up_ref(BIO *b) {
    CRYPTO_add(&b->references, 1, CRYPTO_LOCK_BIO);
    return 1;
}
#endif






/* In 1.1.0 SSLeay has finally been retired. We bidirectionally define the
   values so you can use either one. This is so we can use the new function
   names no matter what OpenSSL we're running on, but users on older pyOpenSSL
   releases won't see issues if they're running OpenSSL 1.1.0 */
#if !defined(SSLEAY_VERSION)
# define SSLeay                  OpenSSL_version_num
# define SSLeay_version          OpenSSL_version
# define SSLEAY_VERSION_NUMBER   OPENSSL_VERSION_NUMBER
# define SSLEAY_VERSION          OPENSSL_VERSION
# define SSLEAY_CFLAGS           OPENSSL_CFLAGS
# define SSLEAY_BUILT_ON         OPENSSL_BUILT_ON
# define SSLEAY_PLATFORM         OPENSSL_PLATFORM
# define SSLEAY_DIR              OPENSSL_DIR
#endif
#if !defined(OPENSSL_VERSION)
# define OpenSSL_version_num     SSLeay
# define OpenSSL_version         SSLeay_version
# define OPENSSL_VERSION         SSLEAY_VERSION
# define OPENSSL_CFLAGS          SSLEAY_CFLAGS
# define OPENSSL_BUILT_ON        SSLEAY_BUILT_ON
# define OPENSSL_PLATFORM        SSLEAY_PLATFORM
# define OPENSSL_DIR             SSLEAY_DIR
#endif
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
static const long Cryptography_HAS_LOCKING_CALLBACKS = 1;
#else
static const long Cryptography_HAS_LOCKING_CALLBACKS = 0;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
static const long Cryptography_HAS_OPENSSL_CLEANUP = 0;

void (*OPENSSL_cleanup)(void) = NULL;

/* This function has a significantly different signature pre-1.1.0. since it is
 * for testing only, we don't bother to expose it on older OpenSSLs.
 */
static const long Cryptography_HAS_MEM_FUNCTIONS = 0;
int (*Cryptography_CRYPTO_set_mem_functions)(
    void *(*)(size_t, const char *, int),
    void *(*)(void *, size_t, const char *, int),
    void (*)(void *, const char *, int)) = NULL;

#else
static const long Cryptography_HAS_OPENSSL_CLEANUP = 1;
static const long Cryptography_HAS_MEM_FUNCTIONS = 1;

int Cryptography_CRYPTO_set_mem_functions(
    void *(*m)(size_t, const char *, int),
    void *(*r)(void *, size_t, const char *, int),
    void (*f)(void *, const char *, int)
) {
    return CRYPTO_set_mem_functions(m, r, f);
}
#endif

void *Cryptography_malloc_wrapper(size_t size, const char *path, int line) {
    return malloc(size);
}

void *Cryptography_realloc_wrapper(void *ptr, size_t size, const char *path,
                                   int line) {
    return realloc(ptr, size);
}

void Cryptography_free_wrapper(void *ptr, const char *path, int line) {
    free(ptr);
}
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_300
#else
static const int CRYPTO_MEM_CHECK_ON=0;
static const int CRYPTO_MEM_CHECK_OFF=0;
static const int CRYPTO_MEM_CHECK_ENABLE=0;
static const int CRYPTO_MEM_CHECK_DISABLE=0;
#endif


#if CRYPTOGRAPHY_OPENSSL_110_OR_GREATER
static const long Cryptography_HAS_SCT = 1;
#else
static const long Cryptography_HAS_SCT = 0;

typedef enum {
    SCT_VERSION_NOT_SET,
    SCT_VERSION_V1
} sct_version_t;
typedef enum {
    CT_LOG_ENTRY_TYPE_NOT_SET,
    CT_LOG_ENTRY_TYPE_X509,
    CT_LOG_ENTRY_TYPE_PRECERT
} ct_log_entry_type_t;
typedef enum {
    SCT_SOURCE_UNKNOWN,
    SCT_SOURCE_TLS_EXTENSION,
    SCT_SOURCE_X509V3_EXTENSION,
    SCT_SOURCE_OCSP_STAPLED_RESPONSE
} sct_source_t;
typedef void SCT;
typedef void Cryptography_STACK_OF_SCT;

sct_version_t (*SCT_get_version)(const SCT *) = NULL;
ct_log_entry_type_t (*SCT_get_log_entry_type)(const SCT *) = NULL;
size_t (*SCT_get0_log_id)(const SCT *, unsigned char **) = NULL;
size_t (*SCT_get0_signature)(const SCT *, unsigned char **) = NULL;
uint64_t (*SCT_get_timestamp)(const SCT *) = NULL;

int (*SCT_set_source)(SCT *, sct_source_t) = NULL;

int (*sk_SCT_num)(const Cryptography_STACK_OF_SCT *) = NULL;
SCT *(*sk_SCT_value)(const Cryptography_STACK_OF_SCT *, int) = NULL;

void (*SCT_LIST_free)(Cryptography_STACK_OF_SCT *) = NULL;
int (*sk_SCT_push)(Cryptography_STACK_OF_SCT *, SCT *) = NULL;
Cryptography_STACK_OF_SCT *(*sk_SCT_new_null)(void) = NULL;
SCT *(*SCT_new)(void) = NULL;
int (*SCT_set1_log_id)(SCT *, unsigned char *, size_t) = NULL;
void (*SCT_set_timestamp)(SCT *, uint64_t) = NULL;
int (*SCT_set_version)(SCT *, sct_version_t) = NULL;
int (*SCT_set_log_entry_type)(SCT *, ct_log_entry_type_t) = NULL;
#endif


/* These functions were added in OpenSSL 1.1.0 */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
void DH_get0_pqg(const DH *dh,
                 const BIGNUM **p, const BIGNUM **q, const BIGNUM **g)
{
    if (p != NULL)
        *p = dh->p;
    if (q != NULL)
        *q = dh->q;
    if (g != NULL)
        *g = dh->g;
}

int DH_set0_pqg(DH *dh, BIGNUM *p, BIGNUM *q, BIGNUM *g)
{
    /* If the fields p and g in d are NULL, the corresponding input
     * parameters MUST be non-NULL.  q may remain NULL.
     */
    if ((dh->p == NULL && p == NULL)
        || (dh->g == NULL && g == NULL))
        return 0;

    if (p != NULL) {
        BN_free(dh->p);
        dh->p = p;
    }
    if (q != NULL) {
        BN_free(dh->q);
        dh->q = q;
    }
    if (g != NULL) {
        BN_free(dh->g);
        dh->g = g;
    }

    if (q != NULL) {
        dh->length = BN_num_bits(q);
    }

    return 1;
}

void DH_get0_key(const DH *dh, const BIGNUM **pub_key, const BIGNUM **priv_key)
{
    if (pub_key != NULL)
        *pub_key = dh->pub_key;
    if (priv_key != NULL)
        *priv_key = dh->priv_key;
}

int DH_set0_key(DH *dh, BIGNUM *pub_key, BIGNUM *priv_key)
{
    /* If the field pub_key in dh is NULL, the corresponding input
     * parameters MUST be non-NULL.  The priv_key field may
     * be left NULL.
     */
    if (dh->pub_key == NULL && pub_key == NULL)
        return 0;

    if (pub_key != NULL) {
        BN_free(dh->pub_key);
        dh->pub_key = pub_key;
    }
    if (priv_key != NULL) {
        BN_free(dh->priv_key);
        dh->priv_key = priv_key;
    }

    return 1;
}
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
#ifndef DH_CHECK_Q_NOT_PRIME
#define DH_CHECK_Q_NOT_PRIME            0x10
#endif

#ifndef DH_CHECK_INVALID_Q_VALUE
#define DH_CHECK_INVALID_Q_VALUE        0x20
#endif

#ifndef DH_CHECK_INVALID_J_VALUE
#define DH_CHECK_INVALID_J_VALUE        0x40
#endif

/* DH_check implementation taken from OpenSSL 1.1.0pre6 */

/*-
 * Check that p is a safe prime and
 * if g is 2, 3 or 5, check that it is a suitable generator
 * where
 * for 2, p mod 24 == 11
 * for 3, p mod 12 == 5
 * for 5, p mod 10 == 3 or 7
 * should hold.
 */

int Cryptography_DH_check(const DH *dh, int *ret)
{
    int ok = 0, r;
    BN_CTX *ctx = NULL;
    BN_ULONG l;
    BIGNUM *t1 = NULL, *t2 = NULL;

    *ret = 0;
    ctx = BN_CTX_new();
    if (ctx == NULL)
        goto err;
    BN_CTX_start(ctx);
    t1 = BN_CTX_get(ctx);
    if (t1 == NULL)
        goto err;
    t2 = BN_CTX_get(ctx);
    if (t2 == NULL)
        goto err;

    if (dh->q) {
        if (BN_cmp(dh->g, BN_value_one()) <= 0)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
        else if (BN_cmp(dh->g, dh->p) >= 0)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
        else {
            /* Check g^q == 1 mod p */
            if (!BN_mod_exp(t1, dh->g, dh->q, dh->p, ctx))
                goto err;
            if (!BN_is_one(t1))
                *ret |= DH_NOT_SUITABLE_GENERATOR;
        }
        r = BN_is_prime_ex(dh->q, BN_prime_checks, ctx, NULL);
        if (r < 0)
            goto err;
        if (!r)
            *ret |= DH_CHECK_Q_NOT_PRIME;
        /* Check p == 1 mod q  i.e. q divides p - 1 */
        if (!BN_div(t1, t2, dh->p, dh->q, ctx))
            goto err;
        if (!BN_is_one(t2))
            *ret |= DH_CHECK_INVALID_Q_VALUE;
        if (dh->j && BN_cmp(dh->j, t1))
            *ret |= DH_CHECK_INVALID_J_VALUE;

    } else if (BN_is_word(dh->g, DH_GENERATOR_2)) {
        l = BN_mod_word(dh->p, 24);
        if (l == (BN_ULONG)-1)
            goto err;
        if (l != 11)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
    } else if (BN_is_word(dh->g, DH_GENERATOR_5)) {
        l = BN_mod_word(dh->p, 10);
        if (l == (BN_ULONG)-1)
            goto err;
        if ((l != 3) && (l != 7))
            *ret |= DH_NOT_SUITABLE_GENERATOR;
    } else
        *ret |= DH_UNABLE_TO_CHECK_GENERATOR;

    r = BN_is_prime_ex(dh->p, BN_prime_checks, ctx, NULL);
    if (r < 0)
        goto err;
    if (!r)
        *ret |= DH_CHECK_P_NOT_PRIME;
    else if (!dh->q) {
        if (!BN_rshift1(t1, dh->p))
            goto err;
        r = BN_is_prime_ex(t1, BN_prime_checks, ctx, NULL);
        if (r < 0)
            goto err;
        if (!r)
            *ret |= DH_CHECK_P_NOT_SAFE_PRIME;
    }
    ok = 1;
 err:
    if (ctx != NULL) {
        BN_CTX_end(ctx);
        BN_CTX_free(ctx);
    }
    return (ok);
}
#else
int Cryptography_DH_check(const DH *dh, int *ret) {
    return DH_check(dh, ret);
}
#endif

/* These functions were added in OpenSSL 1.1.0f commit d0c50e80a8 */
/* Define our own to simplify support across all versions. */
#if defined(EVP_PKEY_DHX) && EVP_PKEY_DHX != -1
DH *Cryptography_d2i_DHxparams_bio(BIO *bp, DH **x) {
    return ASN1_d2i_bio_of(DH, DH_new, d2i_DHxparams, bp, x);
}
int Cryptography_i2d_DHxparams_bio(BIO *bp, DH *x) {
    return ASN1_i2d_bio_of_const(DH, i2d_DHxparams, bp, x);
}
#else
DH *(*Cryptography_d2i_DHxparams_bio)(BIO *bp, DH **x) = NULL;
int (*Cryptography_i2d_DHxparams_bio)(BIO *bp, DH *x) = NULL;
#endif


/* These functions were added in OpenSSL 1.1.0 */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
void DSA_get0_pqg(const DSA *d,
                  const BIGNUM **p, const BIGNUM **q, const BIGNUM **g)
{
    if (p != NULL)
        *p = d->p;
    if (q != NULL)
        *q = d->q;
    if (g != NULL)
        *g = d->g;
}
int DSA_set0_pqg(DSA *d, BIGNUM *p, BIGNUM *q, BIGNUM *g)
{
    /* If the fields p, q and g in d are NULL, the corresponding input
     * parameters MUST be non-NULL.
     */
    if ((d->p == NULL && p == NULL)
        || (d->q == NULL && q == NULL)
        || (d->g == NULL && g == NULL))
        return 0;

    if (p != NULL) {
        BN_free(d->p);
        d->p = p;
    }
    if (q != NULL) {
        BN_free(d->q);
        d->q = q;
    }
    if (g != NULL) {
        BN_free(d->g);
        d->g = g;
    }

    return 1;
}
void DSA_get0_key(const DSA *d,
                  const BIGNUM **pub_key, const BIGNUM **priv_key)
{
    if (pub_key != NULL)
        *pub_key = d->pub_key;
    if (priv_key != NULL)
        *priv_key = d->priv_key;
}
int DSA_set0_key(DSA *d, BIGNUM *pub_key, BIGNUM *priv_key)
{
    /* If the field pub_key in d is NULL, the corresponding input
     * parameters MUST be non-NULL.  The priv_key field may
     * be left NULL.
     */
    if (d->pub_key == NULL && pub_key == NULL)
        return 0;

    if (pub_key != NULL) {
        BN_free(d->pub_key);
        d->pub_key = pub_key;
    }
    if (priv_key != NULL) {
        BN_free(d->priv_key);
        d->priv_key = priv_key;
    }

    return 1;
}
#endif


#if defined(OPENSSL_NO_EC2M)
static const long Cryptography_HAS_EC2M = 0;

int (*EC_POINT_set_affine_coordinates_GF2m)(const EC_GROUP *, EC_POINT *,
    const BIGNUM *, const BIGNUM *, BN_CTX *) = NULL;

int (*EC_POINT_get_affine_coordinates_GF2m)(const EC_GROUP *,
    const EC_POINT *, BIGNUM *, BIGNUM *, BN_CTX *) = NULL;

int (*EC_POINT_set_compressed_coordinates_GF2m)(const EC_GROUP *, EC_POINT *,
    const BIGNUM *, int, BN_CTX *) = NULL;
#else
static const long Cryptography_HAS_EC2M = 1;
#endif

#if (!CRYPTOGRAPHY_IS_LIBRESSL && CRYPTOGRAPHY_OPENSSL_LESS_THAN_102)
static const long Cryptography_HAS_EC_1_0_2 = 0;
const char *(*EC_curve_nid2nist)(int) = NULL;
#else
static const long Cryptography_HAS_EC_1_0_2 = 1;
#endif


#ifndef SSL_CTX_set_ecdh_auto
static const long Cryptography_HAS_SET_ECDH_AUTO = 0;
long (*SSL_CTX_set_ecdh_auto)(SSL_CTX *, int) = NULL;
#else
static const long Cryptography_HAS_SET_ECDH_AUTO = 1;
#endif


static const long Cryptography_HAS_ECDSA = 1;


#ifdef OPENSSL_NO_ENGINE
static const long Cryptography_HAS_ENGINE = 0;

ENGINE *(*ENGINE_by_id)(const char *) = NULL;
int (*ENGINE_init)(ENGINE *) = NULL;
int (*ENGINE_finish)(ENGINE *) = NULL;
ENGINE *(*ENGINE_get_default_RAND)(void) = NULL;
int (*ENGINE_set_default_RAND)(ENGINE *) = NULL;
void (*ENGINE_unregister_RAND)(ENGINE *) = NULL;
int (*ENGINE_ctrl_cmd)(ENGINE *, const char *, long, void *,
                       void (*)(void), int) = NULL;

int (*ENGINE_free)(ENGINE *) = NULL;
const char *(*ENGINE_get_id)(const ENGINE *) = NULL;
const char *(*ENGINE_get_name)(const ENGINE *) = NULL;

#else
static const long Cryptography_HAS_ENGINE = 1;
#endif


static const long Cryptography_HAS_EC_CODES = 1;

#ifdef RSA_R_PKCS_DECODING_ERROR
static const long Cryptography_HAS_RSA_R_PKCS_DECODING_ERROR = 1;
#else
static const long Cryptography_HAS_RSA_R_PKCS_DECODING_ERROR = 0;
static const long RSA_R_PKCS_DECODING_ERROR = 0;
#endif

#ifdef EVP_R_MEMORY_LIMIT_EXCEEDED
static const long Cryptography_HAS_EVP_R_MEMORY_LIMIT_EXCEEDED = 1;
#else
static const long EVP_R_MEMORY_LIMIT_EXCEEDED = 0;
static const long Cryptography_HAS_EVP_R_MEMORY_LIMIT_EXCEEDED = 0;
#endif

#if CRYPTOGRAPHY_OPENSSL_300_OR_GREATER
#else
static const int ERR_LIB_CMP = -42;
static const int ERR_LIB_CRMF = -42;
static const int ERR_LIB_ESS = -42;
static const int ERR_LIB_HTTP = -42;
static const int ERR_LIB_MASK = -42;
static const int ERR_LIB_OFFSET = -42;
static const int ERR_LIB_OSSL_DECODER = -42;
static const int ERR_LIB_OSSL_ENCODER = -42;
static const int ERR_LIB_PROP = -42;
static const int ERR_LIB_PROV = -42;
#endif


#ifdef EVP_PKEY_DHX
const long Cryptography_HAS_EVP_PKEY_DHX = 1;
#else
const long Cryptography_HAS_EVP_PKEY_DHX = 0;
const long EVP_PKEY_DHX = -1;
#endif

EVP_MD_CTX *Cryptography_EVP_MD_CTX_new(void) {
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
    return EVP_MD_CTX_create();
#else
    return EVP_MD_CTX_new();
#endif
}
void Cryptography_EVP_MD_CTX_free(EVP_MD_CTX *ctx) {
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
    EVP_MD_CTX_destroy(ctx);
#else
    EVP_MD_CTX_free(ctx);
#endif
}
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 || defined(OPENSSL_NO_SCRYPT)
static const long Cryptography_HAS_SCRYPT = 0;
int (*EVP_PBE_scrypt)(const char *, size_t, const unsigned char *, size_t,
                      uint64_t, uint64_t, uint64_t, uint64_t, unsigned char *,
                      size_t) = NULL;
#else
static const long Cryptography_HAS_SCRYPT = 1;
#endif

#if CRYPTOGRAPHY_OPENSSL_110_OR_GREATER
static const long Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint = 1;
#else
static const long Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint = 0;
size_t (*EVP_PKEY_get1_tls_encodedpoint)(EVP_PKEY *, unsigned char **) = NULL;
int (*EVP_PKEY_set1_tls_encodedpoint)(EVP_PKEY *, const unsigned char *,
                                      size_t) = NULL;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111
static const long Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY = 0;
static const long Cryptography_HAS_RAW_KEY = 0;
static const long Cryptography_HAS_EVP_DIGESTFINAL_XOF = 0;
int (*EVP_DigestFinalXOF)(EVP_MD_CTX *, unsigned char *, size_t) = NULL;
int (*EVP_DigestSign)(EVP_MD_CTX *, unsigned char *, size_t *,
                      const unsigned char *tbs, size_t) = NULL;
int (*EVP_DigestVerify)(EVP_MD_CTX *, const unsigned char *, size_t,
                        const unsigned char *, size_t) = NULL;
EVP_PKEY *(*EVP_PKEY_new_raw_private_key)(int, ENGINE *, const unsigned char *,
                                       size_t) = NULL;
EVP_PKEY *(*EVP_PKEY_new_raw_public_key)(int, ENGINE *, const unsigned char *,
                                      size_t) = NULL;
int (*EVP_PKEY_get_raw_private_key)(const EVP_PKEY *, unsigned char *,
                                    size_t *) = NULL;
int (*EVP_PKEY_get_raw_public_key)(const EVP_PKEY *, unsigned char *,
                                   size_t *) = NULL;
#else
static const long Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY = 1;
static const long Cryptography_HAS_RAW_KEY = 1;
static const long Cryptography_HAS_EVP_DIGESTFINAL_XOF = 1;
#endif

/* OpenSSL 1.1.0+ does this define for us, but if not present we'll do it */
#if !defined(EVP_CTRL_AEAD_SET_IVLEN)
# define EVP_CTRL_AEAD_SET_IVLEN EVP_CTRL_GCM_SET_IVLEN
#endif
#if !defined(EVP_CTRL_AEAD_GET_TAG)
# define EVP_CTRL_AEAD_GET_TAG EVP_CTRL_GCM_GET_TAG
#endif
#if !defined(EVP_CTRL_AEAD_SET_TAG)
# define EVP_CTRL_AEAD_SET_TAG EVP_CTRL_GCM_SET_TAG
#endif

/* This is tied to X25519 support so we reuse the Cryptography_HAS_X25519
   conditional to remove it. OpenSSL 1.1.0 didn't have this define, but
   1.1.1 will when it is released. We can remove this in the distant
   future when we drop 1.1.0 support. */
#ifndef EVP_PKEY_X25519
#define EVP_PKEY_X25519 NID_X25519
#endif

/* This is tied to X448 support so we reuse the Cryptography_HAS_X448
   conditional to remove it. OpenSSL 1.1.1 adds this define.  We can remove
   this in the distant future when we drop 1.1.0 support. */
#ifndef EVP_PKEY_X448
#define EVP_PKEY_X448 NID_X448
#endif

/* This is tied to ED25519 support so we reuse the Cryptography_HAS_ED25519
   conditional to remove it. */
#ifndef EVP_PKEY_ED25519
#define EVP_PKEY_ED25519 NID_ED25519
#endif

/* This is tied to ED448 support so we reuse the Cryptography_HAS_ED448
   conditional to remove it. */
#ifndef EVP_PKEY_ED448
#define EVP_PKEY_ED448 NID_ED448
#endif

/* This is tied to poly1305 support so we reuse the Cryptography_HAS_POLY1305
   conditional to remove it. */
#ifndef EVP_PKEY_POLY1305
#define EVP_PKEY_POLY1305 NID_poly1305
#endif




HMAC_CTX *Cryptography_HMAC_CTX_new(void) {
#if CRYPTOGRAPHY_OPENSSL_110_OR_GREATER
    return HMAC_CTX_new();
#else
    /* This uses OPENSSL_zalloc in 1.1.0, which is malloc + memset */
    HMAC_CTX *ctx = (HMAC_CTX *)OPENSSL_malloc(sizeof(HMAC_CTX));
    memset(ctx, 0, sizeof(HMAC_CTX));
    return ctx;
#endif
}


void Cryptography_HMAC_CTX_free(HMAC_CTX *ctx) {
#if CRYPTOGRAPHY_OPENSSL_110_OR_GREATER
    return HMAC_CTX_free(ctx);
#else
    if (ctx != NULL) {
        HMAC_CTX_cleanup(ctx);
        OPENSSL_free(ctx);
    }
#endif
}


#ifndef NID_X25519
static const long Cryptography_HAS_X25519 = 0;
static const int NID_X25519 = 0;
#else
static const long Cryptography_HAS_X25519 = 1;
#endif
#ifndef NID_ED25519
static const long Cryptography_HAS_ED25519 = 0;
static const int NID_ED25519 = 0;
#else
static const long Cryptography_HAS_ED25519 = 1;
#endif
#ifndef NID_X448
static const long Cryptography_HAS_X448 = 0;
static const int NID_X448 = 0;
#else
static const long Cryptography_HAS_X448 = 1;
#endif
#ifndef NID_ED448
static const long Cryptography_HAS_ED448 = 0;
static const int NID_ED448 = 0;
#else
static const long Cryptography_HAS_ED448 = 1;
#endif
#ifndef NID_poly1305
static const long Cryptography_HAS_POLY1305 = 0;
static const int NID_poly1305 = 0;
#else
static const long Cryptography_HAS_POLY1305 = 1;
#endif




#if (     CRYPTOGRAPHY_OPENSSL_110_OR_GREATER &&     CRYPTOGRAPHY_OPENSSL_LESS_THAN_110J     )
/* These structs come from ocsp_lcl.h and are needed to de-opaque the struct
   for the getters in OpenSSL 1.1.0 through 1.1.0i */
struct ocsp_responder_id_st {
    int type;
    union {
        X509_NAME *byName;
        ASN1_OCTET_STRING *byKey;
    } value;
};
struct ocsp_response_data_st {
    ASN1_INTEGER *version;
    OCSP_RESPID responderId;
    ASN1_GENERALIZEDTIME *producedAt;
    STACK_OF(OCSP_SINGLERESP) *responses;
    STACK_OF(X509_EXTENSION) *responseExtensions;
};
struct ocsp_basic_response_st {
    OCSP_RESPDATA tbsResponseData;
    X509_ALGOR signatureAlgorithm;
    ASN1_BIT_STRING *signature;
    STACK_OF(X509) *certs;
};
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
/* These functions are all taken from ocsp_cl.c in OpenSSL 1.1.0 */
const OCSP_CERTID *OCSP_SINGLERESP_get0_id(const OCSP_SINGLERESP *single)
{
    return single->certId;
}
const Cryptography_STACK_OF_X509 *OCSP_resp_get0_certs(
    const OCSP_BASICRESP *bs)
{
    return bs->certs;
}
int OCSP_resp_get0_id(const OCSP_BASICRESP *bs,
                      const ASN1_OCTET_STRING **pid,
                      const X509_NAME **pname)
{
    const OCSP_RESPID *rid = bs->tbsResponseData->responderId;

    if (rid->type == V_OCSP_RESPID_NAME) {
        *pname = rid->value.byName;
        *pid = NULL;
    } else if (rid->type == V_OCSP_RESPID_KEY) {
        *pid = rid->value.byKey;
        *pname = NULL;
    } else {
        return 0;
    }
    return 1;
}
const ASN1_GENERALIZEDTIME *OCSP_resp_get0_produced_at(
    const OCSP_BASICRESP* bs)
{
    return bs->tbsResponseData->producedAt;
}
const ASN1_OCTET_STRING *OCSP_resp_get0_signature(const OCSP_BASICRESP *bs)
{
    return bs->signature;
}
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110J
const X509_ALGOR *OCSP_resp_get0_tbs_sigalg(const OCSP_BASICRESP *bs)
{
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
    return bs->signatureAlgorithm;
#else
    return &bs->signatureAlgorithm;
#endif
}

const OCSP_RESPDATA *OCSP_resp_get0_respdata(const OCSP_BASICRESP *bs)
{
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
    return bs->tbsResponseData;
#else
    return &bs->tbsResponseData;
#endif
}
#endif



/* osurandom engine
 *
 * Windows         CryptGenRandom()
 * macOS >= 10.12  getentropy()
 * OpenBSD 5.6+    getentropy()
 * other BSD       getentropy() if SYS_getentropy is defined
 * Linux 3.17+     getrandom() with fallback to /dev/urandom
 * other           /dev/urandom with cached fd
 *
 * The /dev/urandom, getrandom and getentropy code is derived from Python's
 * Python/random.c, written by Antoine Pitrou and Victor Stinner.
 *
 * Copyright 2001-2016 Python Software Foundation; All Rights Reserved.
 */

#ifdef __linux__
#include <poll.h>
#endif

#if CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE
/* OpenSSL has ENGINE support and is older than 1.1.1d (the first version that
 * properly implements fork safety in its RNG) so build the engine. */
static const char *Cryptography_osrandom_engine_id = "osrandom";

/****************************************************************************
 * Windows
 */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine CryptGenRandom()";
static HCRYPTPROV hCryptProv = 0;

static int osrandom_init(ENGINE *e) {
    if (hCryptProv != 0) {
        return 1;
    }
    if (CryptAcquireContext(&hCryptProv, NULL, NULL,
                            PROV_RSA_FULL, CRYPT_VERIFYCONTEXT)) {
        return 1;
    } else {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_INIT,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT,
            __FILE__, __LINE__
        );
        return 0;
    }
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    if (hCryptProv == 0) {
        return 0;
    }

    if (!CryptGenRandom(hCryptProv, (DWORD)size, buffer)) {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM,
            __FILE__, __LINE__
        );
        return 0;
    }
    return 1;
}

static int osrandom_finish(ENGINE *e) {
    if (CryptReleaseContext(hCryptProv, 0)) {
        hCryptProv = 0;
        return 1;
    } else {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_FINISH,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT,
            __FILE__, __LINE__
        );
        return 0;
    }
}

static int osrandom_rand_status(void) {
    return hCryptProv != 0;
}

static const char *osurandom_get_implementation(void) {
    return "CryptGenRandom";
}

#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM */

/****************************************************************************
 * /dev/urandom helpers for all non-BSD Unix platforms
 */
#ifdef CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM

static struct {
    int fd;
    dev_t st_dev;
    ino_t st_ino;
} urandom_cache = { -1 };

static int open_cloexec(const char *path) {
    int open_flags = O_RDONLY;
#ifdef O_CLOEXEC
    open_flags |= O_CLOEXEC;
#endif

    int fd = open(path, open_flags);
    if (fd == -1) {
        return -1;
    }

#ifndef O_CLOEXEC
    int flags = fcntl(fd, F_GETFD);
    if (flags == -1) {
        return -1;
    }
    if (fcntl(fd, F_SETFD, flags | FD_CLOEXEC) == -1) {
        return -1;
    }
#endif
    return fd;
}

#ifdef __linux__
/* On Linux, we open("/dev/random") and use poll() to wait until it's readable
 * before we read from /dev/urandom, this ensures that we don't read from
 * /dev/urandom before the kernel CSPRNG is initialized. This isn't necessary on
 * other platforms because they don't have the same _bug_ as Linux does with
 * /dev/urandom and early boot. */
static int wait_on_devrandom(void) {
    struct pollfd pfd = {};
    int ret = 0;
    int random_fd = open_cloexec("/dev/random");
    if (random_fd < 0) {
        return -1;
    }
    pfd.fd = random_fd;
    pfd.events = POLLIN;
    pfd.revents = 0;
    do {
        ret = poll(&pfd, 1, -1);
    } while (ret < 0 && (errno == EINTR || errno == EAGAIN));
    close(random_fd);
    return ret;
}
#endif

/* return -1 on error */
static int dev_urandom_fd(void) {
    int fd = -1;
    struct stat st;

    /* Check that fd still points to the correct device */
    if (urandom_cache.fd >= 0) {
        if (fstat(urandom_cache.fd, &st)
                || st.st_dev != urandom_cache.st_dev
                || st.st_ino != urandom_cache.st_ino) {
            /* Somebody replaced our FD. Invalidate our cache but don't
             * close the fd. */
            urandom_cache.fd = -1;
        }
    }
    if (urandom_cache.fd < 0) {
#ifdef __linux__
        if (wait_on_devrandom() < 0) {
            goto error;
        }
#endif

        fd = open_cloexec("/dev/urandom");
        if (fd < 0) {
            goto error;
        }
        if (fstat(fd, &st)) {
            goto error;
        }
        /* Another thread initialized the fd */
        if (urandom_cache.fd >= 0) {
            close(fd);
            return urandom_cache.fd;
        }
        urandom_cache.st_dev = st.st_dev;
        urandom_cache.st_ino = st.st_ino;
        urandom_cache.fd = fd;
    }
    return urandom_cache.fd;

  error:
    if (fd != -1) {
        close(fd);
    }
    ERR_Cryptography_OSRandom_error(
        CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD,
        CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED,
        __FILE__, __LINE__
    );
    return -1;
}

static int dev_urandom_read(unsigned char *buffer, int size) {
    int fd;
    int n;

    fd = dev_urandom_fd();
    if (fd < 0) {
        return 0;
    }

    while (size > 0) {
        do {
            n = (int)read(fd, buffer, (size_t)size);
        } while (n < 0 && errno == EINTR);

        if (n <= 0) {
            ERR_Cryptography_OSRandom_error(
                CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ,
                CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED,
                __FILE__, __LINE__
            );
            return 0;
        }
        buffer += n;
        size -= n;
    }
    return 1;
}

static void dev_urandom_close(void) {
    if (urandom_cache.fd >= 0) {
        int fd;
        struct stat st;

        if (fstat(urandom_cache.fd, &st)
                && st.st_dev == urandom_cache.st_dev
                && st.st_ino == urandom_cache.st_ino) {
            fd = urandom_cache.fd;
            urandom_cache.fd = -1;
            close(fd);
        }
    }
}
#endif /* CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM */

/****************************************************************************
 * BSD getentropy
 */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY
static const char *Cryptography_osrandom_engine_name = "osrandom_engine getentropy()";

static int getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_NOT_INIT;

static int osrandom_init(ENGINE *e) {
#if !defined(__APPLE__)
    getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS;
#else
    if (__builtin_available(macOS 10.12, *)) {
        getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS;
    } else {
        getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK;
        int fd = dev_urandom_fd();
        if (fd < 0) {
            return 0;
        }
    }
#endif
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    int len;
    int res;

    switch(getentropy_works) {
#if defined(__APPLE__)
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK:
        return dev_urandom_read(buffer, size);
#endif
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS:
        while (size > 0) {
            /* OpenBSD and macOS restrict maximum buffer size to 256. */
            len = size > 256 ? 256 : size;
/* on mac, availability is already checked using `__builtin_available` above */
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunguarded-availability"
            res = getentropy(buffer, (size_t)len);
#pragma clang diagnostic pop
            if (res < 0) {
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
                    CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED,
                    __FILE__, __LINE__
                );
                return 0;
            }
            buffer += len;
            size -= len;
        }
        return 1;
    }
    __builtin_unreachable();
}

static int osrandom_finish(ENGINE *e) {
    return 1;
}

static int osrandom_rand_status(void) {
    return 1;
}

static const char *osurandom_get_implementation(void) {
    switch(getentropy_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK:
        return "/dev/urandom";
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS:
        return "getentropy";
    }
    __builtin_unreachable();
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY */

/****************************************************************************
 * Linux getrandom engine with fallback to dev_urandom
 */

#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine getrandom()";

static int getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT;

static int osrandom_init(ENGINE *e) {
    /* We try to detect working getrandom until we succeed. */
    if (getrandom_works != CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS) {
        long n;
        char dest[1];
        /* if the kernel CSPRNG is not initialized this will block */
        n = syscall(SYS_getrandom, dest, sizeof(dest), 0);
        if (n == sizeof(dest)) {
            getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS;
        } else {
            int e = errno;
            switch(e) {
            case ENOSYS:
                /* Fallback: Kernel does not support the syscall. */
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK;
                break;
            case EPERM:
                /* Fallback: seccomp prevents syscall */
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK;
                break;
            default:
                /* EINTR cannot occur for buflen < 256. */
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_INIT,
                    CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED,
                    "errno", e
                );
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED;
                break;
            }
        }
    }

    /* fallback to dev urandom */
    if (getrandom_works == CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK) {
        int fd = dev_urandom_fd();
        if (fd < 0) {
            return 0;
        }
    }
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    long n;

    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED,
            __FILE__, __LINE__
        );
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT,
            __FILE__, __LINE__
        );
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return dev_urandom_read(buffer, size);
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        while (size > 0) {
            do {
                n = syscall(SYS_getrandom, buffer, size, 0);
            } while (n < 0 && errno == EINTR);

            if (n <= 0) {
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
                    CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED,
                    __FILE__, __LINE__
                );
                return 0;
            }
            buffer += n;
            size -= (int)n;
        }
        return 1;
    }
    __builtin_unreachable();
}

static int osrandom_finish(ENGINE *e) {
    dev_urandom_close();
    return 1;
}

static int osrandom_rand_status(void) {
    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return urandom_cache.fd >= 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        return 1;
    }
    __builtin_unreachable();
}

static const char *osurandom_get_implementation(void) {
    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        return "<failed>";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        return "<not initialized>";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return "/dev/urandom";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        return "getrandom";
    }
    __builtin_unreachable();
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM */

/****************************************************************************
 * dev_urandom engine for all remaining platforms
 */

#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine /dev/urandom";

static int osrandom_init(ENGINE *e) {
    int fd = dev_urandom_fd();
    if (fd < 0) {
        return 0;
    }
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    return dev_urandom_read(buffer, size);
}

static int osrandom_finish(ENGINE *e) {
    dev_urandom_close();
    return 1;
}

static int osrandom_rand_status(void) {
    return urandom_cache.fd >= 0;
}

static const char *osurandom_get_implementation(void) {
    return "/dev/urandom";
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM */

/****************************************************************************
 * ENGINE boiler plate
 */

/* This replicates the behavior of the OpenSSL FIPS RNG, which returns a
   -1 in the event that there is an error when calling RAND_pseudo_bytes. */
static int osrandom_pseudo_rand_bytes(unsigned char *buffer, int size) {
    int res = osrandom_rand_bytes(buffer, size);
    if (res == 0) {
        return -1;
    } else {
        return res;
    }
}

static RAND_METHOD osrandom_rand = {
    NULL,
    osrandom_rand_bytes,
    NULL,
    NULL,
    osrandom_pseudo_rand_bytes,
    osrandom_rand_status,
};

static const ENGINE_CMD_DEFN osrandom_cmd_defns[] = {
    {CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION,
     "get_implementation",
     "Get CPRNG implementation.",
     ENGINE_CMD_FLAG_NO_INPUT},
     {0, NULL, NULL, 0}
};

static int osrandom_ctrl(ENGINE *e, int cmd, long i, void *p, void (*f) (void)) {
    const char *name;
    size_t len;

    switch (cmd) {
    case CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION:
        /* i: buffer size, p: char* buffer */
        name = osurandom_get_implementation();
        len = strlen(name);
        if ((p == NULL) && (i == 0)) {
            /* return required buffer len */
            return (int)len;
        }
        if ((p == NULL) || i < 0 || ((size_t)i <= len)) {
            /* no buffer or buffer too small */
            ENGINEerr(ENGINE_F_ENGINE_CTRL, ENGINE_R_INVALID_ARGUMENT);
            return 0;
        }
        strcpy((char *)p, name);
        return (int)len;
    default:
        ENGINEerr(ENGINE_F_ENGINE_CTRL, ENGINE_R_CTRL_COMMAND_NOT_IMPLEMENTED);
        return 0;
    }
}

/* error reporting */
#define ERR_FUNC(func) ERR_PACK(0, func, 0)
#define ERR_REASON(reason) ERR_PACK(0, 0, reason)

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_lib_name[] = {
    {0, "osrandom_engine"},
    {0, NULL}
};

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_str_funcs[] = {
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_INIT),
     "osrandom_init"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES),
     "osrandom_rand_bytes"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_FINISH),
     "osrandom_finish"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD),
     "dev_urandom_fd"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ),
     "dev_urandom_read"},
    {0, NULL}
};

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_str_reasons[] = {
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT),
     "CryptAcquireContext() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM),
     "CryptGenRandom() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT),
     "CryptReleaseContext() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED),
     "getentropy() failed"},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED),
     "open('/dev/urandom') failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED),
     "Reading from /dev/urandom fd failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED),
     "getrandom() initialization failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED),
     "getrandom() initialization failed with unexpected errno."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED),
     "getrandom() syscall failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT),
     "getrandom() engine was not properly initialized."},
    {0, NULL}
};

static int Cryptography_OSRandom_lib_error_code = 0;

static void ERR_load_Cryptography_OSRandom_strings(void)
{
    if (Cryptography_OSRandom_lib_error_code == 0) {
        Cryptography_OSRandom_lib_error_code = ERR_get_next_error_library();
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_lib_name);
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_str_funcs);
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_str_reasons);
    }
}

static void ERR_Cryptography_OSRandom_error(int function, int reason,
                                            char *file, int line)
{
    ERR_PUT_error(Cryptography_OSRandom_lib_error_code, function, reason,
                  file, line);
}

/* Returns 1 if successfully added, 2 if engine has previously been added,
   and 0 for error. */
int Cryptography_add_osrandom_engine(void) {
    ENGINE *e;

    ERR_load_Cryptography_OSRandom_strings();

    e = ENGINE_by_id(Cryptography_osrandom_engine_id);
    if (e != NULL) {
        ENGINE_free(e);
        return 2;
    } else {
        ERR_clear_error();
    }

    e = ENGINE_new();
    if (e == NULL) {
        return 0;
    }
    if (!ENGINE_set_id(e, Cryptography_osrandom_engine_id) ||
            !ENGINE_set_name(e, Cryptography_osrandom_engine_name) ||
            !ENGINE_set_RAND(e, &osrandom_rand) ||
            !ENGINE_set_init_function(e, osrandom_init) ||
            !ENGINE_set_finish_function(e, osrandom_finish) ||
            !ENGINE_set_cmd_defns(e, osrandom_cmd_defns) ||
            !ENGINE_set_ctrl_function(e, osrandom_ctrl)) {
        ENGINE_free(e);
        return 0;
    }
    if (!ENGINE_add(e)) {
        ENGINE_free(e);
        return 0;
    }
    if (!ENGINE_free(e)) {
        return 0;
    }

    return 1;
}

#else
/* If OpenSSL has no ENGINE support then we don't want
 * to compile the osrandom engine, but we do need some
 * placeholders */
static const char *Cryptography_osrandom_engine_id = "no-engine-support";
static const char *Cryptography_osrandom_engine_name = "osrandom_engine disabled";

int Cryptography_add_osrandom_engine(void) {
    return 0;
}

#endif


#if !defined(EVP_PKEY_DHX) || EVP_PKEY_DHX == -1
int (*PEM_write_bio_DHxparams)(BIO *, DH *) = NULL;
#endif




static const long Cryptography_HAS_EGD = 0;


static const long Cryptography_HAS_PSS_PADDING = 1;

#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_RSA_OAEP_MD = 1;
static const long Cryptography_HAS_RSA_OAEP_LABEL = 1;
#else
static const long Cryptography_HAS_RSA_OAEP_MD = 0;
static const long Cryptography_HAS_RSA_OAEP_LABEL = 0;
int (*EVP_PKEY_CTX_set_rsa_oaep_md)(EVP_PKEY_CTX *, EVP_MD *) = NULL;
int (*EVP_PKEY_CTX_set0_rsa_oaep_label)(EVP_PKEY_CTX *, unsigned char *,
                                        int) = NULL;
#endif

/* These functions were added in OpenSSL 1.1.0 */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
int RSA_set0_key(RSA *r, BIGNUM *n, BIGNUM *e, BIGNUM *d)
{
    /* If the fields n and e in r are NULL, the corresponding input
     * parameters MUST be non-NULL for n and e.  d may be
     * left NULL (in case only the public key is used).
     */
    if ((r->n == NULL && n == NULL)
        || (r->e == NULL && e == NULL))
        return 0;

    if (n != NULL) {
        BN_free(r->n);
        r->n = n;
    }
    if (e != NULL) {
        BN_free(r->e);
        r->e = e;
    }
    if (d != NULL) {
        BN_free(r->d);
        r->d = d;
    }

    return 1;
}

int RSA_set0_factors(RSA *r, BIGNUM *p, BIGNUM *q)
{
    /* If the fields p and q in r are NULL, the corresponding input
     * parameters MUST be non-NULL.
     */
    if ((r->p == NULL && p == NULL)
        || (r->q == NULL && q == NULL))
        return 0;

    if (p != NULL) {
        BN_free(r->p);
        r->p = p;
    }
    if (q != NULL) {
        BN_free(r->q);
        r->q = q;
    }

    return 1;
}

int RSA_set0_crt_params(RSA *r, BIGNUM *dmp1, BIGNUM *dmq1, BIGNUM *iqmp)
{
    /* If the fields dmp1, dmq1 and iqmp in r are NULL, the corresponding input
     * parameters MUST be non-NULL.
     */
    if ((r->dmp1 == NULL && dmp1 == NULL)
        || (r->dmq1 == NULL && dmq1 == NULL)
        || (r->iqmp == NULL && iqmp == NULL))
        return 0;

    if (dmp1 != NULL) {
        BN_free(r->dmp1);
        r->dmp1 = dmp1;
    }
    if (dmq1 != NULL) {
        BN_free(r->dmq1);
        r->dmq1 = dmq1;
    }
    if (iqmp != NULL) {
        BN_free(r->iqmp);
        r->iqmp = iqmp;
    }

    return 1;
}

void RSA_get0_key(const RSA *r,
                  const BIGNUM **n, const BIGNUM **e, const BIGNUM **d)
{
    if (n != NULL)
        *n = r->n;
    if (e != NULL)
        *e = r->e;
    if (d != NULL)
        *d = r->d;
}

void RSA_get0_factors(const RSA *r, const BIGNUM **p, const BIGNUM **q)
{
    if (p != NULL)
        *p = r->p;
    if (q != NULL)
        *q = r->q;
}

void RSA_get0_crt_params(const RSA *r,
                         const BIGNUM **dmp1, const BIGNUM **dmq1,
                         const BIGNUM **iqmp)
{
    if (dmp1 != NULL)
        *dmp1 = r->dmp1;
    if (dmq1 != NULL)
        *dmq1 = r->dmq1;
    if (iqmp != NULL)
        *iqmp = r->iqmp;
}
#endif


#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102
#error Python 3.7 requires OpenSSL >= 1.0.2
#endif

/* Added in 1.0.2 but we need it in all versions now due to the great
   opaquing. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102
/* from ssl/ssl_lib.c */
const SSL_METHOD *SSL_CTX_get_ssl_method(SSL_CTX *ctx) {
    return ctx->method;
}
#endif

/* Added in 1.1.0 in the great opaquing, but we need to define it for older
   OpenSSLs. Such is our burden. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
/* from ssl/ssl_lib.c */
size_t SSL_get_client_random(const SSL *ssl, unsigned char *out, size_t outlen)
{
    if (outlen == 0)
        return sizeof(ssl->s3->client_random);
    if (outlen > sizeof(ssl->s3->client_random))
        outlen = sizeof(ssl->s3->client_random);
    memcpy(out, ssl->s3->client_random, outlen);
    return outlen;
}
/* Added in 1.1.0 as well */
/* from ssl/ssl_lib.c */
size_t SSL_get_server_random(const SSL *ssl, unsigned char *out, size_t outlen)
{
    if (outlen == 0)
        return sizeof(ssl->s3->server_random);
    if (outlen > sizeof(ssl->s3->server_random))
        outlen = sizeof(ssl->s3->server_random);
    memcpy(out, ssl->s3->server_random, outlen);
    return outlen;
}
/* Added in 1.1.0 as well */
/* from ssl/ssl_lib.c */
size_t SSL_SESSION_get_master_key(const SSL_SESSION *session,
                               unsigned char *out, size_t outlen)
{
    if (session->master_key_length < 0) {
        /* Should never happen */
        return 0;
    }
    if (outlen == 0)
        return session->master_key_length;
    if (outlen > (size_t)session->master_key_length)
        outlen = session->master_key_length;
    memcpy(out, session->master_key, outlen);
    return outlen;
}
/* from ssl/ssl_sess.c */
int SSL_SESSION_has_ticket(const SSL_SESSION *s)
{
    return (s->tlsext_ticklen > 0) ? 1 : 0;
}
/* from ssl/ssl_sess.c */
unsigned long SSL_SESSION_get_ticket_lifetime_hint(const SSL_SESSION *s)
{
    return s->tlsext_tick_lifetime_hint;
}
#endif

static const long Cryptography_HAS_SECURE_RENEGOTIATION = 1;

/* Cryptography now compiles out all SSLv2 bindings. This exists to allow
 * clients that use it to check for SSLv2 support to keep functioning as
 * expected.
 */
static const long Cryptography_HAS_SSL2 = 0;

#ifdef OPENSSL_NO_SSL3_METHOD
static const long Cryptography_HAS_SSL3_METHOD = 0;
const SSL_METHOD* (*SSLv3_method)(void) = NULL;
const SSL_METHOD* (*SSLv3_client_method)(void) = NULL;
const SSL_METHOD* (*SSLv3_server_method)(void) = NULL;
#else
static const long Cryptography_HAS_SSL3_METHOD = 1;
#endif

static const long Cryptography_HAS_TLSEXT_HOSTNAME = 1;
static const long Cryptography_HAS_TLSEXT_STATUS_REQ_CB = 1;
static const long Cryptography_HAS_STATUS_REQ_OCSP_RESP = 1;
static const long Cryptography_HAS_TLSEXT_STATUS_REQ_TYPE = 1;
static const long Cryptography_HAS_RELEASE_BUFFERS = 1;
static const long Cryptography_HAS_OP_NO_COMPRESSION = 1;
static const long Cryptography_HAS_TLSv1_1 = 1;
static const long Cryptography_HAS_TLSv1_2 = 1;
static const long Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING = 1;
static const long Cryptography_HAS_SSL_OP_NO_TICKET = 1;
static const long Cryptography_HAS_SSL_SET_SSL_CTX = 1;
static const long Cryptography_HAS_NEXTPROTONEG = 1;

/* SSL_get0_param was added in OpenSSL 1.0.2. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
X509_VERIFY_PARAM *(*SSL_get0_param)(SSL *) = NULL;
X509_VERIFY_PARAM *(*SSL_CTX_get0_param)(SSL_CTX *ctx) = NULL;
#else
#endif

/* ALPN was added in OpenSSL 1.0.2. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102 && !CRYPTOGRAPHY_IS_LIBRESSL
int (*SSL_CTX_set_alpn_protos)(SSL_CTX *,
                               const unsigned char *,
                               unsigned) = NULL;
int (*SSL_set_alpn_protos)(SSL *, const unsigned char *, unsigned) = NULL;
void (*SSL_CTX_set_alpn_select_cb)(SSL_CTX *,
                                   int (*) (SSL *,
                                            const unsigned char **,
                                            unsigned char *,
                                            const unsigned char *,
                                            unsigned int,
                                            void *),
                                   void *) = NULL;
void (*SSL_get0_alpn_selected)(const SSL *,
                               const unsigned char **,
                               unsigned *) = NULL;
static const long Cryptography_HAS_ALPN = 0;
#else
static const long Cryptography_HAS_ALPN = 1;
#endif

/* SSL_CTX_set_cert_cb was added in OpenSSL 1.0.2. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102
void (*SSL_CTX_set_cert_cb)(SSL_CTX *, int (*)(SSL *, void *), void *) = NULL;
void (*SSL_set_cert_cb)(SSL *, int (*)(SSL *, void *), void *) = NULL;
static const long Cryptography_HAS_SET_CERT_CB = 0;
#else
static const long Cryptography_HAS_SET_CERT_CB = 1;
#endif


/* In OpenSSL 1.0.2i+ the handling of COMP_METHOD when OPENSSL_NO_COMP was
   changed and we no longer need to typedef void */
#if (defined(OPENSSL_NO_COMP) && CRYPTOGRAPHY_OPENSSL_LESS_THAN_102I) ||     CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_COMPRESSION = 0;
typedef void COMP_METHOD;
#else
static const long Cryptography_HAS_COMPRESSION = 1;
#endif

#if defined(SSL_CTRL_GET_SERVER_TMP_KEY)
static const long Cryptography_HAS_GET_SERVER_TMP_KEY = 1;
#else
static const long Cryptography_HAS_GET_SERVER_TMP_KEY = 0;
long (*SSL_get_server_tmp_key)(SSL *, EVP_PKEY **) = NULL;
#endif

/* The setter functions were added in OpenSSL 1.1.0. The getter functions were
   added in OpenSSL 1.1.1. */
#if defined(SSL_CTRL_GET_MAX_PROTO_VERSION)
static const long Cryptography_HAS_CTRL_GET_MAX_PROTO_VERSION = 1;
#else
static const long Cryptography_HAS_CTRL_GET_MAX_PROTO_VERSION = 0;
int (*SSL_CTX_get_min_proto_version)(SSL_CTX *ctx) = NULL;
int (*SSL_CTX_get_max_proto_version)(SSL_CTX *ctx) = NULL;
int (*SSL_get_min_proto_version)(SSL *ssl) = NULL;
int (*SSL_get_max_proto_version)(SSL *ssl) = NULL;
#endif
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
int (*SSL_CTX_set_min_proto_version)(SSL_CTX *ctx, int version) = NULL;
int (*SSL_CTX_set_max_proto_version)(SSL_CTX *ctx, int version) = NULL;
int (*SSL_set_min_proto_version)(SSL *ssl, int version) = NULL;
int (*SSL_set_max_proto_version)(SSL *ssl, int version) = NULL;
#endif

static const long Cryptography_HAS_SSL_CTX_SET_CLIENT_CERT_ENGINE = 1;

static const long Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS = 1;

/* in OpenSSL 1.1.0 the SSL_ST values were renamed to TLS_ST and several were
   removed */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
static const long Cryptography_HAS_SSL_ST = 1;
#else
static const long Cryptography_HAS_SSL_ST = 0;
static const long SSL_ST_BEFORE = 0;
static const long SSL_ST_OK = 0;
static const long SSL_ST_INIT = 0;
static const long SSL_ST_RENEGOTIATE = 0;
#endif
#if CRYPTOGRAPHY_OPENSSL_110_OR_GREATER
static const long Cryptography_HAS_TLS_ST = 1;
#else
static const long Cryptography_HAS_TLS_ST = 0;
static const long TLS_ST_BEFORE = 0;
static const long TLS_ST_OK = 0;
#endif

/* SSLv23_method(), SSLv23_server_method() and SSLv23_client_method() were
   deprecated and the preferred TLS_method(), TLS_server_method() and
   TLS_client_method() functions were introduced in OpenSSL 1.1.0. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
#define TLS_method SSLv23_method
#define TLS_server_method SSLv23_server_method
#define TLS_client_method SSLv23_client_method
#endif

/* LibreSSL 2.9.1 added only the DTLS_*_method functions */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102 && !CRYPTOGRAPHY_LIBRESSL_291_OR_GREATER
static const long Cryptography_HAS_GENERIC_DTLS_METHOD = 0;
const SSL_METHOD *(*DTLS_method)(void) = NULL;
const SSL_METHOD *(*DTLS_server_method)(void) = NULL;
const SSL_METHOD *(*DTLS_client_method)(void) = NULL;
#else
static const long Cryptography_HAS_GENERIC_DTLS_METHOD = 1;
#endif
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102
static const long SSL_OP_NO_DTLSv1 = 0;
static const long SSL_OP_NO_DTLSv1_2 = 0;
long (*DTLS_set_link_mtu)(SSL *, long) = NULL;
long (*DTLS_get_link_min_mtu)(SSL *) = NULL;
#endif

static const long Cryptography_HAS_DTLS = 1;
/* Wrap DTLSv1_get_timeout to avoid cffi to handle a 'struct timeval'. */
long Cryptography_DTLSv1_get_timeout(SSL *ssl, time_t *ptv_sec,
                                     long *ptv_usec) {
    struct timeval tv = { 0 };
    long r = DTLSv1_get_timeout(ssl, &tv);

    if (r == 1) {
        if (ptv_sec) {
            *ptv_sec = tv.tv_sec;
        }

        if (ptv_usec) {
            *ptv_usec = tv.tv_usec;
        }
    }

    return r;
}

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102
static const long Cryptography_HAS_SIGALGS = 0;
const int (*SSL_get_sigalgs)(SSL *, int, int *, int *, int *, unsigned char *,
                             unsigned char *) = NULL;
const long (*SSL_CTX_set1_sigalgs_list)(SSL_CTX *, const char *) = NULL;
#else
static const long Cryptography_HAS_SIGALGS = 1;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL || defined(OPENSSL_NO_PSK)
static const long Cryptography_HAS_PSK = 0;
int (*SSL_CTX_use_psk_identity_hint)(SSL_CTX *, const char *) = NULL;
void (*SSL_CTX_set_psk_server_callback)(SSL_CTX *,
                                        unsigned int (*)(
                                            SSL *,
                                            const char *,
                                            unsigned char *,
                                            unsigned int
                                        )) = NULL;
void (*SSL_CTX_set_psk_client_callback)(SSL_CTX *,
                                        unsigned int (*)(
                                            SSL *,
                                            const char *,
                                            char *,
                                            unsigned int,
                                            unsigned char *,
                                            unsigned int
                                        )) = NULL;
#else
static const long Cryptography_HAS_PSK = 1;
#endif

/*
 * Custom extensions were added in 1.0.2. 1.1.1 is adding a more general
 * SSL_CTX_add_custom_ext function, but we're not binding that yet.
 */
#if CRYPTOGRAPHY_OPENSSL_102_OR_GREATER
static const long Cryptography_HAS_CUSTOM_EXT = 1;
#else
static const long Cryptography_HAS_CUSTOM_EXT = 0;

typedef int (*custom_ext_add_cb)(SSL *, unsigned int,
                                 const unsigned char **,
                                 size_t *, int *,
                                 void *);

typedef void (*custom_ext_free_cb)(SSL *, unsigned int,
                                   const unsigned char *,
                                   void *);

typedef int (*custom_ext_parse_cb)(SSL *, unsigned int,
                                   const unsigned char *,
                                   size_t, int *,
                                   void *);

int (*SSL_CTX_add_client_custom_ext)(SSL_CTX *, unsigned int,
                                     custom_ext_add_cb,
                                     custom_ext_free_cb, void *,
                                     custom_ext_parse_cb,
                                     void *) = NULL;

int (*SSL_CTX_add_server_custom_ext)(SSL_CTX *, unsigned int,
                                     custom_ext_add_cb,
                                     custom_ext_free_cb, void *,
                                     custom_ext_parse_cb,
                                     void *) = NULL;

int (*SSL_extension_supported)(unsigned int) = NULL;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
int (*SSL_CIPHER_is_aead)(const SSL_CIPHER *) = NULL;
int (*SSL_CIPHER_get_cipher_nid)(const SSL_CIPHER *) = NULL;
int (*SSL_CIPHER_get_digest_nid)(const SSL_CIPHER *) = NULL;
int (*SSL_CIPHER_get_kx_nid)(const SSL_CIPHER *) = NULL;
int (*SSL_CIPHER_get_auth_nid)(const SSL_CIPHER *) = NULL;
static const long Cryptography_HAS_CIPHER_DETAILS = 0;
#else
static const long Cryptography_HAS_CIPHER_DETAILS = 1;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111
static const long Cryptography_HAS_TLSv1_3 = 0;
static const long SSL_OP_NO_TLSv1_3 = 0;
static const long SSL_VERIFY_POST_HANDSHAKE = 0;
int (*SSL_CTX_set_ciphersuites)(SSL_CTX *, const char *) = NULL;
int (*SSL_verify_client_post_handshake)(SSL *) = NULL;
void (*SSL_CTX_set_post_handshake_auth)(SSL_CTX *, int) = NULL;
void (*SSL_set_post_handshake_auth)(SSL *, int) = NULL;
uint32_t (*SSL_SESSION_get_max_early_data)(const SSL_SESSION *) = NULL;
int (*SSL_write_early_data)(SSL *, const void *, size_t, size_t *) = NULL;
int (*SSL_read_early_data)(SSL *, void *, size_t, size_t *) = NULL;
int (*SSL_CTX_set_max_early_data)(SSL_CTX *, uint32_t) = NULL;
#else
static const long Cryptography_HAS_TLSv1_3 = 1;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_300
static const long SSL_OP_IGNORE_UNEXPECTED_EOF = 0;
static const long Crytpography_HAS_OP_IGNORE_UNEXPECTED_EOF = 0;
#else
static const long Crytpography_HAS_OP_IGNORE_UNEXPECTED_EOF = 1;
#endif
#ifdef X509_CHECK_FLAG_NEVER_CHECK_SUBJECT
static const long Cryptography_HAS_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT = 1;
#else
static const long Cryptography_HAS_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT = 0;
#endif


/* Added in 1.0.2 beta but we need it in all versions now due to the great
   opaquing. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
/* from x509/x_x509.c version 1.0.2 */
void X509_get0_signature(const ASN1_BIT_STRING **psig,
                         const X509_ALGOR **palg, const X509 *x)
{
    if (psig)
        *psig = x->signature;
    if (palg)
        *palg = x->sig_alg;
}

int X509_get_signature_nid(const X509 *x)
{
    return OBJ_obj2nid(x->sig_alg->algorithm);
}

#endif

/* Added in 1.0.2 but we need it in all versions now due to the great
   opaquing. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102
/* from x509/x_x509.c */
int i2d_re_X509_tbs(X509 *x, unsigned char **pp)
{
    /* in 1.0.2+ this function also sets x->cert_info->enc.modified = 1
       but older OpenSSLs don't have the enc ASN1_ENCODING member in the
       X509 struct.  Setting modified to 1 marks the encoding
       (x->cert_info->enc.enc) as invalid, but since the entire struct isn't
       present we don't care. */
    return i2d_X509_CINF(x->cert_info, pp);
}
#endif

/* X509_REVOKED_dup only exists on 1.0.2+. It is implemented using
   IMPLEMENT_ASN1_DUP_FUNCTION. The below is the equivalent so we have
   it available on all OpenSSLs. */
X509_REVOKED *Cryptography_X509_REVOKED_dup(X509_REVOKED *rev) {
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_102
    return ASN1_item_dup(ASN1_ITEM_rptr(X509_REVOKED), rev);
#else
    return X509_REVOKED_dup(rev);
#endif
}

/* Added in 1.1.0 but we need it in all versions now due to the great
   opaquing. */
#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
int i2d_re_X509_REQ_tbs(X509_REQ *req, unsigned char **pp)
{
    req->req_info->enc.modified = 1;
    return i2d_X509_REQ_INFO(req->req_info, pp);
}
int i2d_re_X509_CRL_tbs(X509_CRL *crl, unsigned char **pp) {
    crl->crl->enc.modified = 1;
    return i2d_X509_CRL_INFO(crl->crl, pp);
}

#if !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
int X509_up_ref(X509 *x) {
   return CRYPTO_add(&x->references, 1, CRYPTO_LOCK_X509);
}

const X509_ALGOR *X509_get0_tbs_sigalg(const X509 *x)
{
    return x->cert_info->signature;
}

/* from x509/x509_req.c */
void X509_REQ_get0_signature(const X509_REQ *req, const ASN1_BIT_STRING **psig,
                             const X509_ALGOR **palg)
{
    if (psig != NULL)
        *psig = req->signature;
    if (palg != NULL)
        *palg = req->sig_alg;
}
void X509_CRL_get0_signature(const X509_CRL *crl, const ASN1_BIT_STRING **psig,
                             const X509_ALGOR **palg)
{
    if (psig != NULL)
        *psig = crl->signature;
    if (palg != NULL)
        *palg = crl->sig_alg;
}
const ASN1_TIME *X509_REVOKED_get0_revocationDate(const X509_REVOKED *x)
{
    return x->revocationDate;
}
const ASN1_INTEGER *X509_REVOKED_get0_serialNumber(const X509_REVOKED *x)
{
    return x->serialNumber;
}
#endif
#endif


#if CRYPTOGRAPHY_OPENSSL_110_OR_GREATER
int Cryptography_X509_NAME_ENTRY_set(X509_NAME_ENTRY *ne) {
    return X509_NAME_ENTRY_set(ne);
}
#else
int Cryptography_X509_NAME_ENTRY_set(X509_NAME_ENTRY *ne) {
    return ne->set;
}
#endif




/* OpenSSL 1.0.2+ verification parameters and error codes */
#if CRYPTOGRAPHY_OPENSSL_102_OR_GREATER
static const long Cryptography_HAS_102_VERIFICATION_ERROR_CODES = 1;
static const long Cryptography_HAS_102_VERIFICATION_PARAMS = 1;
#else
static const long Cryptography_HAS_102_VERIFICATION_ERROR_CODES = 0;
static const long Cryptography_HAS_102_VERIFICATION_PARAMS = 0;

static const long X509_V_ERR_SUITE_B_INVALID_VERSION = 0;
static const long X509_V_ERR_SUITE_B_INVALID_ALGORITHM = 0;
static const long X509_V_ERR_SUITE_B_INVALID_CURVE = 0;
static const long X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM = 0;
static const long X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED = 0;
static const long X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256 = 0;
/* These 3 defines are unavailable in LibreSSL 2.5.x, but may be added
   in the future... */
#ifndef X509_V_ERR_HOSTNAME_MISMATCH
static const long X509_V_ERR_HOSTNAME_MISMATCH = 0;
#endif
#ifndef X509_V_ERR_EMAIL_MISMATCH
static const long X509_V_ERR_EMAIL_MISMATCH = 0;
#endif
#ifndef X509_V_ERR_IP_ADDRESS_MISMATCH
static const long X509_V_ERR_IP_ADDRESS_MISMATCH = 0;
#endif
#ifndef X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT
static const long X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT = 0;
#endif
#ifndef X509_CHECK_FLAG_NO_WILDCARDS
static const long X509_CHECK_FLAG_NO_WILDCARDS = 0;
#endif
#ifndef X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS
static const long X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS = 0;
#endif
#ifndef X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS
static const long X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS = 0;
#endif
#ifndef X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS
static const long X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS = 0;
#endif

/* X509_V_FLAG_TRUSTED_FIRST is also new in 1.0.2+, but it is added separately
   below because it shows up in some earlier 3rd party OpenSSL packages. */
static const long X509_V_FLAG_SUITEB_128_LOS_ONLY = 0;
static const long X509_V_FLAG_SUITEB_192_LOS = 0;
static const long X509_V_FLAG_SUITEB_128_LOS = 0;

#if !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
int (*X509_VERIFY_PARAM_set1_host)(X509_VERIFY_PARAM *, const char *,
                                   size_t) = NULL;
int (*X509_VERIFY_PARAM_set1_email)(X509_VERIFY_PARAM *, const char *,
                                    size_t) = NULL;
int (*X509_VERIFY_PARAM_set1_ip)(X509_VERIFY_PARAM *, const unsigned char *,
                                 size_t) = NULL;
int (*X509_VERIFY_PARAM_set1_ip_asc)(X509_VERIFY_PARAM *, const char *) = NULL;
void (*X509_VERIFY_PARAM_set_hostflags)(X509_VERIFY_PARAM *,
                                        unsigned int) = NULL;
#endif
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 || CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_110_VERIFICATION_PARAMS = 0;
#ifndef X509_CHECK_FLAG_NEVER_CHECK_SUBJECT
static const long X509_CHECK_FLAG_NEVER_CHECK_SUBJECT = 0;
#endif
#else
static const long Cryptography_HAS_110_VERIFICATION_PARAMS = 1;
#endif

/* OpenSSL 1.0.2+ or Solaris's backport */
#ifdef X509_V_FLAG_PARTIAL_CHAIN
static const long Cryptography_HAS_X509_V_FLAG_PARTIAL_CHAIN = 1;
#else
static const long Cryptography_HAS_X509_V_FLAG_PARTIAL_CHAIN = 0;
static const long X509_V_FLAG_PARTIAL_CHAIN = 0;
#endif

/* OpenSSL 1.0.2+, *or* Fedora 20's flavor of OpenSSL 1.0.1e... */
#ifdef X509_V_FLAG_TRUSTED_FIRST
static const long Cryptography_HAS_X509_V_FLAG_TRUSTED_FIRST = 1;
#else
static const long Cryptography_HAS_X509_V_FLAG_TRUSTED_FIRST = 0;
static const long X509_V_FLAG_TRUSTED_FIRST = 0;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110 && !CRYPTOGRAPHY_LIBRESSL_27_OR_GREATER
Cryptography_STACK_OF_X509_OBJECT *X509_STORE_get0_objects(X509_STORE *ctx) {
    return ctx->objs;
}
X509_VERIFY_PARAM *X509_STORE_get0_param(X509_STORE *store) {
    return store->param;
}
int X509_OBJECT_get_type(const X509_OBJECT *x) {
    return x->type;
}

/* from x509/x509_vfy.c */
X509 *X509_STORE_CTX_get0_cert(X509_STORE_CTX *ctx)
{
    return ctx->cert;
}

X509 *X509_OBJECT_get0_X509(X509_OBJECT *x) {
    return x->data.x509;
}
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
static const long Cryptography_HAS_X509_STORE_CTX_GET_ISSUER = 0;
typedef void *X509_STORE_CTX_get_issuer_fn;
X509_STORE_CTX_get_issuer_fn (*X509_STORE_get_get_issuer)(X509_STORE *) = NULL;
void (*X509_STORE_set_get_issuer)(X509_STORE *,
                                  X509_STORE_CTX_get_issuer_fn) = NULL;
#else
static const long Cryptography_HAS_X509_STORE_CTX_GET_ISSUER = 1;
#endif



/* This code is derived from the locking code found in the Python _ssl module's
   locking callback for OpenSSL.

   Copyright 2001-2016 Python Software Foundation; All Rights Reserved.

   It has been subsequently modified to use cross platform locking without
   using CPython APIs by Armin Rigo of the PyPy project.
*/

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110
#ifdef _WIN32
typedef CRITICAL_SECTION Cryptography_mutex;
static __inline void cryptography_mutex_init(Cryptography_mutex *mutex) {
    InitializeCriticalSection(mutex);
}
static __inline void cryptography_mutex_lock(Cryptography_mutex *mutex) {
    EnterCriticalSection(mutex);
}
static __inline void cryptography_mutex_unlock(Cryptography_mutex *mutex) {
    LeaveCriticalSection(mutex);
}
#else
typedef pthread_mutex_t Cryptography_mutex;
#define ASSERT_STATUS(call)                                                 if ((call) != 0) {                                                          perror("Fatal error in callback initialization: " #call);               abort();                                                            }
static inline void cryptography_mutex_init(Cryptography_mutex *mutex) {
#if !defined(pthread_mutexattr_default)
#  define pthread_mutexattr_default ((pthread_mutexattr_t *)NULL)
#endif
    ASSERT_STATUS(pthread_mutex_init(mutex, pthread_mutexattr_default));
}
static inline void cryptography_mutex_lock(Cryptography_mutex *mutex) {
    ASSERT_STATUS(pthread_mutex_lock(mutex));
}
static inline void cryptography_mutex_unlock(Cryptography_mutex *mutex) {
    ASSERT_STATUS(pthread_mutex_unlock(mutex));
}
#endif


static unsigned int _ssl_locks_count = 0;
static Cryptography_mutex *_ssl_locks = NULL;

static void _ssl_thread_locking_function(int mode, int n, const char *file,
                                         int line) {
    /* this function is needed to perform locking on shared data
       structures. (Note that OpenSSL uses a number of global data
       structures that will be implicitly shared whenever multiple
       threads use OpenSSL.) Multi-threaded applications will
       crash at random if it is not set.

       locking_function() must be able to handle up to
       CRYPTO_num_locks() different mutex locks. It sets the n-th
       lock if mode & CRYPTO_LOCK, and releases it otherwise.

       file and line are the file number of the function setting the
       lock. They can be useful for debugging.
    */

    if ((_ssl_locks == NULL) ||
        (n < 0) || ((unsigned)n >= _ssl_locks_count)) {
        return;
    }

    if (mode & CRYPTO_LOCK) {
        cryptography_mutex_lock(_ssl_locks + n);
    } else {
        cryptography_mutex_unlock(_ssl_locks + n);
    }
}

static void init_mutexes(void) {
    int i;
    for (i = 0; i < _ssl_locks_count; i++) {
        cryptography_mutex_init(_ssl_locks + i);
    }
}


int Cryptography_setup_ssl_threads(void) {
    if (_ssl_locks == NULL) {
        _ssl_locks_count = CRYPTO_num_locks();
        _ssl_locks = calloc(_ssl_locks_count, sizeof(Cryptography_mutex));
        if (_ssl_locks == NULL) {
            return 0;
        }
        init_mutexes();
        CRYPTO_set_locking_callback(_ssl_thread_locking_function);
#ifndef _WIN32
        pthread_atfork(NULL, NULL, &init_mutexes);
#endif
    }
    return 1;
}
#else
int (*Cryptography_setup_ssl_threads)(void) = NULL;
#endif

typedef struct {
    char *password;
    int length;
    int called;
    int error;
    int maxsize;
} CRYPTOGRAPHY_PASSWORD_DATA;

int Cryptography_pem_password_cb(char *buf, int size,
                                  int rwflag, void *userdata) {
    /* The password cb is only invoked if OpenSSL decides the private
       key is encrypted. So this path only occurs if it needs a password */
    CRYPTOGRAPHY_PASSWORD_DATA *st = (CRYPTOGRAPHY_PASSWORD_DATA *)userdata;
    st->called += 1;
    st->maxsize = size;
    if (st->length == 0) {
        st->error = -1;
        return 0;
    } else if (st->length < size) {
        memcpy(buf, st->password, st->length);
        return st->length;
    } else {
        st->error = -2;
        return 0;
    }
}

#define CRYPTOGRAPHY_PACKAGE_VERSION "2.7"

/************************************************************/

static void *_cffi_types[] = {
/*  0 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1138), // ACCESS_DESCRIPTION *()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *, int)
/*  1 */ _CFFI_OP(_CFFI_OP_POINTER, 3284), // Cryptography_STACK_OF_ACCESS_DESCRIPTION *
/*  2 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7), // int
/*  3 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  4 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1138), // ACCESS_DESCRIPTION *()(void)
/*  5 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  6 */ _CFFI_OP(_CFFI_OP_FUNCTION, 818), // ASN1_ENUMERATED *()(void)
/*  7 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  8 */ _CFFI_OP(_CFFI_OP_FUNCTION, 9), // ASN1_GENERALIZEDTIME *()(ASN1_GENERALIZEDTIME *, time_t)
/*  9 */ _CFFI_OP(_CFFI_OP_POINTER, 3254), // ASN1_GENERALIZEDTIME *
/* 10 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, _cffi_prim_int(sizeof(time_t), (
           ((time_t)-1) | 0 /* check that time_t is an integer type */
         ) <= 0)), // time_t
/* 11 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 12 */ _CFFI_OP(_CFFI_OP_FUNCTION, 9), // ASN1_GENERALIZEDTIME *()(ASN1_OCTET_STRING *, ASN1_GENERALIZEDTIME * *)
/* 13 */ _CFFI_OP(_CFFI_OP_POINTER, 3263), // ASN1_OCTET_STRING *
/* 14 */ _CFFI_OP(_CFFI_OP_POINTER, 9), // ASN1_GENERALIZEDTIME * *
/* 15 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 16 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3253), // ASN1_GENERALIZEDTIME const *()(OCSP_BASICRESP const *)
/* 17 */ _CFFI_OP(_CFFI_OP_POINTER, 3329), // OCSP_BASICRESP const *
/* 18 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 19 */ _CFFI_OP(_CFFI_OP_FUNCTION, 21), // ASN1_INTEGER *()(BIGNUM *, ASN1_INTEGER *)
/* 20 */ _CFFI_OP(_CFFI_OP_POINTER, 3269), // BIGNUM *
/* 21 */ _CFFI_OP(_CFFI_OP_POINTER, 3256), // ASN1_INTEGER *
/* 22 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 23 */ _CFFI_OP(_CFFI_OP_FUNCTION, 21), // ASN1_INTEGER *()(Cryptography_STACK_OF_ASN1_INTEGER *, int)
/* 24 */ _CFFI_OP(_CFFI_OP_POINTER, 3285), // Cryptography_STACK_OF_ASN1_INTEGER *
/* 25 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 26 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 27 */ _CFFI_OP(_CFFI_OP_FUNCTION, 21), // ASN1_INTEGER *()(X509 *)
/* 28 */ _CFFI_OP(_CFFI_OP_POINTER, 3368), // X509 *
/* 29 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 30 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3255), // ASN1_INTEGER const *()(X509_REVOKED const *)
/* 31 */ _CFFI_OP(_CFFI_OP_POINTER, 3381), // X509_REVOKED const *
/* 32 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 33 */ _CFFI_OP(_CFFI_OP_FUNCTION, 98), // ASN1_ITEM const *()(ASN1_ITEM_EXP *)
/* 34 */ _CFFI_OP(_CFFI_OP_POINTER, 3258), // ASN1_ITEM_EXP *
/* 35 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 36 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3259), // ASN1_NULL *()(void)
/* 37 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 38 */ _CFFI_OP(_CFFI_OP_FUNCTION, 657), // ASN1_OBJECT *()(ASN1_OBJECT const *)
/* 39 */ _CFFI_OP(_CFFI_OP_POINTER, 3261), // ASN1_OBJECT const *
/* 40 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 41 */ _CFFI_OP(_CFFI_OP_FUNCTION, 657), // ASN1_OBJECT *()(Cryptography_STACK_OF_ASN1_OBJECT *, int)
/* 42 */ _CFFI_OP(_CFFI_OP_POINTER, 3286), // Cryptography_STACK_OF_ASN1_OBJECT *
/* 43 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 44 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 45 */ _CFFI_OP(_CFFI_OP_FUNCTION, 657), // ASN1_OBJECT *()(X509_EXTENSION *)
/* 46 */ _CFFI_OP(_CFFI_OP_POINTER, 3375), // X509_EXTENSION *
/* 47 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 48 */ _CFFI_OP(_CFFI_OP_FUNCTION, 657), // ASN1_OBJECT *()(X509_NAME_ENTRY *)
/* 49 */ _CFFI_OP(_CFFI_OP_POINTER, 3378), // X509_NAME_ENTRY *
/* 50 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 51 */ _CFFI_OP(_CFFI_OP_FUNCTION, 657), // ASN1_OBJECT *()(char const *, int)
/* 52 */ _CFFI_OP(_CFFI_OP_POINTER, 3385), // char const *
/* 53 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 54 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 55 */ _CFFI_OP(_CFFI_OP_FUNCTION, 657), // ASN1_OBJECT *()(int)
/* 56 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 57 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 58 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(ASN1_OCTET_STRING *, long)
/* 59 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 60 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9), // long
/* 61 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 62 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(ASN1_OCTET_STRING *, time_t)
/* 63 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 64 */ _CFFI_OP(_CFFI_OP_NOOP, 10),
/* 65 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 66 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509 *)
/* 67 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 68 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 69 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509_CRL *)
/* 70 */ _CFFI_OP(_CFFI_OP_POINTER, 3374), // X509_CRL *
/* 71 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 72 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509_EXTENSION *)
/* 73 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 74 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 75 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509_NAME_ENTRY *)
/* 76 */ _CFFI_OP(_CFFI_OP_NOOP, 49),
/* 77 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 78 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(char const *)
/* 79 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 80 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 81 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(void)
/* 82 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 83 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3262), // ASN1_OCTET_STRING const *()(OCSP_BASICRESP const *)
/* 84 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 85 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 86 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3262), // ASN1_OCTET_STRING const *()(X509_REVOKED const *)
/* 87 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 88 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 89 */ _CFFI_OP(_CFFI_OP_FUNCTION, 866), // ASN1_TYPE *()(ASN1_TYPE * *, unsigned char const * *, long)
/* 90 */ _CFFI_OP(_CFFI_OP_POINTER, 866), // ASN1_TYPE * *
/* 91 */ _CFFI_OP(_CFFI_OP_POINTER, 121), // unsigned char const * *
/* 92 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 93 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 94 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3265), // ASN1_VALUE *()(ASN1_VALUE * *, unsigned char const * *, long, ASN1_ITEM const *)
/* 95 */ _CFFI_OP(_CFFI_OP_POINTER, 3265), // ASN1_VALUE * *
/* 96 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 97 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 98 */ _CFFI_OP(_CFFI_OP_POINTER, 3257), // ASN1_ITEM const *
/* 99 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 100 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2756), // AUTHORITY_KEYID *()(void)
/* 101 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 102 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2759), // BASIC_CONSTRAINTS *()(void)
/* 103 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 104 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(ASN1_INTEGER *, BIGNUM *)
/* 105 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 106 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 107 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 108 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(BIGNUM *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 109 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 110 */ _CFFI_OP(_CFFI_OP_POINTER, 3269), // BIGNUM const *
/* 111 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 112 */ _CFFI_OP(_CFFI_OP_POINTER, 3273), // BN_CTX *
/* 113 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 114 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(BIGNUM const *)
/* 115 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 116 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 117 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(BN_CTX *)
/* 118 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 119 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 120 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(unsigned char const *, int, BIGNUM *)
/* 121 */ _CFFI_OP(_CFFI_OP_POINTER, 3390), // unsigned char const *
/* 122 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 123 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 124 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 125 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(void)
/* 126 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 127 */ _CFFI_OP(_CFFI_OP_FUNCTION, 110), // BIGNUM const *()(EC_KEY const *)
/* 128 */ _CFFI_OP(_CFFI_OP_POINTER, 3306), // EC_KEY const *
/* 129 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 130 */ _CFFI_OP(_CFFI_OP_FUNCTION, 110), // BIGNUM const *()(void)
/* 131 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 132 */ _CFFI_OP(_CFFI_OP_FUNCTION, 137), // BIO *()(BIO_METHOD *)
/* 133 */ _CFFI_OP(_CFFI_OP_POINTER, 3272), // BIO_METHOD *
/* 134 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 135 */ _CFFI_OP(_CFFI_OP_FUNCTION, 137), // BIO *()(PKCS7 *, BIO *)
/* 136 */ _CFFI_OP(_CFFI_OP_POINTER, 3342), // PKCS7 *
/* 137 */ _CFFI_OP(_CFFI_OP_POINTER, 3270), // BIO *
/* 138 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 139 */ _CFFI_OP(_CFFI_OP_FUNCTION, 137), // BIO *()(SSL const *)
/* 140 */ _CFFI_OP(_CFFI_OP_POINTER, 3361), // SSL const *
/* 141 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 142 */ _CFFI_OP(_CFFI_OP_FUNCTION, 137), // BIO *()(char const *, char const *)
/* 143 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 144 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 145 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 146 */ _CFFI_OP(_CFFI_OP_FUNCTION, 137), // BIO *()(int, int)
/* 147 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 148 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 149 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 150 */ _CFFI_OP(_CFFI_OP_FUNCTION, 137), // BIO *()(void const *, int)
/* 151 */ _CFFI_OP(_CFFI_OP_POINTER, 3393), // void const *
/* 152 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 153 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 154 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3271), // BIO_METHOD const *()(void)
/* 155 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 156 */ _CFFI_OP(_CFFI_OP_FUNCTION, 112), // BN_CTX *()(void)
/* 157 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 158 */ _CFFI_OP(_CFFI_OP_FUNCTION, 895), // BN_MONT_CTX *()(void)
/* 159 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 160 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1113), // CMAC_CTX *()(void)
/* 161 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 162 */ _CFFI_OP(_CFFI_OP_FUNCTION, 781), // COMP_METHOD const *()(SSL *)
/* 163 */ _CFFI_OP(_CFFI_OP_POINTER, 3361), // SSL *
/* 164 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 165 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1), // Cryptography_STACK_OF_ACCESS_DESCRIPTION *()(void)
/* 166 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 167 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // Cryptography_STACK_OF_ASN1_INTEGER *()(void)
/* 168 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 169 */ _CFFI_OP(_CFFI_OP_FUNCTION, 42), // Cryptography_STACK_OF_ASN1_OBJECT *()(void)
/* 170 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 171 */ _CFFI_OP(_CFFI_OP_FUNCTION, 240), // Cryptography_STACK_OF_DIST_POINT *()(void)
/* 172 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 173 */ _CFFI_OP(_CFFI_OP_FUNCTION, 371), // Cryptography_STACK_OF_GENERAL_SUBTREE *()(void)
/* 174 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 175 */ _CFFI_OP(_CFFI_OP_FUNCTION, 490), // Cryptography_STACK_OF_POLICYINFO *()(void)
/* 176 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 177 */ _CFFI_OP(_CFFI_OP_FUNCTION, 496), // Cryptography_STACK_OF_POLICYQUALINFO *()(void)
/* 178 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 179 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1183), // Cryptography_STACK_OF_SCT *()(void)
/* 180 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 181 */ _CFFI_OP(_CFFI_OP_FUNCTION, 534), // Cryptography_STACK_OF_SSL_CIPHER *()(SSL const *)
/* 182 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 183 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 184 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // Cryptography_STACK_OF_X509 *()(PKCS7 *, Cryptography_STACK_OF_X509 *, int)
/* 185 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 186 */ _CFFI_OP(_CFFI_OP_POINTER, 3294), // Cryptography_STACK_OF_X509 *
/* 187 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 188 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 189 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // Cryptography_STACK_OF_X509 *()(SSL const *)
/* 190 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 191 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 192 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // Cryptography_STACK_OF_X509 *()(X509_STORE_CTX *)
/* 193 */ _CFFI_OP(_CFFI_OP_POINTER, 3383), // X509_STORE_CTX *
/* 194 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 195 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // Cryptography_STACK_OF_X509 *()(void)
/* 196 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 197 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3293), // Cryptography_STACK_OF_X509 const *()(OCSP_BASICRESP const *)
/* 198 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 199 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 200 */ _CFFI_OP(_CFFI_OP_FUNCTION, 611), // Cryptography_STACK_OF_X509_CRL *()(void)
/* 201 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 202 */ _CFFI_OP(_CFFI_OP_FUNCTION, 683), // Cryptography_STACK_OF_X509_NAME *()(SSL const *)
/* 203 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 204 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 205 */ _CFFI_OP(_CFFI_OP_FUNCTION, 683), // Cryptography_STACK_OF_X509_NAME *()(char const *)
/* 206 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 207 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 208 */ _CFFI_OP(_CFFI_OP_FUNCTION, 683), // Cryptography_STACK_OF_X509_NAME *()(void)
/* 209 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 210 */ _CFFI_OP(_CFFI_OP_FUNCTION, 211), // Cryptography_STACK_OF_X509_NAME_ENTRY *()(Cryptography_STACK_OF_X509_NAME_ENTRY *)
/* 211 */ _CFFI_OP(_CFFI_OP_POINTER, 3297), // Cryptography_STACK_OF_X509_NAME_ENTRY *
/* 212 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 213 */ _CFFI_OP(_CFFI_OP_FUNCTION, 211), // Cryptography_STACK_OF_X509_NAME_ENTRY *()(void)
/* 214 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 215 */ _CFFI_OP(_CFFI_OP_FUNCTION, 716), // Cryptography_STACK_OF_X509_OBJECT *()(X509_STORE *)
/* 216 */ _CFFI_OP(_CFFI_OP_POINTER, 3382), // X509_STORE *
/* 217 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 218 */ _CFFI_OP(_CFFI_OP_FUNCTION, 732), // Cryptography_STACK_OF_X509_REVOKED *()(X509_CRL *)
/* 219 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 220 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 221 */ _CFFI_OP(_CFFI_OP_FUNCTION, 232), // DH *()(BIO *, DH * *)
/* 222 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 223 */ _CFFI_OP(_CFFI_OP_POINTER, 232), // DH * *
/* 224 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 225 */ _CFFI_OP(_CFFI_OP_FUNCTION, 232), // DH *()(BIO *, DH * *, int(*)(char *, int, int, void *), void *)
/* 226 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 227 */ _CFFI_OP(_CFFI_OP_NOOP, 223),
/* 228 */ _CFFI_OP(_CFFI_OP_POINTER, 2302), // int(*)(char *, int, int, void *)
/* 229 */ _CFFI_OP(_CFFI_OP_POINTER, 3393), // void *
/* 230 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 231 */ _CFFI_OP(_CFFI_OP_FUNCTION, 232), // DH *()(DH *)
/* 232 */ _CFFI_OP(_CFFI_OP_POINTER, 3300), // DH *
/* 233 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 234 */ _CFFI_OP(_CFFI_OP_FUNCTION, 232), // DH *()(EVP_PKEY *)
/* 235 */ _CFFI_OP(_CFFI_OP_POINTER, 3317), // EVP_PKEY *
/* 236 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 237 */ _CFFI_OP(_CFFI_OP_FUNCTION, 232), // DH *()(void)
/* 238 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 239 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1159), // DIST_POINT *()(Cryptography_STACK_OF_DIST_POINT *, int)
/* 240 */ _CFFI_OP(_CFFI_OP_POINTER, 3287), // Cryptography_STACK_OF_DIST_POINT *
/* 241 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 242 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 243 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1159), // DIST_POINT *()(void)
/* 244 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 245 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2843), // DIST_POINT_NAME *()(void)
/* 246 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 247 */ _CFFI_OP(_CFFI_OP_FUNCTION, 248), // DSA *()(DSA *)
/* 248 */ _CFFI_OP(_CFFI_OP_POINTER, 3303), // DSA *
/* 249 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 250 */ _CFFI_OP(_CFFI_OP_FUNCTION, 248), // DSA *()(EVP_PKEY *)
/* 251 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 252 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 253 */ _CFFI_OP(_CFFI_OP_FUNCTION, 248), // DSA *()(void)
/* 254 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 255 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2860), // ECDSA_SIG *()(ECDSA_SIG * *, unsigned char const * *, long)
/* 256 */ _CFFI_OP(_CFFI_OP_POINTER, 2860), // ECDSA_SIG * *
/* 257 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 258 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 259 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 260 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2860), // ECDSA_SIG *()(unsigned char const *, int, EC_KEY *)
/* 261 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 262 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 263 */ _CFFI_OP(_CFFI_OP_POINTER, 3306), // EC_KEY *
/* 264 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 265 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2860), // ECDSA_SIG *()(void)
/* 266 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 267 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2863), // EC_GROUP *()(int)
/* 268 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 269 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 270 */ _CFFI_OP(_CFFI_OP_FUNCTION, 286), // EC_GROUP const *()(EC_KEY const *)
/* 271 */ _CFFI_OP(_CFFI_OP_NOOP, 128),
/* 272 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 273 */ _CFFI_OP(_CFFI_OP_FUNCTION, 263), // EC_KEY *()(BIO *, EC_KEY * *)
/* 274 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 275 */ _CFFI_OP(_CFFI_OP_POINTER, 263), // EC_KEY * *
/* 276 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 277 */ _CFFI_OP(_CFFI_OP_FUNCTION, 263), // EC_KEY *()(EVP_PKEY *)
/* 278 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 279 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 280 */ _CFFI_OP(_CFFI_OP_FUNCTION, 263), // EC_KEY *()(int)
/* 281 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 282 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 283 */ _CFFI_OP(_CFFI_OP_FUNCTION, 263), // EC_KEY *()(void)
/* 284 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 285 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1384), // EC_METHOD const *()(EC_GROUP const *)
/* 286 */ _CFFI_OP(_CFFI_OP_POINTER, 3305), // EC_GROUP const *
/* 287 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 288 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1293), // EC_POINT *()(EC_GROUP const *)
/* 289 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 290 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 291 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1293), // EC_POINT *()(EC_POINT const *, EC_GROUP const *)
/* 292 */ _CFFI_OP(_CFFI_OP_POINTER, 3308), // EC_POINT const *
/* 293 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 294 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 295 */ _CFFI_OP(_CFFI_OP_FUNCTION, 292), // EC_POINT const *()(EC_GROUP const *)
/* 296 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 297 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 298 */ _CFFI_OP(_CFFI_OP_FUNCTION, 292), // EC_POINT const *()(EC_KEY const *)
/* 299 */ _CFFI_OP(_CFFI_OP_NOOP, 128),
/* 300 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 301 */ _CFFI_OP(_CFFI_OP_FUNCTION, 340), // ENGINE *()(char const *)
/* 302 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 303 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 304 */ _CFFI_OP(_CFFI_OP_FUNCTION, 340), // ENGINE *()(void)
/* 305 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 306 */ _CFFI_OP(_CFFI_OP_FUNCTION, 479), // EVP_CIPHER const *()(EVP_CIPHER_CTX const *)
/* 307 */ _CFFI_OP(_CFFI_OP_POINTER, 3314), // EVP_CIPHER_CTX const *
/* 308 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 309 */ _CFFI_OP(_CFFI_OP_FUNCTION, 479), // EVP_CIPHER const *()(char const *)
/* 310 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 311 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 312 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1398), // EVP_CIPHER_CTX *()(void)
/* 313 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 314 */ _CFFI_OP(_CFFI_OP_FUNCTION, 396), // EVP_MD const *()(char const *)
/* 315 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 316 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 317 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1434), // EVP_MD_CTX *()(void)
/* 318 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 319 */ _CFFI_OP(_CFFI_OP_FUNCTION, 235), // EVP_PKEY *()(BIO *, EVP_PKEY * *)
/* 320 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 321 */ _CFFI_OP(_CFFI_OP_POINTER, 235), // EVP_PKEY * *
/* 322 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 323 */ _CFFI_OP(_CFFI_OP_FUNCTION, 235), // EVP_PKEY *()(BIO *, EVP_PKEY * *, int(*)(char *, int, int, void *), void *)
/* 324 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 325 */ _CFFI_OP(_CFFI_OP_NOOP, 321),
/* 326 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 327 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 328 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 329 */ _CFFI_OP(_CFFI_OP_FUNCTION, 235), // EVP_PKEY *()(NETSCAPE_SPKI *)
/* 330 */ _CFFI_OP(_CFFI_OP_POINTER, 3326), // NETSCAPE_SPKI *
/* 331 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 332 */ _CFFI_OP(_CFFI_OP_FUNCTION, 235), // EVP_PKEY *()(X509 *)
/* 333 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 334 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 335 */ _CFFI_OP(_CFFI_OP_FUNCTION, 235), // EVP_PKEY *()(X509_REQ *)
/* 336 */ _CFFI_OP(_CFFI_OP_POINTER, 3380), // X509_REQ *
/* 337 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 338 */ _CFFI_OP(_CFFI_OP_FUNCTION, 235), // EVP_PKEY *()(int, ENGINE *, unsigned char const *, size_t)
/* 339 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 340 */ _CFFI_OP(_CFFI_OP_POINTER, 3312), // ENGINE *
/* 341 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 342 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28), // size_t
/* 343 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 344 */ _CFFI_OP(_CFFI_OP_FUNCTION, 235), // EVP_PKEY *()(void)
/* 345 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 346 */ _CFFI_OP(_CFFI_OP_FUNCTION, 351), // EVP_PKEY_CTX *()(EVP_PKEY *, ENGINE *)
/* 347 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 348 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 349 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 350 */ _CFFI_OP(_CFFI_OP_FUNCTION, 351), // EVP_PKEY_CTX *()(EVP_PKEY_CTX *)
/* 351 */ _CFFI_OP(_CFFI_OP_POINTER, 3318), // EVP_PKEY_CTX *
/* 352 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 353 */ _CFFI_OP(_CFFI_OP_FUNCTION, 351), // EVP_PKEY_CTX *()(int, ENGINE *)
/* 354 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 355 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 356 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 357 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1006), // GENERAL_NAME *()(GENERAL_NAMES *, int)
/* 358 */ _CFFI_OP(_CFFI_OP_POINTER, 3321), // GENERAL_NAMES *
/* 359 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 360 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 361 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1006), // GENERAL_NAME *()(void)
/* 362 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 363 */ _CFFI_OP(_CFFI_OP_FUNCTION, 358), // GENERAL_NAMES *()(GENERAL_NAMES * *, unsigned char const * *, long)
/* 364 */ _CFFI_OP(_CFFI_OP_POINTER, 358), // GENERAL_NAMES * *
/* 365 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 366 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 367 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 368 */ _CFFI_OP(_CFFI_OP_FUNCTION, 358), // GENERAL_NAMES *()(void)
/* 369 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 370 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1166), // GENERAL_SUBTREE *()(Cryptography_STACK_OF_GENERAL_SUBTREE *, int)
/* 371 */ _CFFI_OP(_CFFI_OP_POINTER, 3288), // Cryptography_STACK_OF_GENERAL_SUBTREE *
/* 372 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 373 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 374 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1166), // GENERAL_SUBTREE *()(void)
/* 375 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 376 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1594), // HMAC_CTX *()(void)
/* 377 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 378 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2908), // ISSUING_DIST_POINT *()(void)
/* 379 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 380 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2911), // NAME_CONSTRAINTS *()(void)
/* 381 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 382 */ _CFFI_OP(_CFFI_OP_FUNCTION, 330), // NETSCAPE_SPKI *()(char const *, int)
/* 383 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 384 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 385 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 386 */ _CFFI_OP(_CFFI_OP_FUNCTION, 330), // NETSCAPE_SPKI *()(void)
/* 387 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 388 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2917), // NOTICEREF *()(void)
/* 389 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 390 */ _CFFI_OP(_CFFI_OP_FUNCTION, 429), // OCSP_BASICRESP *()(OCSP_RESPONSE *)
/* 391 */ _CFFI_OP(_CFFI_OP_POINTER, 3336), // OCSP_RESPONSE *
/* 392 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 393 */ _CFFI_OP(_CFFI_OP_FUNCTION, 429), // OCSP_BASICRESP *()(void)
/* 394 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 395 */ _CFFI_OP(_CFFI_OP_FUNCTION, 408), // OCSP_CERTID *()(EVP_MD const *, X509 const *, X509 const *)
/* 396 */ _CFFI_OP(_CFFI_OP_POINTER, 3315), // EVP_MD const *
/* 397 */ _CFFI_OP(_CFFI_OP_POINTER, 3368), // X509 const *
/* 398 */ _CFFI_OP(_CFFI_OP_NOOP, 397),
/* 399 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 400 */ _CFFI_OP(_CFFI_OP_FUNCTION, 408), // OCSP_CERTID *()(OCSP_ONEREQ *)
/* 401 */ _CFFI_OP(_CFFI_OP_POINTER, 3332), // OCSP_ONEREQ *
/* 402 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 403 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3330), // OCSP_CERTID const *()(OCSP_SINGLERESP const *)
/* 404 */ _CFFI_OP(_CFFI_OP_POINTER, 3337), // OCSP_SINGLERESP const *
/* 405 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 406 */ _CFFI_OP(_CFFI_OP_FUNCTION, 401), // OCSP_ONEREQ *()(OCSP_REQUEST *, OCSP_CERTID *)
/* 407 */ _CFFI_OP(_CFFI_OP_POINTER, 3333), // OCSP_REQUEST *
/* 408 */ _CFFI_OP(_CFFI_OP_POINTER, 3331), // OCSP_CERTID *
/* 409 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 410 */ _CFFI_OP(_CFFI_OP_FUNCTION, 401), // OCSP_ONEREQ *()(OCSP_REQUEST *, int)
/* 411 */ _CFFI_OP(_CFFI_OP_NOOP, 407),
/* 412 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 413 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 414 */ _CFFI_OP(_CFFI_OP_FUNCTION, 407), // OCSP_REQUEST *()(BIO *, OCSP_REQUEST * *)
/* 415 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 416 */ _CFFI_OP(_CFFI_OP_POINTER, 407), // OCSP_REQUEST * *
/* 417 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 418 */ _CFFI_OP(_CFFI_OP_FUNCTION, 407), // OCSP_REQUEST *()(void)
/* 419 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 420 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3334), // OCSP_RESPDATA const *()(OCSP_BASICRESP const *)
/* 421 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 422 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 423 */ _CFFI_OP(_CFFI_OP_FUNCTION, 391), // OCSP_RESPONSE *()(BIO *, OCSP_RESPONSE * *)
/* 424 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 425 */ _CFFI_OP(_CFFI_OP_POINTER, 391), // OCSP_RESPONSE * *
/* 426 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 427 */ _CFFI_OP(_CFFI_OP_FUNCTION, 391), // OCSP_RESPONSE *()(int, OCSP_BASICRESP *)
/* 428 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 429 */ _CFFI_OP(_CFFI_OP_POINTER, 3329), // OCSP_BASICRESP *
/* 430 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 431 */ _CFFI_OP(_CFFI_OP_FUNCTION, 644), // OCSP_SINGLERESP *()(OCSP_BASICRESP *, OCSP_CERTID *, int, int, ASN1_OCTET_STRING *, ASN1_OCTET_STRING *, ASN1_OCTET_STRING *)
/* 432 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 433 */ _CFFI_OP(_CFFI_OP_NOOP, 408),
/* 434 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 435 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 436 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 437 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 438 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 439 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 440 */ _CFFI_OP(_CFFI_OP_FUNCTION, 644), // OCSP_SINGLERESP *()(OCSP_BASICRESP *, int)
/* 441 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 442 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 443 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 444 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2936), // OTHERNAME *()(void)
/* 445 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 446 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1018), // PKCS12 *()(BIO *, PKCS12 * *)
/* 447 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 448 */ _CFFI_OP(_CFFI_OP_POINTER, 1018), // PKCS12 * *
/* 449 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 450 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1018), // PKCS12 *()(char *, char *, EVP_PKEY *, X509 *, Cryptography_STACK_OF_X509 *, int, int, int, int, int)
/* 451 */ _CFFI_OP(_CFFI_OP_POINTER, 3385), // char *
/* 452 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 453 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 454 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 455 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 456 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 457 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 458 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 459 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 460 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 461 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 462 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // PKCS7 *()(BIO *, BIO * *)
/* 463 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 464 */ _CFFI_OP(_CFFI_OP_POINTER, 137), // BIO * *
/* 465 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 466 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // PKCS7 *()(BIO *, PKCS7 * *)
/* 467 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 468 */ _CFFI_OP(_CFFI_OP_POINTER, 136), // PKCS7 * *
/* 469 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 470 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // PKCS7 *()(BIO *, PKCS7 * *, int(*)(char *, int, int, void *), void *)
/* 471 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 472 */ _CFFI_OP(_CFFI_OP_NOOP, 468),
/* 473 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 474 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 475 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 476 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // PKCS7 *()(Cryptography_STACK_OF_X509 *, BIO *, EVP_CIPHER const *, int)
/* 477 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 478 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 479 */ _CFFI_OP(_CFFI_OP_POINTER, 3313), // EVP_CIPHER const *
/* 480 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 481 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 482 */ _CFFI_OP(_CFFI_OP_FUNCTION, 136), // PKCS7 *()(X509 *, EVP_PKEY *, Cryptography_STACK_OF_X509 *, BIO *, int)
/* 483 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 484 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 485 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 486 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 487 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 488 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 489 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1173), // POLICYINFO *()(Cryptography_STACK_OF_POLICYINFO *, int)
/* 490 */ _CFFI_OP(_CFFI_OP_POINTER, 3289), // Cryptography_STACK_OF_POLICYINFO *
/* 491 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 492 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 493 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1173), // POLICYINFO *()(void)
/* 494 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 495 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1180), // POLICYQUALINFO *()(Cryptography_STACK_OF_POLICYQUALINFO *, int)
/* 496 */ _CFFI_OP(_CFFI_OP_POINTER, 3290), // Cryptography_STACK_OF_POLICYQUALINFO *
/* 497 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 498 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 499 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1180), // POLICYQUALINFO *()(void)
/* 500 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 501 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2951), // POLICY_CONSTRAINTS *()(void)
/* 502 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 503 */ _CFFI_OP(_CFFI_OP_FUNCTION, 517), // RSA *()(BIO *, RSA * *)
/* 504 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 505 */ _CFFI_OP(_CFFI_OP_POINTER, 517), // RSA * *
/* 506 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 507 */ _CFFI_OP(_CFFI_OP_FUNCTION, 517), // RSA *()(BIO *, RSA * *, int(*)(char *, int, int, void *), void *)
/* 508 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 509 */ _CFFI_OP(_CFFI_OP_NOOP, 505),
/* 510 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 511 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 512 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 513 */ _CFFI_OP(_CFFI_OP_FUNCTION, 517), // RSA *()(EVP_PKEY *)
/* 514 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 515 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 516 */ _CFFI_OP(_CFFI_OP_FUNCTION, 517), // RSA *()(RSA *)
/* 517 */ _CFFI_OP(_CFFI_OP_POINTER, 3357), // RSA *
/* 518 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 519 */ _CFFI_OP(_CFFI_OP_FUNCTION, 517), // RSA *()(void)
/* 520 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 521 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1184), // SCT *()(Cryptography_STACK_OF_SCT const *, int)
/* 522 */ _CFFI_OP(_CFFI_OP_POINTER, 3291), // Cryptography_STACK_OF_SCT const *
/* 523 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 524 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 525 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1184), // SCT *()(void)
/* 526 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 527 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3359), // SRTP_PROTECTION_PROFILE *()(SSL *)
/* 528 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 529 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 530 */ _CFFI_OP(_CFFI_OP_FUNCTION, 163), // SSL *()(SSL_CTX *)
/* 531 */ _CFFI_OP(_CFFI_OP_POINTER, 3363), // SSL_CTX *
/* 532 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 533 */ _CFFI_OP(_CFFI_OP_FUNCTION, 771), // SSL_CIPHER const *()(Cryptography_STACK_OF_SSL_CIPHER *, int)
/* 534 */ _CFFI_OP(_CFFI_OP_POINTER, 3292), // Cryptography_STACK_OF_SSL_CIPHER *
/* 535 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 536 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 537 */ _CFFI_OP(_CFFI_OP_FUNCTION, 771), // SSL_CIPHER const *()(SSL const *)
/* 538 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 539 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 540 */ _CFFI_OP(_CFFI_OP_FUNCTION, 531), // SSL_CTX *()(SSL *, SSL_CTX *)
/* 541 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 542 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 543 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 544 */ _CFFI_OP(_CFFI_OP_FUNCTION, 531), // SSL_CTX *()(SSL const *)
/* 545 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 546 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 547 */ _CFFI_OP(_CFFI_OP_FUNCTION, 531), // SSL_CTX *()(SSL_METHOD *)
/* 548 */ _CFFI_OP(_CFFI_OP_POINTER, 3365), // SSL_METHOD *
/* 549 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 550 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3364), // SSL_METHOD const *()(SSL_CTX *)
/* 551 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 552 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 553 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3364), // SSL_METHOD const *()(void)
/* 554 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 555 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1761), // SSL_SESSION *()(SSL *)
/* 556 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 557 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 558 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1761), // SSL_SESSION *()(SSL const *)
/* 559 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 560 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 561 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3102), // USERNOTICE *()(void)
/* 562 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 563 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(BIO *, X509 * *)
/* 564 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 565 */ _CFFI_OP(_CFFI_OP_POINTER, 28), // X509 * *
/* 566 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 567 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(BIO *, X509 * *, int(*)(char *, int, int, void *), void *)
/* 568 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 569 */ _CFFI_OP(_CFFI_OP_NOOP, 565),
/* 570 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 571 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 572 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 573 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(Cryptography_STACK_OF_X509 *, int)
/* 574 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 575 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 576 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 577 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(SSL const *)
/* 578 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 579 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 580 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(X509 *)
/* 581 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 582 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 583 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(X509_OBJECT *)
/* 584 */ _CFFI_OP(_CFFI_OP_POINTER, 3379), // X509_OBJECT *
/* 585 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 586 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(X509_STORE_CTX *)
/* 587 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 588 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 589 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(void)
/* 590 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 591 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3370), // X509V3_EXT_METHOD const *()(X509_EXTENSION *)
/* 592 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 593 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 594 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3372), // X509_ALGOR const *()(OCSP_BASICRESP const *)
/* 595 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 596 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 597 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3372), // X509_ALGOR const *()(X509 const *)
/* 598 */ _CFFI_OP(_CFFI_OP_NOOP, 397),
/* 599 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 600 */ _CFFI_OP(_CFFI_OP_FUNCTION, 70), // X509_CRL *()(BIO *, X509_CRL * *)
/* 601 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 602 */ _CFFI_OP(_CFFI_OP_POINTER, 70), // X509_CRL * *
/* 603 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 604 */ _CFFI_OP(_CFFI_OP_FUNCTION, 70), // X509_CRL *()(BIO *, X509_CRL * *, int(*)(char *, int, int, void *), void *)
/* 605 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 606 */ _CFFI_OP(_CFFI_OP_NOOP, 602),
/* 607 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 608 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 609 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 610 */ _CFFI_OP(_CFFI_OP_FUNCTION, 70), // X509_CRL *()(Cryptography_STACK_OF_X509_CRL *, int)
/* 611 */ _CFFI_OP(_CFFI_OP_POINTER, 3295), // Cryptography_STACK_OF_X509_CRL *
/* 612 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 613 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 614 */ _CFFI_OP(_CFFI_OP_FUNCTION, 70), // X509_CRL *()(X509_CRL *)
/* 615 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 616 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 617 */ _CFFI_OP(_CFFI_OP_FUNCTION, 70), // X509_CRL *()(void)
/* 618 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 619 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(CONF *, X509V3_CTX *, char *, char *)
/* 620 */ _CFFI_OP(_CFFI_OP_POINTER, 3278), // CONF *
/* 621 */ _CFFI_OP(_CFFI_OP_POINTER, 3369), // X509V3_CTX *
/* 622 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 623 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 624 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 625 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(Cryptography_LHASH_OF_CONF_VALUE *, X509V3_CTX *, int, char *)
/* 626 */ _CFFI_OP(_CFFI_OP_POINTER, 3283), // Cryptography_LHASH_OF_CONF_VALUE *
/* 627 */ _CFFI_OP(_CFFI_OP_NOOP, 621),
/* 628 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 629 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 630 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 631 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(OCSP_BASICRESP *, int)
/* 632 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 633 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 634 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 635 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(OCSP_ONEREQ *, int)
/* 636 */ _CFFI_OP(_CFFI_OP_NOOP, 401),
/* 637 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 638 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 639 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(OCSP_REQUEST *, int)
/* 640 */ _CFFI_OP(_CFFI_OP_NOOP, 407),
/* 641 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 642 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 643 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(OCSP_SINGLERESP *, int)
/* 644 */ _CFFI_OP(_CFFI_OP_POINTER, 3337), // OCSP_SINGLERESP *
/* 645 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 646 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 647 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(X509 *, int)
/* 648 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 649 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 650 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 651 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(X509_CRL *, int)
/* 652 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 653 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 654 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 655 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(X509_EXTENSION * *, ASN1_OBJECT *, int, ASN1_OCTET_STRING *)
/* 656 */ _CFFI_OP(_CFFI_OP_POINTER, 46), // X509_EXTENSION * *
/* 657 */ _CFFI_OP(_CFFI_OP_POINTER, 3261), // ASN1_OBJECT *
/* 658 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 659 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 660 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 661 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(X509_EXTENSION *)
/* 662 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 663 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 664 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(X509_EXTENSIONS *, int)
/* 665 */ _CFFI_OP(_CFFI_OP_POINTER, 3376), // X509_EXTENSIONS *
/* 666 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 667 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 668 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(X509_REVOKED *, int)
/* 669 */ _CFFI_OP(_CFFI_OP_POINTER, 3381), // X509_REVOKED *
/* 670 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 671 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 672 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // X509_EXTENSION *()(int, int, void *)
/* 673 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 674 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 675 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 676 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 677 */ _CFFI_OP(_CFFI_OP_FUNCTION, 665), // X509_EXTENSIONS *()(X509_REQ *)
/* 678 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 679 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 680 */ _CFFI_OP(_CFFI_OP_FUNCTION, 665), // X509_EXTENSIONS *()(void)
/* 681 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 682 */ _CFFI_OP(_CFFI_OP_FUNCTION, 693), // X509_NAME *()(Cryptography_STACK_OF_X509_NAME *, int)
/* 683 */ _CFFI_OP(_CFFI_OP_POINTER, 3296), // Cryptography_STACK_OF_X509_NAME *
/* 684 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 685 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 686 */ _CFFI_OP(_CFFI_OP_FUNCTION, 693), // X509_NAME *()(X509 *)
/* 687 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 688 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 689 */ _CFFI_OP(_CFFI_OP_FUNCTION, 693), // X509_NAME *()(X509_CRL *)
/* 690 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 691 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 692 */ _CFFI_OP(_CFFI_OP_FUNCTION, 693), // X509_NAME *()(X509_NAME *)
/* 693 */ _CFFI_OP(_CFFI_OP_POINTER, 3377), // X509_NAME *
/* 694 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 695 */ _CFFI_OP(_CFFI_OP_FUNCTION, 693), // X509_NAME *()(X509_REQ *)
/* 696 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 697 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 698 */ _CFFI_OP(_CFFI_OP_FUNCTION, 693), // X509_NAME *()(void)
/* 699 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 700 */ _CFFI_OP(_CFFI_OP_FUNCTION, 49), // X509_NAME_ENTRY *()(Cryptography_STACK_OF_X509_NAME_ENTRY *, int)
/* 701 */ _CFFI_OP(_CFFI_OP_NOOP, 211),
/* 702 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 703 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 704 */ _CFFI_OP(_CFFI_OP_FUNCTION, 49), // X509_NAME_ENTRY *()(X509_NAME *, int)
/* 705 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 706 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 707 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 708 */ _CFFI_OP(_CFFI_OP_FUNCTION, 49), // X509_NAME_ENTRY *()(X509_NAME_ENTRY * *, ASN1_OBJECT *, int, unsigned char const *, int)
/* 709 */ _CFFI_OP(_CFFI_OP_POINTER, 49), // X509_NAME_ENTRY * *
/* 710 */ _CFFI_OP(_CFFI_OP_NOOP, 657),
/* 711 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 712 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 713 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 714 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 715 */ _CFFI_OP(_CFFI_OP_FUNCTION, 584), // X509_OBJECT *()(Cryptography_STACK_OF_X509_OBJECT *, int)
/* 716 */ _CFFI_OP(_CFFI_OP_POINTER, 3298), // Cryptography_STACK_OF_X509_OBJECT *
/* 717 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 718 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 719 */ _CFFI_OP(_CFFI_OP_FUNCTION, 336), // X509_REQ *()(BIO *, X509_REQ * *)
/* 720 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 721 */ _CFFI_OP(_CFFI_OP_POINTER, 336), // X509_REQ * *
/* 722 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 723 */ _CFFI_OP(_CFFI_OP_FUNCTION, 336), // X509_REQ *()(BIO *, X509_REQ * *, int(*)(char *, int, int, void *), void *)
/* 724 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 725 */ _CFFI_OP(_CFFI_OP_NOOP, 721),
/* 726 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 727 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 728 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 729 */ _CFFI_OP(_CFFI_OP_FUNCTION, 336), // X509_REQ *()(void)
/* 730 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 731 */ _CFFI_OP(_CFFI_OP_FUNCTION, 669), // X509_REVOKED *()(Cryptography_STACK_OF_X509_REVOKED *, int)
/* 732 */ _CFFI_OP(_CFFI_OP_POINTER, 3299), // Cryptography_STACK_OF_X509_REVOKED *
/* 733 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 734 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 735 */ _CFFI_OP(_CFFI_OP_FUNCTION, 669), // X509_REVOKED *()(X509_REVOKED *)
/* 736 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 737 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 738 */ _CFFI_OP(_CFFI_OP_FUNCTION, 669), // X509_REVOKED *()(void)
/* 739 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 740 */ _CFFI_OP(_CFFI_OP_FUNCTION, 216), // X509_STORE *()(SSL_CTX const *)
/* 741 */ _CFFI_OP(_CFFI_OP_POINTER, 3363), // SSL_CTX const *
/* 742 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 743 */ _CFFI_OP(_CFFI_OP_FUNCTION, 216), // X509_STORE *()(void)
/* 744 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 745 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // X509_STORE_CTX *()(void)
/* 746 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 747 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1769), // X509_VERIFY_PARAM *()(SSL *)
/* 748 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 749 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 750 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1769), // X509_VERIFY_PARAM *()(SSL_CTX *)
/* 751 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 752 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 753 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1769), // X509_VERIFY_PARAM *()(X509_STORE *)
/* 754 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 755 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 756 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1769), // X509_VERIFY_PARAM *()(X509_STORE_CTX *)
/* 757 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 758 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 759 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1769), // X509_VERIFY_PARAM *()(void)
/* 760 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 761 */ _CFFI_OP(_CFFI_OP_FUNCTION, 451), // char *()(BIGNUM const *)
/* 762 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 763 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 764 */ _CFFI_OP(_CFFI_OP_FUNCTION, 451), // char *()(NETSCAPE_SPKI *)
/* 765 */ _CFFI_OP(_CFFI_OP_NOOP, 330),
/* 766 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 767 */ _CFFI_OP(_CFFI_OP_FUNCTION, 451), // char *()(SSL *)
/* 768 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 769 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 770 */ _CFFI_OP(_CFFI_OP_FUNCTION, 451), // char *()(SSL_CIPHER const *, char *, int)
/* 771 */ _CFFI_OP(_CFFI_OP_POINTER, 3362), // SSL_CIPHER const *
/* 772 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 773 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 774 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 775 */ _CFFI_OP(_CFFI_OP_FUNCTION, 451), // char *()(X509_NAME *, char *, int)
/* 776 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 777 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 778 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 779 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 780 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(COMP_METHOD const *)
/* 781 */ _CFFI_OP(_CFFI_OP_POINTER, 3277), // COMP_METHOD const *
/* 782 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 783 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(ENGINE const *)
/* 784 */ _CFFI_OP(_CFFI_OP_POINTER, 3312), // ENGINE const *
/* 785 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 786 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(SSL const *)
/* 787 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 788 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 789 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(SSL const *, int)
/* 790 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 791 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 792 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 793 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(SSL_CIPHER const *)
/* 794 */ _CFFI_OP(_CFFI_OP_NOOP, 771),
/* 795 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 796 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(int)
/* 797 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 798 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 799 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(long)
/* 800 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 801 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 802 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(unsigned long)
/* 803 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10), // unsigned long
/* 804 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 805 */ _CFFI_OP(_CFFI_OP_FUNCTION, 52), // char const *()(void)
/* 806 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 807 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1737), // ct_log_entry_type_t()(SCT const *)
/* 808 */ _CFFI_OP(_CFFI_OP_POINTER, 3358), // SCT const *
/* 809 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 810 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(AES_KEY *, unsigned char const *, unsigned char *, unsigned char const *, unsigned int)
/* 811 */ _CFFI_OP(_CFFI_OP_POINTER, 3251), // AES_KEY *
/* 812 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 813 */ _CFFI_OP(_CFFI_OP_POINTER, 3390), // unsigned char *
/* 814 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 815 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8), // unsigned int
/* 816 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 817 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_ENUMERATED *, long)
/* 818 */ _CFFI_OP(_CFFI_OP_POINTER, 3252), // ASN1_ENUMERATED *
/* 819 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 820 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 821 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_INTEGER *, long)
/* 822 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 823 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 824 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 825 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OBJECT const *)
/* 826 */ _CFFI_OP(_CFFI_OP_NOOP, 39),
/* 827 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 828 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OBJECT const *, ASN1_OBJECT const *)
/* 829 */ _CFFI_OP(_CFFI_OP_NOOP, 39),
/* 830 */ _CFFI_OP(_CFFI_OP_NOOP, 39),
/* 831 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 832 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING * *, ASN1_OBJECT * *, ASN1_OCTET_STRING * *, ASN1_INTEGER * *, OCSP_CERTID *)
/* 833 */ _CFFI_OP(_CFFI_OP_POINTER, 13), // ASN1_OCTET_STRING * *
/* 834 */ _CFFI_OP(_CFFI_OP_POINTER, 657), // ASN1_OBJECT * *
/* 835 */ _CFFI_OP(_CFFI_OP_NOOP, 833),
/* 836 */ _CFFI_OP(_CFFI_OP_POINTER, 21), // ASN1_INTEGER * *
/* 837 */ _CFFI_OP(_CFFI_OP_NOOP, 408),
/* 838 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 839 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *)
/* 840 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 841 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 842 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, char const *)
/* 843 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 844 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 845 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 846 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, int)
/* 847 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 848 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 849 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 850 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, int, int)
/* 851 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 852 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 853 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 854 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 855 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, unsigned char const *, int)
/* 856 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 857 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 858 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 859 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 860 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, void const *, int)
/* 861 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 862 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 863 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 864 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 865 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_TYPE *, unsigned char * *)
/* 866 */ _CFFI_OP(_CFFI_OP_POINTER, 3264), // ASN1_TYPE *
/* 867 */ _CFFI_OP(_CFFI_OP_POINTER, 813), // unsigned char * *
/* 868 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 869 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM * *, char const *)
/* 870 */ _CFFI_OP(_CFFI_OP_POINTER, 20), // BIGNUM * *
/* 871 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 872 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 873 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *)
/* 874 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 875 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 876 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 877 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *)
/* 878 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 879 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 880 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 881 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 882 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 883 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 884 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 885 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 886 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 887 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 888 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 889 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BIGNUM const *, BN_CTX *, BN_MONT_CTX *)
/* 890 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 891 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 892 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 893 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 894 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 895 */ _CFFI_OP(_CFFI_OP_POINTER, 3275), // BN_MONT_CTX *
/* 896 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 897 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 898 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 899 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 900 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 901 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 902 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 903 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BN_ULONG)
/* 904 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 905 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, _cffi_prim_int(sizeof(BN_ULONG), (
           ((BN_ULONG)-1) | 0 /* check that BN_ULONG is an integer type */
         ) <= 0)), // BN_ULONG
/* 906 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 907 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, int, int, BIGNUM const *, BIGNUM const *, BN_GENCB *)
/* 908 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 909 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 910 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 911 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 912 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 913 */ _CFFI_OP(_CFFI_OP_POINTER, 3274), // BN_GENCB *
/* 914 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 915 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *)
/* 916 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 917 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 918 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *, BIGNUM const *)
/* 919 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 920 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 921 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 922 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *, int, BN_CTX *, BN_GENCB *)
/* 923 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 924 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 925 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 926 */ _CFFI_OP(_CFFI_OP_NOOP, 913),
/* 927 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 928 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *, unsigned char *)
/* 929 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 930 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 931 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 932 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *)
/* 933 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 934 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 935 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, ASN1_INTEGER *)
/* 936 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 937 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 938 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 939 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, ASN1_OCTET_STRING *)
/* 940 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 941 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 942 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 943 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, DH *)
/* 944 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 945 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 946 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 947 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, DSA *)
/* 948 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 949 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 950 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 951 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, DSA *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 952 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 953 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 954 */ _CFFI_OP(_CFFI_OP_NOOP, 479),
/* 955 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 956 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 957 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 958 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 959 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 960 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EC_KEY *)
/* 961 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 962 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 963 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 964 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EC_KEY *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 965 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 966 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 967 */ _CFFI_OP(_CFFI_OP_NOOP, 479),
/* 968 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 969 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 970 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 971 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 972 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 973 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EVP_PKEY *)
/* 974 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 975 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 976 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 977 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EVP_PKEY *, EVP_CIPHER const *, char *, int, int(*)(char *, int, int, void *), void *)
/* 978 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 979 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 980 */ _CFFI_OP(_CFFI_OP_NOOP, 479),
/* 981 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 982 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 983 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 984 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 985 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 986 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EVP_PKEY *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 987 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 988 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 989 */ _CFFI_OP(_CFFI_OP_NOOP, 479),
/* 990 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 991 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 992 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 993 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 994 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 995 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EVP_PKEY *, int, char *, int, int(*)(char *, int, int, void *), void *)
/* 996 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 997 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 998 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 999 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 1000 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1001 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 1002 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1003 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1004 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, GENERAL_NAME *)
/* 1005 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1006 */ _CFFI_OP(_CFFI_OP_POINTER, 3320), // GENERAL_NAME *
/* 1007 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1008 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, OCSP_REQUEST *)
/* 1009 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1010 */ _CFFI_OP(_CFFI_OP_NOOP, 407),
/* 1011 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1012 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, OCSP_RESPONSE *)
/* 1013 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1014 */ _CFFI_OP(_CFFI_OP_NOOP, 391),
/* 1015 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1016 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, PKCS12 *)
/* 1017 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1018 */ _CFFI_OP(_CFFI_OP_POINTER, 3341), // PKCS12 *
/* 1019 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1020 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, PKCS7 *)
/* 1021 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1022 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1023 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1024 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, PKCS7 *, BIO *, int)
/* 1025 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1026 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1027 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1028 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1029 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1030 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA *)
/* 1031 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1032 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 1033 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1034 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 1035 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1036 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 1037 */ _CFFI_OP(_CFFI_OP_NOOP, 479),
/* 1038 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1039 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1040 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 1041 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1042 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1043 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA const *)
/* 1044 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1045 */ _CFFI_OP(_CFFI_OP_POINTER, 3357), // RSA const *
/* 1046 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1047 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA const *, int)
/* 1048 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1049 */ _CFFI_OP(_CFFI_OP_NOOP, 1045),
/* 1050 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1051 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1052 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, SSL_SESSION const *)
/* 1053 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1054 */ _CFFI_OP(_CFFI_OP_POINTER, 3366), // SSL_SESSION const *
/* 1055 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1056 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509 *)
/* 1057 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1058 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1059 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1060 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509 *, unsigned long, unsigned long)
/* 1061 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1062 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1063 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1064 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1065 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1066 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_CRL *)
/* 1067 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1068 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 1069 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1070 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_EXTENSION *, unsigned long, int)
/* 1071 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1072 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1073 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1074 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1075 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1076 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_NAME *, int, unsigned long)
/* 1077 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1078 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 1079 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1080 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1081 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1082 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_REQ *)
/* 1083 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1084 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 1085 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1086 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_REQ *, unsigned long, unsigned long)
/* 1087 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1088 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 1089 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1090 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1091 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1092 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, char *, int)
/* 1093 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1094 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 1095 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1096 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1097 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, void *, int)
/* 1098 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1099 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1100 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1101 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1102 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, void const *, int)
/* 1103 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1104 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 1105 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1106 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1107 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BN_MONT_CTX *, BIGNUM const *, BN_CTX *)
/* 1108 */ _CFFI_OP(_CFFI_OP_NOOP, 895),
/* 1109 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1110 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1111 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1112 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, CMAC_CTX const *)
/* 1113 */ _CFFI_OP(_CFFI_OP_POINTER, 3276), // CMAC_CTX *
/* 1114 */ _CFFI_OP(_CFFI_OP_POINTER, 3276), // CMAC_CTX const *
/* 1115 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1116 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, unsigned char *, size_t *)
/* 1117 */ _CFFI_OP(_CFFI_OP_NOOP, 1113),
/* 1118 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1119 */ _CFFI_OP(_CFFI_OP_POINTER, 342), // size_t *
/* 1120 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1121 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, void const *, size_t)
/* 1122 */ _CFFI_OP(_CFFI_OP_NOOP, 1113),
/* 1123 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 1124 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1125 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1126 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, void const *, size_t, EVP_CIPHER const *, ENGINE *)
/* 1127 */ _CFFI_OP(_CFFI_OP_NOOP, 1113),
/* 1128 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 1129 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1130 */ _CFFI_OP(_CFFI_OP_NOOP, 479),
/* 1131 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 1132 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1133 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *)
/* 1134 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 1135 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1136 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *, ACCESS_DESCRIPTION *)
/* 1137 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 1138 */ _CFFI_OP(_CFFI_OP_POINTER, 3250), // ACCESS_DESCRIPTION *
/* 1139 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1140 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_INTEGER *)
/* 1141 */ _CFFI_OP(_CFFI_OP_NOOP, 24),
/* 1142 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1143 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_INTEGER *, ASN1_INTEGER *)
/* 1144 */ _CFFI_OP(_CFFI_OP_NOOP, 24),
/* 1145 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 1146 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1147 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_OBJECT *)
/* 1148 */ _CFFI_OP(_CFFI_OP_NOOP, 42),
/* 1149 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1150 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_OBJECT *, ASN1_OBJECT *)
/* 1151 */ _CFFI_OP(_CFFI_OP_NOOP, 42),
/* 1152 */ _CFFI_OP(_CFFI_OP_NOOP, 657),
/* 1153 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1154 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_DIST_POINT *)
/* 1155 */ _CFFI_OP(_CFFI_OP_NOOP, 240),
/* 1156 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1157 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_DIST_POINT *, DIST_POINT *)
/* 1158 */ _CFFI_OP(_CFFI_OP_NOOP, 240),
/* 1159 */ _CFFI_OP(_CFFI_OP_POINTER, 3301), // DIST_POINT *
/* 1160 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1161 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_GENERAL_SUBTREE *)
/* 1162 */ _CFFI_OP(_CFFI_OP_NOOP, 371),
/* 1163 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1164 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_GENERAL_SUBTREE *, GENERAL_SUBTREE *)
/* 1165 */ _CFFI_OP(_CFFI_OP_NOOP, 371),
/* 1166 */ _CFFI_OP(_CFFI_OP_POINTER, 3322), // GENERAL_SUBTREE *
/* 1167 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1168 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYINFO *)
/* 1169 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1170 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1171 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYINFO *, POLICYINFO *)
/* 1172 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1173 */ _CFFI_OP(_CFFI_OP_POINTER, 3354), // POLICYINFO *
/* 1174 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1175 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYQUALINFO *)
/* 1176 */ _CFFI_OP(_CFFI_OP_NOOP, 496),
/* 1177 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1178 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYQUALINFO *, POLICYQUALINFO *)
/* 1179 */ _CFFI_OP(_CFFI_OP_NOOP, 496),
/* 1180 */ _CFFI_OP(_CFFI_OP_POINTER, 3355), // POLICYQUALINFO *
/* 1181 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1182 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_SCT *, SCT *)
/* 1183 */ _CFFI_OP(_CFFI_OP_POINTER, 3291), // Cryptography_STACK_OF_SCT *
/* 1184 */ _CFFI_OP(_CFFI_OP_POINTER, 3358), // SCT *
/* 1185 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1186 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_SCT const *)
/* 1187 */ _CFFI_OP(_CFFI_OP_NOOP, 522),
/* 1188 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1189 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_SSL_CIPHER *)
/* 1190 */ _CFFI_OP(_CFFI_OP_NOOP, 534),
/* 1191 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1192 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509 *)
/* 1193 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1194 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1195 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509 *, X509 *)
/* 1196 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1197 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1198 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1199 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_CRL *)
/* 1200 */ _CFFI_OP(_CFFI_OP_NOOP, 611),
/* 1201 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1202 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_CRL *, X509_CRL *)
/* 1203 */ _CFFI_OP(_CFFI_OP_NOOP, 611),
/* 1204 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 1205 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1206 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME *)
/* 1207 */ _CFFI_OP(_CFFI_OP_NOOP, 683),
/* 1208 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1209 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME *, X509_NAME *)
/* 1210 */ _CFFI_OP(_CFFI_OP_NOOP, 683),
/* 1211 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 1212 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1213 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME_ENTRY *)
/* 1214 */ _CFFI_OP(_CFFI_OP_NOOP, 211),
/* 1215 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1216 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME_ENTRY *, X509_NAME_ENTRY *)
/* 1217 */ _CFFI_OP(_CFFI_OP_NOOP, 211),
/* 1218 */ _CFFI_OP(_CFFI_OP_NOOP, 49),
/* 1219 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1220 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_OBJECT *)
/* 1221 */ _CFFI_OP(_CFFI_OP_NOOP, 716),
/* 1222 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1223 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_REVOKED *)
/* 1224 */ _CFFI_OP(_CFFI_OP_NOOP, 732),
/* 1225 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1226 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *)
/* 1227 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 1228 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1229 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *, BIGNUM *, BIGNUM *)
/* 1230 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 1231 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1232 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1233 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1234 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 1235 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 1236 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1237 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1238 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1239 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1240 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *, int, int, BN_GENCB *)
/* 1241 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 1242 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1243 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1244 */ _CFFI_OP(_CFFI_OP_NOOP, 913),
/* 1245 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1246 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH const *)
/* 1247 */ _CFFI_OP(_CFFI_OP_POINTER, 3300), // DH const *
/* 1248 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1249 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH const *, int *)
/* 1250 */ _CFFI_OP(_CFFI_OP_NOOP, 1247),
/* 1251 */ _CFFI_OP(_CFFI_OP_POINTER, 2), // int *
/* 1252 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1253 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *)
/* 1254 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 1255 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1256 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *, BIGNUM *, BIGNUM *)
/* 1257 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 1258 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1259 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1260 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1261 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 1262 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 1263 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1264 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1265 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1266 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1267 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *, int, unsigned char *, int, int *, unsigned long *, BN_GENCB *)
/* 1268 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 1269 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1270 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1271 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1272 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1273 */ _CFFI_OP(_CFFI_OP_POINTER, 803), // unsigned long *
/* 1274 */ _CFFI_OP(_CFFI_OP_NOOP, 913),
/* 1275 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1276 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA const *)
/* 1277 */ _CFFI_OP(_CFFI_OP_POINTER, 3303), // DSA const *
/* 1278 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1279 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ECDSA_SIG const *, unsigned char * *)
/* 1280 */ _CFFI_OP(_CFFI_OP_POINTER, 3304), // ECDSA_SIG const *
/* 1281 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 1282 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1283 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *)
/* 1284 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1285 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1286 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, BIGNUM *, BN_CTX *)
/* 1287 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1288 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1289 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1290 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1291 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 1292 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1293 */ _CFFI_OP(_CFFI_OP_POINTER, 3308), // EC_POINT *
/* 1294 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1295 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1296 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1297 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1298 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, EC_POINT const *, BIGNUM const *, BN_CTX *)
/* 1299 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1300 */ _CFFI_OP(_CFFI_OP_NOOP, 1293),
/* 1301 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1302 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1303 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1304 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1305 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1306 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, int, BN_CTX *)
/* 1307 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1308 */ _CFFI_OP(_CFFI_OP_NOOP, 1293),
/* 1309 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1310 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1311 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1312 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1313 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BN_CTX *)
/* 1314 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1315 */ _CFFI_OP(_CFFI_OP_NOOP, 1293),
/* 1316 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1317 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1318 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, EC_POINT const *, BN_CTX *)
/* 1319 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1320 */ _CFFI_OP(_CFFI_OP_NOOP, 1293),
/* 1321 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1322 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1323 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1324 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, EC_POINT const *, EC_POINT const *, BN_CTX *)
/* 1325 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1326 */ _CFFI_OP(_CFFI_OP_NOOP, 1293),
/* 1327 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1328 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1329 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1330 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1331 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, unsigned char const *, size_t, BN_CTX *)
/* 1332 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1333 */ _CFFI_OP(_CFFI_OP_NOOP, 1293),
/* 1334 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1335 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1336 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1337 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1338 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *)
/* 1339 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1340 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1341 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1342 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *, BIGNUM *, BIGNUM *, BN_CTX *)
/* 1343 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1344 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1345 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1346 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1347 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1348 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1349 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *, BN_CTX *)
/* 1350 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1351 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1352 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1353 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1354 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *, EC_POINT const *, BN_CTX *)
/* 1355 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1356 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1357 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1358 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1359 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1360 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *)
/* 1361 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 1362 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1363 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, BIGNUM *, BIGNUM *)
/* 1364 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 1365 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1366 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1367 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1368 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, BIGNUM const *)
/* 1369 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 1370 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1371 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1372 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, EC_GROUP const *)
/* 1373 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 1374 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 1375 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1376 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, EC_POINT const *)
/* 1377 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 1378 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 1379 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1380 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY const *)
/* 1381 */ _CFFI_OP(_CFFI_OP_NOOP, 128),
/* 1382 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1383 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_METHOD const *)
/* 1384 */ _CFFI_OP(_CFFI_OP_POINTER, 3307), // EC_METHOD const *
/* 1385 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1386 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ENGINE *)
/* 1387 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 1388 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1389 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ENGINE *, char const *, long, void *, void(*)(void), int)
/* 1390 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 1391 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1392 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1393 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1394 */ _CFFI_OP(_CFFI_OP_POINTER, 3240), // void(*)(void)
/* 1395 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1396 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1397 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *)
/* 1398 */ _CFFI_OP(_CFFI_OP_POINTER, 3314), // EVP_CIPHER_CTX *
/* 1399 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1400 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, EVP_CIPHER const *, ENGINE *, unsigned char const *, unsigned char const *, int)
/* 1401 */ _CFFI_OP(_CFFI_OP_NOOP, 1398),
/* 1402 */ _CFFI_OP(_CFFI_OP_NOOP, 479),
/* 1403 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 1404 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1405 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1406 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1407 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1408 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, int)
/* 1409 */ _CFFI_OP(_CFFI_OP_NOOP, 1398),
/* 1410 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1411 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1412 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, int, int, void *)
/* 1413 */ _CFFI_OP(_CFFI_OP_NOOP, 1398),
/* 1414 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1415 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1416 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1417 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1418 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, unsigned char *, int *)
/* 1419 */ _CFFI_OP(_CFFI_OP_NOOP, 1398),
/* 1420 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1421 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1422 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1423 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, unsigned char *, int *, unsigned char const *, int)
/* 1424 */ _CFFI_OP(_CFFI_OP_NOOP, 1398),
/* 1425 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1426 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1427 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1428 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1429 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1430 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD const *)
/* 1431 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1432 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1433 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_MD const *)
/* 1434 */ _CFFI_OP(_CFFI_OP_POINTER, 3316), // EVP_MD_CTX *
/* 1435 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1436 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1437 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_MD const *, ENGINE *)
/* 1438 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1439 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1440 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 1441 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1442 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_MD_CTX const *)
/* 1443 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1444 */ _CFFI_OP(_CFFI_OP_POINTER, 3316), // EVP_MD_CTX const *
/* 1445 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1446 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_PKEY_CTX * *, EVP_MD const *, ENGINE *, EVP_PKEY *)
/* 1447 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1448 */ _CFFI_OP(_CFFI_OP_POINTER, 351), // EVP_PKEY_CTX * *
/* 1449 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1450 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 1451 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1452 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1453 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, size_t *)
/* 1454 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1455 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1456 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1457 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1458 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, size_t *, unsigned char const *, size_t)
/* 1459 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1460 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1461 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1462 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1463 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1464 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1465 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, size_t)
/* 1466 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1467 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1468 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1469 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1470 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, unsigned int *)
/* 1471 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1472 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1473 */ _CFFI_OP(_CFFI_OP_POINTER, 815), // unsigned int *
/* 1474 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1475 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, unsigned int *, EVP_PKEY *)
/* 1476 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1477 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1478 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 1479 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1480 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1481 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char const *, size_t, unsigned char const *, size_t)
/* 1482 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1483 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1484 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1485 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1486 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1487 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1488 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char const *, unsigned int, EVP_PKEY *)
/* 1489 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1490 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1491 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1492 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1493 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1494 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, void const *, size_t)
/* 1495 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 1496 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 1497 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1498 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1499 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX const *)
/* 1500 */ _CFFI_OP(_CFFI_OP_NOOP, 1444),
/* 1501 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1502 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *)
/* 1503 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1504 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1505 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, DH *)
/* 1506 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1507 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 1508 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1509 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, DSA *)
/* 1510 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1511 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 1512 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1513 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, EC_KEY *)
/* 1514 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1515 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 1516 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1517 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, RSA *)
/* 1518 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1519 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 1520 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1521 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, int)
/* 1522 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1523 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1524 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1525 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, unsigned char const *, size_t)
/* 1526 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1527 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1528 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1529 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1530 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY const *, EVP_PKEY const *)
/* 1531 */ _CFFI_OP(_CFFI_OP_POINTER, 3317), // EVP_PKEY const *
/* 1532 */ _CFFI_OP(_CFFI_OP_NOOP, 1531),
/* 1533 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1534 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY const *, unsigned char *, size_t *)
/* 1535 */ _CFFI_OP(_CFFI_OP_NOOP, 1531),
/* 1536 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1537 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1538 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1539 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *)
/* 1540 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1541 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1542 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_MD *)
/* 1543 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1544 */ _CFFI_OP(_CFFI_OP_POINTER, 3315), // EVP_MD *
/* 1545 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1546 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_MD const *)
/* 1547 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1548 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1549 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1550 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_PKEY * *)
/* 1551 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1552 */ _CFFI_OP(_CFFI_OP_NOOP, 321),
/* 1553 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1554 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_PKEY *)
/* 1555 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1556 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1557 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1558 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char *, int)
/* 1559 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1560 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1561 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1562 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1563 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char *, size_t *)
/* 1564 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1565 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1566 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1567 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1568 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char *, size_t *, unsigned char const *, size_t)
/* 1569 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1570 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1571 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1572 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1573 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1574 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1575 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char const *, size_t, unsigned char const *, size_t)
/* 1576 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 1577 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1578 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1579 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1580 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1581 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1582 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(GENERAL_NAMES *)
/* 1583 */ _CFFI_OP(_CFFI_OP_NOOP, 358),
/* 1584 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1585 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(GENERAL_NAMES *, GENERAL_NAME *)
/* 1586 */ _CFFI_OP(_CFFI_OP_NOOP, 358),
/* 1587 */ _CFFI_OP(_CFFI_OP_NOOP, 1006),
/* 1588 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1589 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(GENERAL_NAMES *, unsigned char * *)
/* 1590 */ _CFFI_OP(_CFFI_OP_NOOP, 358),
/* 1591 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 1592 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1593 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, HMAC_CTX *)
/* 1594 */ _CFFI_OP(_CFFI_OP_POINTER, 3323), // HMAC_CTX *
/* 1595 */ _CFFI_OP(_CFFI_OP_NOOP, 1594),
/* 1596 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1597 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, unsigned char *, unsigned int *)
/* 1598 */ _CFFI_OP(_CFFI_OP_NOOP, 1594),
/* 1599 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1600 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 1601 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1602 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, unsigned char const *, size_t)
/* 1603 */ _CFFI_OP(_CFFI_OP_NOOP, 1594),
/* 1604 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1605 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1606 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1607 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, void const *, int, EVP_MD const *, ENGINE *)
/* 1608 */ _CFFI_OP(_CFFI_OP_NOOP, 1594),
/* 1609 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 1610 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1611 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1612 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 1613 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1614 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(NETSCAPE_SPKI *, EVP_PKEY *)
/* 1615 */ _CFFI_OP(_CFFI_OP_NOOP, 330),
/* 1616 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1617 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1618 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(NETSCAPE_SPKI *, EVP_PKEY *, EVP_MD const *)
/* 1619 */ _CFFI_OP(_CFFI_OP_NOOP, 330),
/* 1620 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1621 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1622 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1623 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *)
/* 1624 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 1625 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1626 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *, X509 *)
/* 1627 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 1628 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1629 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1630 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *, X509 *, EVP_PKEY *, EVP_MD const *, Cryptography_STACK_OF_X509 *, unsigned long)
/* 1631 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 1632 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1633 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1634 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 1635 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1636 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1637 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1638 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *, X509_EXTENSION *, int)
/* 1639 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 1640 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1641 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1642 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1643 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *, unsigned char *, int)
/* 1644 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 1645 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1646 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1647 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1648 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP const *, ASN1_OCTET_STRING const * *, X509_NAME const * *)
/* 1649 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1650 */ _CFFI_OP(_CFFI_OP_POINTER, 3262), // ASN1_OCTET_STRING const * *
/* 1651 */ _CFFI_OP(_CFFI_OP_POINTER, 2157), // X509_NAME const * *
/* 1652 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1653 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_ONEREQ *)
/* 1654 */ _CFFI_OP(_CFFI_OP_NOOP, 401),
/* 1655 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1656 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_REQUEST *)
/* 1657 */ _CFFI_OP(_CFFI_OP_NOOP, 407),
/* 1658 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1659 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_REQUEST *, X509_EXTENSION *, int)
/* 1660 */ _CFFI_OP(_CFFI_OP_NOOP, 407),
/* 1661 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1662 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1663 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1664 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_REQUEST *, unsigned char *, int)
/* 1665 */ _CFFI_OP(_CFFI_OP_NOOP, 407),
/* 1666 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1667 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1668 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1669 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_RESPDATA *, unsigned char * *)
/* 1670 */ _CFFI_OP(_CFFI_OP_POINTER, 3335), // OCSP_RESPDATA *
/* 1671 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 1672 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1673 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_RESPONSE *)
/* 1674 */ _CFFI_OP(_CFFI_OP_NOOP, 391),
/* 1675 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1676 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_SINGLERESP *)
/* 1677 */ _CFFI_OP(_CFFI_OP_NOOP, 644),
/* 1678 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1679 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_SINGLERESP *, int *, ASN1_GENERALIZEDTIME * *, ASN1_GENERALIZEDTIME * *, ASN1_GENERALIZEDTIME * *)
/* 1680 */ _CFFI_OP(_CFFI_OP_NOOP, 644),
/* 1681 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1682 */ _CFFI_OP(_CFFI_OP_NOOP, 14),
/* 1683 */ _CFFI_OP(_CFFI_OP_NOOP, 14),
/* 1684 */ _CFFI_OP(_CFFI_OP_NOOP, 14),
/* 1685 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1686 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS12 *, char const *, EVP_PKEY * *, X509 * *, Cryptography_STACK_OF_X509 * *)
/* 1687 */ _CFFI_OP(_CFFI_OP_NOOP, 1018),
/* 1688 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1689 */ _CFFI_OP(_CFFI_OP_NOOP, 321),
/* 1690 */ _CFFI_OP(_CFFI_OP_NOOP, 565),
/* 1691 */ _CFFI_OP(_CFFI_OP_POINTER, 186), // Cryptography_STACK_OF_X509 * *
/* 1692 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1693 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS7 *)
/* 1694 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1695 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1696 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS7 *, Cryptography_STACK_OF_X509 *, X509_STORE *, BIO *, BIO *, int)
/* 1697 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1698 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1699 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 1700 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1701 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1702 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1703 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1704 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS7 *, EVP_PKEY *, X509 *, BIO *, int)
/* 1705 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 1706 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1707 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1708 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 1709 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1710 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1711 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, BIGNUM *, BIGNUM *)
/* 1712 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 1713 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1714 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1715 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1716 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 1717 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 1718 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1719 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1720 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1721 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1722 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, BN_CTX *)
/* 1723 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 1724 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 1725 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1726 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, int, BIGNUM *, BN_GENCB *)
/* 1727 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 1728 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1729 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1730 */ _CFFI_OP(_CFFI_OP_NOOP, 913),
/* 1731 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1732 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA const *)
/* 1733 */ _CFFI_OP(_CFFI_OP_NOOP, 1045),
/* 1734 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1735 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, ct_log_entry_type_t)
/* 1736 */ _CFFI_OP(_CFFI_OP_NOOP, 1184),
/* 1737 */ _CFFI_OP(_CFFI_OP_ENUM, 0), // ct_log_entry_type_t
/* 1738 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1739 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, sct_source_t)
/* 1740 */ _CFFI_OP(_CFFI_OP_NOOP, 1184),
/* 1741 */ _CFFI_OP(_CFFI_OP_ENUM, 2), // sct_source_t
/* 1742 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1743 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, sct_version_t)
/* 1744 */ _CFFI_OP(_CFFI_OP_NOOP, 1184),
/* 1745 */ _CFFI_OP(_CFFI_OP_ENUM, 3), // sct_version_t
/* 1746 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1747 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, unsigned char *, size_t)
/* 1748 */ _CFFI_OP(_CFFI_OP_NOOP, 1184),
/* 1749 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1750 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1751 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1752 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *)
/* 1753 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1754 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1755 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, EVP_PKEY *)
/* 1756 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1757 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1758 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1759 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, SSL_SESSION *)
/* 1760 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1761 */ _CFFI_OP(_CFFI_OP_POINTER, 3366), // SSL_SESSION *
/* 1762 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1763 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, X509 *)
/* 1764 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1765 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1766 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1767 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, X509_VERIFY_PARAM *)
/* 1768 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1769 */ _CFFI_OP(_CFFI_OP_POINTER, 3384), // X509_VERIFY_PARAM *
/* 1770 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1771 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, char *)
/* 1772 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1773 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 1774 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1775 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, char const *)
/* 1776 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1777 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1778 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1779 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, char const *, int)
/* 1780 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1781 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1782 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1783 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1784 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int *, void *)
/* 1785 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1786 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1787 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1788 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1789 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int)
/* 1790 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1791 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1792 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1793 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int, int *, int *, int *, unsigned char *, unsigned char *)
/* 1794 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1795 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1796 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1797 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1798 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1799 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1800 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1801 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1802 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int, void *)
/* 1803 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1804 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1805 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1806 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1807 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, size_t)
/* 1808 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1809 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1810 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1811 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 1812 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1813 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 1814 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1815 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1816 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1817 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1818 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1819 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char *, size_t, char const *, size_t, unsigned char const *, size_t, int)
/* 1820 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1821 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1822 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1823 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1824 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1825 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1826 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1827 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1828 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1829 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char *, unsigned int *)
/* 1830 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1831 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1832 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 1833 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1834 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 1835 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1836 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 1837 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 1838 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1839 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1840 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1841 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1842 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char const * *, unsigned int *, void *)
/* 1843 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1844 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 1845 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 1846 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1847 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1848 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char const *, int)
/* 1849 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1850 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1851 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1852 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1853 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char const *, unsigned int)
/* 1854 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1855 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1856 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1857 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1858 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *)
/* 1859 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1860 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1861 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 1862 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1863 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1864 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1865 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1866 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned int, unsigned char const *, size_t, int *, void *)
/* 1867 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1868 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1869 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1870 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1871 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1872 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1873 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1874 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void *)
/* 1875 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1876 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1877 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1878 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void *, int)
/* 1879 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1880 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1881 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1882 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1883 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void *, size_t, size_t *)
/* 1884 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1885 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1886 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1887 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1888 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1889 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void const *, int)
/* 1890 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1891 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 1892 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1893 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1894 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void const *, size_t, size_t *)
/* 1895 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 1896 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 1897 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1898 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 1899 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1900 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL const *)
/* 1901 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 1902 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1903 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL const *, int)
/* 1904 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 1905 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1906 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1907 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CIPHER const *)
/* 1908 */ _CFFI_OP(_CFFI_OP_NOOP, 771),
/* 1909 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1910 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CIPHER const *, int *)
/* 1911 */ _CFFI_OP(_CFFI_OP_NOOP, 771),
/* 1912 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 1913 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1914 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *)
/* 1915 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1916 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1917 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, EVP_PKEY *)
/* 1918 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1919 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 1920 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1921 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, X509 *)
/* 1922 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1923 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1924 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1925 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, X509_VERIFY_PARAM *)
/* 1926 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1927 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 1928 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1929 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, char const *)
/* 1930 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1931 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1932 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1933 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, char const *, char const *)
/* 1934 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1935 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1936 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1937 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1938 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, char const *, int)
/* 1939 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1940 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 1941 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1942 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1943 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, int)
/* 1944 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1945 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1946 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1947 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, int, unsigned char const *)
/* 1948 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1949 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1950 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1951 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1952 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, int, void *)
/* 1953 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1954 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1955 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1956 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1957 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, size_t)
/* 1958 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1959 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1960 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1961 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, uint32_t)
/* 1962 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1963 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 22), // uint32_t
/* 1964 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1965 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, unsigned char const *, unsigned int)
/* 1966 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1967 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1968 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1969 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1970 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, unsigned int, int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *), void(*)(SSL *, unsigned int, unsigned char const *, void *), void *, int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *), void *)
/* 1971 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 1972 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1973 */ _CFFI_OP(_CFFI_OP_POINTER, 1858), // int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *)
/* 1974 */ _CFFI_OP(_CFFI_OP_POINTER, 2997), // void(*)(SSL *, unsigned int, unsigned char const *, void *)
/* 1975 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1976 */ _CFFI_OP(_CFFI_OP_POINTER, 1866), // int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *)
/* 1977 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 1978 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1979 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX const *)
/* 1980 */ _CFFI_OP(_CFFI_OP_NOOP, 741),
/* 1981 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1982 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_SESSION *, unsigned char const *, unsigned int)
/* 1983 */ _CFFI_OP(_CFFI_OP_NOOP, 1761),
/* 1984 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 1985 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1986 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1987 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_SESSION const *)
/* 1988 */ _CFFI_OP(_CFFI_OP_NOOP, 1054),
/* 1989 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1990 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 * *, X509_STORE_CTX *, X509 *)
/* 1991 */ _CFFI_OP(_CFFI_OP_NOOP, 565),
/* 1992 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1993 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1994 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1995 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *)
/* 1996 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1997 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1998 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, ASN1_INTEGER *)
/* 1999 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2000 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 2001 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2002 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, ASN1_OCTET_STRING *)
/* 2003 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2004 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2005 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2006 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, EVP_PKEY *)
/* 2007 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2008 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2009 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2010 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, EVP_PKEY *, EVP_MD const *)
/* 2011 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2012 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2013 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 2014 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2015 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, X509_EXTENSION *, int)
/* 2016 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2017 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2018 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2019 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2020 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, X509_NAME *)
/* 2021 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2022 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2023 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2024 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, int, int)
/* 2025 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2026 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2027 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2028 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2029 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, long)
/* 2030 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2031 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2032 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2033 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, unsigned char * *)
/* 2034 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2035 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2036 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2037 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 const *)
/* 2038 */ _CFFI_OP(_CFFI_OP_NOOP, 397),
/* 2039 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2040 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 const *, EVP_MD const *, unsigned char *, unsigned int *)
/* 2041 */ _CFFI_OP(_CFFI_OP_NOOP, 397),
/* 2042 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 2043 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2044 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 2045 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2046 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 const *, X509 const *)
/* 2047 */ _CFFI_OP(_CFFI_OP_NOOP, 397),
/* 2048 */ _CFFI_OP(_CFFI_OP_NOOP, 397),
/* 2049 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2050 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *)
/* 2051 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2052 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2053 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, ASN1_OCTET_STRING *)
/* 2054 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2055 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2056 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2057 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, EVP_PKEY *)
/* 2058 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2059 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2060 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2061 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, EVP_PKEY *, EVP_MD const *)
/* 2062 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2063 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2064 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 2065 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2066 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_EXTENSION *, int)
/* 2067 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2068 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2069 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2070 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2071 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_NAME *)
/* 2072 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2073 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2074 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2075 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_REVOKED * *, ASN1_INTEGER *)
/* 2076 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2077 */ _CFFI_OP(_CFFI_OP_POINTER, 669), // X509_REVOKED * *
/* 2078 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 2079 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2080 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_REVOKED *)
/* 2081 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2082 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 2083 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2084 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, long)
/* 2085 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2086 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2087 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2088 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, unsigned char * *)
/* 2089 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2090 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2091 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2092 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL const *, X509_CRL const *)
/* 2093 */ _CFFI_OP(_CFFI_OP_POINTER, 3374), // X509_CRL const *
/* 2094 */ _CFFI_OP(_CFFI_OP_NOOP, 2093),
/* 2095 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2096 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSION *)
/* 2097 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2098 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2099 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSIONS *)
/* 2100 */ _CFFI_OP(_CFFI_OP_NOOP, 665),
/* 2101 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2102 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSIONS *, X509_EXTENSION *)
/* 2103 */ _CFFI_OP(_CFFI_OP_NOOP, 665),
/* 2104 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2105 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2106 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSIONS *, X509_EXTENSION *, int)
/* 2107 */ _CFFI_OP(_CFFI_OP_NOOP, 665),
/* 2108 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2109 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2110 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2111 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *)
/* 2112 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2113 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2114 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, ASN1_OBJECT *, int, unsigned char *, int, int, int)
/* 2115 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2116 */ _CFFI_OP(_CFFI_OP_NOOP, 657),
/* 2117 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2118 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2119 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2120 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2121 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2122 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2123 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, X509_NAME_ENTRY *, int, int)
/* 2124 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2125 */ _CFFI_OP(_CFFI_OP_NOOP, 49),
/* 2126 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2127 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2128 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2129 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, char const *, int, unsigned char const *, int, int, int)
/* 2130 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2131 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2132 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2133 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2134 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2135 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2136 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2137 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2138 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, int, int)
/* 2139 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2140 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2141 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2142 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2143 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, int, int, unsigned char *, int, int, int)
/* 2144 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2145 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2146 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2147 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2148 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2149 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2150 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2151 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2152 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, unsigned char * *)
/* 2153 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2154 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2155 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2156 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME const *, X509_NAME const *)
/* 2157 */ _CFFI_OP(_CFFI_OP_POINTER, 3377), // X509_NAME const *
/* 2158 */ _CFFI_OP(_CFFI_OP_NOOP, 2157),
/* 2159 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2160 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME_ENTRY *)
/* 2161 */ _CFFI_OP(_CFFI_OP_NOOP, 49),
/* 2162 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2163 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_OBJECT const *)
/* 2164 */ _CFFI_OP(_CFFI_OP_POINTER, 3379), // X509_OBJECT const *
/* 2165 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2166 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, EVP_PKEY *)
/* 2167 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 2168 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2169 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2170 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, EVP_PKEY *, EVP_MD const *)
/* 2171 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 2172 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2173 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 2174 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2175 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, X509_EXTENSIONS *)
/* 2176 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 2177 */ _CFFI_OP(_CFFI_OP_NOOP, 665),
/* 2178 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2179 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, X509_NAME *)
/* 2180 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 2181 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 2182 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2183 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, long)
/* 2184 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 2185 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2186 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2187 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, unsigned char * *)
/* 2188 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 2189 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2190 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2191 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *)
/* 2192 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 2193 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2194 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, ASN1_INTEGER *)
/* 2195 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 2196 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 2197 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2198 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, ASN1_OCTET_STRING *)
/* 2199 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 2200 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2201 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2202 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, X509_EXTENSION *, int)
/* 2203 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 2204 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2205 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2206 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2207 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, int, void *, int, unsigned long)
/* 2208 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 2209 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2210 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2211 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2212 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2213 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2214 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *)
/* 2215 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2216 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2217 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, X509 *)
/* 2218 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2219 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2220 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2221 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, X509_CRL *)
/* 2222 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2223 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2224 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2225 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, X509_VERIFY_PARAM *)
/* 2226 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2227 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2228 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2229 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, char const *, char const *)
/* 2230 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2231 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2232 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2233 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2234 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, unsigned long)
/* 2235 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2236 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2237 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2238 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *)
/* 2239 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2240 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2241 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, X509_STORE *, X509 *, Cryptography_STACK_OF_X509 *)
/* 2242 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2243 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2244 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2245 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2246 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2247 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, char const *)
/* 2248 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2249 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2250 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2251 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, int, void *)
/* 2252 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2253 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2254 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2255 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2256 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, void *)
/* 2257 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2258 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2259 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2260 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, ASN1_OBJECT *)
/* 2261 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2262 */ _CFFI_OP(_CFFI_OP_NOOP, 657),
/* 2263 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2264 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, Cryptography_STACK_OF_ASN1_OBJECT *)
/* 2265 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2266 */ _CFFI_OP(_CFFI_OP_NOOP, 42),
/* 2267 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2268 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, char const *)
/* 2269 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2270 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2271 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2272 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, char const *, size_t)
/* 2273 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2274 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2275 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2276 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2277 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, int)
/* 2278 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2279 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2280 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2281 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, unsigned char const *, size_t)
/* 2282 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2283 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2284 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2285 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2286 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, unsigned long)
/* 2287 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2288 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2289 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2290 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM const *)
/* 2291 */ _CFFI_OP(_CFFI_OP_POINTER, 3384), // X509_VERIFY_PARAM const *
/* 2292 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2293 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char *)
/* 2294 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 2295 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2296 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char *, int, ASN1_OBJECT const *, int)
/* 2297 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 2298 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2299 */ _CFFI_OP(_CFFI_OP_NOOP, 39),
/* 2300 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2301 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2302 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char *, int, int, void *)
/* 2303 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 2304 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2305 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2306 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2307 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2308 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *)
/* 2309 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2310 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2311 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *, char const *, char const *)
/* 2312 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2313 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2314 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2315 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2316 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *, int, unsigned char const *, int, int, EVP_MD const *, int, unsigned char *)
/* 2317 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2318 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2319 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2320 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2321 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2322 */ _CFFI_OP(_CFFI_OP_NOOP, 396),
/* 2323 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2324 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2325 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2326 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *, int, unsigned char const *, int, int, int, unsigned char *)
/* 2327 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2328 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2329 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2330 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2331 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2332 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2333 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2334 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2335 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *, size_t, unsigned char const *, size_t, uint64_t, uint64_t, uint64_t, uint64_t, unsigned char *, size_t)
/* 2336 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2337 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2338 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2339 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2340 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24), // uint64_t
/* 2341 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2342 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2343 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2344 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2345 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2346 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2347 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int)
/* 2348 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2349 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2350 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, SSL *, unsigned char const *, long)
/* 2351 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2352 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2353 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2354 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2355 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2356 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, SSL_CTX *, unsigned char const *, long)
/* 2357 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2358 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2359 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2360 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2361 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2362 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, X509_STORE_CTX *)
/* 2363 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2364 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2365 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2366 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, int)
/* 2367 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2368 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2369 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2370 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char *, unsigned int *, DSA *)
/* 2371 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2372 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2373 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2374 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2375 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 2376 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 2377 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2378 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char *, unsigned int *, EC_KEY *)
/* 2379 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2380 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2381 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2382 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2383 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 2384 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 2385 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2386 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char const *, int, DSA *)
/* 2387 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2388 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2389 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2390 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2391 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2392 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 2393 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2394 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char const *, int, EC_KEY *)
/* 2395 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2396 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2397 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2398 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2399 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2400 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 2401 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2402 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, unsigned char *, RSA *, int)
/* 2403 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2404 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2405 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2406 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 2407 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2408 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2409 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(long, void *, CRYPTO_EX_new *, CRYPTO_EX_dup *, CRYPTO_EX_free *)
/* 2410 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2411 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2412 */ _CFFI_OP(_CFFI_OP_POINTER, 3282), // CRYPTO_EX_new *
/* 2413 */ _CFFI_OP(_CFFI_OP_POINTER, 3280), // CRYPTO_EX_dup *
/* 2414 */ _CFFI_OP(_CFFI_OP_POINTER, 3281), // CRYPTO_EX_free *
/* 2415 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2416 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char * *, ASN1_OCTET_STRING *)
/* 2417 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2418 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2419 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2420 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char * *, unsigned char *, unsigned char const *, unsigned int, unsigned char const *, unsigned int)
/* 2421 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2422 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2423 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2424 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2425 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2426 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2427 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2428 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char *, BIGNUM const *, DH *)
/* 2429 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2430 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 2431 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 2432 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2433 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char *, int)
/* 2434 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2435 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2436 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2437 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char const *, int, AES_KEY *)
/* 2438 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2439 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2440 */ _CFFI_OP(_CFFI_OP_NOOP, 811),
/* 2441 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2442 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char const *, int, ECDSA_SIG const *, EC_KEY *)
/* 2443 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 2444 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2445 */ _CFFI_OP(_CFFI_OP_NOOP, 1280),
/* 2446 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 2447 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2448 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned int)
/* 2449 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2450 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2451 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned long)
/* 2452 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2453 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2454 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(void *(*)(size_t, char const *, int), void *(*)(void *, size_t, char const *, int), void(*)(void *, char const *, int))
/* 2455 */ _CFFI_OP(_CFFI_OP_POINTER, 2710), // void *(*)(size_t, char const *, int)
/* 2456 */ _CFFI_OP(_CFFI_OP_POINTER, 2715), // void *(*)(void *, size_t, char const *, int)
/* 2457 */ _CFFI_OP(_CFFI_OP_POINTER, 3230), // void(*)(void *, char const *, int)
/* 2458 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2459 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(void *, size_t, EC_POINT const *, EC_KEY *, void *(*)(void const *, size_t, void *, size_t *))
/* 2460 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2461 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2462 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 2463 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 2464 */ _CFFI_OP(_CFFI_OP_POINTER, 2726), // void *(*)(void const *, size_t, void *, size_t *)
/* 2465 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2466 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(void)
/* 2467 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2468 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2995), // int(*()(SSL const *))(int, X509_STORE_CTX *)
/* 2469 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 2470 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2471 */ _CFFI_OP(_CFFI_OP_FUNCTION, 228), // int(*()(SSL_CTX *))(char *, int, int, void *)
/* 2472 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2473 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2474 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2995), // int(*()(SSL_CTX const *))(int, X509_STORE_CTX *)
/* 2475 */ _CFFI_OP(_CFFI_OP_NOOP, 741),
/* 2476 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2477 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3155), // int(*()(X509_STORE *))(X509 * *, X509_STORE_CTX *, X509 *)
/* 2478 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 2479 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2480 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(ASN1_ENUMERATED *)
/* 2481 */ _CFFI_OP(_CFFI_OP_NOOP, 818),
/* 2482 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2483 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(BIO *, char * *)
/* 2484 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2485 */ _CFFI_OP(_CFFI_OP_POINTER, 451), // char * *
/* 2486 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2487 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(BIO *, char *)
/* 2488 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2489 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 2490 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2491 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(BIO *, int)
/* 2492 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2493 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2494 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2495 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(BIO *, long)
/* 2496 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2497 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2498 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2499 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL *)
/* 2500 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2501 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2502 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL *, EVP_PKEY * *)
/* 2503 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2504 */ _CFFI_OP(_CFFI_OP_NOOP, 321),
/* 2505 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2506 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL *, long)
/* 2507 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2508 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2509 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2510 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL *, time_t *, long *)
/* 2511 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2512 */ _CFFI_OP(_CFFI_OP_POINTER, 10), // time_t *
/* 2513 */ _CFFI_OP(_CFFI_OP_POINTER, 60), // long *
/* 2514 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2515 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL *, unsigned char *, int)
/* 2516 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2517 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2518 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2519 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2520 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL *, unsigned char const * *)
/* 2521 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2522 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 2523 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2524 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL const *)
/* 2525 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 2526 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2527 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_CTX *)
/* 2528 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2529 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2530 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_CTX *, char const *)
/* 2531 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2532 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2533 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2534 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_CTX *, int(*)(SSL *, void *))
/* 2535 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2536 */ _CFFI_OP(_CFFI_OP_POINTER, 1874), // int(*)(SSL *, void *)
/* 2537 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2538 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_CTX *, int)
/* 2539 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2540 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2541 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2542 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_CTX *, long)
/* 2543 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2544 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2545 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2546 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_CTX *, void *)
/* 2547 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2548 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2549 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2550 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_CTX const *)
/* 2551 */ _CFFI_OP(_CFFI_OP_NOOP, 741),
/* 2552 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2553 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(SSL_SESSION const *)
/* 2554 */ _CFFI_OP(_CFFI_OP_NOOP, 1054),
/* 2555 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2556 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(X509 *)
/* 2557 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2558 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2559 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(X509_CRL *)
/* 2560 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 2561 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2562 */ _CFFI_OP(_CFFI_OP_FUNCTION, 60), // long()(X509_REQ *)
/* 2563 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 2564 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2565 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1745), // sct_version_t()(SCT const *)
/* 2566 */ _CFFI_OP(_CFFI_OP_NOOP, 808),
/* 2567 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2568 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(BIO *)
/* 2569 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2570 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2571 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(EC_GROUP const *, EC_POINT const *, point_conversion_form_t, unsigned char *, size_t, BN_CTX *)
/* 2572 */ _CFFI_OP(_CFFI_OP_NOOP, 286),
/* 2573 */ _CFFI_OP(_CFFI_OP_NOOP, 292),
/* 2574 */ _CFFI_OP(_CFFI_OP_ENUM, 1), // point_conversion_form_t
/* 2575 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2576 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2577 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 2578 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2579 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(EC_builtin_curve *, size_t)
/* 2580 */ _CFFI_OP(_CFFI_OP_POINTER, 3309), // EC_builtin_curve *
/* 2581 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2582 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2583 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(EVP_PKEY *, unsigned char * *)
/* 2584 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2585 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2586 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2587 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(SCT const *, unsigned char * *)
/* 2588 */ _CFFI_OP(_CFFI_OP_NOOP, 808),
/* 2589 */ _CFFI_OP(_CFFI_OP_NOOP, 867),
/* 2590 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2591 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(SSL const *)
/* 2592 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 2593 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2594 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(SSL const *, unsigned char *, size_t)
/* 2595 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 2596 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2597 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2598 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2599 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(SSL const *, void *, size_t)
/* 2600 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 2601 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2602 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2603 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2604 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(SSL_CTX const *)
/* 2605 */ _CFFI_OP(_CFFI_OP_NOOP, 741),
/* 2606 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2607 */ _CFFI_OP(_CFFI_OP_FUNCTION, 342), // size_t()(SSL_SESSION const *, unsigned char *, size_t)
/* 2608 */ _CFFI_OP(_CFFI_OP_NOOP, 1054),
/* 2609 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2610 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2611 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2612 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1963), // uint32_t()(SSL_SESSION const *)
/* 2613 */ _CFFI_OP(_CFFI_OP_NOOP, 1054),
/* 2614 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2615 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2340), // uint64_t()(SCT const *)
/* 2616 */ _CFFI_OP(_CFFI_OP_NOOP, 808),
/* 2617 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2618 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2340), // uint64_t()(SSL_CIPHER const *)
/* 2619 */ _CFFI_OP(_CFFI_OP_NOOP, 771),
/* 2620 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2621 */ _CFFI_OP(_CFFI_OP_FUNCTION, 813), // unsigned char *()(ASN1_OCTET_STRING *)
/* 2622 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2623 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2624 */ _CFFI_OP(_CFFI_OP_FUNCTION, 813), // unsigned char *()(X509 *, int *)
/* 2625 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2626 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 2627 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2628 */ _CFFI_OP(_CFFI_OP_FUNCTION, 121), // unsigned char const *()(SSL_SESSION const *, unsigned int *)
/* 2629 */ _CFFI_OP(_CFFI_OP_NOOP, 1054),
/* 2630 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 2631 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2632 */ _CFFI_OP(_CFFI_OP_FUNCTION, 815), // unsigned int()(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int)
/* 2633 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2634 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2635 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 2636 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2637 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2638 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2639 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2640 */ _CFFI_OP(_CFFI_OP_FUNCTION, 815), // unsigned int()(SSL *, char const *, unsigned char *, unsigned int)
/* 2641 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2642 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2643 */ _CFFI_OP(_CFFI_OP_NOOP, 813),
/* 2644 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2645 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2646 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(SSL *)
/* 2647 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2648 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2649 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(SSL *, unsigned long)
/* 2650 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2651 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2652 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2653 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(SSL_CTX *)
/* 2654 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2655 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2656 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(SSL_CTX *, DH *)
/* 2657 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2658 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 2659 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2660 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(SSL_CTX *, EC_KEY *)
/* 2661 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2662 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 2663 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2664 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(SSL_CTX *, X509 *)
/* 2665 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2666 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2667 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2668 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(SSL_CTX *, unsigned long)
/* 2669 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2670 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2671 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2672 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(X509 *)
/* 2673 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2674 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2675 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(X509_VERIFY_PARAM *)
/* 2676 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 2677 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2678 */ _CFFI_OP(_CFFI_OP_FUNCTION, 803), // unsigned long()(void)
/* 2679 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2680 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(SSL const *, int)
/* 2681 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 2682 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2683 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2684 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(SSL_CTX *)
/* 2685 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 2686 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2687 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(SSL_CTX const *, int)
/* 2688 */ _CFFI_OP(_CFFI_OP_NOOP, 741),
/* 2689 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2690 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2691 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(X509 *, int, int *, int *)
/* 2692 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2693 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2694 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 2695 */ _CFFI_OP(_CFFI_OP_NOOP, 1251),
/* 2696 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2697 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(X509V3_CTX *)
/* 2698 */ _CFFI_OP(_CFFI_OP_NOOP, 621),
/* 2699 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2700 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(X509_EXTENSION *)
/* 2701 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2702 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2703 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(X509_STORE_CTX *, int)
/* 2704 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2705 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2706 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2707 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(size_t)
/* 2708 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2709 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2710 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(size_t, char const *, int)
/* 2711 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2712 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2713 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2714 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2715 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(void *, size_t, char const *, int)
/* 2716 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2717 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2718 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 2719 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2720 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2721 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(void *, unsigned char const * *, long)
/* 2722 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2723 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 2724 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2725 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2726 */ _CFFI_OP(_CFFI_OP_FUNCTION, 229), // void *()(void const *, size_t, void *, size_t *)
/* 2727 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 2728 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2729 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2730 */ _CFFI_OP(_CFFI_OP_NOOP, 1119),
/* 2731 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2732 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ACCESS_DESCRIPTION *)
/* 2733 */ _CFFI_OP(_CFFI_OP_NOOP, 1138),
/* 2734 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2735 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ASN1_ENUMERATED *)
/* 2736 */ _CFFI_OP(_CFFI_OP_NOOP, 818),
/* 2737 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2738 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ASN1_GENERALIZEDTIME *)
/* 2739 */ _CFFI_OP(_CFFI_OP_NOOP, 9),
/* 2740 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2741 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ASN1_INTEGER *)
/* 2742 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 2743 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2744 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ASN1_OBJECT *)
/* 2745 */ _CFFI_OP(_CFFI_OP_NOOP, 657),
/* 2746 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2747 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ASN1_OCTET_STRING *)
/* 2748 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2749 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2750 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ASN1_OCTET_STRING const * *, X509_ALGOR const * *, X509 const *)
/* 2751 */ _CFFI_OP(_CFFI_OP_NOOP, 1650),
/* 2752 */ _CFFI_OP(_CFFI_OP_POINTER, 3372), // X509_ALGOR const * *
/* 2753 */ _CFFI_OP(_CFFI_OP_NOOP, 397),
/* 2754 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2755 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(AUTHORITY_KEYID *)
/* 2756 */ _CFFI_OP(_CFFI_OP_POINTER, 3267), // AUTHORITY_KEYID *
/* 2757 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2758 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(BASIC_CONSTRAINTS *)
/* 2759 */ _CFFI_OP(_CFFI_OP_POINTER, 3268), // BASIC_CONSTRAINTS *
/* 2760 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2761 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(BIGNUM *)
/* 2762 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 2763 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2764 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(BIGNUM *, int)
/* 2765 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 2766 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2767 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2768 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(BIO *)
/* 2769 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2770 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2771 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(BN_CTX *)
/* 2772 */ _CFFI_OP(_CFFI_OP_NOOP, 112),
/* 2773 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2774 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(BN_MONT_CTX *)
/* 2775 */ _CFFI_OP(_CFFI_OP_NOOP, 895),
/* 2776 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2777 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(CMAC_CTX *)
/* 2778 */ _CFFI_OP(_CFFI_OP_NOOP, 1113),
/* 2779 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2780 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *)
/* 2781 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 2782 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2783 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *, void(*)(ACCESS_DESCRIPTION *))
/* 2784 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 2785 */ _CFFI_OP(_CFFI_OP_POINTER, 2732), // void(*)(ACCESS_DESCRIPTION *)
/* 2786 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2787 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_ASN1_INTEGER *)
/* 2788 */ _CFFI_OP(_CFFI_OP_NOOP, 24),
/* 2789 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2790 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_ASN1_OBJECT *)
/* 2791 */ _CFFI_OP(_CFFI_OP_NOOP, 42),
/* 2792 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2793 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_DIST_POINT *)
/* 2794 */ _CFFI_OP(_CFFI_OP_NOOP, 240),
/* 2795 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2796 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_DIST_POINT *, void(*)(DIST_POINT *))
/* 2797 */ _CFFI_OP(_CFFI_OP_NOOP, 240),
/* 2798 */ _CFFI_OP(_CFFI_OP_POINTER, 2839), // void(*)(DIST_POINT *)
/* 2799 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2800 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_GENERAL_SUBTREE *)
/* 2801 */ _CFFI_OP(_CFFI_OP_NOOP, 371),
/* 2802 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2803 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_POLICYINFO *)
/* 2804 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2805 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2806 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_POLICYINFO *, void(*)(POLICYINFO *))
/* 2807 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2808 */ _CFFI_OP(_CFFI_OP_POINTER, 2944), // void(*)(POLICYINFO *)
/* 2809 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2810 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_POLICYQUALINFO *)
/* 2811 */ _CFFI_OP(_CFFI_OP_NOOP, 496),
/* 2812 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2813 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_SCT *)
/* 2814 */ _CFFI_OP(_CFFI_OP_NOOP, 1183),
/* 2815 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2816 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_X509 *)
/* 2817 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2818 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2819 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_X509_CRL *)
/* 2820 */ _CFFI_OP(_CFFI_OP_NOOP, 611),
/* 2821 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2822 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(Cryptography_STACK_OF_X509_NAME *)
/* 2823 */ _CFFI_OP(_CFFI_OP_NOOP, 683),
/* 2824 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2825 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DH *)
/* 2826 */ _CFFI_OP(_CFFI_OP_NOOP, 232),
/* 2827 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2828 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DH const *, BIGNUM const * *, BIGNUM const * *)
/* 2829 */ _CFFI_OP(_CFFI_OP_NOOP, 1247),
/* 2830 */ _CFFI_OP(_CFFI_OP_POINTER, 110), // BIGNUM const * *
/* 2831 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2832 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2833 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DH const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2834 */ _CFFI_OP(_CFFI_OP_NOOP, 1247),
/* 2835 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2836 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2837 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2838 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2839 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DIST_POINT *)
/* 2840 */ _CFFI_OP(_CFFI_OP_NOOP, 1159),
/* 2841 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2842 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DIST_POINT_NAME *)
/* 2843 */ _CFFI_OP(_CFFI_OP_POINTER, 3302), // DIST_POINT_NAME *
/* 2844 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2845 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DSA *)
/* 2846 */ _CFFI_OP(_CFFI_OP_NOOP, 248),
/* 2847 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2848 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DSA const *, BIGNUM const * *, BIGNUM const * *)
/* 2849 */ _CFFI_OP(_CFFI_OP_NOOP, 1277),
/* 2850 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2851 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2852 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2853 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(DSA const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2854 */ _CFFI_OP(_CFFI_OP_NOOP, 1277),
/* 2855 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2856 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2857 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2858 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2859 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ECDSA_SIG *)
/* 2860 */ _CFFI_OP(_CFFI_OP_POINTER, 3304), // ECDSA_SIG *
/* 2861 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2862 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EC_GROUP *)
/* 2863 */ _CFFI_OP(_CFFI_OP_POINTER, 3305), // EC_GROUP *
/* 2864 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2865 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EC_KEY *)
/* 2866 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 2867 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2868 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EC_KEY *, int)
/* 2869 */ _CFFI_OP(_CFFI_OP_NOOP, 263),
/* 2870 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2871 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2872 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EC_POINT *)
/* 2873 */ _CFFI_OP(_CFFI_OP_NOOP, 1293),
/* 2874 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2875 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ENGINE *)
/* 2876 */ _CFFI_OP(_CFFI_OP_NOOP, 340),
/* 2877 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2878 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EVP_CIPHER_CTX *)
/* 2879 */ _CFFI_OP(_CFFI_OP_NOOP, 1398),
/* 2880 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2881 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EVP_MD_CTX *)
/* 2882 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 2883 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2884 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EVP_MD_CTX *, int)
/* 2885 */ _CFFI_OP(_CFFI_OP_NOOP, 1434),
/* 2886 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2887 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2888 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EVP_PKEY *)
/* 2889 */ _CFFI_OP(_CFFI_OP_NOOP, 235),
/* 2890 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2891 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(EVP_PKEY_CTX *)
/* 2892 */ _CFFI_OP(_CFFI_OP_NOOP, 351),
/* 2893 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2894 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(GENERAL_NAME *)
/* 2895 */ _CFFI_OP(_CFFI_OP_NOOP, 1006),
/* 2896 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2897 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(GENERAL_NAMES *)
/* 2898 */ _CFFI_OP(_CFFI_OP_NOOP, 358),
/* 2899 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2900 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(GENERAL_NAMES *, void(*)(GENERAL_NAME *))
/* 2901 */ _CFFI_OP(_CFFI_OP_NOOP, 358),
/* 2902 */ _CFFI_OP(_CFFI_OP_POINTER, 2894), // void(*)(GENERAL_NAME *)
/* 2903 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2904 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(HMAC_CTX *)
/* 2905 */ _CFFI_OP(_CFFI_OP_NOOP, 1594),
/* 2906 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2907 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(ISSUING_DIST_POINT *)
/* 2908 */ _CFFI_OP(_CFFI_OP_POINTER, 3324), // ISSUING_DIST_POINT *
/* 2909 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2910 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(NAME_CONSTRAINTS *)
/* 2911 */ _CFFI_OP(_CFFI_OP_POINTER, 3325), // NAME_CONSTRAINTS *
/* 2912 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2913 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(NETSCAPE_SPKI *)
/* 2914 */ _CFFI_OP(_CFFI_OP_NOOP, 330),
/* 2915 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2916 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(NOTICEREF *)
/* 2917 */ _CFFI_OP(_CFFI_OP_POINTER, 3327), // NOTICEREF *
/* 2918 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2919 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(OBJ_NAME const *, void *)
/* 2920 */ _CFFI_OP(_CFFI_OP_POINTER, 3328), // OBJ_NAME const *
/* 2921 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2922 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2923 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(OCSP_BASICRESP *)
/* 2924 */ _CFFI_OP(_CFFI_OP_NOOP, 429),
/* 2925 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2926 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(OCSP_CERTID *)
/* 2927 */ _CFFI_OP(_CFFI_OP_NOOP, 408),
/* 2928 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2929 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(OCSP_REQUEST *)
/* 2930 */ _CFFI_OP(_CFFI_OP_NOOP, 407),
/* 2931 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2932 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(OCSP_RESPONSE *)
/* 2933 */ _CFFI_OP(_CFFI_OP_NOOP, 391),
/* 2934 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2935 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(OTHERNAME *)
/* 2936 */ _CFFI_OP(_CFFI_OP_POINTER, 3340), // OTHERNAME *
/* 2937 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2938 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(PKCS12 *)
/* 2939 */ _CFFI_OP(_CFFI_OP_NOOP, 1018),
/* 2940 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2941 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(PKCS7 *)
/* 2942 */ _CFFI_OP(_CFFI_OP_NOOP, 136),
/* 2943 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2944 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(POLICYINFO *)
/* 2945 */ _CFFI_OP(_CFFI_OP_NOOP, 1173),
/* 2946 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2947 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(POLICYQUALINFO *)
/* 2948 */ _CFFI_OP(_CFFI_OP_NOOP, 1180),
/* 2949 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2950 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(POLICY_CONSTRAINTS *)
/* 2951 */ _CFFI_OP(_CFFI_OP_POINTER, 3356), // POLICY_CONSTRAINTS *
/* 2952 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2953 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(RSA *)
/* 2954 */ _CFFI_OP(_CFFI_OP_NOOP, 517),
/* 2955 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2956 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(RSA const *, BIGNUM const * *, BIGNUM const * *)
/* 2957 */ _CFFI_OP(_CFFI_OP_NOOP, 1045),
/* 2958 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2959 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2960 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2961 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(RSA const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2962 */ _CFFI_OP(_CFFI_OP_NOOP, 1045),
/* 2963 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2964 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2965 */ _CFFI_OP(_CFFI_OP_NOOP, 2830),
/* 2966 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2967 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SCT *, uint64_t)
/* 2968 */ _CFFI_OP(_CFFI_OP_NOOP, 1184),
/* 2969 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2970 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2971 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *)
/* 2972 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2973 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2974 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, BIO *, BIO *)
/* 2975 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2976 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2977 */ _CFFI_OP(_CFFI_OP_NOOP, 137),
/* 2978 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2979 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, char *)
/* 2980 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2981 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 2982 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2983 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, int(*)(SSL *, void *), void *)
/* 2984 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2985 */ _CFFI_OP(_CFFI_OP_NOOP, 2536),
/* 2986 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 2987 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2988 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, int)
/* 2989 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2990 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2991 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2992 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, int, int(*)(int, X509_STORE_CTX *))
/* 2993 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2994 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2995 */ _CFFI_OP(_CFFI_OP_POINTER, 2362), // int(*)(int, X509_STORE_CTX *)
/* 2996 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2997 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, unsigned int, unsigned char const *, void *)
/* 2998 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 2999 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 3000 */ _CFFI_OP(_CFFI_OP_NOOP, 121),
/* 3001 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3002 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3003 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, void(*)(SSL const *, int, int))
/* 3004 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 3005 */ _CFFI_OP(_CFFI_OP_POINTER, 3011), // void(*)(SSL const *, int, int)
/* 3006 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3007 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL *, void(*)(int, int, int, void const *, size_t, SSL *, void *))
/* 3008 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 3009 */ _CFFI_OP(_CFFI_OP_POINTER, 3208), // void(*)(int, int, int, void const *, size_t, SSL *, void *)
/* 3010 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3011 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL const *, int, int)
/* 3012 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 3013 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3014 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3015 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3016 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL const *, unsigned char const * *, unsigned int *)
/* 3017 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 3018 */ _CFFI_OP(_CFFI_OP_NOOP, 91),
/* 3019 */ _CFFI_OP(_CFFI_OP_NOOP, 1473),
/* 3020 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3021 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *)
/* 3022 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3023 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3024 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, Cryptography_STACK_OF_X509_NAME *)
/* 3025 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3026 */ _CFFI_OP(_CFFI_OP_NOOP, 683),
/* 3027 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3028 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, X509_STORE *)
/* 3029 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3030 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 3031 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3032 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(SSL *, int *, void *))
/* 3033 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3034 */ _CFFI_OP(_CFFI_OP_POINTER, 1784), // int(*)(SSL *, int *, void *)
/* 3035 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3036 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(SSL *, unsigned char * *, unsigned char *, unsigned char const *, unsigned int, void *), void *)
/* 3037 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3038 */ _CFFI_OP(_CFFI_OP_POINTER, 1811), // int(*)(SSL *, unsigned char * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 3039 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3040 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3041 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(SSL *, unsigned char *, unsigned int *))
/* 3042 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3043 */ _CFFI_OP(_CFFI_OP_POINTER, 1829), // int(*)(SSL *, unsigned char *, unsigned int *)
/* 3044 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3045 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *), void *)
/* 3046 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3047 */ _CFFI_OP(_CFFI_OP_POINTER, 1834), // int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 3048 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3049 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3050 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(SSL *, unsigned char const * *, unsigned int *, void *), void *)
/* 3051 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3052 */ _CFFI_OP(_CFFI_OP_POINTER, 1842), // int(*)(SSL *, unsigned char const * *, unsigned int *, void *)
/* 3053 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3054 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3055 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(SSL *, void *), void *)
/* 3056 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3057 */ _CFFI_OP(_CFFI_OP_NOOP, 2536),
/* 3058 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3059 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3060 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(X509_STORE_CTX *, void *), void *)
/* 3061 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3062 */ _CFFI_OP(_CFFI_OP_POINTER, 2256), // int(*)(X509_STORE_CTX *, void *)
/* 3063 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3064 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3065 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int(*)(char *, int, int, void *))
/* 3066 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3067 */ _CFFI_OP(_CFFI_OP_NOOP, 228),
/* 3068 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3069 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int)
/* 3070 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3071 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3072 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3073 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, int, int(*)(int, X509_STORE_CTX *))
/* 3074 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3075 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3076 */ _CFFI_OP(_CFFI_OP_NOOP, 2995),
/* 3077 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3078 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int))
/* 3079 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3080 */ _CFFI_OP(_CFFI_OP_POINTER, 2632), // unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int)
/* 3081 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3082 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int))
/* 3083 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3084 */ _CFFI_OP(_CFFI_OP_POINTER, 2640), // unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int)
/* 3085 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3086 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, void *)
/* 3087 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3088 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3089 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3090 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, void(*)(SSL const *, int, int))
/* 3091 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3092 */ _CFFI_OP(_CFFI_OP_NOOP, 3005),
/* 3093 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3094 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_CTX *, void(*)(int, int, int, void const *, size_t, SSL *, void *))
/* 3095 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3096 */ _CFFI_OP(_CFFI_OP_NOOP, 3009),
/* 3097 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3098 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(SSL_SESSION *)
/* 3099 */ _CFFI_OP(_CFFI_OP_NOOP, 1761),
/* 3100 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3101 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(USERNOTICE *)
/* 3102 */ _CFFI_OP(_CFFI_OP_POINTER, 3367), // USERNOTICE *
/* 3103 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3104 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509 *)
/* 3105 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 3106 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3107 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509V3_CTX *, X509 *, X509 *, X509_REQ *, X509_CRL *, int)
/* 3108 */ _CFFI_OP(_CFFI_OP_NOOP, 621),
/* 3109 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 3110 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 3111 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 3112 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 3113 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3114 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3115 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_CRL *)
/* 3116 */ _CFFI_OP(_CFFI_OP_NOOP, 70),
/* 3117 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3118 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_CRL const *, ASN1_OCTET_STRING const * *, X509_ALGOR const * *)
/* 3119 */ _CFFI_OP(_CFFI_OP_NOOP, 2093),
/* 3120 */ _CFFI_OP(_CFFI_OP_NOOP, 1650),
/* 3121 */ _CFFI_OP(_CFFI_OP_NOOP, 2752),
/* 3122 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3123 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_EXTENSION *)
/* 3124 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 3125 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3126 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_EXTENSIONS *)
/* 3127 */ _CFFI_OP(_CFFI_OP_NOOP, 665),
/* 3128 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3129 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_EXTENSIONS *, void(*)(X509_EXTENSION *))
/* 3130 */ _CFFI_OP(_CFFI_OP_NOOP, 665),
/* 3131 */ _CFFI_OP(_CFFI_OP_POINTER, 3123), // void(*)(X509_EXTENSION *)
/* 3132 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3133 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_NAME *)
/* 3134 */ _CFFI_OP(_CFFI_OP_NOOP, 693),
/* 3135 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3136 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_NAME_ENTRY *)
/* 3137 */ _CFFI_OP(_CFFI_OP_NOOP, 49),
/* 3138 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3139 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_REQ *)
/* 3140 */ _CFFI_OP(_CFFI_OP_NOOP, 336),
/* 3141 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3142 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_REQ const *, ASN1_OCTET_STRING const * *, X509_ALGOR const * *)
/* 3143 */ _CFFI_OP(_CFFI_OP_POINTER, 3380), // X509_REQ const *
/* 3144 */ _CFFI_OP(_CFFI_OP_NOOP, 1650),
/* 3145 */ _CFFI_OP(_CFFI_OP_NOOP, 2752),
/* 3146 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3147 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_REVOKED *)
/* 3148 */ _CFFI_OP(_CFFI_OP_NOOP, 669),
/* 3149 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3150 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE *)
/* 3151 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 3152 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3153 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE *, int(*)(X509 * *, X509_STORE_CTX *, X509 *))
/* 3154 */ _CFFI_OP(_CFFI_OP_NOOP, 216),
/* 3155 */ _CFFI_OP(_CFFI_OP_POINTER, 1990), // int(*)(X509 * *, X509_STORE_CTX *, X509 *)
/* 3156 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3157 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE_CTX *)
/* 3158 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 3159 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3160 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE_CTX *, Cryptography_STACK_OF_X509 *)
/* 3161 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 3162 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 3163 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3164 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE_CTX *, Cryptography_STACK_OF_X509_CRL *)
/* 3165 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 3166 */ _CFFI_OP(_CFFI_OP_NOOP, 611),
/* 3167 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3168 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE_CTX *, X509 *)
/* 3169 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 3170 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 3171 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3172 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE_CTX *, X509_VERIFY_PARAM *)
/* 3173 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 3174 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 3175 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3176 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE_CTX *, int(*)(int, X509_STORE_CTX *))
/* 3177 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 3178 */ _CFFI_OP(_CFFI_OP_NOOP, 2995),
/* 3179 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3180 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_STORE_CTX *, int)
/* 3181 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 3182 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3183 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3184 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_VERIFY_PARAM *)
/* 3185 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 3186 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3187 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_VERIFY_PARAM *, int)
/* 3188 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 3189 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3190 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3191 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_VERIFY_PARAM *, time_t)
/* 3192 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 3193 */ _CFFI_OP(_CFFI_OP_NOOP, 10),
/* 3194 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3195 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(X509_VERIFY_PARAM *, unsigned int)
/* 3196 */ _CFFI_OP(_CFFI_OP_NOOP, 1769),
/* 3197 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 3198 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3199 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(char const *)
/* 3200 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 3201 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3202 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(int, int, char const *, int)
/* 3203 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3204 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3205 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 3206 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3207 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3208 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(int, int, int, void const *, size_t, SSL *, void *)
/* 3209 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3210 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3211 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3212 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 3213 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 3214 */ _CFFI_OP(_CFFI_OP_NOOP, 163),
/* 3215 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3216 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3217 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(int, void(*)(OBJ_NAME const *, void *), void *)
/* 3218 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3219 */ _CFFI_OP(_CFFI_OP_POINTER, 2919), // void(*)(OBJ_NAME const *, void *)
/* 3220 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3221 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3222 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(unsigned long, char *, size_t)
/* 3223 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 3224 */ _CFFI_OP(_CFFI_OP_NOOP, 451),
/* 3225 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 3226 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3227 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(void *)
/* 3228 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3229 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3230 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(void *, char const *, int)
/* 3231 */ _CFFI_OP(_CFFI_OP_NOOP, 229),
/* 3232 */ _CFFI_OP(_CFFI_OP_NOOP, 52),
/* 3233 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3234 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3235 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(void const *, int, double)
/* 3236 */ _CFFI_OP(_CFFI_OP_NOOP, 151),
/* 3237 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 3238 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 14), // double
/* 3239 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3240 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3393), // void()(void)
/* 3241 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3242 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3005), // void(*()(SSL const *))(SSL const *, int, int)
/* 3243 */ _CFFI_OP(_CFFI_OP_NOOP, 140),
/* 3244 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3245 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3005), // void(*()(SSL_CTX *))(SSL const *, int, int)
/* 3246 */ _CFFI_OP(_CFFI_OP_NOOP, 531),
/* 3247 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3248 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3392), // void(*()(void))(int, int, char const *, int)
/* 3249 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3250 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 4), // ACCESS_DESCRIPTION
/* 3251 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 5), // AES_KEY
/* 3252 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 6), // ASN1_ENUMERATED
/* 3253 */ _CFFI_OP(_CFFI_OP_POINTER, 3254), // ASN1_GENERALIZEDTIME const *
/* 3254 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 7), // ASN1_GENERALIZEDTIME
/* 3255 */ _CFFI_OP(_CFFI_OP_POINTER, 3256), // ASN1_INTEGER const *
/* 3256 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 8), // ASN1_INTEGER
/* 3257 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 9), // ASN1_ITEM
/* 3258 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 10), // ASN1_ITEM_EXP
/* 3259 */ _CFFI_OP(_CFFI_OP_POINTER, 3260), // ASN1_NULL *
/* 3260 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 11), // ASN1_NULL
/* 3261 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 12), // ASN1_OBJECT
/* 3262 */ _CFFI_OP(_CFFI_OP_POINTER, 3263), // ASN1_OCTET_STRING const *
/* 3263 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 118), // ASN1_OCTET_STRING
/* 3264 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 13), // ASN1_TYPE
/* 3265 */ _CFFI_OP(_CFFI_OP_POINTER, 3266), // ASN1_VALUE *
/* 3266 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 14), // ASN1_VALUE
/* 3267 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 15), // AUTHORITY_KEYID
/* 3268 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 16), // BASIC_CONSTRAINTS
/* 3269 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 17), // BIGNUM
/* 3270 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 18), // BIO
/* 3271 */ _CFFI_OP(_CFFI_OP_POINTER, 3272), // BIO_METHOD const *
/* 3272 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 19), // BIO_METHOD
/* 3273 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 20), // BN_CTX
/* 3274 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 21), // BN_GENCB
/* 3275 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 22), // BN_MONT_CTX
/* 3276 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 23), // CMAC_CTX
/* 3277 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 24), // COMP_METHOD
/* 3278 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 25), // CONF
/* 3279 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 26), // CRYPTOGRAPHY_PASSWORD_DATA
/* 3280 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 27), // CRYPTO_EX_dup
/* 3281 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 28), // CRYPTO_EX_free
/* 3282 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 29), // CRYPTO_EX_new
/* 3283 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 30), // Cryptography_LHASH_OF_CONF_VALUE
/* 3284 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 31), // Cryptography_STACK_OF_ACCESS_DESCRIPTION
/* 3285 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 32), // Cryptography_STACK_OF_ASN1_INTEGER
/* 3286 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 33), // Cryptography_STACK_OF_ASN1_OBJECT
/* 3287 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 34), // Cryptography_STACK_OF_DIST_POINT
/* 3288 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 35), // Cryptography_STACK_OF_GENERAL_SUBTREE
/* 3289 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 36), // Cryptography_STACK_OF_POLICYINFO
/* 3290 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 37), // Cryptography_STACK_OF_POLICYQUALINFO
/* 3291 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 38), // Cryptography_STACK_OF_SCT
/* 3292 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 39), // Cryptography_STACK_OF_SSL_CIPHER
/* 3293 */ _CFFI_OP(_CFFI_OP_POINTER, 3294), // Cryptography_STACK_OF_X509 const *
/* 3294 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 40), // Cryptography_STACK_OF_X509
/* 3295 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 41), // Cryptography_STACK_OF_X509_CRL
/* 3296 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 42), // Cryptography_STACK_OF_X509_NAME
/* 3297 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 43), // Cryptography_STACK_OF_X509_NAME_ENTRY
/* 3298 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 44), // Cryptography_STACK_OF_X509_OBJECT
/* 3299 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 45), // Cryptography_STACK_OF_X509_REVOKED
/* 3300 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 46), // DH
/* 3301 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 47), // DIST_POINT
/* 3302 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 48), // DIST_POINT_NAME
/* 3303 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 49), // DSA
/* 3304 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 50), // ECDSA_SIG
/* 3305 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 51), // EC_GROUP
/* 3306 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 52), // EC_KEY
/* 3307 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 53), // EC_METHOD
/* 3308 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 54), // EC_POINT
/* 3309 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 55), // EC_builtin_curve
/* 3310 */ _CFFI_OP(_CFFI_OP_POINTER, 3311), // EDIPARTYNAME *
/* 3311 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 56), // EDIPARTYNAME
/* 3312 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 57), // ENGINE
/* 3313 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 58), // EVP_CIPHER
/* 3314 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 59), // EVP_CIPHER_CTX
/* 3315 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 60), // EVP_MD
/* 3316 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 61), // EVP_MD_CTX
/* 3317 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 62), // EVP_PKEY
/* 3318 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 63), // EVP_PKEY_CTX
/* 3319 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 64), // EXTENDED_KEY_USAGE
/* 3320 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 65), // GENERAL_NAME
/* 3321 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 119), // GENERAL_NAMES
/* 3322 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 66), // GENERAL_SUBTREE
/* 3323 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 67), // HMAC_CTX
/* 3324 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 68), // ISSUING_DIST_POINT
/* 3325 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 69), // NAME_CONSTRAINTS
/* 3326 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 70), // NETSCAPE_SPKI
/* 3327 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 71), // NOTICEREF
/* 3328 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 72), // OBJ_NAME
/* 3329 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 73), // OCSP_BASICRESP
/* 3330 */ _CFFI_OP(_CFFI_OP_POINTER, 3331), // OCSP_CERTID const *
/* 3331 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 74), // OCSP_CERTID
/* 3332 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 75), // OCSP_ONEREQ
/* 3333 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 76), // OCSP_REQUEST
/* 3334 */ _CFFI_OP(_CFFI_OP_POINTER, 3335), // OCSP_RESPDATA const *
/* 3335 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 77), // OCSP_RESPDATA
/* 3336 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 78), // OCSP_RESPONSE
/* 3337 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 79), // OCSP_SINGLERESP
/* 3338 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 80), // OSSL_LIB_CTX
/* 3339 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 81), // OSSL_PROVIDER
/* 3340 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 82), // OTHERNAME
/* 3341 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 83), // PKCS12
/* 3342 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 84), // PKCS7
/* 3343 */ _CFFI_OP(_CFFI_OP_POINTER, 3344), // PKCS7_DIGEST *
/* 3344 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 85), // PKCS7_DIGEST
/* 3345 */ _CFFI_OP(_CFFI_OP_POINTER, 3346), // PKCS7_ENCRYPT *
/* 3346 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 86), // PKCS7_ENCRYPT
/* 3347 */ _CFFI_OP(_CFFI_OP_POINTER, 3348), // PKCS7_ENVELOPE *
/* 3348 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 87), // PKCS7_ENVELOPE
/* 3349 */ _CFFI_OP(_CFFI_OP_POINTER, 3350), // PKCS7_SIGNED *
/* 3350 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 88), // PKCS7_SIGNED
/* 3351 */ _CFFI_OP(_CFFI_OP_POINTER, 3352), // PKCS7_SIGN_ENVELOPE *
/* 3352 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 89), // PKCS7_SIGN_ENVELOPE
/* 3353 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 90), // PKCS8_PRIV_KEY_INFO
/* 3354 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 91), // POLICYINFO
/* 3355 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 92), // POLICYQUALINFO
/* 3356 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 93), // POLICY_CONSTRAINTS
/* 3357 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 94), // RSA
/* 3358 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 95), // SCT
/* 3359 */ _CFFI_OP(_CFFI_OP_POINTER, 3360), // SRTP_PROTECTION_PROFILE *
/* 3360 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 96), // SRTP_PROTECTION_PROFILE
/* 3361 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 97), // SSL
/* 3362 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 98), // SSL_CIPHER
/* 3363 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 99), // SSL_CTX
/* 3364 */ _CFFI_OP(_CFFI_OP_POINTER, 3365), // SSL_METHOD const *
/* 3365 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 100), // SSL_METHOD
/* 3366 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 101), // SSL_SESSION
/* 3367 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 102), // USERNOTICE
/* 3368 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 103), // X509
/* 3369 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 104), // X509V3_CTX
/* 3370 */ _CFFI_OP(_CFFI_OP_POINTER, 3371), // X509V3_EXT_METHOD const *
/* 3371 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 105), // X509V3_EXT_METHOD
/* 3372 */ _CFFI_OP(_CFFI_OP_POINTER, 3373), // X509_ALGOR const *
/* 3373 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 106), // X509_ALGOR
/* 3374 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 107), // X509_CRL
/* 3375 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 108), // X509_EXTENSION
/* 3376 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 109), // X509_EXTENSIONS
/* 3377 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 110), // X509_NAME
/* 3378 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 111), // X509_NAME_ENTRY
/* 3379 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 112), // X509_OBJECT
/* 3380 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 113), // X509_REQ
/* 3381 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 114), // X509_REVOKED
/* 3382 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 115), // X509_STORE
/* 3383 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 116), // X509_STORE_CTX
/* 3384 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 117), // X509_VERIFY_PARAM
/* 3385 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 2), // char
/* 3386 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 0), // union $1
/* 3387 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 1), // union $2
/* 3388 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 2), // union $3
/* 3389 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 3), // union $4
/* 3390 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 4), // unsigned char
/* 3391 */ _CFFI_OP(_CFFI_OP_POINTER, 2721), // void *(*)(void *, unsigned char const * *, long)
/* 3392 */ _CFFI_OP(_CFFI_OP_POINTER, 3202), // void(*)(int, int, char const *, int)
/* 3393 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 0), // void
};

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_ACCESS_DESCRIPTION(ACCESS_DESCRIPTION *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->method; (void)tmp; }
  { GENERAL_NAME * *tmp = &p->location; (void)tmp; }
}
struct _cffi_align_typedef_ACCESS_DESCRIPTION { char x; ACCESS_DESCRIPTION y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_AUTHORITY_KEYID(AUTHORITY_KEYID *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OCTET_STRING * *tmp = &p->keyid; (void)tmp; }
  { GENERAL_NAMES * *tmp = &p->issuer; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->serial; (void)tmp; }
}
struct _cffi_align_typedef_AUTHORITY_KEYID { char x; AUTHORITY_KEYID y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_BASIC_CONSTRAINTS(BASIC_CONSTRAINTS *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->ca) | 0);  /* check that 'BASIC_CONSTRAINTS.ca' is an integer */
  { ASN1_INTEGER * *tmp = &p->pathlen; (void)tmp; }
}
struct _cffi_align_typedef_BASIC_CONSTRAINTS { char x; BASIC_CONSTRAINTS y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_CRYPTOGRAPHY_PASSWORD_DATA(CRYPTOGRAPHY_PASSWORD_DATA *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { char * *tmp = &p->password; (void)tmp; }
  (void)((p->length) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.length' is an integer */
  (void)((p->called) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.called' is an integer */
  (void)((p->error) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.error' is an integer */
  (void)((p->maxsize) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.maxsize' is an integer */
}
struct _cffi_align_typedef_CRYPTOGRAPHY_PASSWORD_DATA { char x; CRYPTOGRAPHY_PASSWORD_DATA y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_DIST_POINT(DIST_POINT *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { DIST_POINT_NAME * *tmp = &p->distpoint; (void)tmp; }
  { ASN1_OCTET_STRING * *tmp = &p->reasons; (void)tmp; }
  { GENERAL_NAMES * *tmp = &p->CRLissuer; (void)tmp; }
}
struct _cffi_align_typedef_DIST_POINT { char x; DIST_POINT y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_DIST_POINT_NAME(DIST_POINT_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'DIST_POINT_NAME.type' is an integer */
  /* cannot generate 'union $2' in field 'name': unknown type name */
}
struct _cffi_align_typedef_DIST_POINT_NAME { char x; DIST_POINT_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_EC_builtin_curve(EC_builtin_curve *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->nid) | 0);  /* check that 'EC_builtin_curve.nid' is an integer */
  { char const * *tmp = &p->comment; (void)tmp; }
}
struct _cffi_align_typedef_EC_builtin_curve { char x; EC_builtin_curve y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_EDIPARTYNAME(EDIPARTYNAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
}
struct _cffi_align_typedef_EDIPARTYNAME { char x; EDIPARTYNAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_GENERAL_NAME(GENERAL_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'GENERAL_NAME.type' is an integer */
  /* cannot generate 'union $1' in field 'd': unknown type name */
}
struct _cffi_align_typedef_GENERAL_NAME { char x; GENERAL_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_GENERAL_SUBTREE(GENERAL_SUBTREE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { GENERAL_NAME * *tmp = &p->base; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->minimum; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->maximum; (void)tmp; }
}
struct _cffi_align_typedef_GENERAL_SUBTREE { char x; GENERAL_SUBTREE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_ISSUING_DIST_POINT(ISSUING_DIST_POINT *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { DIST_POINT_NAME * *tmp = &p->distpoint; (void)tmp; }
  (void)((p->onlyuser) | 0);  /* check that 'ISSUING_DIST_POINT.onlyuser' is an integer */
  (void)((p->onlyCA) | 0);  /* check that 'ISSUING_DIST_POINT.onlyCA' is an integer */
  { ASN1_OCTET_STRING * *tmp = &p->onlysomereasons; (void)tmp; }
  (void)((p->indirectCRL) | 0);  /* check that 'ISSUING_DIST_POINT.indirectCRL' is an integer */
  (void)((p->onlyattr) | 0);  /* check that 'ISSUING_DIST_POINT.onlyattr' is an integer */
}
struct _cffi_align_typedef_ISSUING_DIST_POINT { char x; ISSUING_DIST_POINT y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_NAME_CONSTRAINTS(NAME_CONSTRAINTS *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_GENERAL_SUBTREE * *tmp = &p->permittedSubtrees; (void)tmp; }
  { Cryptography_STACK_OF_GENERAL_SUBTREE * *tmp = &p->excludedSubtrees; (void)tmp; }
}
struct _cffi_align_typedef_NAME_CONSTRAINTS { char x; NAME_CONSTRAINTS y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_NOTICEREF(NOTICEREF *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OCTET_STRING * *tmp = &p->organization; (void)tmp; }
  { Cryptography_STACK_OF_ASN1_INTEGER * *tmp = &p->noticenos; (void)tmp; }
}
struct _cffi_align_typedef_NOTICEREF { char x; NOTICEREF y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_OBJ_NAME(OBJ_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'OBJ_NAME.type' is an integer */
  (void)((p->alias) | 0);  /* check that 'OBJ_NAME.alias' is an integer */
  { char const * *tmp = &p->name; (void)tmp; }
  { char const * *tmp = &p->data; (void)tmp; }
}
struct _cffi_align_typedef_OBJ_NAME { char x; OBJ_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_OTHERNAME(OTHERNAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->type_id; (void)tmp; }
  { ASN1_TYPE * *tmp = &p->value; (void)tmp; }
}
struct _cffi_align_typedef_OTHERNAME { char x; OTHERNAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7(PKCS7 *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->type; (void)tmp; }
  /* cannot generate 'union $4' in field 'd': unknown type name */
}
struct _cffi_align_typedef_PKCS7 { char x; PKCS7 y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7_SIGNED(PKCS7_SIGNED *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_X509 * *tmp = &p->cert; (void)tmp; }
  { Cryptography_STACK_OF_X509_CRL * *tmp = &p->crl; (void)tmp; }
}
struct _cffi_align_typedef_PKCS7_SIGNED { char x; PKCS7_SIGNED y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7_SIGN_ENVELOPE(PKCS7_SIGN_ENVELOPE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_X509 * *tmp = &p->cert; (void)tmp; }
  { Cryptography_STACK_OF_X509_CRL * *tmp = &p->crl; (void)tmp; }
}
struct _cffi_align_typedef_PKCS7_SIGN_ENVELOPE { char x; PKCS7_SIGN_ENVELOPE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_POLICYINFO(POLICYINFO *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->policyid; (void)tmp; }
  { Cryptography_STACK_OF_POLICYQUALINFO * *tmp = &p->qualifiers; (void)tmp; }
}
struct _cffi_align_typedef_POLICYINFO { char x; POLICYINFO y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_POLICYQUALINFO(POLICYQUALINFO *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->pqualid; (void)tmp; }
  /* cannot generate 'union $3' in field 'd': unknown type name */
}
struct _cffi_align_typedef_POLICYQUALINFO { char x; POLICYQUALINFO y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_POLICY_CONSTRAINTS(POLICY_CONSTRAINTS *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_INTEGER * *tmp = &p->requireExplicitPolicy; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->inhibitPolicyMapping; (void)tmp; }
}
struct _cffi_align_typedef_POLICY_CONSTRAINTS { char x; POLICY_CONSTRAINTS y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_SRTP_PROTECTION_PROFILE(SRTP_PROTECTION_PROFILE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { char const * *tmp = &p->name; (void)tmp; }
  (void)((p->id) | 0);  /* check that 'SRTP_PROTECTION_PROFILE.id' is an integer */
}
struct _cffi_align_typedef_SRTP_PROTECTION_PROFILE { char x; SRTP_PROTECTION_PROFILE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_USERNOTICE(USERNOTICE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { NOTICEREF * *tmp = &p->noticeref; (void)tmp; }
  { ASN1_OCTET_STRING * *tmp = &p->exptext; (void)tmp; }
}
struct _cffi_align_typedef_USERNOTICE { char x; USERNOTICE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_X509V3_CTX(X509V3_CTX *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { X509 * *tmp = &p->issuer_cert; (void)tmp; }
  { X509 * *tmp = &p->subject_cert; (void)tmp; }
}
struct _cffi_align_typedef_X509V3_CTX { char x; X509V3_CTX y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_X509V3_EXT_METHOD(X509V3_EXT_METHOD *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_ITEM_EXP * *tmp = &p->it; (void)tmp; }
  { void *(* *tmp)(void *, unsigned char const * *, long) = &p->d2i; (void)tmp; }
}
struct _cffi_align_typedef_X509V3_EXT_METHOD { char x; X509V3_EXT_METHOD y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_X509_ALGOR(X509_ALGOR *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->algorithm; (void)tmp; }
}
struct _cffi_align_typedef_X509_ALGOR { char x; X509_ALGOR y; };

static int _cffi_const_CT_LOG_ENTRY_TYPE_NOT_SET(unsigned long long *o)
{
  int n = (CT_LOG_ENTRY_TYPE_NOT_SET) <= 0;
  *o = (unsigned long long)((CT_LOG_ENTRY_TYPE_NOT_SET) | 0);  /* check that CT_LOG_ENTRY_TYPE_NOT_SET is an integer */
  return n;
}

static int _cffi_const_CT_LOG_ENTRY_TYPE_X509(unsigned long long *o)
{
  int n = (CT_LOG_ENTRY_TYPE_X509) <= 0;
  *o = (unsigned long long)((CT_LOG_ENTRY_TYPE_X509) | 0);  /* check that CT_LOG_ENTRY_TYPE_X509 is an integer */
  return n;
}

static int _cffi_const_CT_LOG_ENTRY_TYPE_PRECERT(unsigned long long *o)
{
  int n = (CT_LOG_ENTRY_TYPE_PRECERT) <= 0;
  *o = (unsigned long long)((CT_LOG_ENTRY_TYPE_PRECERT) | 0);  /* check that CT_LOG_ENTRY_TYPE_PRECERT is an integer */
  return n;
}

static int _cffi_const_POINT_CONVERSION_COMPRESSED(unsigned long long *o)
{
  int n = (POINT_CONVERSION_COMPRESSED) <= 0;
  *o = (unsigned long long)((POINT_CONVERSION_COMPRESSED) | 0);  /* check that POINT_CONVERSION_COMPRESSED is an integer */
  return n;
}

static int _cffi_const_POINT_CONVERSION_UNCOMPRESSED(unsigned long long *o)
{
  int n = (POINT_CONVERSION_UNCOMPRESSED) <= 0;
  *o = (unsigned long long)((POINT_CONVERSION_UNCOMPRESSED) | 0);  /* check that POINT_CONVERSION_UNCOMPRESSED is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_UNKNOWN(unsigned long long *o)
{
  int n = (SCT_SOURCE_UNKNOWN) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_UNKNOWN) | 0);  /* check that SCT_SOURCE_UNKNOWN is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_TLS_EXTENSION(unsigned long long *o)
{
  int n = (SCT_SOURCE_TLS_EXTENSION) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_TLS_EXTENSION) | 0);  /* check that SCT_SOURCE_TLS_EXTENSION is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_X509V3_EXTENSION(unsigned long long *o)
{
  int n = (SCT_SOURCE_X509V3_EXTENSION) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_X509V3_EXTENSION) | 0);  /* check that SCT_SOURCE_X509V3_EXTENSION is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_OCSP_STAPLED_RESPONSE(unsigned long long *o)
{
  int n = (SCT_SOURCE_OCSP_STAPLED_RESPONSE) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_OCSP_STAPLED_RESPONSE) | 0);  /* check that SCT_SOURCE_OCSP_STAPLED_RESPONSE is an integer */
  return n;
}

static int _cffi_const_SCT_VERSION_NOT_SET(unsigned long long *o)
{
  int n = (SCT_VERSION_NOT_SET) <= 0;
  *o = (unsigned long long)((SCT_VERSION_NOT_SET) | 0);  /* check that SCT_VERSION_NOT_SET is an integer */
  return n;
}

static int _cffi_const_SCT_VERSION_V1(unsigned long long *o)
{
  int n = (SCT_VERSION_V1) <= 0;
  *o = (unsigned long long)((SCT_VERSION_V1) | 0);  /* check that SCT_VERSION_V1 is an integer */
  return n;
}

static int _cffi_const_ASN1_R_BOOLEAN_IS_WRONG_LENGTH(unsigned long long *o)
{
  int n = (ASN1_R_BOOLEAN_IS_WRONG_LENGTH) <= 0;
  *o = (unsigned long long)((ASN1_R_BOOLEAN_IS_WRONG_LENGTH) | 0);  /* check that ASN1_R_BOOLEAN_IS_WRONG_LENGTH is an integer */
  return n;
}

static int _cffi_const_ASN1_R_BUFFER_TOO_SMALL(unsigned long long *o)
{
  int n = (ASN1_R_BUFFER_TOO_SMALL) <= 0;
  *o = (unsigned long long)((ASN1_R_BUFFER_TOO_SMALL) | 0);  /* check that ASN1_R_BUFFER_TOO_SMALL is an integer */
  return n;
}

static int _cffi_const_ASN1_R_CIPHER_HAS_NO_OBJECT_IDENTIFIER(unsigned long long *o)
{
  int n = (ASN1_R_CIPHER_HAS_NO_OBJECT_IDENTIFIER) <= 0;
  *o = (unsigned long long)((ASN1_R_CIPHER_HAS_NO_OBJECT_IDENTIFIER) | 0);  /* check that ASN1_R_CIPHER_HAS_NO_OBJECT_IDENTIFIER is an integer */
  return n;
}

static int _cffi_const_ASN1_R_DATA_IS_WRONG(unsigned long long *o)
{
  int n = (ASN1_R_DATA_IS_WRONG) <= 0;
  *o = (unsigned long long)((ASN1_R_DATA_IS_WRONG) | 0);  /* check that ASN1_R_DATA_IS_WRONG is an integer */
  return n;
}

static int _cffi_const_ASN1_R_DECODE_ERROR(unsigned long long *o)
{
  int n = (ASN1_R_DECODE_ERROR) <= 0;
  *o = (unsigned long long)((ASN1_R_DECODE_ERROR) | 0);  /* check that ASN1_R_DECODE_ERROR is an integer */
  return n;
}

static int _cffi_const_ASN1_R_DEPTH_EXCEEDED(unsigned long long *o)
{
  int n = (ASN1_R_DEPTH_EXCEEDED) <= 0;
  *o = (unsigned long long)((ASN1_R_DEPTH_EXCEEDED) | 0);  /* check that ASN1_R_DEPTH_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_ASN1_R_ENCODE_ERROR(unsigned long long *o)
{
  int n = (ASN1_R_ENCODE_ERROR) <= 0;
  *o = (unsigned long long)((ASN1_R_ENCODE_ERROR) | 0);  /* check that ASN1_R_ENCODE_ERROR is an integer */
  return n;
}

static int _cffi_const_ASN1_R_ERROR_GETTING_TIME(unsigned long long *o)
{
  int n = (ASN1_R_ERROR_GETTING_TIME) <= 0;
  *o = (unsigned long long)((ASN1_R_ERROR_GETTING_TIME) | 0);  /* check that ASN1_R_ERROR_GETTING_TIME is an integer */
  return n;
}

static int _cffi_const_ASN1_R_ERROR_LOADING_SECTION(unsigned long long *o)
{
  int n = (ASN1_R_ERROR_LOADING_SECTION) <= 0;
  *o = (unsigned long long)((ASN1_R_ERROR_LOADING_SECTION) | 0);  /* check that ASN1_R_ERROR_LOADING_SECTION is an integer */
  return n;
}

static int _cffi_const_ASN1_R_HEADER_TOO_LONG(unsigned long long *o)
{
  int n = (ASN1_R_HEADER_TOO_LONG) <= 0;
  *o = (unsigned long long)((ASN1_R_HEADER_TOO_LONG) | 0);  /* check that ASN1_R_HEADER_TOO_LONG is an integer */
  return n;
}

static int _cffi_const_ASN1_R_MSTRING_WRONG_TAG(unsigned long long *o)
{
  int n = (ASN1_R_MSTRING_WRONG_TAG) <= 0;
  *o = (unsigned long long)((ASN1_R_MSTRING_WRONG_TAG) | 0);  /* check that ASN1_R_MSTRING_WRONG_TAG is an integer */
  return n;
}

static int _cffi_const_ASN1_R_NESTED_ASN1_STRING(unsigned long long *o)
{
  int n = (ASN1_R_NESTED_ASN1_STRING) <= 0;
  *o = (unsigned long long)((ASN1_R_NESTED_ASN1_STRING) | 0);  /* check that ASN1_R_NESTED_ASN1_STRING is an integer */
  return n;
}

static int _cffi_const_ASN1_R_NO_CONTENT_TYPE(unsigned long long *o)
{
  int n = (ASN1_R_NO_CONTENT_TYPE) <= 0;
  *o = (unsigned long long)((ASN1_R_NO_CONTENT_TYPE) | 0);  /* check that ASN1_R_NO_CONTENT_TYPE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_NO_MATCHING_CHOICE_TYPE(unsigned long long *o)
{
  int n = (ASN1_R_NO_MATCHING_CHOICE_TYPE) <= 0;
  *o = (unsigned long long)((ASN1_R_NO_MATCHING_CHOICE_TYPE) | 0);  /* check that ASN1_R_NO_MATCHING_CHOICE_TYPE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_NO_MULTIPART_BODY_FAILURE(unsigned long long *o)
{
  int n = (ASN1_R_NO_MULTIPART_BODY_FAILURE) <= 0;
  *o = (unsigned long long)((ASN1_R_NO_MULTIPART_BODY_FAILURE) | 0);  /* check that ASN1_R_NO_MULTIPART_BODY_FAILURE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_NO_MULTIPART_BOUNDARY(unsigned long long *o)
{
  int n = (ASN1_R_NO_MULTIPART_BOUNDARY) <= 0;
  *o = (unsigned long long)((ASN1_R_NO_MULTIPART_BOUNDARY) | 0);  /* check that ASN1_R_NO_MULTIPART_BOUNDARY is an integer */
  return n;
}

static int _cffi_const_ASN1_R_UNKNOWN_MESSAGE_DIGEST_ALGORITHM(unsigned long long *o)
{
  int n = (ASN1_R_UNKNOWN_MESSAGE_DIGEST_ALGORITHM) <= 0;
  *o = (unsigned long long)((ASN1_R_UNKNOWN_MESSAGE_DIGEST_ALGORITHM) | 0);  /* check that ASN1_R_UNKNOWN_MESSAGE_DIGEST_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_ASN1_R_UNKNOWN_OBJECT_TYPE(unsigned long long *o)
{
  int n = (ASN1_R_UNKNOWN_OBJECT_TYPE) <= 0;
  *o = (unsigned long long)((ASN1_R_UNKNOWN_OBJECT_TYPE) | 0);  /* check that ASN1_R_UNKNOWN_OBJECT_TYPE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_UNKNOWN_PUBLIC_KEY_TYPE(unsigned long long *o)
{
  int n = (ASN1_R_UNKNOWN_PUBLIC_KEY_TYPE) <= 0;
  *o = (unsigned long long)((ASN1_R_UNKNOWN_PUBLIC_KEY_TYPE) | 0);  /* check that ASN1_R_UNKNOWN_PUBLIC_KEY_TYPE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_UNKNOWN_TAG(unsigned long long *o)
{
  int n = (ASN1_R_UNKNOWN_TAG) <= 0;
  *o = (unsigned long long)((ASN1_R_UNKNOWN_TAG) | 0);  /* check that ASN1_R_UNKNOWN_TAG is an integer */
  return n;
}

static int _cffi_const_ASN1_R_UNSUPPORTED_ANY_DEFINED_BY_TYPE(unsigned long long *o)
{
  int n = (ASN1_R_UNSUPPORTED_ANY_DEFINED_BY_TYPE) <= 0;
  *o = (unsigned long long)((ASN1_R_UNSUPPORTED_ANY_DEFINED_BY_TYPE) | 0);  /* check that ASN1_R_UNSUPPORTED_ANY_DEFINED_BY_TYPE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_UNSUPPORTED_PUBLIC_KEY_TYPE(unsigned long long *o)
{
  int n = (ASN1_R_UNSUPPORTED_PUBLIC_KEY_TYPE) <= 0;
  *o = (unsigned long long)((ASN1_R_UNSUPPORTED_PUBLIC_KEY_TYPE) | 0);  /* check that ASN1_R_UNSUPPORTED_PUBLIC_KEY_TYPE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_UNSUPPORTED_TYPE(unsigned long long *o)
{
  int n = (ASN1_R_UNSUPPORTED_TYPE) <= 0;
  *o = (unsigned long long)((ASN1_R_UNSUPPORTED_TYPE) | 0);  /* check that ASN1_R_UNSUPPORTED_TYPE is an integer */
  return n;
}

static int _cffi_const_ASN1_R_WRONG_TAG(unsigned long long *o)
{
  int n = (ASN1_R_WRONG_TAG) <= 0;
  *o = (unsigned long long)((ASN1_R_WRONG_TAG) | 0);  /* check that ASN1_R_WRONG_TAG is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_IS_LIBRESSL(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_IS_LIBRESSL) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_IS_LIBRESSL) | 0);  /* check that CRYPTOGRAPHY_IS_LIBRESSL is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_LIBRESSL_28_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_LIBRESSL_28_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_LIBRESSL_28_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_LIBRESSL_28_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE) | 0);  /* check that CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_102L_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_102L_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_102L_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_102L_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_110_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_110_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_110_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_110_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_111D_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_300_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_300_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_300_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_300_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_102(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_102) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_102) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_102 is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_102I(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_102I) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_102I) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_102I is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_111(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_111) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_300(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_300) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_300) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_300 is an integer */
  return n;
}

static void _cffi_const_CRYPTOGRAPHY_PACKAGE_VERSION(char *o)
{
  *(char const * *)o = CRYPTOGRAPHY_PACKAGE_VERSION;
}

static int _cffi_const_CRYPTO_MEM_CHECK_DISABLE(unsigned long long *o)
{
  int n = (CRYPTO_MEM_CHECK_DISABLE) <= 0;
  *o = (unsigned long long)((CRYPTO_MEM_CHECK_DISABLE) | 0);  /* check that CRYPTO_MEM_CHECK_DISABLE is an integer */
  return n;
}

static int _cffi_const_CRYPTO_MEM_CHECK_ENABLE(unsigned long long *o)
{
  int n = (CRYPTO_MEM_CHECK_ENABLE) <= 0;
  *o = (unsigned long long)((CRYPTO_MEM_CHECK_ENABLE) | 0);  /* check that CRYPTO_MEM_CHECK_ENABLE is an integer */
  return n;
}

static int _cffi_const_CRYPTO_MEM_CHECK_OFF(unsigned long long *o)
{
  int n = (CRYPTO_MEM_CHECK_OFF) <= 0;
  *o = (unsigned long long)((CRYPTO_MEM_CHECK_OFF) | 0);  /* check that CRYPTO_MEM_CHECK_OFF is an integer */
  return n;
}

static int _cffi_const_CRYPTO_MEM_CHECK_ON(unsigned long long *o)
{
  int n = (CRYPTO_MEM_CHECK_ON) <= 0;
  *o = (unsigned long long)((CRYPTO_MEM_CHECK_ON) | 0);  /* check that CRYPTO_MEM_CHECK_ON is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_102_VERIFICATION_ERROR_CODES(unsigned long long *o)
{
  int n = (Cryptography_HAS_102_VERIFICATION_ERROR_CODES) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_102_VERIFICATION_ERROR_CODES) | 0);  /* check that Cryptography_HAS_102_VERIFICATION_ERROR_CODES is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_102_VERIFICATION_PARAMS(unsigned long long *o)
{
  int n = (Cryptography_HAS_102_VERIFICATION_PARAMS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_102_VERIFICATION_PARAMS) | 0);  /* check that Cryptography_HAS_102_VERIFICATION_PARAMS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_110_VERIFICATION_PARAMS(unsigned long long *o)
{
  int n = (Cryptography_HAS_110_VERIFICATION_PARAMS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_110_VERIFICATION_PARAMS) | 0);  /* check that Cryptography_HAS_110_VERIFICATION_PARAMS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ALPN(unsigned long long *o)
{
  int n = (Cryptography_HAS_ALPN) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ALPN) | 0);  /* check that Cryptography_HAS_ALPN is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_CIPHER_DETAILS(unsigned long long *o)
{
  int n = (Cryptography_HAS_CIPHER_DETAILS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_CIPHER_DETAILS) | 0);  /* check that Cryptography_HAS_CIPHER_DETAILS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_COMPRESSION(unsigned long long *o)
{
  int n = (Cryptography_HAS_COMPRESSION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_COMPRESSION) | 0);  /* check that Cryptography_HAS_COMPRESSION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_CTRL_GET_MAX_PROTO_VERSION(unsigned long long *o)
{
  int n = (Cryptography_HAS_CTRL_GET_MAX_PROTO_VERSION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_CTRL_GET_MAX_PROTO_VERSION) | 0);  /* check that Cryptography_HAS_CTRL_GET_MAX_PROTO_VERSION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_CUSTOM_EXT(unsigned long long *o)
{
  int n = (Cryptography_HAS_CUSTOM_EXT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_CUSTOM_EXT) | 0);  /* check that Cryptography_HAS_CUSTOM_EXT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_DTLS(unsigned long long *o)
{
  int n = (Cryptography_HAS_DTLS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_DTLS) | 0);  /* check that Cryptography_HAS_DTLS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EC2M(unsigned long long *o)
{
  int n = (Cryptography_HAS_EC2M) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EC2M) | 0);  /* check that Cryptography_HAS_EC2M is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ECDSA(unsigned long long *o)
{
  int n = (Cryptography_HAS_ECDSA) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ECDSA) | 0);  /* check that Cryptography_HAS_ECDSA is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EC_1_0_2(unsigned long long *o)
{
  int n = (Cryptography_HAS_EC_1_0_2) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EC_1_0_2) | 0);  /* check that Cryptography_HAS_EC_1_0_2 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EC_CODES(unsigned long long *o)
{
  int n = (Cryptography_HAS_EC_CODES) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EC_CODES) | 0);  /* check that Cryptography_HAS_EC_CODES is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ED25519(unsigned long long *o)
{
  int n = (Cryptography_HAS_ED25519) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ED25519) | 0);  /* check that Cryptography_HAS_ED25519 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ED448(unsigned long long *o)
{
  int n = (Cryptography_HAS_ED448) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ED448) | 0);  /* check that Cryptography_HAS_ED448 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EGD(unsigned long long *o)
{
  int n = (Cryptography_HAS_EGD) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EGD) | 0);  /* check that Cryptography_HAS_EGD is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ENGINE(unsigned long long *o)
{
  int n = (Cryptography_HAS_ENGINE) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ENGINE) | 0);  /* check that Cryptography_HAS_ENGINE is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_DIGESTFINAL_XOF(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_DIGESTFINAL_XOF) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_DIGESTFINAL_XOF) | 0);  /* check that Cryptography_HAS_EVP_DIGESTFINAL_XOF is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_PKEY_DHX(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_PKEY_DHX) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_PKEY_DHX) | 0);  /* check that Cryptography_HAS_EVP_PKEY_DHX is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint) | 0);  /* check that Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_R_MEMORY_LIMIT_EXCEEDED(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_R_MEMORY_LIMIT_EXCEEDED) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_R_MEMORY_LIMIT_EXCEEDED) | 0);  /* check that Cryptography_HAS_EVP_R_MEMORY_LIMIT_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_GENERIC_DTLS_METHOD(unsigned long long *o)
{
  int n = (Cryptography_HAS_GENERIC_DTLS_METHOD) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_GENERIC_DTLS_METHOD) | 0);  /* check that Cryptography_HAS_GENERIC_DTLS_METHOD is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_GET_SERVER_TMP_KEY(unsigned long long *o)
{
  int n = (Cryptography_HAS_GET_SERVER_TMP_KEY) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_GET_SERVER_TMP_KEY) | 0);  /* check that Cryptography_HAS_GET_SERVER_TMP_KEY is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_LOCKING_CALLBACKS(unsigned long long *o)
{
  int n = (Cryptography_HAS_LOCKING_CALLBACKS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_LOCKING_CALLBACKS) | 0);  /* check that Cryptography_HAS_LOCKING_CALLBACKS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_MEM_FUNCTIONS(unsigned long long *o)
{
  int n = (Cryptography_HAS_MEM_FUNCTIONS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_MEM_FUNCTIONS) | 0);  /* check that Cryptography_HAS_MEM_FUNCTIONS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_NEXTPROTONEG(unsigned long long *o)
{
  int n = (Cryptography_HAS_NEXTPROTONEG) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_NEXTPROTONEG) | 0);  /* check that Cryptography_HAS_NEXTPROTONEG is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY(unsigned long long *o)
{
  int n = (Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY) | 0);  /* check that Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_OPENSSL_CLEANUP(unsigned long long *o)
{
  int n = (Cryptography_HAS_OPENSSL_CLEANUP) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_OPENSSL_CLEANUP) | 0);  /* check that Cryptography_HAS_OPENSSL_CLEANUP is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_OP_NO_COMPRESSION(unsigned long long *o)
{
  int n = (Cryptography_HAS_OP_NO_COMPRESSION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_OP_NO_COMPRESSION) | 0);  /* check that Cryptography_HAS_OP_NO_COMPRESSION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_POLY1305(unsigned long long *o)
{
  int n = (Cryptography_HAS_POLY1305) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_POLY1305) | 0);  /* check that Cryptography_HAS_POLY1305 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PROVIDERS(unsigned long long *o)
{
  int n = (Cryptography_HAS_PROVIDERS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PROVIDERS) | 0);  /* check that Cryptography_HAS_PROVIDERS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PSK(unsigned long long *o)
{
  int n = (Cryptography_HAS_PSK) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PSK) | 0);  /* check that Cryptography_HAS_PSK is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PSS_PADDING(unsigned long long *o)
{
  int n = (Cryptography_HAS_PSS_PADDING) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PSS_PADDING) | 0);  /* check that Cryptography_HAS_PSS_PADDING is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RAW_KEY(unsigned long long *o)
{
  int n = (Cryptography_HAS_RAW_KEY) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RAW_KEY) | 0);  /* check that Cryptography_HAS_RAW_KEY is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RELEASE_BUFFERS(unsigned long long *o)
{
  int n = (Cryptography_HAS_RELEASE_BUFFERS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RELEASE_BUFFERS) | 0);  /* check that Cryptography_HAS_RELEASE_BUFFERS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RSA_OAEP_LABEL(unsigned long long *o)
{
  int n = (Cryptography_HAS_RSA_OAEP_LABEL) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RSA_OAEP_LABEL) | 0);  /* check that Cryptography_HAS_RSA_OAEP_LABEL is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RSA_OAEP_MD(unsigned long long *o)
{
  int n = (Cryptography_HAS_RSA_OAEP_MD) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RSA_OAEP_MD) | 0);  /* check that Cryptography_HAS_RSA_OAEP_MD is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RSA_R_PKCS_DECODING_ERROR(unsigned long long *o)
{
  int n = (Cryptography_HAS_RSA_R_PKCS_DECODING_ERROR) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RSA_R_PKCS_DECODING_ERROR) | 0);  /* check that Cryptography_HAS_RSA_R_PKCS_DECODING_ERROR is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SCRYPT(unsigned long long *o)
{
  int n = (Cryptography_HAS_SCRYPT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SCRYPT) | 0);  /* check that Cryptography_HAS_SCRYPT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SCT(unsigned long long *o)
{
  int n = (Cryptography_HAS_SCT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SCT) | 0);  /* check that Cryptography_HAS_SCT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SECURE_RENEGOTIATION(unsigned long long *o)
{
  int n = (Cryptography_HAS_SECURE_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SECURE_RENEGOTIATION) | 0);  /* check that Cryptography_HAS_SECURE_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SET_CERT_CB(unsigned long long *o)
{
  int n = (Cryptography_HAS_SET_CERT_CB) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SET_CERT_CB) | 0);  /* check that Cryptography_HAS_SET_CERT_CB is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SET_ECDH_AUTO(unsigned long long *o)
{
  int n = (Cryptography_HAS_SET_ECDH_AUTO) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SET_ECDH_AUTO) | 0);  /* check that Cryptography_HAS_SET_ECDH_AUTO is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SIGALGS(unsigned long long *o)
{
  int n = (Cryptography_HAS_SIGALGS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SIGALGS) | 0);  /* check that Cryptography_HAS_SIGALGS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL2(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL2) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL2) | 0);  /* check that Cryptography_HAS_SSL2 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL3_METHOD(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL3_METHOD) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL3_METHOD) | 0);  /* check that Cryptography_HAS_SSL3_METHOD is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS) | 0);  /* check that Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_CTX_SET_CLIENT_CERT_ENGINE(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_CTX_SET_CLIENT_CERT_ENGINE) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_CTX_SET_CLIENT_CERT_ENGINE) | 0);  /* check that Cryptography_HAS_SSL_CTX_SET_CLIENT_CERT_ENGINE is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING) | 0);  /* check that Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_OP_NO_TICKET(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_OP_NO_TICKET) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_OP_NO_TICKET) | 0);  /* check that Cryptography_HAS_SSL_OP_NO_TICKET is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_SET_SSL_CTX(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_SET_SSL_CTX) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_SET_SSL_CTX) | 0);  /* check that Cryptography_HAS_SSL_SET_SSL_CTX is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_ST(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_ST) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_ST) | 0);  /* check that Cryptography_HAS_SSL_ST is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_STATUS_REQ_OCSP_RESP(unsigned long long *o)
{
  int n = (Cryptography_HAS_STATUS_REQ_OCSP_RESP) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_STATUS_REQ_OCSP_RESP) | 0);  /* check that Cryptography_HAS_STATUS_REQ_OCSP_RESP is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSEXT_HOSTNAME(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSEXT_HOSTNAME) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSEXT_HOSTNAME) | 0);  /* check that Cryptography_HAS_TLSEXT_HOSTNAME is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSEXT_STATUS_REQ_CB(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSEXT_STATUS_REQ_CB) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSEXT_STATUS_REQ_CB) | 0);  /* check that Cryptography_HAS_TLSEXT_STATUS_REQ_CB is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSEXT_STATUS_REQ_TYPE(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSEXT_STATUS_REQ_TYPE) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSEXT_STATUS_REQ_TYPE) | 0);  /* check that Cryptography_HAS_TLSEXT_STATUS_REQ_TYPE is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLS_ST(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLS_ST) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLS_ST) | 0);  /* check that Cryptography_HAS_TLS_ST is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_1(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_1) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_1) | 0);  /* check that Cryptography_HAS_TLSv1_1 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_2(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_2) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_2) | 0);  /* check that Cryptography_HAS_TLSv1_2 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_3(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_3) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_3) | 0);  /* check that Cryptography_HAS_TLSv1_3 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X25519(unsigned long long *o)
{
  int n = (Cryptography_HAS_X25519) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X25519) | 0);  /* check that Cryptography_HAS_X25519 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X448(unsigned long long *o)
{
  int n = (Cryptography_HAS_X448) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X448) | 0);  /* check that Cryptography_HAS_X448 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT(unsigned long long *o)
{
  int n = (Cryptography_HAS_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) | 0);  /* check that Cryptography_HAS_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X509_STORE_CTX_GET_ISSUER(unsigned long long *o)
{
  int n = (Cryptography_HAS_X509_STORE_CTX_GET_ISSUER) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X509_STORE_CTX_GET_ISSUER) | 0);  /* check that Cryptography_HAS_X509_STORE_CTX_GET_ISSUER is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X509_V_FLAG_PARTIAL_CHAIN(unsigned long long *o)
{
  int n = (Cryptography_HAS_X509_V_FLAG_PARTIAL_CHAIN) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X509_V_FLAG_PARTIAL_CHAIN) | 0);  /* check that Cryptography_HAS_X509_V_FLAG_PARTIAL_CHAIN is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X509_V_FLAG_TRUSTED_FIRST(unsigned long long *o)
{
  int n = (Cryptography_HAS_X509_V_FLAG_TRUSTED_FIRST) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X509_V_FLAG_TRUSTED_FIRST) | 0);  /* check that Cryptography_HAS_X509_V_FLAG_TRUSTED_FIRST is an integer */
  return n;
}

static void _cffi_const_Cryptography_osrandom_engine_id(char *o)
{
  *(char const * *)o = Cryptography_osrandom_engine_id;
}

static void _cffi_const_Cryptography_osrandom_engine_name(char *o)
{
  *(char const * *)o = Cryptography_osrandom_engine_name;
}

static int _cffi_const_Crytpography_HAS_OP_IGNORE_UNEXPECTED_EOF(unsigned long long *o)
{
  int n = (Crytpography_HAS_OP_IGNORE_UNEXPECTED_EOF) <= 0;
  *o = (unsigned long long)((Crytpography_HAS_OP_IGNORE_UNEXPECTED_EOF) | 0);  /* check that Crytpography_HAS_OP_IGNORE_UNEXPECTED_EOF is an integer */
  return n;
}

static int _cffi_const_DH_NOT_SUITABLE_GENERATOR(unsigned long long *o)
{
  int n = (DH_NOT_SUITABLE_GENERATOR) <= 0;
  *o = (unsigned long long)((DH_NOT_SUITABLE_GENERATOR) | 0);  /* check that DH_NOT_SUITABLE_GENERATOR is an integer */
  return n;
}

static int _cffi_const_DH_R_INVALID_PUBKEY(unsigned long long *o)
{
  int n = (DH_R_INVALID_PUBKEY) <= 0;
  *o = (unsigned long long)((DH_R_INVALID_PUBKEY) | 0);  /* check that DH_R_INVALID_PUBKEY is an integer */
  return n;
}

static int _cffi_const_EC_R_UNKNOWN_GROUP(unsigned long long *o)
{
  int n = (EC_R_UNKNOWN_GROUP) <= 0;
  *o = (unsigned long long)((EC_R_UNKNOWN_GROUP) | 0);  /* check that EC_R_UNKNOWN_GROUP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ASN1(unsigned long long *o)
{
  int n = (ERR_LIB_ASN1) <= 0;
  *o = (unsigned long long)((ERR_LIB_ASN1) | 0);  /* check that ERR_LIB_ASN1 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ASYNC(unsigned long long *o)
{
  int n = (ERR_LIB_ASYNC) <= 0;
  *o = (unsigned long long)((ERR_LIB_ASYNC) | 0);  /* check that ERR_LIB_ASYNC is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_BIO(unsigned long long *o)
{
  int n = (ERR_LIB_BIO) <= 0;
  *o = (unsigned long long)((ERR_LIB_BIO) | 0);  /* check that ERR_LIB_BIO is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_BN(unsigned long long *o)
{
  int n = (ERR_LIB_BN) <= 0;
  *o = (unsigned long long)((ERR_LIB_BN) | 0);  /* check that ERR_LIB_BN is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_BUF(unsigned long long *o)
{
  int n = (ERR_LIB_BUF) <= 0;
  *o = (unsigned long long)((ERR_LIB_BUF) | 0);  /* check that ERR_LIB_BUF is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_CMP(unsigned long long *o)
{
  int n = (ERR_LIB_CMP) <= 0;
  *o = (unsigned long long)((ERR_LIB_CMP) | 0);  /* check that ERR_LIB_CMP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_CMS(unsigned long long *o)
{
  int n = (ERR_LIB_CMS) <= 0;
  *o = (unsigned long long)((ERR_LIB_CMS) | 0);  /* check that ERR_LIB_CMS is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_COMP(unsigned long long *o)
{
  int n = (ERR_LIB_COMP) <= 0;
  *o = (unsigned long long)((ERR_LIB_COMP) | 0);  /* check that ERR_LIB_COMP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_CONF(unsigned long long *o)
{
  int n = (ERR_LIB_CONF) <= 0;
  *o = (unsigned long long)((ERR_LIB_CONF) | 0);  /* check that ERR_LIB_CONF is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_CRMF(unsigned long long *o)
{
  int n = (ERR_LIB_CRMF) <= 0;
  *o = (unsigned long long)((ERR_LIB_CRMF) | 0);  /* check that ERR_LIB_CRMF is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_CRYPTO(unsigned long long *o)
{
  int n = (ERR_LIB_CRYPTO) <= 0;
  *o = (unsigned long long)((ERR_LIB_CRYPTO) | 0);  /* check that ERR_LIB_CRYPTO is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_CT(unsigned long long *o)
{
  int n = (ERR_LIB_CT) <= 0;
  *o = (unsigned long long)((ERR_LIB_CT) | 0);  /* check that ERR_LIB_CT is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_DH(unsigned long long *o)
{
  int n = (ERR_LIB_DH) <= 0;
  *o = (unsigned long long)((ERR_LIB_DH) | 0);  /* check that ERR_LIB_DH is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_DSA(unsigned long long *o)
{
  int n = (ERR_LIB_DSA) <= 0;
  *o = (unsigned long long)((ERR_LIB_DSA) | 0);  /* check that ERR_LIB_DSA is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_DSO(unsigned long long *o)
{
  int n = (ERR_LIB_DSO) <= 0;
  *o = (unsigned long long)((ERR_LIB_DSO) | 0);  /* check that ERR_LIB_DSO is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_EC(unsigned long long *o)
{
  int n = (ERR_LIB_EC) <= 0;
  *o = (unsigned long long)((ERR_LIB_EC) | 0);  /* check that ERR_LIB_EC is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ECDH(unsigned long long *o)
{
  int n = (ERR_LIB_ECDH) <= 0;
  *o = (unsigned long long)((ERR_LIB_ECDH) | 0);  /* check that ERR_LIB_ECDH is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ECDSA(unsigned long long *o)
{
  int n = (ERR_LIB_ECDSA) <= 0;
  *o = (unsigned long long)((ERR_LIB_ECDSA) | 0);  /* check that ERR_LIB_ECDSA is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ENGINE(unsigned long long *o)
{
  int n = (ERR_LIB_ENGINE) <= 0;
  *o = (unsigned long long)((ERR_LIB_ENGINE) | 0);  /* check that ERR_LIB_ENGINE is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ESS(unsigned long long *o)
{
  int n = (ERR_LIB_ESS) <= 0;
  *o = (unsigned long long)((ERR_LIB_ESS) | 0);  /* check that ERR_LIB_ESS is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_EVP(unsigned long long *o)
{
  int n = (ERR_LIB_EVP) <= 0;
  *o = (unsigned long long)((ERR_LIB_EVP) | 0);  /* check that ERR_LIB_EVP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_FIPS(unsigned long long *o)
{
  int n = (ERR_LIB_FIPS) <= 0;
  *o = (unsigned long long)((ERR_LIB_FIPS) | 0);  /* check that ERR_LIB_FIPS is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_HMAC(unsigned long long *o)
{
  int n = (ERR_LIB_HMAC) <= 0;
  *o = (unsigned long long)((ERR_LIB_HMAC) | 0);  /* check that ERR_LIB_HMAC is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_HTTP(unsigned long long *o)
{
  int n = (ERR_LIB_HTTP) <= 0;
  *o = (unsigned long long)((ERR_LIB_HTTP) | 0);  /* check that ERR_LIB_HTTP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_KDF(unsigned long long *o)
{
  int n = (ERR_LIB_KDF) <= 0;
  *o = (unsigned long long)((ERR_LIB_KDF) | 0);  /* check that ERR_LIB_KDF is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_MASK(unsigned long long *o)
{
  int n = (ERR_LIB_MASK) <= 0;
  *o = (unsigned long long)((ERR_LIB_MASK) | 0);  /* check that ERR_LIB_MASK is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_NONE(unsigned long long *o)
{
  int n = (ERR_LIB_NONE) <= 0;
  *o = (unsigned long long)((ERR_LIB_NONE) | 0);  /* check that ERR_LIB_NONE is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_OBJ(unsigned long long *o)
{
  int n = (ERR_LIB_OBJ) <= 0;
  *o = (unsigned long long)((ERR_LIB_OBJ) | 0);  /* check that ERR_LIB_OBJ is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_OCSP(unsigned long long *o)
{
  int n = (ERR_LIB_OCSP) <= 0;
  *o = (unsigned long long)((ERR_LIB_OCSP) | 0);  /* check that ERR_LIB_OCSP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_OFFSET(unsigned long long *o)
{
  int n = (ERR_LIB_OFFSET) <= 0;
  *o = (unsigned long long)((ERR_LIB_OFFSET) | 0);  /* check that ERR_LIB_OFFSET is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_OSSL_DECODER(unsigned long long *o)
{
  int n = (ERR_LIB_OSSL_DECODER) <= 0;
  *o = (unsigned long long)((ERR_LIB_OSSL_DECODER) | 0);  /* check that ERR_LIB_OSSL_DECODER is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_OSSL_ENCODER(unsigned long long *o)
{
  int n = (ERR_LIB_OSSL_ENCODER) <= 0;
  *o = (unsigned long long)((ERR_LIB_OSSL_ENCODER) | 0);  /* check that ERR_LIB_OSSL_ENCODER is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_OSSL_STORE(unsigned long long *o)
{
  int n = (ERR_LIB_OSSL_STORE) <= 0;
  *o = (unsigned long long)((ERR_LIB_OSSL_STORE) | 0);  /* check that ERR_LIB_OSSL_STORE is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PEM(unsigned long long *o)
{
  int n = (ERR_LIB_PEM) <= 0;
  *o = (unsigned long long)((ERR_LIB_PEM) | 0);  /* check that ERR_LIB_PEM is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PKCS12(unsigned long long *o)
{
  int n = (ERR_LIB_PKCS12) <= 0;
  *o = (unsigned long long)((ERR_LIB_PKCS12) | 0);  /* check that ERR_LIB_PKCS12 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PKCS7(unsigned long long *o)
{
  int n = (ERR_LIB_PKCS7) <= 0;
  *o = (unsigned long long)((ERR_LIB_PKCS7) | 0);  /* check that ERR_LIB_PKCS7 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PROP(unsigned long long *o)
{
  int n = (ERR_LIB_PROP) <= 0;
  *o = (unsigned long long)((ERR_LIB_PROP) | 0);  /* check that ERR_LIB_PROP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PROV(unsigned long long *o)
{
  int n = (ERR_LIB_PROV) <= 0;
  *o = (unsigned long long)((ERR_LIB_PROV) | 0);  /* check that ERR_LIB_PROV is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_RAND(unsigned long long *o)
{
  int n = (ERR_LIB_RAND) <= 0;
  *o = (unsigned long long)((ERR_LIB_RAND) | 0);  /* check that ERR_LIB_RAND is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_RSA(unsigned long long *o)
{
  int n = (ERR_LIB_RSA) <= 0;
  *o = (unsigned long long)((ERR_LIB_RSA) | 0);  /* check that ERR_LIB_RSA is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_SM2(unsigned long long *o)
{
  int n = (ERR_LIB_SM2) <= 0;
  *o = (unsigned long long)((ERR_LIB_SM2) | 0);  /* check that ERR_LIB_SM2 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_SSL(unsigned long long *o)
{
  int n = (ERR_LIB_SSL) <= 0;
  *o = (unsigned long long)((ERR_LIB_SSL) | 0);  /* check that ERR_LIB_SSL is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_SYS(unsigned long long *o)
{
  int n = (ERR_LIB_SYS) <= 0;
  *o = (unsigned long long)((ERR_LIB_SYS) | 0);  /* check that ERR_LIB_SYS is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_TS(unsigned long long *o)
{
  int n = (ERR_LIB_TS) <= 0;
  *o = (unsigned long long)((ERR_LIB_TS) | 0);  /* check that ERR_LIB_TS is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_UI(unsigned long long *o)
{
  int n = (ERR_LIB_UI) <= 0;
  *o = (unsigned long long)((ERR_LIB_UI) | 0);  /* check that ERR_LIB_UI is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_USER(unsigned long long *o)
{
  int n = (ERR_LIB_USER) <= 0;
  *o = (unsigned long long)((ERR_LIB_USER) | 0);  /* check that ERR_LIB_USER is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_X509(unsigned long long *o)
{
  int n = (ERR_LIB_X509) <= 0;
  *o = (unsigned long long)((ERR_LIB_X509) | 0);  /* check that ERR_LIB_X509 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_X509V3(unsigned long long *o)
{
  int n = (ERR_LIB_X509V3) <= 0;
  *o = (unsigned long long)((ERR_LIB_X509V3) | 0);  /* check that ERR_LIB_X509V3 is an integer */
  return n;
}

static int _cffi_const_ERR_R_MALLOC_FAILURE(unsigned long long *o)
{
  int n = (ERR_R_MALLOC_FAILURE) <= 0;
  *o = (unsigned long long)((ERR_R_MALLOC_FAILURE) | 0);  /* check that ERR_R_MALLOC_FAILURE is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_GET_TAG(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_GET_TAG) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_GET_TAG) | 0);  /* check that EVP_CTRL_AEAD_GET_TAG is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_SET_IVLEN(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_SET_IVLEN) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_SET_IVLEN) | 0);  /* check that EVP_CTRL_AEAD_SET_IVLEN is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_SET_TAG(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_SET_TAG) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_SET_TAG) | 0);  /* check that EVP_CTRL_AEAD_SET_TAG is an integer */
  return n;
}

static int _cffi_const_EVP_F_EVP_ENCRYPTFINAL_EX(unsigned long long *o)
{
  int n = (EVP_F_EVP_ENCRYPTFINAL_EX) <= 0;
  *o = (unsigned long long)((EVP_F_EVP_ENCRYPTFINAL_EX) | 0);  /* check that EVP_F_EVP_ENCRYPTFINAL_EX is an integer */
  return n;
}

static int _cffi_const_EVP_MAX_MD_SIZE(unsigned long long *o)
{
  int n = (EVP_MAX_MD_SIZE) <= 0;
  *o = (unsigned long long)((EVP_MAX_MD_SIZE) | 0);  /* check that EVP_MAX_MD_SIZE is an integer */
  return n;
}

static int _cffi_const_EVP_MD_CTX_FLAG_NON_FIPS_ALLOW(unsigned long long *o)
{
  int n = (EVP_MD_CTX_FLAG_NON_FIPS_ALLOW) <= 0;
  *o = (unsigned long long)((EVP_MD_CTX_FLAG_NON_FIPS_ALLOW) | 0);  /* check that EVP_MD_CTX_FLAG_NON_FIPS_ALLOW is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DH(unsigned long long *o)
{
  int n = (EVP_PKEY_DH) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DH) | 0);  /* check that EVP_PKEY_DH is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DHX(unsigned long long *o)
{
  int n = (EVP_PKEY_DHX) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DHX) | 0);  /* check that EVP_PKEY_DHX is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DSA(unsigned long long *o)
{
  int n = (EVP_PKEY_DSA) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DSA) | 0);  /* check that EVP_PKEY_DSA is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_EC(unsigned long long *o)
{
  int n = (EVP_PKEY_EC) <= 0;
  *o = (unsigned long long)((EVP_PKEY_EC) | 0);  /* check that EVP_PKEY_EC is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_ED25519(unsigned long long *o)
{
  int n = (EVP_PKEY_ED25519) <= 0;
  *o = (unsigned long long)((EVP_PKEY_ED25519) | 0);  /* check that EVP_PKEY_ED25519 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_ED448(unsigned long long *o)
{
  int n = (EVP_PKEY_ED448) <= 0;
  *o = (unsigned long long)((EVP_PKEY_ED448) | 0);  /* check that EVP_PKEY_ED448 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_POLY1305(unsigned long long *o)
{
  int n = (EVP_PKEY_POLY1305) <= 0;
  *o = (unsigned long long)((EVP_PKEY_POLY1305) | 0);  /* check that EVP_PKEY_POLY1305 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_RSA(unsigned long long *o)
{
  int n = (EVP_PKEY_RSA) <= 0;
  *o = (unsigned long long)((EVP_PKEY_RSA) | 0);  /* check that EVP_PKEY_RSA is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_X25519(unsigned long long *o)
{
  int n = (EVP_PKEY_X25519) <= 0;
  *o = (unsigned long long)((EVP_PKEY_X25519) | 0);  /* check that EVP_PKEY_X25519 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_X448(unsigned long long *o)
{
  int n = (EVP_PKEY_X448) <= 0;
  *o = (unsigned long long)((EVP_PKEY_X448) | 0);  /* check that EVP_PKEY_X448 is an integer */
  return n;
}

static int _cffi_const_EVP_R_AES_KEY_SETUP_FAILED(unsigned long long *o)
{
  int n = (EVP_R_AES_KEY_SETUP_FAILED) <= 0;
  *o = (unsigned long long)((EVP_R_AES_KEY_SETUP_FAILED) | 0);  /* check that EVP_R_AES_KEY_SETUP_FAILED is an integer */
  return n;
}

static int _cffi_const_EVP_R_BAD_DECRYPT(unsigned long long *o)
{
  int n = (EVP_R_BAD_DECRYPT) <= 0;
  *o = (unsigned long long)((EVP_R_BAD_DECRYPT) | 0);  /* check that EVP_R_BAD_DECRYPT is an integer */
  return n;
}

static int _cffi_const_EVP_R_CAMELLIA_KEY_SETUP_FAILED(unsigned long long *o)
{
  int n = (EVP_R_CAMELLIA_KEY_SETUP_FAILED) <= 0;
  *o = (unsigned long long)((EVP_R_CAMELLIA_KEY_SETUP_FAILED) | 0);  /* check that EVP_R_CAMELLIA_KEY_SETUP_FAILED is an integer */
  return n;
}

static int _cffi_const_EVP_R_CIPHER_PARAMETER_ERROR(unsigned long long *o)
{
  int n = (EVP_R_CIPHER_PARAMETER_ERROR) <= 0;
  *o = (unsigned long long)((EVP_R_CIPHER_PARAMETER_ERROR) | 0);  /* check that EVP_R_CIPHER_PARAMETER_ERROR is an integer */
  return n;
}

static int _cffi_const_EVP_R_CTRL_NOT_IMPLEMENTED(unsigned long long *o)
{
  int n = (EVP_R_CTRL_NOT_IMPLEMENTED) <= 0;
  *o = (unsigned long long)((EVP_R_CTRL_NOT_IMPLEMENTED) | 0);  /* check that EVP_R_CTRL_NOT_IMPLEMENTED is an integer */
  return n;
}

static int _cffi_const_EVP_R_CTRL_OPERATION_NOT_IMPLEMENTED(unsigned long long *o)
{
  int n = (EVP_R_CTRL_OPERATION_NOT_IMPLEMENTED) <= 0;
  *o = (unsigned long long)((EVP_R_CTRL_OPERATION_NOT_IMPLEMENTED) | 0);  /* check that EVP_R_CTRL_OPERATION_NOT_IMPLEMENTED is an integer */
  return n;
}

static int _cffi_const_EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH(unsigned long long *o)
{
  int n = (EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) <= 0;
  *o = (unsigned long long)((EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) | 0);  /* check that EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH is an integer */
  return n;
}

static int _cffi_const_EVP_R_DECODE_ERROR(unsigned long long *o)
{
  int n = (EVP_R_DECODE_ERROR) <= 0;
  *o = (unsigned long long)((EVP_R_DECODE_ERROR) | 0);  /* check that EVP_R_DECODE_ERROR is an integer */
  return n;
}

static int _cffi_const_EVP_R_DIFFERENT_KEY_TYPES(unsigned long long *o)
{
  int n = (EVP_R_DIFFERENT_KEY_TYPES) <= 0;
  *o = (unsigned long long)((EVP_R_DIFFERENT_KEY_TYPES) | 0);  /* check that EVP_R_DIFFERENT_KEY_TYPES is an integer */
  return n;
}

static int _cffi_const_EVP_R_INITIALIZATION_ERROR(unsigned long long *o)
{
  int n = (EVP_R_INITIALIZATION_ERROR) <= 0;
  *o = (unsigned long long)((EVP_R_INITIALIZATION_ERROR) | 0);  /* check that EVP_R_INITIALIZATION_ERROR is an integer */
  return n;
}

static int _cffi_const_EVP_R_INPUT_NOT_INITIALIZED(unsigned long long *o)
{
  int n = (EVP_R_INPUT_NOT_INITIALIZED) <= 0;
  *o = (unsigned long long)((EVP_R_INPUT_NOT_INITIALIZED) | 0);  /* check that EVP_R_INPUT_NOT_INITIALIZED is an integer */
  return n;
}

static int _cffi_const_EVP_R_INVALID_KEY_LENGTH(unsigned long long *o)
{
  int n = (EVP_R_INVALID_KEY_LENGTH) <= 0;
  *o = (unsigned long long)((EVP_R_INVALID_KEY_LENGTH) | 0);  /* check that EVP_R_INVALID_KEY_LENGTH is an integer */
  return n;
}

static int _cffi_const_EVP_R_MEMORY_LIMIT_EXCEEDED(unsigned long long *o)
{
  int n = (EVP_R_MEMORY_LIMIT_EXCEEDED) <= 0;
  *o = (unsigned long long)((EVP_R_MEMORY_LIMIT_EXCEEDED) | 0);  /* check that EVP_R_MEMORY_LIMIT_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_EVP_R_MISSING_PARAMETERS(unsigned long long *o)
{
  int n = (EVP_R_MISSING_PARAMETERS) <= 0;
  *o = (unsigned long long)((EVP_R_MISSING_PARAMETERS) | 0);  /* check that EVP_R_MISSING_PARAMETERS is an integer */
  return n;
}

static int _cffi_const_EVP_R_NO_CIPHER_SET(unsigned long long *o)
{
  int n = (EVP_R_NO_CIPHER_SET) <= 0;
  *o = (unsigned long long)((EVP_R_NO_CIPHER_SET) | 0);  /* check that EVP_R_NO_CIPHER_SET is an integer */
  return n;
}

static int _cffi_const_EVP_R_NO_DIGEST_SET(unsigned long long *o)
{
  int n = (EVP_R_NO_DIGEST_SET) <= 0;
  *o = (unsigned long long)((EVP_R_NO_DIGEST_SET) | 0);  /* check that EVP_R_NO_DIGEST_SET is an integer */
  return n;
}

static int _cffi_const_EVP_R_PUBLIC_KEY_NOT_RSA(unsigned long long *o)
{
  int n = (EVP_R_PUBLIC_KEY_NOT_RSA) <= 0;
  *o = (unsigned long long)((EVP_R_PUBLIC_KEY_NOT_RSA) | 0);  /* check that EVP_R_PUBLIC_KEY_NOT_RSA is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNKNOWN_PBE_ALGORITHM(unsigned long long *o)
{
  int n = (EVP_R_UNKNOWN_PBE_ALGORITHM) <= 0;
  *o = (unsigned long long)((EVP_R_UNKNOWN_PBE_ALGORITHM) | 0);  /* check that EVP_R_UNKNOWN_PBE_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNSUPPORTED_CIPHER(unsigned long long *o)
{
  int n = (EVP_R_UNSUPPORTED_CIPHER) <= 0;
  *o = (unsigned long long)((EVP_R_UNSUPPORTED_CIPHER) | 0);  /* check that EVP_R_UNSUPPORTED_CIPHER is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNSUPPORTED_KEYLENGTH(unsigned long long *o)
{
  int n = (EVP_R_UNSUPPORTED_KEYLENGTH) <= 0;
  *o = (unsigned long long)((EVP_R_UNSUPPORTED_KEYLENGTH) | 0);  /* check that EVP_R_UNSUPPORTED_KEYLENGTH is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNSUPPORTED_KEY_DERIVATION_FUNCTION(unsigned long long *o)
{
  int n = (EVP_R_UNSUPPORTED_KEY_DERIVATION_FUNCTION) <= 0;
  *o = (unsigned long long)((EVP_R_UNSUPPORTED_KEY_DERIVATION_FUNCTION) | 0);  /* check that EVP_R_UNSUPPORTED_KEY_DERIVATION_FUNCTION is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM(unsigned long long *o)
{
  int n = (EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM) <= 0;
  *o = (unsigned long long)((EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM) | 0);  /* check that EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNSUPPORTED_SALT_TYPE(unsigned long long *o)
{
  int n = (EVP_R_UNSUPPORTED_SALT_TYPE) <= 0;
  *o = (unsigned long long)((EVP_R_UNSUPPORTED_SALT_TYPE) | 0);  /* check that EVP_R_UNSUPPORTED_SALT_TYPE is an integer */
  return n;
}

static int _cffi_const_EVP_R_WRONG_FINAL_BLOCK_LENGTH(unsigned long long *o)
{
  int n = (EVP_R_WRONG_FINAL_BLOCK_LENGTH) <= 0;
  *o = (unsigned long long)((EVP_R_WRONG_FINAL_BLOCK_LENGTH) | 0);  /* check that EVP_R_WRONG_FINAL_BLOCK_LENGTH is an integer */
  return n;
}

static int _cffi_const_GEN_DIRNAME(unsigned long long *o)
{
  int n = (GEN_DIRNAME) <= 0;
  *o = (unsigned long long)((GEN_DIRNAME) | 0);  /* check that GEN_DIRNAME is an integer */
  return n;
}

static int _cffi_const_GEN_DNS(unsigned long long *o)
{
  int n = (GEN_DNS) <= 0;
  *o = (unsigned long long)((GEN_DNS) | 0);  /* check that GEN_DNS is an integer */
  return n;
}

static int _cffi_const_GEN_EDIPARTY(unsigned long long *o)
{
  int n = (GEN_EDIPARTY) <= 0;
  *o = (unsigned long long)((GEN_EDIPARTY) | 0);  /* check that GEN_EDIPARTY is an integer */
  return n;
}

static int _cffi_const_GEN_EMAIL(unsigned long long *o)
{
  int n = (GEN_EMAIL) <= 0;
  *o = (unsigned long long)((GEN_EMAIL) | 0);  /* check that GEN_EMAIL is an integer */
  return n;
}

static int _cffi_const_GEN_IPADD(unsigned long long *o)
{
  int n = (GEN_IPADD) <= 0;
  *o = (unsigned long long)((GEN_IPADD) | 0);  /* check that GEN_IPADD is an integer */
  return n;
}

static int _cffi_const_GEN_OTHERNAME(unsigned long long *o)
{
  int n = (GEN_OTHERNAME) <= 0;
  *o = (unsigned long long)((GEN_OTHERNAME) | 0);  /* check that GEN_OTHERNAME is an integer */
  return n;
}

static int _cffi_const_GEN_RID(unsigned long long *o)
{
  int n = (GEN_RID) <= 0;
  *o = (unsigned long long)((GEN_RID) | 0);  /* check that GEN_RID is an integer */
  return n;
}

static int _cffi_const_GEN_URI(unsigned long long *o)
{
  int n = (GEN_URI) <= 0;
  *o = (unsigned long long)((GEN_URI) | 0);  /* check that GEN_URI is an integer */
  return n;
}

static int _cffi_const_GEN_X400(unsigned long long *o)
{
  int n = (GEN_X400) <= 0;
  *o = (unsigned long long)((GEN_X400) | 0);  /* check that GEN_X400 is an integer */
  return n;
}

static int _cffi_const_MBSTRING_UTF8(unsigned long long *o)
{
  int n = (MBSTRING_UTF8) <= 0;
  *o = (unsigned long long)((MBSTRING_UTF8) | 0);  /* check that MBSTRING_UTF8 is an integer */
  return n;
}

static int _cffi_const_NID_ED25519(unsigned long long *o)
{
  int n = (NID_ED25519) <= 0;
  *o = (unsigned long long)((NID_ED25519) | 0);  /* check that NID_ED25519 is an integer */
  return n;
}

static int _cffi_const_NID_ED448(unsigned long long *o)
{
  int n = (NID_ED448) <= 0;
  *o = (unsigned long long)((NID_ED448) | 0);  /* check that NID_ED448 is an integer */
  return n;
}

static int _cffi_const_NID_X25519(unsigned long long *o)
{
  int n = (NID_X25519) <= 0;
  *o = (unsigned long long)((NID_X25519) | 0);  /* check that NID_X25519 is an integer */
  return n;
}

static int _cffi_const_NID_X448(unsigned long long *o)
{
  int n = (NID_X448) <= 0;
  *o = (unsigned long long)((NID_X448) | 0);  /* check that NID_X448 is an integer */
  return n;
}

static int _cffi_const_NID_X9_62_prime256v1(unsigned long long *o)
{
  int n = (NID_X9_62_prime256v1) <= 0;
  *o = (unsigned long long)((NID_X9_62_prime256v1) | 0);  /* check that NID_X9_62_prime256v1 is an integer */
  return n;
}

static int _cffi_const_NID_ad_OCSP(unsigned long long *o)
{
  int n = (NID_ad_OCSP) <= 0;
  *o = (unsigned long long)((NID_ad_OCSP) | 0);  /* check that NID_ad_OCSP is an integer */
  return n;
}

static int _cffi_const_NID_ad_ca_issuers(unsigned long long *o)
{
  int n = (NID_ad_ca_issuers) <= 0;
  *o = (unsigned long long)((NID_ad_ca_issuers) | 0);  /* check that NID_ad_ca_issuers is an integer */
  return n;
}

static int _cffi_const_NID_crl_distribution_points(unsigned long long *o)
{
  int n = (NID_crl_distribution_points) <= 0;
  *o = (unsigned long long)((NID_crl_distribution_points) | 0);  /* check that NID_crl_distribution_points is an integer */
  return n;
}

static int _cffi_const_NID_crl_reason(unsigned long long *o)
{
  int n = (NID_crl_reason) <= 0;
  *o = (unsigned long long)((NID_crl_reason) | 0);  /* check that NID_crl_reason is an integer */
  return n;
}

static int _cffi_const_NID_info_access(unsigned long long *o)
{
  int n = (NID_info_access) <= 0;
  *o = (unsigned long long)((NID_info_access) | 0);  /* check that NID_info_access is an integer */
  return n;
}

static int _cffi_const_NID_pbe_WithSHA1And3_Key_TripleDES_CBC(unsigned long long *o)
{
  int n = (NID_pbe_WithSHA1And3_Key_TripleDES_CBC) <= 0;
  *o = (unsigned long long)((NID_pbe_WithSHA1And3_Key_TripleDES_CBC) | 0);  /* check that NID_pbe_WithSHA1And3_Key_TripleDES_CBC is an integer */
  return n;
}

static int _cffi_const_NID_poly1305(unsigned long long *o)
{
  int n = (NID_poly1305) <= 0;
  *o = (unsigned long long)((NID_poly1305) | 0);  /* check that NID_poly1305 is an integer */
  return n;
}

static int _cffi_const_NID_subject_alt_name(unsigned long long *o)
{
  int n = (NID_subject_alt_name) <= 0;
  *o = (unsigned long long)((NID_subject_alt_name) | 0);  /* check that NID_subject_alt_name is an integer */
  return n;
}

static int _cffi_const_NID_undef(unsigned long long *o)
{
  int n = (NID_undef) <= 0;
  *o = (unsigned long long)((NID_undef) | 0);  /* check that NID_undef is an integer */
  return n;
}

static int _cffi_const_OBJ_NAME_TYPE_MD_METH(unsigned long long *o)
{
  int n = (OBJ_NAME_TYPE_MD_METH) <= 0;
  *o = (unsigned long long)((OBJ_NAME_TYPE_MD_METH) | 0);  /* check that OBJ_NAME_TYPE_MD_METH is an integer */
  return n;
}

static int _cffi_const_OCSP_NOCERTS(unsigned long long *o)
{
  int n = (OCSP_NOCERTS) <= 0;
  *o = (unsigned long long)((OCSP_NOCERTS) | 0);  /* check that OCSP_NOCERTS is an integer */
  return n;
}

static int _cffi_const_OCSP_RESPID_KEY(unsigned long long *o)
{
  int n = (OCSP_RESPID_KEY) <= 0;
  *o = (unsigned long long)((OCSP_RESPID_KEY) | 0);  /* check that OCSP_RESPID_KEY is an integer */
  return n;
}

static int _cffi_const_OPENSSL_BUILT_ON(unsigned long long *o)
{
  int n = (OPENSSL_BUILT_ON) <= 0;
  *o = (unsigned long long)((OPENSSL_BUILT_ON) | 0);  /* check that OPENSSL_BUILT_ON is an integer */
  return n;
}

static int _cffi_const_OPENSSL_CFLAGS(unsigned long long *o)
{
  int n = (OPENSSL_CFLAGS) <= 0;
  *o = (unsigned long long)((OPENSSL_CFLAGS) | 0);  /* check that OPENSSL_CFLAGS is an integer */
  return n;
}

static int _cffi_const_OPENSSL_DIR(unsigned long long *o)
{
  int n = (OPENSSL_DIR) <= 0;
  *o = (unsigned long long)((OPENSSL_DIR) | 0);  /* check that OPENSSL_DIR is an integer */
  return n;
}

static int _cffi_const_OPENSSL_EC_NAMED_CURVE(unsigned long long *o)
{
  int n = (OPENSSL_EC_NAMED_CURVE) <= 0;
  *o = (unsigned long long)((OPENSSL_EC_NAMED_CURVE) | 0);  /* check that OPENSSL_EC_NAMED_CURVE is an integer */
  return n;
}

static int _cffi_const_OPENSSL_NPN_NEGOTIATED(unsigned long long *o)
{
  int n = (OPENSSL_NPN_NEGOTIATED) <= 0;
  *o = (unsigned long long)((OPENSSL_NPN_NEGOTIATED) | 0);  /* check that OPENSSL_NPN_NEGOTIATED is an integer */
  return n;
}

static int _cffi_const_OPENSSL_PLATFORM(unsigned long long *o)
{
  int n = (OPENSSL_PLATFORM) <= 0;
  *o = (unsigned long long)((OPENSSL_PLATFORM) | 0);  /* check that OPENSSL_PLATFORM is an integer */
  return n;
}

static int _cffi_const_OPENSSL_VERSION(unsigned long long *o)
{
  int n = (OPENSSL_VERSION) <= 0;
  *o = (unsigned long long)((OPENSSL_VERSION) | 0);  /* check that OPENSSL_VERSION is an integer */
  return n;
}

static int _cffi_const_OPENSSL_VERSION_NUMBER(unsigned long long *o)
{
  int n = (OPENSSL_VERSION_NUMBER) <= 0;
  *o = (unsigned long long)((OPENSSL_VERSION_NUMBER) | 0);  /* check that OPENSSL_VERSION_NUMBER is an integer */
  return n;
}

static void _cffi_const_OPENSSL_VERSION_TEXT(char *o)
{
  *(char const * *)o = OPENSSL_VERSION_TEXT;
}

static int _cffi_const_PEM_R_BAD_BASE64_DECODE(unsigned long long *o)
{
  int n = (PEM_R_BAD_BASE64_DECODE) <= 0;
  *o = (unsigned long long)((PEM_R_BAD_BASE64_DECODE) | 0);  /* check that PEM_R_BAD_BASE64_DECODE is an integer */
  return n;
}

static int _cffi_const_PEM_R_BAD_DECRYPT(unsigned long long *o)
{
  int n = (PEM_R_BAD_DECRYPT) <= 0;
  *o = (unsigned long long)((PEM_R_BAD_DECRYPT) | 0);  /* check that PEM_R_BAD_DECRYPT is an integer */
  return n;
}

static int _cffi_const_PEM_R_BAD_END_LINE(unsigned long long *o)
{
  int n = (PEM_R_BAD_END_LINE) <= 0;
  *o = (unsigned long long)((PEM_R_BAD_END_LINE) | 0);  /* check that PEM_R_BAD_END_LINE is an integer */
  return n;
}

static int _cffi_const_PEM_R_BAD_IV_CHARS(unsigned long long *o)
{
  int n = (PEM_R_BAD_IV_CHARS) <= 0;
  *o = (unsigned long long)((PEM_R_BAD_IV_CHARS) | 0);  /* check that PEM_R_BAD_IV_CHARS is an integer */
  return n;
}

static int _cffi_const_PEM_R_BAD_PASSWORD_READ(unsigned long long *o)
{
  int n = (PEM_R_BAD_PASSWORD_READ) <= 0;
  *o = (unsigned long long)((PEM_R_BAD_PASSWORD_READ) | 0);  /* check that PEM_R_BAD_PASSWORD_READ is an integer */
  return n;
}

static int _cffi_const_PEM_R_ERROR_CONVERTING_PRIVATE_KEY(unsigned long long *o)
{
  int n = (PEM_R_ERROR_CONVERTING_PRIVATE_KEY) <= 0;
  *o = (unsigned long long)((PEM_R_ERROR_CONVERTING_PRIVATE_KEY) | 0);  /* check that PEM_R_ERROR_CONVERTING_PRIVATE_KEY is an integer */
  return n;
}

static int _cffi_const_PEM_R_NOT_DEK_INFO(unsigned long long *o)
{
  int n = (PEM_R_NOT_DEK_INFO) <= 0;
  *o = (unsigned long long)((PEM_R_NOT_DEK_INFO) | 0);  /* check that PEM_R_NOT_DEK_INFO is an integer */
  return n;
}

static int _cffi_const_PEM_R_NOT_ENCRYPTED(unsigned long long *o)
{
  int n = (PEM_R_NOT_ENCRYPTED) <= 0;
  *o = (unsigned long long)((PEM_R_NOT_ENCRYPTED) | 0);  /* check that PEM_R_NOT_ENCRYPTED is an integer */
  return n;
}

static int _cffi_const_PEM_R_NOT_PROC_TYPE(unsigned long long *o)
{
  int n = (PEM_R_NOT_PROC_TYPE) <= 0;
  *o = (unsigned long long)((PEM_R_NOT_PROC_TYPE) | 0);  /* check that PEM_R_NOT_PROC_TYPE is an integer */
  return n;
}

static int _cffi_const_PEM_R_NO_START_LINE(unsigned long long *o)
{
  int n = (PEM_R_NO_START_LINE) <= 0;
  *o = (unsigned long long)((PEM_R_NO_START_LINE) | 0);  /* check that PEM_R_NO_START_LINE is an integer */
  return n;
}

static int _cffi_const_PEM_R_PROBLEMS_GETTING_PASSWORD(unsigned long long *o)
{
  int n = (PEM_R_PROBLEMS_GETTING_PASSWORD) <= 0;
  *o = (unsigned long long)((PEM_R_PROBLEMS_GETTING_PASSWORD) | 0);  /* check that PEM_R_PROBLEMS_GETTING_PASSWORD is an integer */
  return n;
}

static int _cffi_const_PEM_R_READ_KEY(unsigned long long *o)
{
  int n = (PEM_R_READ_KEY) <= 0;
  *o = (unsigned long long)((PEM_R_READ_KEY) | 0);  /* check that PEM_R_READ_KEY is an integer */
  return n;
}

static int _cffi_const_PEM_R_SHORT_HEADER(unsigned long long *o)
{
  int n = (PEM_R_SHORT_HEADER) <= 0;
  *o = (unsigned long long)((PEM_R_SHORT_HEADER) | 0);  /* check that PEM_R_SHORT_HEADER is an integer */
  return n;
}

static int _cffi_const_PEM_R_UNSUPPORTED_CIPHER(unsigned long long *o)
{
  int n = (PEM_R_UNSUPPORTED_CIPHER) <= 0;
  *o = (unsigned long long)((PEM_R_UNSUPPORTED_CIPHER) | 0);  /* check that PEM_R_UNSUPPORTED_CIPHER is an integer */
  return n;
}

static int _cffi_const_PEM_R_UNSUPPORTED_ENCRYPTION(unsigned long long *o)
{
  int n = (PEM_R_UNSUPPORTED_ENCRYPTION) <= 0;
  *o = (unsigned long long)((PEM_R_UNSUPPORTED_ENCRYPTION) | 0);  /* check that PEM_R_UNSUPPORTED_ENCRYPTION is an integer */
  return n;
}

static int _cffi_const_PKCS12_R_PKCS12_CIPHERFINAL_ERROR(unsigned long long *o)
{
  int n = (PKCS12_R_PKCS12_CIPHERFINAL_ERROR) <= 0;
  *o = (unsigned long long)((PKCS12_R_PKCS12_CIPHERFINAL_ERROR) | 0);  /* check that PKCS12_R_PKCS12_CIPHERFINAL_ERROR is an integer */
  return n;
}

static int _cffi_const_PKCS7_BINARY(unsigned long long *o)
{
  int n = (PKCS7_BINARY) <= 0;
  *o = (unsigned long long)((PKCS7_BINARY) | 0);  /* check that PKCS7_BINARY is an integer */
  return n;
}

static int _cffi_const_PKCS7_DETACHED(unsigned long long *o)
{
  int n = (PKCS7_DETACHED) <= 0;
  *o = (unsigned long long)((PKCS7_DETACHED) | 0);  /* check that PKCS7_DETACHED is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOATTR(unsigned long long *o)
{
  int n = (PKCS7_NOATTR) <= 0;
  *o = (unsigned long long)((PKCS7_NOATTR) | 0);  /* check that PKCS7_NOATTR is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOCERTS(unsigned long long *o)
{
  int n = (PKCS7_NOCERTS) <= 0;
  *o = (unsigned long long)((PKCS7_NOCERTS) | 0);  /* check that PKCS7_NOCERTS is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOCHAIN(unsigned long long *o)
{
  int n = (PKCS7_NOCHAIN) <= 0;
  *o = (unsigned long long)((PKCS7_NOCHAIN) | 0);  /* check that PKCS7_NOCHAIN is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOINTERN(unsigned long long *o)
{
  int n = (PKCS7_NOINTERN) <= 0;
  *o = (unsigned long long)((PKCS7_NOINTERN) | 0);  /* check that PKCS7_NOINTERN is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOSIGS(unsigned long long *o)
{
  int n = (PKCS7_NOSIGS) <= 0;
  *o = (unsigned long long)((PKCS7_NOSIGS) | 0);  /* check that PKCS7_NOSIGS is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOSMIMECAP(unsigned long long *o)
{
  int n = (PKCS7_NOSMIMECAP) <= 0;
  *o = (unsigned long long)((PKCS7_NOSMIMECAP) | 0);  /* check that PKCS7_NOSMIMECAP is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOVERIFY(unsigned long long *o)
{
  int n = (PKCS7_NOVERIFY) <= 0;
  *o = (unsigned long long)((PKCS7_NOVERIFY) | 0);  /* check that PKCS7_NOVERIFY is an integer */
  return n;
}

static int _cffi_const_PKCS7_STREAM(unsigned long long *o)
{
  int n = (PKCS7_STREAM) <= 0;
  *o = (unsigned long long)((PKCS7_STREAM) | 0);  /* check that PKCS7_STREAM is an integer */
  return n;
}

static int _cffi_const_PKCS7_TEXT(unsigned long long *o)
{
  int n = (PKCS7_TEXT) <= 0;
  *o = (unsigned long long)((PKCS7_TEXT) | 0);  /* check that PKCS7_TEXT is an integer */
  return n;
}

static int _cffi_const_PROV_R_BAD_DECRYPT(unsigned long long *o)
{
  int n = (PROV_R_BAD_DECRYPT) <= 0;
  *o = (unsigned long long)((PROV_R_BAD_DECRYPT) | 0);  /* check that PROV_R_BAD_DECRYPT is an integer */
  return n;
}

static int _cffi_const_PROV_R_WRONG_FINAL_BLOCK_LENGTH(unsigned long long *o)
{
  int n = (PROV_R_WRONG_FINAL_BLOCK_LENGTH) <= 0;
  *o = (unsigned long long)((PROV_R_WRONG_FINAL_BLOCK_LENGTH) | 0);  /* check that PROV_R_WRONG_FINAL_BLOCK_LENGTH is an integer */
  return n;
}

static int _cffi_const_PROV_R_XTS_DUPLICATED_KEYS(unsigned long long *o)
{
  int n = (PROV_R_XTS_DUPLICATED_KEYS) <= 0;
  *o = (unsigned long long)((PROV_R_XTS_DUPLICATED_KEYS) | 0);  /* check that PROV_R_XTS_DUPLICATED_KEYS is an integer */
  return n;
}

static int _cffi_const_RSA_F4(unsigned long long *o)
{
  int n = (RSA_F4) <= 0;
  *o = (unsigned long long)((RSA_F4) | 0);  /* check that RSA_F4 is an integer */
  return n;
}

static int _cffi_const_RSA_NO_PADDING(unsigned long long *o)
{
  int n = (RSA_NO_PADDING) <= 0;
  *o = (unsigned long long)((RSA_NO_PADDING) | 0);  /* check that RSA_NO_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_OAEP_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_OAEP_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_OAEP_PADDING) | 0);  /* check that RSA_PKCS1_OAEP_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_PADDING) | 0);  /* check that RSA_PKCS1_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_PSS_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_PSS_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_PSS_PADDING) | 0);  /* check that RSA_PKCS1_PSS_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_R_BLOCK_TYPE_IS_NOT_01(unsigned long long *o)
{
  int n = (RSA_R_BLOCK_TYPE_IS_NOT_01) <= 0;
  *o = (unsigned long long)((RSA_R_BLOCK_TYPE_IS_NOT_01) | 0);  /* check that RSA_R_BLOCK_TYPE_IS_NOT_01 is an integer */
  return n;
}

static int _cffi_const_RSA_R_BLOCK_TYPE_IS_NOT_02(unsigned long long *o)
{
  int n = (RSA_R_BLOCK_TYPE_IS_NOT_02) <= 0;
  *o = (unsigned long long)((RSA_R_BLOCK_TYPE_IS_NOT_02) | 0);  /* check that RSA_R_BLOCK_TYPE_IS_NOT_02 is an integer */
  return n;
}

static int _cffi_const_RSA_R_DATA_TOO_LARGE_FOR_KEY_SIZE(unsigned long long *o)
{
  int n = (RSA_R_DATA_TOO_LARGE_FOR_KEY_SIZE) <= 0;
  *o = (unsigned long long)((RSA_R_DATA_TOO_LARGE_FOR_KEY_SIZE) | 0);  /* check that RSA_R_DATA_TOO_LARGE_FOR_KEY_SIZE is an integer */
  return n;
}

static int _cffi_const_RSA_R_DATA_TOO_LARGE_FOR_MODULUS(unsigned long long *o)
{
  int n = (RSA_R_DATA_TOO_LARGE_FOR_MODULUS) <= 0;
  *o = (unsigned long long)((RSA_R_DATA_TOO_LARGE_FOR_MODULUS) | 0);  /* check that RSA_R_DATA_TOO_LARGE_FOR_MODULUS is an integer */
  return n;
}

static int _cffi_const_RSA_R_DIGEST_TOO_BIG_FOR_RSA_KEY(unsigned long long *o)
{
  int n = (RSA_R_DIGEST_TOO_BIG_FOR_RSA_KEY) <= 0;
  *o = (unsigned long long)((RSA_R_DIGEST_TOO_BIG_FOR_RSA_KEY) | 0);  /* check that RSA_R_DIGEST_TOO_BIG_FOR_RSA_KEY is an integer */
  return n;
}

static int _cffi_const_RSA_R_OAEP_DECODING_ERROR(unsigned long long *o)
{
  int n = (RSA_R_OAEP_DECODING_ERROR) <= 0;
  *o = (unsigned long long)((RSA_R_OAEP_DECODING_ERROR) | 0);  /* check that RSA_R_OAEP_DECODING_ERROR is an integer */
  return n;
}

static int _cffi_const_RSA_R_PKCS_DECODING_ERROR(unsigned long long *o)
{
  int n = (RSA_R_PKCS_DECODING_ERROR) <= 0;
  *o = (unsigned long long)((RSA_R_PKCS_DECODING_ERROR) | 0);  /* check that RSA_R_PKCS_DECODING_ERROR is an integer */
  return n;
}

static int _cffi_const_SSL3_MT_CHANGE_CIPHER_SPEC(unsigned long long *o)
{
  int n = (SSL3_MT_CHANGE_CIPHER_SPEC) <= 0;
  *o = (unsigned long long)((SSL3_MT_CHANGE_CIPHER_SPEC) | 0);  /* check that SSL3_MT_CHANGE_CIPHER_SPEC is an integer */
  return n;
}

static int _cffi_const_SSL3_RANDOM_SIZE(unsigned long long *o)
{
  int n = (SSL3_RANDOM_SIZE) <= 0;
  *o = (unsigned long long)((SSL3_RANDOM_SIZE) | 0);  /* check that SSL3_RANDOM_SIZE is an integer */
  return n;
}

static int _cffi_const_SSL3_RT_ALERT(unsigned long long *o)
{
  int n = (SSL3_RT_ALERT) <= 0;
  *o = (unsigned long long)((SSL3_RT_ALERT) | 0);  /* check that SSL3_RT_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL3_RT_APPLICATION_DATA(unsigned long long *o)
{
  int n = (SSL3_RT_APPLICATION_DATA) <= 0;
  *o = (unsigned long long)((SSL3_RT_APPLICATION_DATA) | 0);  /* check that SSL3_RT_APPLICATION_DATA is an integer */
  return n;
}

static int _cffi_const_SSL3_RT_CHANGE_CIPHER_SPEC(unsigned long long *o)
{
  int n = (SSL3_RT_CHANGE_CIPHER_SPEC) <= 0;
  *o = (unsigned long long)((SSL3_RT_CHANGE_CIPHER_SPEC) | 0);  /* check that SSL3_RT_CHANGE_CIPHER_SPEC is an integer */
  return n;
}

static int _cffi_const_SSL3_RT_HANDSHAKE(unsigned long long *o)
{
  int n = (SSL3_RT_HANDSHAKE) <= 0;
  *o = (unsigned long long)((SSL3_RT_HANDSHAKE) | 0);  /* check that SSL3_RT_HANDSHAKE is an integer */
  return n;
}

static int _cffi_const_SSL3_RT_HEADER(unsigned long long *o)
{
  int n = (SSL3_RT_HEADER) <= 0;
  *o = (unsigned long long)((SSL3_RT_HEADER) | 0);  /* check that SSL3_RT_HEADER is an integer */
  return n;
}

static int _cffi_const_SSL3_RT_INNER_CONTENT_TYPE(unsigned long long *o)
{
  int n = (SSL3_RT_INNER_CONTENT_TYPE) <= 0;
  *o = (unsigned long long)((SSL3_RT_INNER_CONTENT_TYPE) | 0);  /* check that SSL3_RT_INNER_CONTENT_TYPE is an integer */
  return n;
}

static int _cffi_const_SSL3_VERSION(unsigned long long *o)
{
  int n = (SSL3_VERSION) <= 0;
  *o = (unsigned long long)((SSL3_VERSION) | 0);  /* check that SSL3_VERSION is an integer */
  return n;
}

static int _cffi_const_SSLEAY_BUILT_ON(unsigned long long *o)
{
  int n = (SSLEAY_BUILT_ON) <= 0;
  *o = (unsigned long long)((SSLEAY_BUILT_ON) | 0);  /* check that SSLEAY_BUILT_ON is an integer */
  return n;
}

static int _cffi_const_SSLEAY_CFLAGS(unsigned long long *o)
{
  int n = (SSLEAY_CFLAGS) <= 0;
  *o = (unsigned long long)((SSLEAY_CFLAGS) | 0);  /* check that SSLEAY_CFLAGS is an integer */
  return n;
}

static int _cffi_const_SSLEAY_DIR(unsigned long long *o)
{
  int n = (SSLEAY_DIR) <= 0;
  *o = (unsigned long long)((SSLEAY_DIR) | 0);  /* check that SSLEAY_DIR is an integer */
  return n;
}

static int _cffi_const_SSLEAY_PLATFORM(unsigned long long *o)
{
  int n = (SSLEAY_PLATFORM) <= 0;
  *o = (unsigned long long)((SSLEAY_PLATFORM) | 0);  /* check that SSLEAY_PLATFORM is an integer */
  return n;
}

static int _cffi_const_SSLEAY_VERSION(unsigned long long *o)
{
  int n = (SSLEAY_VERSION) <= 0;
  *o = (unsigned long long)((SSLEAY_VERSION) | 0);  /* check that SSLEAY_VERSION is an integer */
  return n;
}

static int _cffi_const_SSL_AD_ACCESS_DENIED(unsigned long long *o)
{
  int n = (SSL_AD_ACCESS_DENIED) <= 0;
  *o = (unsigned long long)((SSL_AD_ACCESS_DENIED) | 0);  /* check that SSL_AD_ACCESS_DENIED is an integer */
  return n;
}

static int _cffi_const_SSL_AD_BAD_CERTIFICATE(unsigned long long *o)
{
  int n = (SSL_AD_BAD_CERTIFICATE) <= 0;
  *o = (unsigned long long)((SSL_AD_BAD_CERTIFICATE) | 0);  /* check that SSL_AD_BAD_CERTIFICATE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_BAD_CERTIFICATE_HASH_VALUE(unsigned long long *o)
{
  int n = (SSL_AD_BAD_CERTIFICATE_HASH_VALUE) <= 0;
  *o = (unsigned long long)((SSL_AD_BAD_CERTIFICATE_HASH_VALUE) | 0);  /* check that SSL_AD_BAD_CERTIFICATE_HASH_VALUE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_BAD_CERTIFICATE_STATUS_RESPONSE(unsigned long long *o)
{
  int n = (SSL_AD_BAD_CERTIFICATE_STATUS_RESPONSE) <= 0;
  *o = (unsigned long long)((SSL_AD_BAD_CERTIFICATE_STATUS_RESPONSE) | 0);  /* check that SSL_AD_BAD_CERTIFICATE_STATUS_RESPONSE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_BAD_RECORD_MAC(unsigned long long *o)
{
  int n = (SSL_AD_BAD_RECORD_MAC) <= 0;
  *o = (unsigned long long)((SSL_AD_BAD_RECORD_MAC) | 0);  /* check that SSL_AD_BAD_RECORD_MAC is an integer */
  return n;
}

static int _cffi_const_SSL_AD_CERTIFICATE_EXPIRED(unsigned long long *o)
{
  int n = (SSL_AD_CERTIFICATE_EXPIRED) <= 0;
  *o = (unsigned long long)((SSL_AD_CERTIFICATE_EXPIRED) | 0);  /* check that SSL_AD_CERTIFICATE_EXPIRED is an integer */
  return n;
}

static int _cffi_const_SSL_AD_CERTIFICATE_REVOKED(unsigned long long *o)
{
  int n = (SSL_AD_CERTIFICATE_REVOKED) <= 0;
  *o = (unsigned long long)((SSL_AD_CERTIFICATE_REVOKED) | 0);  /* check that SSL_AD_CERTIFICATE_REVOKED is an integer */
  return n;
}

static int _cffi_const_SSL_AD_CERTIFICATE_UNKNOWN(unsigned long long *o)
{
  int n = (SSL_AD_CERTIFICATE_UNKNOWN) <= 0;
  *o = (unsigned long long)((SSL_AD_CERTIFICATE_UNKNOWN) | 0);  /* check that SSL_AD_CERTIFICATE_UNKNOWN is an integer */
  return n;
}

static int _cffi_const_SSL_AD_CERTIFICATE_UNOBTAINABLE(unsigned long long *o)
{
  int n = (SSL_AD_CERTIFICATE_UNOBTAINABLE) <= 0;
  *o = (unsigned long long)((SSL_AD_CERTIFICATE_UNOBTAINABLE) | 0);  /* check that SSL_AD_CERTIFICATE_UNOBTAINABLE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_CLOSE_NOTIFY(unsigned long long *o)
{
  int n = (SSL_AD_CLOSE_NOTIFY) <= 0;
  *o = (unsigned long long)((SSL_AD_CLOSE_NOTIFY) | 0);  /* check that SSL_AD_CLOSE_NOTIFY is an integer */
  return n;
}

static int _cffi_const_SSL_AD_DECODE_ERROR(unsigned long long *o)
{
  int n = (SSL_AD_DECODE_ERROR) <= 0;
  *o = (unsigned long long)((SSL_AD_DECODE_ERROR) | 0);  /* check that SSL_AD_DECODE_ERROR is an integer */
  return n;
}

static int _cffi_const_SSL_AD_DECOMPRESSION_FAILURE(unsigned long long *o)
{
  int n = (SSL_AD_DECOMPRESSION_FAILURE) <= 0;
  *o = (unsigned long long)((SSL_AD_DECOMPRESSION_FAILURE) | 0);  /* check that SSL_AD_DECOMPRESSION_FAILURE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_DECRYPT_ERROR(unsigned long long *o)
{
  int n = (SSL_AD_DECRYPT_ERROR) <= 0;
  *o = (unsigned long long)((SSL_AD_DECRYPT_ERROR) | 0);  /* check that SSL_AD_DECRYPT_ERROR is an integer */
  return n;
}

static int _cffi_const_SSL_AD_HANDSHAKE_FAILURE(unsigned long long *o)
{
  int n = (SSL_AD_HANDSHAKE_FAILURE) <= 0;
  *o = (unsigned long long)((SSL_AD_HANDSHAKE_FAILURE) | 0);  /* check that SSL_AD_HANDSHAKE_FAILURE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_ILLEGAL_PARAMETER(unsigned long long *o)
{
  int n = (SSL_AD_ILLEGAL_PARAMETER) <= 0;
  *o = (unsigned long long)((SSL_AD_ILLEGAL_PARAMETER) | 0);  /* check that SSL_AD_ILLEGAL_PARAMETER is an integer */
  return n;
}

static int _cffi_const_SSL_AD_INSUFFICIENT_SECURITY(unsigned long long *o)
{
  int n = (SSL_AD_INSUFFICIENT_SECURITY) <= 0;
  *o = (unsigned long long)((SSL_AD_INSUFFICIENT_SECURITY) | 0);  /* check that SSL_AD_INSUFFICIENT_SECURITY is an integer */
  return n;
}

static int _cffi_const_SSL_AD_INTERNAL_ERROR(unsigned long long *o)
{
  int n = (SSL_AD_INTERNAL_ERROR) <= 0;
  *o = (unsigned long long)((SSL_AD_INTERNAL_ERROR) | 0);  /* check that SSL_AD_INTERNAL_ERROR is an integer */
  return n;
}

static int _cffi_const_SSL_AD_NO_RENEGOTIATION(unsigned long long *o)
{
  int n = (SSL_AD_NO_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((SSL_AD_NO_RENEGOTIATION) | 0);  /* check that SSL_AD_NO_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_SSL_AD_PROTOCOL_VERSION(unsigned long long *o)
{
  int n = (SSL_AD_PROTOCOL_VERSION) <= 0;
  *o = (unsigned long long)((SSL_AD_PROTOCOL_VERSION) | 0);  /* check that SSL_AD_PROTOCOL_VERSION is an integer */
  return n;
}

static int _cffi_const_SSL_AD_RECORD_OVERFLOW(unsigned long long *o)
{
  int n = (SSL_AD_RECORD_OVERFLOW) <= 0;
  *o = (unsigned long long)((SSL_AD_RECORD_OVERFLOW) | 0);  /* check that SSL_AD_RECORD_OVERFLOW is an integer */
  return n;
}

static int _cffi_const_SSL_AD_UNEXPECTED_MESSAGE(unsigned long long *o)
{
  int n = (SSL_AD_UNEXPECTED_MESSAGE) <= 0;
  *o = (unsigned long long)((SSL_AD_UNEXPECTED_MESSAGE) | 0);  /* check that SSL_AD_UNEXPECTED_MESSAGE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_UNKNOWN_CA(unsigned long long *o)
{
  int n = (SSL_AD_UNKNOWN_CA) <= 0;
  *o = (unsigned long long)((SSL_AD_UNKNOWN_CA) | 0);  /* check that SSL_AD_UNKNOWN_CA is an integer */
  return n;
}

static int _cffi_const_SSL_AD_UNKNOWN_PSK_IDENTITY(unsigned long long *o)
{
  int n = (SSL_AD_UNKNOWN_PSK_IDENTITY) <= 0;
  *o = (unsigned long long)((SSL_AD_UNKNOWN_PSK_IDENTITY) | 0);  /* check that SSL_AD_UNKNOWN_PSK_IDENTITY is an integer */
  return n;
}

static int _cffi_const_SSL_AD_UNRECOGNIZED_NAME(unsigned long long *o)
{
  int n = (SSL_AD_UNRECOGNIZED_NAME) <= 0;
  *o = (unsigned long long)((SSL_AD_UNRECOGNIZED_NAME) | 0);  /* check that SSL_AD_UNRECOGNIZED_NAME is an integer */
  return n;
}

static int _cffi_const_SSL_AD_UNSUPPORTED_CERTIFICATE(unsigned long long *o)
{
  int n = (SSL_AD_UNSUPPORTED_CERTIFICATE) <= 0;
  *o = (unsigned long long)((SSL_AD_UNSUPPORTED_CERTIFICATE) | 0);  /* check that SSL_AD_UNSUPPORTED_CERTIFICATE is an integer */
  return n;
}

static int _cffi_const_SSL_AD_UNSUPPORTED_EXTENSION(unsigned long long *o)
{
  int n = (SSL_AD_UNSUPPORTED_EXTENSION) <= 0;
  *o = (unsigned long long)((SSL_AD_UNSUPPORTED_EXTENSION) | 0);  /* check that SSL_AD_UNSUPPORTED_EXTENSION is an integer */
  return n;
}

static int _cffi_const_SSL_AD_USER_CANCELLED(unsigned long long *o)
{
  int n = (SSL_AD_USER_CANCELLED) <= 0;
  *o = (unsigned long long)((SSL_AD_USER_CANCELLED) | 0);  /* check that SSL_AD_USER_CANCELLED is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ACCEPT_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_ACCEPT_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_ACCEPT_EXIT) | 0);  /* check that SSL_CB_ACCEPT_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ACCEPT_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_ACCEPT_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_ACCEPT_LOOP) | 0);  /* check that SSL_CB_ACCEPT_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_ALERT) | 0);  /* check that SSL_CB_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_CONNECT_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_CONNECT_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_CONNECT_EXIT) | 0);  /* check that SSL_CB_CONNECT_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_CONNECT_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_CONNECT_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_CONNECT_LOOP) | 0);  /* check that SSL_CB_CONNECT_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_EXIT) | 0);  /* check that SSL_CB_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_HANDSHAKE_DONE(unsigned long long *o)
{
  int n = (SSL_CB_HANDSHAKE_DONE) <= 0;
  *o = (unsigned long long)((SSL_CB_HANDSHAKE_DONE) | 0);  /* check that SSL_CB_HANDSHAKE_DONE is an integer */
  return n;
}

static int _cffi_const_SSL_CB_HANDSHAKE_START(unsigned long long *o)
{
  int n = (SSL_CB_HANDSHAKE_START) <= 0;
  *o = (unsigned long long)((SSL_CB_HANDSHAKE_START) | 0);  /* check that SSL_CB_HANDSHAKE_START is an integer */
  return n;
}

static int _cffi_const_SSL_CB_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_LOOP) | 0);  /* check that SSL_CB_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_READ(unsigned long long *o)
{
  int n = (SSL_CB_READ) <= 0;
  *o = (unsigned long long)((SSL_CB_READ) | 0);  /* check that SSL_CB_READ is an integer */
  return n;
}

static int _cffi_const_SSL_CB_READ_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_READ_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_READ_ALERT) | 0);  /* check that SSL_CB_READ_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_WRITE(unsigned long long *o)
{
  int n = (SSL_CB_WRITE) <= 0;
  *o = (unsigned long long)((SSL_CB_WRITE) | 0);  /* check that SSL_CB_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_CB_WRITE_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_WRITE_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_WRITE_ALERT) | 0);  /* check that SSL_CB_WRITE_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_NONE(unsigned long long *o)
{
  int n = (SSL_ERROR_NONE) <= 0;
  *o = (unsigned long long)((SSL_ERROR_NONE) | 0);  /* check that SSL_ERROR_NONE is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_SSL(unsigned long long *o)
{
  int n = (SSL_ERROR_SSL) <= 0;
  *o = (unsigned long long)((SSL_ERROR_SSL) | 0);  /* check that SSL_ERROR_SSL is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_SYSCALL(unsigned long long *o)
{
  int n = (SSL_ERROR_SYSCALL) <= 0;
  *o = (unsigned long long)((SSL_ERROR_SYSCALL) | 0);  /* check that SSL_ERROR_SYSCALL is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_CONNECT(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_CONNECT) | 0);  /* check that SSL_ERROR_WANT_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_READ(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_READ) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_READ) | 0);  /* check that SSL_ERROR_WANT_READ is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_WRITE(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_WRITE) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_WRITE) | 0);  /* check that SSL_ERROR_WANT_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_X509_LOOKUP(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_X509_LOOKUP) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_X509_LOOKUP) | 0);  /* check that SSL_ERROR_WANT_X509_LOOKUP is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_ZERO_RETURN(unsigned long long *o)
{
  int n = (SSL_ERROR_ZERO_RETURN) <= 0;
  *o = (unsigned long long)((SSL_ERROR_ZERO_RETURN) | 0);  /* check that SSL_ERROR_ZERO_RETURN is an integer */
  return n;
}

static int _cffi_const_SSL_FILETYPE_ASN1(unsigned long long *o)
{
  int n = (SSL_FILETYPE_ASN1) <= 0;
  *o = (unsigned long long)((SSL_FILETYPE_ASN1) | 0);  /* check that SSL_FILETYPE_ASN1 is an integer */
  return n;
}

static int _cffi_const_SSL_FILETYPE_PEM(unsigned long long *o)
{
  int n = (SSL_FILETYPE_PEM) <= 0;
  *o = (unsigned long long)((SSL_FILETYPE_PEM) | 0);  /* check that SSL_FILETYPE_PEM is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER(unsigned long long *o)
{
  int n = (SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER) <= 0;
  *o = (unsigned long long)((SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER) | 0);  /* check that SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_AUTO_RETRY(unsigned long long *o)
{
  int n = (SSL_MODE_AUTO_RETRY) <= 0;
  *o = (unsigned long long)((SSL_MODE_AUTO_RETRY) | 0);  /* check that SSL_MODE_AUTO_RETRY is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_ENABLE_PARTIAL_WRITE(unsigned long long *o)
{
  int n = (SSL_MODE_ENABLE_PARTIAL_WRITE) <= 0;
  *o = (unsigned long long)((SSL_MODE_ENABLE_PARTIAL_WRITE) | 0);  /* check that SSL_MODE_ENABLE_PARTIAL_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_RELEASE_BUFFERS(unsigned long long *o)
{
  int n = (SSL_MODE_RELEASE_BUFFERS) <= 0;
  *o = (unsigned long long)((SSL_MODE_RELEASE_BUFFERS) | 0);  /* check that SSL_MODE_RELEASE_BUFFERS is an integer */
  return n;
}

static int _cffi_const_SSL_OP_ALL(unsigned long long *o)
{
  int n = (SSL_OP_ALL) <= 0;
  *o = (unsigned long long)((SSL_OP_ALL) | 0);  /* check that SSL_OP_ALL is an integer */
  return n;
}

static int _cffi_const_SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION(unsigned long long *o)
{
  int n = (SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION) | 0);  /* check that SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_SSL_OP_CIPHER_SERVER_PREFERENCE(unsigned long long *o)
{
  int n = (SSL_OP_CIPHER_SERVER_PREFERENCE) <= 0;
  *o = (unsigned long long)((SSL_OP_CIPHER_SERVER_PREFERENCE) | 0);  /* check that SSL_OP_CIPHER_SERVER_PREFERENCE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_COOKIE_EXCHANGE(unsigned long long *o)
{
  int n = (SSL_OP_COOKIE_EXCHANGE) <= 0;
  *o = (unsigned long long)((SSL_OP_COOKIE_EXCHANGE) | 0);  /* check that SSL_OP_COOKIE_EXCHANGE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS(unsigned long long *o)
{
  int n = (SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS) <= 0;
  *o = (unsigned long long)((SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS) | 0);  /* check that SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS is an integer */
  return n;
}

static int _cffi_const_SSL_OP_EPHEMERAL_RSA(unsigned long long *o)
{
  int n = (SSL_OP_EPHEMERAL_RSA) <= 0;
  *o = (unsigned long long)((SSL_OP_EPHEMERAL_RSA) | 0);  /* check that SSL_OP_EPHEMERAL_RSA is an integer */
  return n;
}

static int _cffi_const_SSL_OP_IGNORE_UNEXPECTED_EOF(unsigned long long *o)
{
  int n = (SSL_OP_IGNORE_UNEXPECTED_EOF) <= 0;
  *o = (unsigned long long)((SSL_OP_IGNORE_UNEXPECTED_EOF) | 0);  /* check that SSL_OP_IGNORE_UNEXPECTED_EOF is an integer */
  return n;
}

static int _cffi_const_SSL_OP_LEGACY_SERVER_CONNECT(unsigned long long *o)
{
  int n = (SSL_OP_LEGACY_SERVER_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_OP_LEGACY_SERVER_CONNECT) | 0);  /* check that SSL_OP_LEGACY_SERVER_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER(unsigned long long *o)
{
  int n = (SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER) <= 0;
  *o = (unsigned long long)((SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER) | 0);  /* check that SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MICROSOFT_SESS_ID_BUG(unsigned long long *o)
{
  int n = (SSL_OP_MICROSOFT_SESS_ID_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_MICROSOFT_SESS_ID_BUG) | 0);  /* check that SSL_OP_MICROSOFT_SESS_ID_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MSIE_SSLV2_RSA_PADDING(unsigned long long *o)
{
  int n = (SSL_OP_MSIE_SSLV2_RSA_PADDING) <= 0;
  *o = (unsigned long long)((SSL_OP_MSIE_SSLV2_RSA_PADDING) | 0);  /* check that SSL_OP_MSIE_SSLV2_RSA_PADDING is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_CA_DN_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_CA_DN_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_CA_DN_BUG) | 0);  /* check that SSL_OP_NETSCAPE_CA_DN_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_CHALLENGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_CHALLENGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_CHALLENGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_CHALLENGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_COMPRESSION(unsigned long long *o)
{
  int n = (SSL_OP_NO_COMPRESSION) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_COMPRESSION) | 0);  /* check that SSL_OP_NO_COMPRESSION is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_DTLSv1(unsigned long long *o)
{
  int n = (SSL_OP_NO_DTLSv1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_DTLSv1) | 0);  /* check that SSL_OP_NO_DTLSv1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_DTLSv1_2(unsigned long long *o)
{
  int n = (SSL_OP_NO_DTLSv1_2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_DTLSv1_2) | 0);  /* check that SSL_OP_NO_DTLSv1_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_QUERY_MTU(unsigned long long *o)
{
  int n = (SSL_OP_NO_QUERY_MTU) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_QUERY_MTU) | 0);  /* check that SSL_OP_NO_QUERY_MTU is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_SSLv2(unsigned long long *o)
{
  int n = (SSL_OP_NO_SSLv2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_SSLv2) | 0);  /* check that SSL_OP_NO_SSLv2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_SSLv3(unsigned long long *o)
{
  int n = (SSL_OP_NO_SSLv3) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_SSLv3) | 0);  /* check that SSL_OP_NO_SSLv3 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TICKET(unsigned long long *o)
{
  int n = (SSL_OP_NO_TICKET) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TICKET) | 0);  /* check that SSL_OP_NO_TICKET is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1) | 0);  /* check that SSL_OP_NO_TLSv1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_1(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_1) | 0);  /* check that SSL_OP_NO_TLSv1_1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_2(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_2) | 0);  /* check that SSL_OP_NO_TLSv1_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_3(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_3) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_3) | 0);  /* check that SSL_OP_NO_TLSv1_3 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_PKCS1_CHECK_1(unsigned long long *o)
{
  int n = (SSL_OP_PKCS1_CHECK_1) <= 0;
  *o = (unsigned long long)((SSL_OP_PKCS1_CHECK_1) | 0);  /* check that SSL_OP_PKCS1_CHECK_1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_PKCS1_CHECK_2(unsigned long long *o)
{
  int n = (SSL_OP_PKCS1_CHECK_2) <= 0;
  *o = (unsigned long long)((SSL_OP_PKCS1_CHECK_2) | 0);  /* check that SSL_OP_PKCS1_CHECK_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SINGLE_DH_USE(unsigned long long *o)
{
  int n = (SSL_OP_SINGLE_DH_USE) <= 0;
  *o = (unsigned long long)((SSL_OP_SINGLE_DH_USE) | 0);  /* check that SSL_OP_SINGLE_DH_USE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SINGLE_ECDH_USE(unsigned long long *o)
{
  int n = (SSL_OP_SINGLE_ECDH_USE) <= 0;
  *o = (unsigned long long)((SSL_OP_SINGLE_ECDH_USE) | 0);  /* check that SSL_OP_SINGLE_ECDH_USE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SSLEAY_080_CLIENT_DH_BUG(unsigned long long *o)
{
  int n = (SSL_OP_SSLEAY_080_CLIENT_DH_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_SSLEAY_080_CLIENT_DH_BUG) | 0);  /* check that SSL_OP_SSLEAY_080_CLIENT_DH_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG) | 0);  /* check that SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_BLOCK_PADDING_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_BLOCK_PADDING_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_BLOCK_PADDING_BUG) | 0);  /* check that SSL_OP_TLS_BLOCK_PADDING_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_D5_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_D5_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_D5_BUG) | 0);  /* check that SSL_OP_TLS_D5_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_ROLLBACK_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_ROLLBACK_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_ROLLBACK_BUG) | 0);  /* check that SSL_OP_TLS_ROLLBACK_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_RECEIVED_SHUTDOWN(unsigned long long *o)
{
  int n = (SSL_RECEIVED_SHUTDOWN) <= 0;
  *o = (unsigned long long)((SSL_RECEIVED_SHUTDOWN) | 0);  /* check that SSL_RECEIVED_SHUTDOWN is an integer */
  return n;
}

static int _cffi_const_SSL_SENT_SHUTDOWN(unsigned long long *o)
{
  int n = (SSL_SENT_SHUTDOWN) <= 0;
  *o = (unsigned long long)((SSL_SENT_SHUTDOWN) | 0);  /* check that SSL_SENT_SHUTDOWN is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_BOTH(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_BOTH) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_BOTH) | 0);  /* check that SSL_SESS_CACHE_BOTH is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_CLIENT(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_CLIENT) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_CLIENT) | 0);  /* check that SSL_SESS_CACHE_CLIENT is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_AUTO_CLEAR(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_AUTO_CLEAR) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_AUTO_CLEAR) | 0);  /* check that SSL_SESS_CACHE_NO_AUTO_CLEAR is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL_LOOKUP(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL_LOOKUP) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL_LOOKUP) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL_LOOKUP is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL_STORE(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL_STORE) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL_STORE) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL_STORE is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_OFF(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_OFF) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_OFF) | 0);  /* check that SSL_SESS_CACHE_OFF is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_SERVER(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_SERVER) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_SERVER) | 0);  /* check that SSL_SESS_CACHE_SERVER is an integer */
  return n;
}

static int _cffi_const_SSL_ST_ACCEPT(unsigned long long *o)
{
  int n = (SSL_ST_ACCEPT) <= 0;
  *o = (unsigned long long)((SSL_ST_ACCEPT) | 0);  /* check that SSL_ST_ACCEPT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_BEFORE(unsigned long long *o)
{
  int n = (SSL_ST_BEFORE) <= 0;
  *o = (unsigned long long)((SSL_ST_BEFORE) | 0);  /* check that SSL_ST_BEFORE is an integer */
  return n;
}

static int _cffi_const_SSL_ST_CONNECT(unsigned long long *o)
{
  int n = (SSL_ST_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_ST_CONNECT) | 0);  /* check that SSL_ST_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_INIT(unsigned long long *o)
{
  int n = (SSL_ST_INIT) <= 0;
  *o = (unsigned long long)((SSL_ST_INIT) | 0);  /* check that SSL_ST_INIT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_MASK(unsigned long long *o)
{
  int n = (SSL_ST_MASK) <= 0;
  *o = (unsigned long long)((SSL_ST_MASK) | 0);  /* check that SSL_ST_MASK is an integer */
  return n;
}

static int _cffi_const_SSL_ST_OK(unsigned long long *o)
{
  int n = (SSL_ST_OK) <= 0;
  *o = (unsigned long long)((SSL_ST_OK) | 0);  /* check that SSL_ST_OK is an integer */
  return n;
}

static int _cffi_const_SSL_ST_RENEGOTIATE(unsigned long long *o)
{
  int n = (SSL_ST_RENEGOTIATE) <= 0;
  *o = (unsigned long long)((SSL_ST_RENEGOTIATE) | 0);  /* check that SSL_ST_RENEGOTIATE is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_ALERT_FATAL(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_ALERT_FATAL) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_ALERT_FATAL) | 0);  /* check that SSL_TLSEXT_ERR_ALERT_FATAL is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_ALERT_WARNING(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_ALERT_WARNING) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_ALERT_WARNING) | 0);  /* check that SSL_TLSEXT_ERR_ALERT_WARNING is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_NOACK(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_NOACK) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_NOACK) | 0);  /* check that SSL_TLSEXT_ERR_NOACK is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_OK(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_OK) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_OK) | 0);  /* check that SSL_TLSEXT_ERR_OK is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_CLIENT_ONCE(unsigned long long *o)
{
  int n = (SSL_VERIFY_CLIENT_ONCE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_CLIENT_ONCE) | 0);  /* check that SSL_VERIFY_CLIENT_ONCE is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_FAIL_IF_NO_PEER_CERT(unsigned long long *o)
{
  int n = (SSL_VERIFY_FAIL_IF_NO_PEER_CERT) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_FAIL_IF_NO_PEER_CERT) | 0);  /* check that SSL_VERIFY_FAIL_IF_NO_PEER_CERT is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_NONE(unsigned long long *o)
{
  int n = (SSL_VERIFY_NONE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_NONE) | 0);  /* check that SSL_VERIFY_NONE is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_PEER(unsigned long long *o)
{
  int n = (SSL_VERIFY_PEER) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_PEER) | 0);  /* check that SSL_VERIFY_PEER is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_POST_HANDSHAKE(unsigned long long *o)
{
  int n = (SSL_VERIFY_POST_HANDSHAKE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_POST_HANDSHAKE) | 0);  /* check that SSL_VERIFY_POST_HANDSHAKE is an integer */
  return n;
}

static int _cffi_const_TLS1_1_VERSION(unsigned long long *o)
{
  int n = (TLS1_1_VERSION) <= 0;
  *o = (unsigned long long)((TLS1_1_VERSION) | 0);  /* check that TLS1_1_VERSION is an integer */
  return n;
}

static int _cffi_const_TLS1_2_VERSION(unsigned long long *o)
{
  int n = (TLS1_2_VERSION) <= 0;
  *o = (unsigned long long)((TLS1_2_VERSION) | 0);  /* check that TLS1_2_VERSION is an integer */
  return n;
}

static int _cffi_const_TLS1_VERSION(unsigned long long *o)
{
  int n = (TLS1_VERSION) <= 0;
  *o = (unsigned long long)((TLS1_VERSION) | 0);  /* check that TLS1_VERSION is an integer */
  return n;
}

static int _cffi_const_TLSEXT_NAMETYPE_host_name(unsigned long long *o)
{
  int n = (TLSEXT_NAMETYPE_host_name) <= 0;
  *o = (unsigned long long)((TLSEXT_NAMETYPE_host_name) | 0);  /* check that TLSEXT_NAMETYPE_host_name is an integer */
  return n;
}

static int _cffi_const_TLSEXT_STATUSTYPE_ocsp(unsigned long long *o)
{
  int n = (TLSEXT_STATUSTYPE_ocsp) <= 0;
  *o = (unsigned long long)((TLSEXT_STATUSTYPE_ocsp) | 0);  /* check that TLSEXT_STATUSTYPE_ocsp is an integer */
  return n;
}

static int _cffi_const_TLS_ST_BEFORE(unsigned long long *o)
{
  int n = (TLS_ST_BEFORE) <= 0;
  *o = (unsigned long long)((TLS_ST_BEFORE) | 0);  /* check that TLS_ST_BEFORE is an integer */
  return n;
}

static int _cffi_const_TLS_ST_OK(unsigned long long *o)
{
  int n = (TLS_ST_OK) <= 0;
  *o = (unsigned long long)((TLS_ST_OK) | 0);  /* check that TLS_ST_OK is an integer */
  return n;
}

static int _cffi_const_V_ASN1_GENERALIZEDTIME(unsigned long long *o)
{
  int n = (V_ASN1_GENERALIZEDTIME) <= 0;
  *o = (unsigned long long)((V_ASN1_GENERALIZEDTIME) | 0);  /* check that V_ASN1_GENERALIZEDTIME is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT) | 0);  /* check that X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) | 0);  /* check that X509_CHECK_FLAG_NEVER_CHECK_SUBJECT is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NO_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NO_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NO_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_NO_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS) | 0);  /* check that X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS is an integer */
  return n;
}

static int _cffi_const_X509_LU_CRL(unsigned long long *o)
{
  int n = (X509_LU_CRL) <= 0;
  *o = (unsigned long long)((X509_LU_CRL) | 0);  /* check that X509_LU_CRL is an integer */
  return n;
}

static int _cffi_const_X509_LU_X509(unsigned long long *o)
{
  int n = (X509_LU_X509) <= 0;
  *o = (unsigned long long)((X509_LU_X509) | 0);  /* check that X509_LU_X509 is an integer */
  return n;
}

static int _cffi_const_X509_R_CERT_ALREADY_IN_HASH_TABLE(unsigned long long *o)
{
  int n = (X509_R_CERT_ALREADY_IN_HASH_TABLE) <= 0;
  *o = (unsigned long long)((X509_R_CERT_ALREADY_IN_HASH_TABLE) | 0);  /* check that X509_R_CERT_ALREADY_IN_HASH_TABLE is an integer */
  return n;
}

static int _cffi_const_X509_R_KEY_VALUES_MISMATCH(unsigned long long *o)
{
  int n = (X509_R_KEY_VALUES_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_R_KEY_VALUES_MISMATCH) | 0);  /* check that X509_R_KEY_VALUES_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH) | 0);  /* check that X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_AKID_SKID_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_AKID_SKID_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_AKID_SKID_MISMATCH) | 0);  /* check that X509_V_ERR_AKID_SKID_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_APPLICATION_VERIFICATION(unsigned long long *o)
{
  int n = (X509_V_ERR_APPLICATION_VERIFICATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_APPLICATION_VERIFICATION) | 0);  /* check that X509_V_ERR_APPLICATION_VERIFICATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_CHAIN_TOO_LONG(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_CHAIN_TOO_LONG) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_CHAIN_TOO_LONG) | 0);  /* check that X509_V_ERR_CERT_CHAIN_TOO_LONG is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_HAS_EXPIRED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_HAS_EXPIRED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_HAS_EXPIRED) | 0);  /* check that X509_V_ERR_CERT_HAS_EXPIRED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_NOT_YET_VALID(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_NOT_YET_VALID) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_NOT_YET_VALID) | 0);  /* check that X509_V_ERR_CERT_NOT_YET_VALID is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_REJECTED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_REJECTED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_REJECTED) | 0);  /* check that X509_V_ERR_CERT_REJECTED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_REVOKED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_REVOKED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_REVOKED) | 0);  /* check that X509_V_ERR_CERT_REVOKED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_SIGNATURE_FAILURE(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_SIGNATURE_FAILURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_SIGNATURE_FAILURE) | 0);  /* check that X509_V_ERR_CERT_SIGNATURE_FAILURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_UNTRUSTED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_UNTRUSTED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_UNTRUSTED) | 0);  /* check that X509_V_ERR_CERT_UNTRUSTED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_HAS_EXPIRED(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_HAS_EXPIRED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_HAS_EXPIRED) | 0);  /* check that X509_V_ERR_CRL_HAS_EXPIRED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_NOT_YET_VALID(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_NOT_YET_VALID) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_NOT_YET_VALID) | 0);  /* check that X509_V_ERR_CRL_NOT_YET_VALID is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_PATH_VALIDATION_ERROR(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_PATH_VALIDATION_ERROR) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_PATH_VALIDATION_ERROR) | 0);  /* check that X509_V_ERR_CRL_PATH_VALIDATION_ERROR is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_SIGNATURE_FAILURE(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_SIGNATURE_FAILURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_SIGNATURE_FAILURE) | 0);  /* check that X509_V_ERR_CRL_SIGNATURE_FAILURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT(unsigned long long *o)
{
  int n = (X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) <= 0;
  *o = (unsigned long long)((X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) | 0);  /* check that X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_DIFFERENT_CRL_SCOPE(unsigned long long *o)
{
  int n = (X509_V_ERR_DIFFERENT_CRL_SCOPE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_DIFFERENT_CRL_SCOPE) | 0);  /* check that X509_V_ERR_DIFFERENT_CRL_SCOPE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_EMAIL_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_EMAIL_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_EMAIL_MISMATCH) | 0);  /* check that X509_V_ERR_EMAIL_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_EXCLUDED_VIOLATION(unsigned long long *o)
{
  int n = (X509_V_ERR_EXCLUDED_VIOLATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_EXCLUDED_VIOLATION) | 0);  /* check that X509_V_ERR_EXCLUDED_VIOLATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_HOSTNAME_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_HOSTNAME_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_HOSTNAME_MISMATCH) | 0);  /* check that X509_V_ERR_HOSTNAME_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_CA(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_CA) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_CA) | 0);  /* check that X509_V_ERR_INVALID_CA is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_EXTENSION) | 0);  /* check that X509_V_ERR_INVALID_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_NON_CA(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_NON_CA) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_NON_CA) | 0);  /* check that X509_V_ERR_INVALID_NON_CA is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_POLICY_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_POLICY_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_POLICY_EXTENSION) | 0);  /* check that X509_V_ERR_INVALID_POLICY_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_PURPOSE(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_PURPOSE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_PURPOSE) | 0);  /* check that X509_V_ERR_INVALID_PURPOSE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_IP_ADDRESS_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_IP_ADDRESS_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_IP_ADDRESS_MISMATCH) | 0);  /* check that X509_V_ERR_IP_ADDRESS_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_CERTSIGN(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_CERTSIGN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_CERTSIGN) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_CERTSIGN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_CRL_SIGN(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_CRL_SIGN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_CRL_SIGN) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_CRL_SIGN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_NO_EXPLICIT_POLICY(unsigned long long *o)
{
  int n = (X509_V_ERR_NO_EXPLICIT_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_NO_EXPLICIT_POLICY) | 0);  /* check that X509_V_ERR_NO_EXPLICIT_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_OUT_OF_MEM(unsigned long long *o)
{
  int n = (X509_V_ERR_OUT_OF_MEM) <= 0;
  *o = (unsigned long long)((X509_V_ERR_OUT_OF_MEM) | 0);  /* check that X509_V_ERR_OUT_OF_MEM is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PATH_LENGTH_EXCEEDED(unsigned long long *o)
{
  int n = (X509_V_ERR_PATH_LENGTH_EXCEEDED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PATH_LENGTH_EXCEEDED) | 0);  /* check that X509_V_ERR_PATH_LENGTH_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PERMITTED_VIOLATION(unsigned long long *o)
{
  int n = (X509_V_ERR_PERMITTED_VIOLATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PERMITTED_VIOLATION) | 0);  /* check that X509_V_ERR_PERMITTED_VIOLATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED(unsigned long long *o)
{
  int n = (X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED) | 0);  /* check that X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED(unsigned long long *o)
{
  int n = (X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED) | 0);  /* check that X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN(unsigned long long *o)
{
  int n = (X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN) | 0);  /* check that X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUBJECT_ISSUER_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_SUBJECT_ISSUER_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUBJECT_ISSUER_MISMATCH) | 0);  /* check that X509_V_ERR_SUBJECT_ISSUER_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUBTREE_MINMAX(unsigned long long *o)
{
  int n = (X509_V_ERR_SUBTREE_MINMAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUBTREE_MINMAX) | 0);  /* check that X509_V_ERR_SUBTREE_MINMAX is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256) | 0);  /* check that X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256 is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_ALGORITHM(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_ALGORITHM) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_ALGORITHM) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_CURVE(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_CURVE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_CURVE) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_CURVE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_VERSION(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_VERSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_VERSION) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_VERSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED) | 0);  /* check that X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_CRL(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_CRL) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_CRL) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_CRL is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION) | 0);  /* check that X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION) | 0);  /* check that X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNNESTED_RESOURCE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNNESTED_RESOURCE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNNESTED_RESOURCE) | 0);  /* check that X509_V_ERR_UNNESTED_RESOURCE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX) | 0);  /* check that X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE) | 0);  /* check that X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE) | 0);  /* check that X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_NAME_SYNTAX(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_NAME_SYNTAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_NAME_SYNTAX) | 0);  /* check that X509_V_ERR_UNSUPPORTED_NAME_SYNTAX is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_ALLOW_PROXY_CERTS(unsigned long long *o)
{
  int n = (X509_V_FLAG_ALLOW_PROXY_CERTS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_ALLOW_PROXY_CERTS) | 0);  /* check that X509_V_FLAG_ALLOW_PROXY_CERTS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CB_ISSUER_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_CB_ISSUER_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CB_ISSUER_CHECK) | 0);  /* check that X509_V_FLAG_CB_ISSUER_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CHECK_SS_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_FLAG_CHECK_SS_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CHECK_SS_SIGNATURE) | 0);  /* check that X509_V_FLAG_CHECK_SS_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CRL_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_CRL_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CRL_CHECK) | 0);  /* check that X509_V_FLAG_CRL_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CRL_CHECK_ALL(unsigned long long *o)
{
  int n = (X509_V_FLAG_CRL_CHECK_ALL) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CRL_CHECK_ALL) | 0);  /* check that X509_V_FLAG_CRL_CHECK_ALL is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_EXPLICIT_POLICY(unsigned long long *o)
{
  int n = (X509_V_FLAG_EXPLICIT_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_EXPLICIT_POLICY) | 0);  /* check that X509_V_FLAG_EXPLICIT_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_EXTENDED_CRL_SUPPORT(unsigned long long *o)
{
  int n = (X509_V_FLAG_EXTENDED_CRL_SUPPORT) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_EXTENDED_CRL_SUPPORT) | 0);  /* check that X509_V_FLAG_EXTENDED_CRL_SUPPORT is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_IGNORE_CRITICAL(unsigned long long *o)
{
  int n = (X509_V_FLAG_IGNORE_CRITICAL) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_IGNORE_CRITICAL) | 0);  /* check that X509_V_FLAG_IGNORE_CRITICAL is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_INHIBIT_ANY(unsigned long long *o)
{
  int n = (X509_V_FLAG_INHIBIT_ANY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_INHIBIT_ANY) | 0);  /* check that X509_V_FLAG_INHIBIT_ANY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_INHIBIT_MAP(unsigned long long *o)
{
  int n = (X509_V_FLAG_INHIBIT_MAP) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_INHIBIT_MAP) | 0);  /* check that X509_V_FLAG_INHIBIT_MAP is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_NOTIFY_POLICY(unsigned long long *o)
{
  int n = (X509_V_FLAG_NOTIFY_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_NOTIFY_POLICY) | 0);  /* check that X509_V_FLAG_NOTIFY_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_PARTIAL_CHAIN(unsigned long long *o)
{
  int n = (X509_V_FLAG_PARTIAL_CHAIN) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_PARTIAL_CHAIN) | 0);  /* check that X509_V_FLAG_PARTIAL_CHAIN is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_POLICY_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_POLICY_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_POLICY_CHECK) | 0);  /* check that X509_V_FLAG_POLICY_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_SUITEB_128_LOS(unsigned long long *o)
{
  int n = (X509_V_FLAG_SUITEB_128_LOS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_SUITEB_128_LOS) | 0);  /* check that X509_V_FLAG_SUITEB_128_LOS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_SUITEB_128_LOS_ONLY(unsigned long long *o)
{
  int n = (X509_V_FLAG_SUITEB_128_LOS_ONLY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_SUITEB_128_LOS_ONLY) | 0);  /* check that X509_V_FLAG_SUITEB_128_LOS_ONLY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_SUITEB_192_LOS(unsigned long long *o)
{
  int n = (X509_V_FLAG_SUITEB_192_LOS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_SUITEB_192_LOS) | 0);  /* check that X509_V_FLAG_SUITEB_192_LOS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_TRUSTED_FIRST(unsigned long long *o)
{
  int n = (X509_V_FLAG_TRUSTED_FIRST) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_TRUSTED_FIRST) | 0);  /* check that X509_V_FLAG_TRUSTED_FIRST is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_USE_CHECK_TIME(unsigned long long *o)
{
  int n = (X509_V_FLAG_USE_CHECK_TIME) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_USE_CHECK_TIME) | 0);  /* check that X509_V_FLAG_USE_CHECK_TIME is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_USE_DELTAS(unsigned long long *o)
{
  int n = (X509_V_FLAG_USE_DELTAS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_USE_DELTAS) | 0);  /* check that X509_V_FLAG_USE_DELTAS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_X509_STRICT(unsigned long long *o)
{
  int n = (X509_V_FLAG_X509_STRICT) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_X509_STRICT) | 0);  /* check that X509_V_FLAG_X509_STRICT is an integer */
  return n;
}

static int _cffi_const_X509_V_OK(unsigned long long *o)
{
  int n = (X509_V_OK) <= 0;
  *o = (unsigned long long)((X509_V_OK) | 0);  /* check that X509_V_OK is an integer */
  return n;
}

static void _cffi_d_ACCESS_DESCRIPTION_free(ACCESS_DESCRIPTION * x0)
{
  ACCESS_DESCRIPTION_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ACCESS_DESCRIPTION_free(PyObject *self, PyObject *arg0)
{
  ACCESS_DESCRIPTION * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1138), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ACCESS_DESCRIPTION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1138), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ACCESS_DESCRIPTION_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ACCESS_DESCRIPTION_free _cffi_d_ACCESS_DESCRIPTION_free
#endif

static ACCESS_DESCRIPTION * _cffi_d_ACCESS_DESCRIPTION_new(void)
{
  return ACCESS_DESCRIPTION_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ACCESS_DESCRIPTION_new(PyObject *self, PyObject *noarg)
{
  ACCESS_DESCRIPTION * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ACCESS_DESCRIPTION_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1138));
  return pyresult;
}
#else
#  define _cffi_f_ACCESS_DESCRIPTION_new _cffi_d_ACCESS_DESCRIPTION_new
#endif

static int _cffi_d_AES_set_decrypt_key(unsigned char const * x0, int x1, AES_KEY * x2)
{
  return AES_set_decrypt_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AES_set_decrypt_key(PyObject *self, PyObject *args)
{
  unsigned char const * x0;
  int x1;
  AES_KEY * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "AES_set_decrypt_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(811), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (AES_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(811), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AES_set_decrypt_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_AES_set_decrypt_key _cffi_d_AES_set_decrypt_key
#endif

static int _cffi_d_AES_set_encrypt_key(unsigned char const * x0, int x1, AES_KEY * x2)
{
  return AES_set_encrypt_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AES_set_encrypt_key(PyObject *self, PyObject *args)
{
  unsigned char const * x0;
  int x1;
  AES_KEY * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "AES_set_encrypt_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(811), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (AES_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(811), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AES_set_encrypt_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_AES_set_encrypt_key _cffi_d_AES_set_encrypt_key
#endif

static int _cffi_d_AES_unwrap_key(AES_KEY * x0, unsigned char const * x1, unsigned char * x2, unsigned char const * x3, unsigned int x4)
{
  return AES_unwrap_key(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AES_unwrap_key(PyObject *self, PyObject *args)
{
  AES_KEY * x0;
  unsigned char const * x1;
  unsigned char * x2;
  unsigned char const * x3;
  unsigned int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "AES_unwrap_key", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(811), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (AES_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(811), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, unsigned int);
  if (x4 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AES_unwrap_key(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_AES_unwrap_key _cffi_d_AES_unwrap_key
#endif

static int _cffi_d_AES_wrap_key(AES_KEY * x0, unsigned char const * x1, unsigned char * x2, unsigned char const * x3, unsigned int x4)
{
  return AES_wrap_key(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AES_wrap_key(PyObject *self, PyObject *args)
{
  AES_KEY * x0;
  unsigned char const * x1;
  unsigned char * x2;
  unsigned char const * x3;
  unsigned int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "AES_wrap_key", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(811), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (AES_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(811), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, unsigned int);
  if (x4 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AES_wrap_key(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_AES_wrap_key _cffi_d_AES_wrap_key
#endif

static void _cffi_d_ASN1_BIT_STRING_free(ASN1_OCTET_STRING * x0)
{
  ASN1_BIT_STRING_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_BIT_STRING_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_free _cffi_d_ASN1_BIT_STRING_free
#endif

static int _cffi_d_ASN1_BIT_STRING_get_bit(ASN1_OCTET_STRING * x0, int x1)
{
  return ASN1_BIT_STRING_get_bit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_get_bit(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_BIT_STRING_get_bit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_BIT_STRING_get_bit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_get_bit _cffi_d_ASN1_BIT_STRING_get_bit
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_BIT_STRING_new(void)
{
  return ASN1_BIT_STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_BIT_STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_new _cffi_d_ASN1_BIT_STRING_new
#endif

static int _cffi_d_ASN1_BIT_STRING_set_bit(ASN1_OCTET_STRING * x0, int x1, int x2)
{
  return ASN1_BIT_STRING_set_bit(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_set_bit(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  int x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_BIT_STRING_set_bit", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_BIT_STRING_set_bit(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_set_bit _cffi_d_ASN1_BIT_STRING_set_bit
#endif

static void _cffi_d_ASN1_ENUMERATED_free(ASN1_ENUMERATED * x0)
{
  ASN1_ENUMERATED_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_free(PyObject *self, PyObject *arg0)
{
  ASN1_ENUMERATED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(818), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(818), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_ENUMERATED_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_free _cffi_d_ASN1_ENUMERATED_free
#endif

static long _cffi_d_ASN1_ENUMERATED_get(ASN1_ENUMERATED * x0)
{
  return ASN1_ENUMERATED_get(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_get(PyObject *self, PyObject *arg0)
{
  ASN1_ENUMERATED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(818), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(818), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_get(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_get _cffi_d_ASN1_ENUMERATED_get
#endif

static ASN1_ENUMERATED * _cffi_d_ASN1_ENUMERATED_new(void)
{
  return ASN1_ENUMERATED_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_new(PyObject *self, PyObject *noarg)
{
  ASN1_ENUMERATED * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(818));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_new _cffi_d_ASN1_ENUMERATED_new
#endif

static int _cffi_d_ASN1_ENUMERATED_set(ASN1_ENUMERATED * x0, long x1)
{
  return ASN1_ENUMERATED_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_set(PyObject *self, PyObject *args)
{
  ASN1_ENUMERATED * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_ENUMERATED_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(818), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(818), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_set _cffi_d_ASN1_ENUMERATED_set
#endif

static void _cffi_d_ASN1_GENERALIZEDTIME_free(ASN1_GENERALIZEDTIME * x0)
{
  ASN1_GENERALIZEDTIME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_GENERALIZEDTIME_free(PyObject *self, PyObject *arg0)
{
  ASN1_GENERALIZEDTIME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(9), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(9), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_GENERALIZEDTIME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_GENERALIZEDTIME_free _cffi_d_ASN1_GENERALIZEDTIME_free
#endif

static ASN1_GENERALIZEDTIME * _cffi_d_ASN1_GENERALIZEDTIME_set(ASN1_GENERALIZEDTIME * x0, time_t x1)
{
  return ASN1_GENERALIZEDTIME_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_GENERALIZEDTIME_set(PyObject *self, PyObject *args)
{
  ASN1_GENERALIZEDTIME * x0;
  time_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_GENERALIZEDTIME * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_GENERALIZEDTIME_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(9), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(9), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, time_t);
  if (x1 == (time_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_GENERALIZEDTIME_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(9));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_GENERALIZEDTIME_set _cffi_d_ASN1_GENERALIZEDTIME_set
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_IA5STRING_new(void)
{
  return ASN1_IA5STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_IA5STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_IA5STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_IA5STRING_new _cffi_d_ASN1_IA5STRING_new
#endif

static void _cffi_d_ASN1_INTEGER_free(ASN1_INTEGER * x0)
{
  ASN1_INTEGER_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_free(PyObject *self, PyObject *arg0)
{
  ASN1_INTEGER * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_INTEGER_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_INTEGER_free _cffi_d_ASN1_INTEGER_free
#endif

static int _cffi_d_ASN1_INTEGER_set(ASN1_INTEGER * x0, long x1)
{
  return ASN1_INTEGER_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_set(PyObject *self, PyObject *args)
{
  ASN1_INTEGER * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_INTEGER_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_INTEGER_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_INTEGER_set _cffi_d_ASN1_INTEGER_set
#endif

static BIGNUM * _cffi_d_ASN1_INTEGER_to_BN(ASN1_INTEGER * x0, BIGNUM * x1)
{
  return ASN1_INTEGER_to_BN(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_to_BN(PyObject *self, PyObject *args)
{
  ASN1_INTEGER * x0;
  BIGNUM * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_INTEGER_to_BN", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_INTEGER_to_BN(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_INTEGER_to_BN _cffi_d_ASN1_INTEGER_to_BN
#endif

static ASN1_ITEM const * _cffi_d_ASN1_ITEM_ptr(ASN1_ITEM_EXP * x0)
{
  return ASN1_ITEM_ptr(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ITEM_ptr(PyObject *self, PyObject *arg0)
{
  ASN1_ITEM_EXP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_ITEM const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(34), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ITEM_EXP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(34), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ITEM_ptr(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(98));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ITEM_ptr _cffi_d_ASN1_ITEM_ptr
#endif

static ASN1_NULL * _cffi_d_ASN1_NULL_new(void)
{
  return ASN1_NULL_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_NULL_new(PyObject *self, PyObject *noarg)
{
  ASN1_NULL * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_NULL_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3259));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_NULL_new _cffi_d_ASN1_NULL_new
#endif

static void _cffi_d_ASN1_OBJECT_free(ASN1_OBJECT * x0)
{
  ASN1_OBJECT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OBJECT_free(PyObject *self, PyObject *arg0)
{
  ASN1_OBJECT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(657), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(657), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_OBJECT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_OBJECT_free _cffi_d_ASN1_OBJECT_free
#endif

static void _cffi_d_ASN1_OCTET_STRING_free(ASN1_OCTET_STRING * x0)
{
  ASN1_OCTET_STRING_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_OCTET_STRING_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_free _cffi_d_ASN1_OCTET_STRING_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_OCTET_STRING_new(void)
{
  return ASN1_OCTET_STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_OCTET_STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_new _cffi_d_ASN1_OCTET_STRING_new
#endif

static int _cffi_d_ASN1_OCTET_STRING_set(ASN1_OCTET_STRING * x0, unsigned char const * x1, int x2)
{
  return ASN1_OCTET_STRING_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_set(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  unsigned char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_OCTET_STRING_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_OCTET_STRING_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_set _cffi_d_ASN1_OCTET_STRING_set
#endif

static unsigned char * _cffi_d_ASN1_STRING_data(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_data(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_data(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_data(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(813));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_data _cffi_d_ASN1_STRING_data
#endif

static int _cffi_d_ASN1_STRING_length(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_length(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_length(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_length(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_length _cffi_d_ASN1_STRING_length
#endif

static int _cffi_d_ASN1_STRING_set(ASN1_OCTET_STRING * x0, void const * x1, int x2)
{
  return ASN1_STRING_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_set(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  void const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_STRING_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_set _cffi_d_ASN1_STRING_set
#endif

static int _cffi_d_ASN1_STRING_set_default_mask_asc(char * x0)
{
  return ASN1_STRING_set_default_mask_asc(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_set_default_mask_asc(PyObject *self, PyObject *arg0)
{
  char * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(451), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(451), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_set_default_mask_asc(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_set_default_mask_asc _cffi_d_ASN1_STRING_set_default_mask_asc
#endif

static int _cffi_d_ASN1_STRING_to_UTF8(unsigned char * * x0, ASN1_OCTET_STRING * x1)
{
  return ASN1_STRING_to_UTF8(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_to_UTF8(PyObject *self, PyObject *args)
{
  unsigned char * * x0;
  ASN1_OCTET_STRING * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_STRING_to_UTF8", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(867), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(867), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_to_UTF8(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_to_UTF8 _cffi_d_ASN1_STRING_to_UTF8
#endif

static int _cffi_d_ASN1_STRING_type(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_type(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_type _cffi_d_ASN1_STRING_type
#endif

static void _cffi_d_ASN1_TIME_free(ASN1_OCTET_STRING * x0)
{
  ASN1_TIME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_TIME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_TIME_free _cffi_d_ASN1_TIME_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_TIME_new(void)
{
  return ASN1_TIME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_new _cffi_d_ASN1_TIME_new
#endif

static int _cffi_d_ASN1_TIME_print(BIO * x0, ASN1_OCTET_STRING * x1)
{
  return ASN1_TIME_print(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_print(PyObject *self, PyObject *args)
{
  BIO * x0;
  ASN1_OCTET_STRING * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_print", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_print(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_print _cffi_d_ASN1_TIME_print
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_TIME_set(ASN1_OCTET_STRING * x0, time_t x1)
{
  return ASN1_TIME_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_set(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  time_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, time_t);
  if (x1 == (time_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_set _cffi_d_ASN1_TIME_set
#endif

static int _cffi_d_ASN1_TIME_set_string(ASN1_OCTET_STRING * x0, char const * x1)
{
  return ASN1_TIME_set_string(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_set_string(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_set_string", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_set_string(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_set_string _cffi_d_ASN1_TIME_set_string
#endif

static ASN1_GENERALIZEDTIME * _cffi_d_ASN1_TIME_to_generalizedtime(ASN1_OCTET_STRING * x0, ASN1_GENERALIZEDTIME * * x1)
{
  return ASN1_TIME_to_generalizedtime(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_to_generalizedtime(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  ASN1_GENERALIZEDTIME * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_GENERALIZEDTIME * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_to_generalizedtime", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_to_generalizedtime(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(9));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_to_generalizedtime _cffi_d_ASN1_TIME_to_generalizedtime
#endif

static void _cffi_d_ASN1_UTF8STRING_free(ASN1_OCTET_STRING * x0)
{
  ASN1_UTF8STRING_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_UTF8STRING_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_UTF8STRING_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_UTF8STRING_free _cffi_d_ASN1_UTF8STRING_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_UTF8STRING_new(void)
{
  return ASN1_UTF8STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_UTF8STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_UTF8STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_UTF8STRING_new _cffi_d_ASN1_UTF8STRING_new
#endif

static ASN1_VALUE * _cffi_d_ASN1_item_d2i(ASN1_VALUE * * x0, unsigned char const * * x1, long x2, ASN1_ITEM const * x3)
{
  return ASN1_item_d2i(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_item_d2i(PyObject *self, PyObject *args)
{
  ASN1_VALUE * * x0;
  unsigned char const * * x1;
  long x2;
  ASN1_ITEM const * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_VALUE * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ASN1_item_d2i", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(95), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_VALUE * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(95), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(91), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(91), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, long);
  if (x2 == (long)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(98), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ASN1_ITEM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(98), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_item_d2i(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3265));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_item_d2i _cffi_d_ASN1_item_d2i
#endif

static void _cffi_d_AUTHORITY_KEYID_free(AUTHORITY_KEYID * x0)
{
  AUTHORITY_KEYID_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AUTHORITY_KEYID_free(PyObject *self, PyObject *arg0)
{
  AUTHORITY_KEYID * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2756), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (AUTHORITY_KEYID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2756), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { AUTHORITY_KEYID_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_AUTHORITY_KEYID_free _cffi_d_AUTHORITY_KEYID_free
#endif

static AUTHORITY_KEYID * _cffi_d_AUTHORITY_KEYID_new(void)
{
  return AUTHORITY_KEYID_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AUTHORITY_KEYID_new(PyObject *self, PyObject *noarg)
{
  AUTHORITY_KEYID * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AUTHORITY_KEYID_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2756));
  return pyresult;
}
#else
#  define _cffi_f_AUTHORITY_KEYID_new _cffi_d_AUTHORITY_KEYID_new
#endif

static void _cffi_d_BASIC_CONSTRAINTS_free(BASIC_CONSTRAINTS * x0)
{
  BASIC_CONSTRAINTS_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BASIC_CONSTRAINTS_free(PyObject *self, PyObject *arg0)
{
  BASIC_CONSTRAINTS * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2759), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BASIC_CONSTRAINTS *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2759), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BASIC_CONSTRAINTS_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BASIC_CONSTRAINTS_free _cffi_d_BASIC_CONSTRAINTS_free
#endif

static BASIC_CONSTRAINTS * _cffi_d_BASIC_CONSTRAINTS_new(void)
{
  return BASIC_CONSTRAINTS_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BASIC_CONSTRAINTS_new(PyObject *self, PyObject *noarg)
{
  BASIC_CONSTRAINTS * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BASIC_CONSTRAINTS_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2759));
  return pyresult;
}
#else
#  define _cffi_f_BASIC_CONSTRAINTS_new _cffi_d_BASIC_CONSTRAINTS_new
#endif

static void _cffi_d_BIO_clear_retry_flags(BIO * x0)
{
  BIO_clear_retry_flags(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_clear_retry_flags(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_clear_retry_flags(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_clear_retry_flags _cffi_d_BIO_clear_retry_flags
#endif

static size_t _cffi_d_BIO_ctrl_pending(BIO * x0)
{
  return BIO_ctrl_pending(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_ctrl_pending(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_ctrl_pending(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_ctrl_pending _cffi_d_BIO_ctrl_pending
#endif

static int _cffi_d_BIO_free(BIO * x0)
{
  return BIO_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_free(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_free _cffi_d_BIO_free
#endif

static void _cffi_d_BIO_free_all(BIO * x0)
{
  BIO_free_all(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_free_all(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_free_all(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_free_all _cffi_d_BIO_free_all
#endif

static long _cffi_d_BIO_get_mem_data(BIO * x0, char * * x1)
{
  return BIO_get_mem_data(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_get_mem_data(PyObject *self, PyObject *args)
{
  BIO * x0;
  char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_get_mem_data", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2485), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2485), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_get_mem_data(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_get_mem_data _cffi_d_BIO_get_mem_data
#endif

static int _cffi_d_BIO_gets(BIO * x0, char * x1, int x2)
{
  return BIO_gets(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_gets(PyObject *self, PyObject *args)
{
  BIO * x0;
  char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_gets", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(451), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(451), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_gets(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_gets _cffi_d_BIO_gets
#endif

static BIO * _cffi_d_BIO_new(BIO_METHOD * x0)
{
  return BIO_new(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new(PyObject *self, PyObject *arg0)
{
  BIO_METHOD * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(133), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(133), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(137));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new _cffi_d_BIO_new
#endif

static BIO * _cffi_d_BIO_new_dgram(int x0, int x1)
{
  return BIO_new_dgram(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_dgram(PyObject *self, PyObject *args)
{
  int x0;
  int x1;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_dgram", 2, 2, &arg0, &arg1))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_dgram(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(137));
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_dgram _cffi_d_BIO_new_dgram
#endif

static BIO * _cffi_d_BIO_new_file(char const * x0, char const * x1)
{
  return BIO_new_file(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_file(PyObject *self, PyObject *args)
{
  char const * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_file", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_file(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(137));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_file _cffi_d_BIO_new_file
#endif

static BIO * _cffi_d_BIO_new_mem_buf(void const * x0, int x1)
{
  return BIO_new_mem_buf(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_mem_buf(PyObject *self, PyObject *args)
{
  void const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_mem_buf", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_mem_buf(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(137));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_mem_buf _cffi_d_BIO_new_mem_buf
#endif

static int _cffi_d_BIO_read(BIO * x0, void * x1, int x2)
{
  return BIO_read(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_read(PyObject *self, PyObject *args)
{
  BIO * x0;
  void * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_read", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_read(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_read _cffi_d_BIO_read
#endif

static long _cffi_d_BIO_read_filename(BIO * x0, char * x1)
{
  return BIO_read_filename(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_read_filename(PyObject *self, PyObject *args)
{
  BIO * x0;
  char * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_read_filename", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(451), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(451), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_read_filename(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_read_filename _cffi_d_BIO_read_filename
#endif

static int _cffi_d_BIO_reset(BIO * x0)
{
  return BIO_reset(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_reset(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_reset(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_reset _cffi_d_BIO_reset
#endif

static BIO_METHOD const * _cffi_d_BIO_s_datagram(void)
{
  return BIO_s_datagram();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_s_datagram(PyObject *self, PyObject *noarg)
{
  BIO_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_s_datagram(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3271));
  return pyresult;
}
#else
#  define _cffi_f_BIO_s_datagram _cffi_d_BIO_s_datagram
#endif

static BIO_METHOD const * _cffi_d_BIO_s_file(void)
{
  return BIO_s_file();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_s_file(PyObject *self, PyObject *noarg)
{
  BIO_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_s_file(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3271));
  return pyresult;
}
#else
#  define _cffi_f_BIO_s_file _cffi_d_BIO_s_file
#endif

static BIO_METHOD const * _cffi_d_BIO_s_mem(void)
{
  return BIO_s_mem();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_s_mem(PyObject *self, PyObject *noarg)
{
  BIO_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_s_mem(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3271));
  return pyresult;
}
#else
#  define _cffi_f_BIO_s_mem _cffi_d_BIO_s_mem
#endif

static long _cffi_d_BIO_set_mem_eof_return(BIO * x0, int x1)
{
  return BIO_set_mem_eof_return(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_set_mem_eof_return(PyObject *self, PyObject *args)
{
  BIO * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_set_mem_eof_return", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_set_mem_eof_return(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_set_mem_eof_return _cffi_d_BIO_set_mem_eof_return
#endif

static long _cffi_d_BIO_set_nbio(BIO * x0, long x1)
{
  return BIO_set_nbio(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_set_nbio(PyObject *self, PyObject *args)
{
  BIO * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_set_nbio", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_set_nbio(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_set_nbio _cffi_d_BIO_set_nbio
#endif

static void _cffi_d_BIO_set_retry_read(BIO * x0)
{
  BIO_set_retry_read(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_set_retry_read(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_set_retry_read(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_set_retry_read _cffi_d_BIO_set_retry_read
#endif

static int _cffi_d_BIO_should_io_special(BIO * x0)
{
  return BIO_should_io_special(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_io_special(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_io_special(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_io_special _cffi_d_BIO_should_io_special
#endif

static int _cffi_d_BIO_should_read(BIO * x0)
{
  return BIO_should_read(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_read(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_read(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_read _cffi_d_BIO_should_read
#endif

static int _cffi_d_BIO_should_retry(BIO * x0)
{
  return BIO_should_retry(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_retry(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_retry(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_retry _cffi_d_BIO_should_retry
#endif

static int _cffi_d_BIO_should_write(BIO * x0)
{
  return BIO_should_write(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_write(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_write(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_write _cffi_d_BIO_should_write
#endif

static int _cffi_d_BIO_up_ref(BIO * x0)
{
  return BIO_up_ref(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_up_ref(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_up_ref(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_up_ref _cffi_d_BIO_up_ref
#endif

static int _cffi_d_BIO_write(BIO * x0, void const * x1, int x2)
{
  return BIO_write(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_write(PyObject *self, PyObject *args)
{
  BIO * x0;
  void const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_write", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_write(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_write _cffi_d_BIO_write
#endif

static void _cffi_d_BN_CTX_end(BN_CTX * x0)
{
  BN_CTX_end(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_end(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_end(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_end _cffi_d_BN_CTX_end
#endif

static void _cffi_d_BN_CTX_free(BN_CTX * x0)
{
  BN_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_free(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_free _cffi_d_BN_CTX_free
#endif

static BIGNUM * _cffi_d_BN_CTX_get(BN_CTX * x0)
{
  return BN_CTX_get(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_get(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_CTX_get(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_CTX_get _cffi_d_BN_CTX_get
#endif

static BN_CTX * _cffi_d_BN_CTX_new(void)
{
  return BN_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_new(PyObject *self, PyObject *noarg)
{
  BN_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(112));
  return pyresult;
}
#else
#  define _cffi_f_BN_CTX_new _cffi_d_BN_CTX_new
#endif

static void _cffi_d_BN_CTX_start(BN_CTX * x0)
{
  BN_CTX_start(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_start(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_start(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_start _cffi_d_BN_CTX_start
#endif

static void _cffi_d_BN_MONT_CTX_free(BN_MONT_CTX * x0)
{
  BN_MONT_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_free(PyObject *self, PyObject *arg0)
{
  BN_MONT_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(895), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(895), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_MONT_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_MONT_CTX_free _cffi_d_BN_MONT_CTX_free
#endif

static BN_MONT_CTX * _cffi_d_BN_MONT_CTX_new(void)
{
  return BN_MONT_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_new(PyObject *self, PyObject *noarg)
{
  BN_MONT_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_MONT_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(895));
  return pyresult;
}
#else
#  define _cffi_f_BN_MONT_CTX_new _cffi_d_BN_MONT_CTX_new
#endif

static int _cffi_d_BN_MONT_CTX_set(BN_MONT_CTX * x0, BIGNUM const * x1, BN_CTX * x2)
{
  return BN_MONT_CTX_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_set(PyObject *self, PyObject *args)
{
  BN_MONT_CTX * x0;
  BIGNUM const * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_MONT_CTX_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(895), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(895), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_MONT_CTX_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_MONT_CTX_set _cffi_d_BN_MONT_CTX_set
#endif

static int _cffi_d_BN_add(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2)
{
  return BN_add(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_add(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_add", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_add(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_add _cffi_d_BN_add
#endif

static BIGNUM * _cffi_d_BN_bin2bn(unsigned char const * x0, int x1, BIGNUM * x2)
{
  return BN_bin2bn(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bin2bn(PyObject *self, PyObject *args)
{
  unsigned char const * x0;
  int x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_bin2bn", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bin2bn(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bin2bn _cffi_d_BN_bin2bn
#endif

static int _cffi_d_BN_bn2bin(BIGNUM const * x0, unsigned char * x1)
{
  return BN_bn2bin(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bn2bin(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  unsigned char * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_bn2bin", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bn2bin(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bn2bin _cffi_d_BN_bn2bin
#endif

static char * _cffi_d_BN_bn2hex(BIGNUM const * x0)
{
  return BN_bn2hex(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bn2hex(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bn2hex(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(451));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bn2hex _cffi_d_BN_bn2hex
#endif

static void _cffi_d_BN_clear_free(BIGNUM * x0)
{
  BN_clear_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_clear_free(PyObject *self, PyObject *arg0)
{
  BIGNUM * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_clear_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_clear_free _cffi_d_BN_clear_free
#endif

static int _cffi_d_BN_cmp(BIGNUM const * x0, BIGNUM const * x1)
{
  return BN_cmp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_cmp(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_cmp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_cmp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_cmp _cffi_d_BN_cmp
#endif

static BIGNUM * _cffi_d_BN_dup(BIGNUM const * x0)
{
  return BN_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_dup(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_dup _cffi_d_BN_dup
#endif

static void _cffi_d_BN_free(BIGNUM * x0)
{
  BN_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_free(PyObject *self, PyObject *arg0)
{
  BIGNUM * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_free _cffi_d_BN_free
#endif

static int _cffi_d_BN_generate_prime_ex(BIGNUM * x0, int x1, int x2, BIGNUM const * x3, BIGNUM const * x4, BN_GENCB * x5)
{
  return BN_generate_prime_ex(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_generate_prime_ex(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  int x1;
  int x2;
  BIGNUM const * x3;
  BIGNUM const * x4;
  BN_GENCB * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_generate_prime_ex", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(913), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(913), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_generate_prime_ex(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_generate_prime_ex _cffi_d_BN_generate_prime_ex
#endif

static int _cffi_d_BN_hex2bn(BIGNUM * * x0, char const * x1)
{
  return BN_hex2bn(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_hex2bn(PyObject *self, PyObject *args)
{
  BIGNUM * * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_hex2bn", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(870), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(870), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_hex2bn(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_hex2bn _cffi_d_BN_hex2bn
#endif

static int _cffi_d_BN_is_negative(BIGNUM const * x0)
{
  return BN_is_negative(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_is_negative(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_is_negative(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_is_negative _cffi_d_BN_is_negative
#endif

static int _cffi_d_BN_is_prime_ex(BIGNUM const * x0, int x1, BN_CTX * x2, BN_GENCB * x3)
{
  return BN_is_prime_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_is_prime_ex(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  int x1;
  BN_CTX * x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_is_prime_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(913), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(913), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_is_prime_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_is_prime_ex _cffi_d_BN_is_prime_ex
#endif

static int _cffi_d_BN_mod(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_mod(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_mod", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod _cffi_d_BN_mod
#endif

static int _cffi_d_BN_mod_add(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_add(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_add(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_add", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_add(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_add _cffi_d_BN_mod_add
#endif

static int _cffi_d_BN_mod_exp(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_exp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp _cffi_d_BN_mod_exp
#endif

static int _cffi_d_BN_mod_exp_mont(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4, BN_MONT_CTX * x5)
{
  return BN_mod_exp_mont(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp_mont(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  BN_MONT_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp_mont", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(895), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(895), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp_mont(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp_mont _cffi_d_BN_mod_exp_mont
#endif

static int _cffi_d_BN_mod_exp_mont_consttime(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4, BN_MONT_CTX * x5)
{
  return BN_mod_exp_mont_consttime(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp_mont_consttime(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  BN_MONT_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp_mont_consttime", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(895), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(895), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp_mont_consttime(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp_mont_consttime _cffi_d_BN_mod_exp_mont_consttime
#endif

static BIGNUM * _cffi_d_BN_mod_inverse(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_mod_inverse(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_inverse(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_mod_inverse", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_inverse(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_inverse _cffi_d_BN_mod_inverse
#endif

static int _cffi_d_BN_mod_mul(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_mul(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_mul(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_mul", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_mul(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_mul _cffi_d_BN_mod_mul
#endif

static int _cffi_d_BN_mod_sub(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_sub(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_sub(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_sub", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_sub(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_sub _cffi_d_BN_mod_sub
#endif

static BIGNUM * _cffi_d_BN_new(void)
{
  return BN_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_new(PyObject *self, PyObject *noarg)
{
  BIGNUM * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  return pyresult;
}
#else
#  define _cffi_f_BN_new _cffi_d_BN_new
#endif

static int _cffi_d_BN_nnmod(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_nnmod(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_nnmod(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_nnmod", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_nnmod(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_nnmod _cffi_d_BN_nnmod
#endif

static int _cffi_d_BN_num_bits(BIGNUM const * x0)
{
  return BN_num_bits(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_num_bits(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_num_bits(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_num_bits _cffi_d_BN_num_bits
#endif

static int _cffi_d_BN_num_bytes(BIGNUM const * x0)
{
  return BN_num_bytes(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_num_bytes(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_num_bytes(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_num_bytes _cffi_d_BN_num_bytes
#endif

static int _cffi_d_BN_prime_checks_for_size(int x0)
{
  return BN_prime_checks_for_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_prime_checks_for_size(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_prime_checks_for_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_BN_prime_checks_for_size _cffi_d_BN_prime_checks_for_size
#endif

static int _cffi_d_BN_rand_range(BIGNUM * x0, BIGNUM const * x1)
{
  return BN_rand_range(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_rand_range(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_rand_range", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_rand_range(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_rand_range _cffi_d_BN_rand_range
#endif

static void _cffi_d_BN_set_flags(BIGNUM * x0, int x1)
{
  BN_set_flags(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_set_flags(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_set_flags", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_set_flags(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_set_flags _cffi_d_BN_set_flags
#endif

static int _cffi_d_BN_set_word(BIGNUM * x0, BN_ULONG x1)
{
  return BN_set_word(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_set_word(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BN_ULONG x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_set_word", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, BN_ULONG);
  if (x1 == (BN_ULONG)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_set_word(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_set_word _cffi_d_BN_set_word
#endif

static int _cffi_d_BN_sub(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2)
{
  return BN_sub(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_sub(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_sub", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_sub(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_sub _cffi_d_BN_sub
#endif

static ASN1_INTEGER * _cffi_d_BN_to_ASN1_INTEGER(BIGNUM * x0, ASN1_INTEGER * x1)
{
  return BN_to_ASN1_INTEGER(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_to_ASN1_INTEGER(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  ASN1_INTEGER * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_INTEGER * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_to_ASN1_INTEGER", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_to_ASN1_INTEGER(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(21));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_to_ASN1_INTEGER _cffi_d_BN_to_ASN1_INTEGER
#endif

static BIGNUM const * _cffi_d_BN_value_one(void)
{
  return BN_value_one();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_value_one(PyObject *self, PyObject *noarg)
{
  BIGNUM const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_value_one(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(110));
  return pyresult;
}
#else
#  define _cffi_f_BN_value_one _cffi_d_BN_value_one
#endif

static void _cffi_d_CERTIFICATEPOLICIES_free(Cryptography_STACK_OF_POLICYINFO * x0)
{
  CERTIFICATEPOLICIES_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CERTIFICATEPOLICIES_free(PyObject *self, PyObject *arg0)
{
  Cryptography_STACK_OF_POLICYINFO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(490), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_POLICYINFO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(490), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { CERTIFICATEPOLICIES_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_CERTIFICATEPOLICIES_free _cffi_d_CERTIFICATEPOLICIES_free
#endif

static int _cffi_d_CMAC_CTX_copy(CMAC_CTX * x0, CMAC_CTX const * x1)
{
  return CMAC_CTX_copy(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_copy(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  CMAC_CTX const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "CMAC_CTX_copy", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1113), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1113), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1114), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (CMAC_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1114), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_CTX_copy(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_CTX_copy _cffi_d_CMAC_CTX_copy
#endif

static void _cffi_d_CMAC_CTX_free(CMAC_CTX * x0)
{
  CMAC_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_free(PyObject *self, PyObject *arg0)
{
  CMAC_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1113), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1113), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { CMAC_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_CMAC_CTX_free _cffi_d_CMAC_CTX_free
#endif

static CMAC_CTX * _cffi_d_CMAC_CTX_new(void)
{
  return CMAC_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_new(PyObject *self, PyObject *noarg)
{
  CMAC_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1113));
  return pyresult;
}
#else
#  define _cffi_f_CMAC_CTX_new _cffi_d_CMAC_CTX_new
#endif

static int _cffi_d_CMAC_Final(CMAC_CTX * x0, unsigned char * x1, size_t * x2)
{
  return CMAC_Final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Final(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "CMAC_Final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1113), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1113), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Final _cffi_d_CMAC_Final
#endif

static int _cffi_d_CMAC_Init(CMAC_CTX * x0, void const * x1, size_t x2, EVP_CIPHER const * x3, ENGINE * x4)
{
  return CMAC_Init(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Init(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  void const * x1;
  size_t x2;
  EVP_CIPHER const * x3;
  ENGINE * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "CMAC_Init", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1113), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1113), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(479), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(479), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Init(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Init _cffi_d_CMAC_Init
#endif

static int _cffi_d_CMAC_Update(CMAC_CTX * x0, void const * x1, size_t x2)
{
  return CMAC_Update(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Update(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "CMAC_Update", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1113), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1113), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Update(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Update _cffi_d_CMAC_Update
#endif

static void _cffi_d_CRL_DIST_POINTS_free(Cryptography_STACK_OF_DIST_POINT * x0)
{
  CRL_DIST_POINTS_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CRL_DIST_POINTS_free(PyObject *self, PyObject *arg0)
{
  Cryptography_STACK_OF_DIST_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(240), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_DIST_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(240), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { CRL_DIST_POINTS_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_CRL_DIST_POINTS_free _cffi_d_CRL_DIST_POINTS_free
#endif

static void(* _cffi_d_CRYPTO_get_locking_callback(void))(int, int, char const *, int)
{
  return CRYPTO_get_locking_callback();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CRYPTO_get_locking_callback(PyObject *self, PyObject *noarg)
{
  void(* result)(int, int, char const *, int);
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CRYPTO_get_locking_callback(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3392));
  return pyresult;
}
#else
#  define _cffi_f_CRYPTO_get_locking_callback _cffi_d_CRYPTO_get_locking_callback
#endif

static int _cffi_d_Cryptography_CRYPTO_set_mem_functions(void *(* x0)(size_t, char const *, int), void *(* x1)(void *, size_t, char const *, int), void(* x2)(void *, char const *, int))
{
  return Cryptography_CRYPTO_set_mem_functions(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_CRYPTO_set_mem_functions(PyObject *self, PyObject *args)
{
  void *(* x0)(size_t, char const *, int);
  void *(* x1)(void *, size_t, char const *, int);
  void(* x2)(void *, char const *, int);
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_CRYPTO_set_mem_functions", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = (void *(*)(size_t, char const *, int))_cffi_to_c_pointer(arg0, _cffi_type(2455));
  if (x0 == (void *(*)(size_t, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  x1 = (void *(*)(void *, size_t, char const *, int))_cffi_to_c_pointer(arg1, _cffi_type(2456));
  if (x1 == (void *(*)(void *, size_t, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  x2 = (void(*)(void *, char const *, int))_cffi_to_c_pointer(arg2, _cffi_type(2457));
  if (x2 == (void(*)(void *, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_CRYPTO_set_mem_functions(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_CRYPTO_set_mem_functions _cffi_d_Cryptography_CRYPTO_set_mem_functions
#endif

static int _cffi_d_Cryptography_DH_check(DH const * x0, int * x1)
{
  return Cryptography_DH_check(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_DH_check(PyObject *self, PyObject *args)
{
  DH const * x0;
  int * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_DH_check", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1247), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1247), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1251), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1251), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_DH_check(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_DH_check _cffi_d_Cryptography_DH_check
#endif

static long _cffi_d_Cryptography_DTLSv1_get_timeout(SSL * x0, time_t * x1, long * x2)
{
  return Cryptography_DTLSv1_get_timeout(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_DTLSv1_get_timeout(PyObject *self, PyObject *args)
{
  SSL * x0;
  time_t * x1;
  long * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_DTLSv1_get_timeout", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(163), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(163), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2512), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (time_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2512), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2513), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (long *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2513), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_DTLSv1_get_timeout(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_DTLSv1_get_timeout _cffi_d_Cryptography_DTLSv1_get_timeout
#endif

static void _cffi_d_Cryptography_EVP_MD_CTX_free(EVP_MD_CTX * x0)
{
  Cryptography_EVP_MD_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_EVP_MD_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_MD_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { Cryptography_EVP_MD_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_Cryptography_EVP_MD_CTX_free _cffi_d_Cryptography_EVP_MD_CTX_free
#endif

static EVP_MD_CTX * _cffi_d_Cryptography_EVP_MD_CTX_new(void)
{
  return Cryptography_EVP_MD_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_EVP_MD_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_MD_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_EVP_MD_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1434));
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_EVP_MD_CTX_new _cffi_d_Cryptography_EVP_MD_CTX_new
#endif

static void _cffi_d_Cryptography_HMAC_CTX_free(HMAC_CTX * x0)
{
  Cryptography_HMAC_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_HMAC_CTX_free(PyObject *self, PyObject *arg0)
{
  HMAC_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { Cryptography_HMAC_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_Cryptography_HMAC_CTX_free _cffi_d_Cryptography_HMAC_CTX_free
#endif

static HMAC_CTX * _cffi_d_Cryptography_HMAC_CTX_new(void)
{
  return Cryptography_HMAC_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_HMAC_CTX_new(PyObject *self, PyObject *noarg)
{
  HMAC_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_HMAC_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1594));
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_HMAC_CTX_new _cffi_d_Cryptography_HMAC_CTX_new
#endif

static int _cffi_d_Cryptography_X509_NAME_ENTRY_set(X509_NAME_ENTRY * x0)
{
  return Cryptography_X509_NAME_ENTRY_set(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_X509_NAME_ENTRY_set(PyObject *self, PyObject *arg0)
{
  X509_NAME_ENTRY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(49), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (X509_NAME_ENTRY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(49), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_X509_NAME_ENTRY_set(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_X509_NAME_ENTRY_set _cffi_d_Cryptography_X509_NAME_ENTRY_set
#endif

static X509_REVOKED * _cffi_d_Cryptography_X509_REVOKED_dup(X509_REVOKED * x0)
{
  return Cryptography_X509_REVOKED_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_X509_REVOKED_dup(PyObject *self, PyObject *arg0)
{
  X509_REVOKED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_REVOKED * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(669), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (X509_REVOKED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(669), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_X509_REVOKED_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(669));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_X509_REVOKED_dup _cffi_d_Cryptography_X509_REVOKED_dup
#endif

static int _cffi_d_Cryptography_add_osrandom_engine(void)
{
  return Cryptography_add_osrandom_engine();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_add_osrandom_engine(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_add_osrandom_engine(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_add_osrandom_engine _cffi_d_Cryptography_add_osrandom_engine
#endif

static DH * _cffi_d_Cryptography_d2i_DHxparams_bio(BIO * x0, DH * * x1)
{
  return Cryptography_d2i_DHxparams_bio(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_d2i_DHxparams_bio(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_d2i_DHxparams_bio", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(223), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(223), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_d2i_DHxparams_bio(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(232));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_d2i_DHxparams_bio _cffi_d_Cryptography_d2i_DHxparams_bio
#endif

static void _cffi_d_Cryptography_free_wrapper(void * x0, char const * x1, int x2)
{
  Cryptography_free_wrapper(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_free_wrapper(PyObject *self, PyObject *args)
{
  void * x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_free_wrapper", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { Cryptography_free_wrapper(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_Cryptography_free_wrapper _cffi_d_Cryptography_free_wrapper
#endif

static int _cffi_d_Cryptography_i2d_DHxparams_bio(BIO * x0, DH * x1)
{
  return Cryptography_i2d_DHxparams_bio(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_i2d_DHxparams_bio(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_i2d_DHxparams_bio", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_i2d_DHxparams_bio(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_i2d_DHxparams_bio _cffi_d_Cryptography_i2d_DHxparams_bio
#endif

static void * _cffi_d_Cryptography_malloc_wrapper(size_t x0, char const * x1, int x2)
{
  return Cryptography_malloc_wrapper(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_malloc_wrapper(PyObject *self, PyObject *args)
{
  size_t x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_malloc_wrapper", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = _cffi_to_c_int(arg0, size_t);
  if (x0 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_malloc_wrapper(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(229));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_malloc_wrapper _cffi_d_Cryptography_malloc_wrapper
#endif

static int _cffi_d_Cryptography_pem_password_cb(char * x0, int x1, int x2, void * x3)
{
  return Cryptography_pem_password_cb(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_pem_password_cb(PyObject *self, PyObject *args)
{
  char * x0;
  int x1;
  int x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "Cryptography_pem_password_cb", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(451), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(451), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_pem_password_cb(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_pem_password_cb _cffi_d_Cryptography_pem_password_cb
#endif

static void * _cffi_d_Cryptography_realloc_wrapper(void * x0, size_t x1, char const * x2, int x3)
{
  return Cryptography_realloc_wrapper(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_realloc_wrapper(PyObject *self, PyObject *args)
{
  void * x0;
  size_t x1;
  char const * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "Cryptography_realloc_wrapper", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_realloc_wrapper(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(229));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_realloc_wrapper _cffi_d_Cryptography_realloc_wrapper
#endif

static int _cffi_d_Cryptography_setup_ssl_threads(void)
{
  return Cryptography_setup_ssl_threads();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_setup_ssl_threads(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_setup_ssl_threads(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_setup_ssl_threads _cffi_d_Cryptography_setup_ssl_threads
#endif

static int _cffi_d_DH_compute_key(unsigned char * x0, BIGNUM const * x1, DH * x2)
{
  return DH_compute_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_compute_key(PyObject *self, PyObject *args)
{
  unsigned char * x0;
  BIGNUM const * x1;
  DH * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_compute_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_compute_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_compute_key _cffi_d_DH_compute_key
#endif

static void _cffi_d_DH_free(DH * x0)
{
  DH_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_free(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_free _cffi_d_DH_free
#endif

static int _cffi_d_DH_generate_key(DH * x0)
{
  return DH_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_generate_key(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_generate_key _cffi_d_DH_generate_key
#endif

static int _cffi_d_DH_generate_parameters_ex(DH * x0, int x1, int x2, BN_GENCB * x3)
{
  return DH_generate_parameters_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_generate_parameters_ex(PyObject *self, PyObject *args)
{
  DH * x0;
  int x1;
  int x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_generate_parameters_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(913), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(913), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_generate_parameters_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_generate_parameters_ex _cffi_d_DH_generate_parameters_ex
#endif

static void _cffi_d_DH_get0_key(DH const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  DH_get0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_get0_key(PyObject *self, PyObject *args)
{
  DH const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_get0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1247), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1247), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_get0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_get0_key _cffi_d_DH_get0_key
#endif

static void _cffi_d_DH_get0_pqg(DH const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  DH_get0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_get0_pqg(PyObject *self, PyObject *args)
{
  DH const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_get0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1247), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1247), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_get0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_get0_pqg _cffi_d_DH_get0_pqg
#endif

static DH * _cffi_d_DH_new(void)
{
  return DH_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_new(PyObject *self, PyObject *noarg)
{
  DH * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(232));
  return pyresult;
}
#else
#  define _cffi_f_DH_new _cffi_d_DH_new
#endif

static int _cffi_d_DH_set0_key(DH * x0, BIGNUM * x1, BIGNUM * x2)
{
  return DH_set0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_set0_key(PyObject *self, PyObject *args)
{
  DH * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_set0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_set0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_set0_key _cffi_d_DH_set0_key
#endif

static int _cffi_d_DH_set0_pqg(DH * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return DH_set0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_set0_pqg(PyObject *self, PyObject *args)
{
  DH * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_set0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_set0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_set0_pqg _cffi_d_DH_set0_pqg
#endif

static int _cffi_d_DH_size(DH const * x0)
{
  return DH_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_size(PyObject *self, PyObject *arg0)
{
  DH const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1247), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1247), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_size _cffi_d_DH_size
#endif

static DH * _cffi_d_DHparams_dup(DH * x0)
{
  return DHparams_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DHparams_dup(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DHparams_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(232));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DHparams_dup _cffi_d_DHparams_dup
#endif

static void _cffi_d_DIST_POINT_NAME_free(DIST_POINT_NAME * x0)
{
  DIST_POINT_NAME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_NAME_free(PyObject *self, PyObject *arg0)
{
  DIST_POINT_NAME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2843), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DIST_POINT_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2843), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DIST_POINT_NAME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DIST_POINT_NAME_free _cffi_d_DIST_POINT_NAME_free
#endif

static DIST_POINT_NAME * _cffi_d_DIST_POINT_NAME_new(void)
{
  return DIST_POINT_NAME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_NAME_new(PyObject *self, PyObject *noarg)
{
  DIST_POINT_NAME * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DIST_POINT_NAME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2843));
  return pyresult;
}
#else
#  define _cffi_f_DIST_POINT_NAME_new _cffi_d_DIST_POINT_NAME_new
#endif

static void _cffi_d_DIST_POINT_free(DIST_POINT * x0)
{
  DIST_POINT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_free(PyObject *self, PyObject *arg0)
{
  DIST_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1159), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DIST_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1159), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DIST_POINT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DIST_POINT_free _cffi_d_DIST_POINT_free
#endif

static DIST_POINT * _cffi_d_DIST_POINT_new(void)
{
  return DIST_POINT_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_new(PyObject *self, PyObject *noarg)
{
  DIST_POINT * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DIST_POINT_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1159));
  return pyresult;
}
#else
#  define _cffi_f_DIST_POINT_new _cffi_d_DIST_POINT_new
#endif

static void _cffi_d_DSA_free(DSA * x0)
{
  DSA_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_free(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_free _cffi_d_DSA_free
#endif

static int _cffi_d_DSA_generate_key(DSA * x0)
{
  return DSA_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_generate_key(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_generate_key _cffi_d_DSA_generate_key
#endif

static int _cffi_d_DSA_generate_parameters_ex(DSA * x0, int x1, unsigned char * x2, int x3, int * x4, unsigned long * x5, BN_GENCB * x6)
{
  return DSA_generate_parameters_ex(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_generate_parameters_ex(PyObject *self, PyObject *args)
{
  DSA * x0;
  int x1;
  unsigned char * x2;
  int x3;
  int * x4;
  unsigned long * x5;
  BN_GENCB * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "DSA_generate_parameters_ex", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1251), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1251), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1273), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned long *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1273), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(913), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(913), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_generate_parameters_ex(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_generate_parameters_ex _cffi_d_DSA_generate_parameters_ex
#endif

static void _cffi_d_DSA_get0_key(DSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  DSA_get0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_get0_key(PyObject *self, PyObject *args)
{
  DSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DSA_get0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1277), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1277), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_get0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_get0_key _cffi_d_DSA_get0_key
#endif

static void _cffi_d_DSA_get0_pqg(DSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  DSA_get0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_get0_pqg(PyObject *self, PyObject *args)
{
  DSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DSA_get0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1277), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1277), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2830), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2830), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_get0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_get0_pqg _cffi_d_DSA_get0_pqg
#endif

static DSA * _cffi_d_DSA_new(void)
{
  return DSA_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_new(PyObject *self, PyObject *noarg)
{
  DSA * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(248));
  return pyresult;
}
#else
#  define _cffi_f_DSA_new _cffi_d_DSA_new
#endif

static int _cffi_d_DSA_set0_key(DSA * x0, BIGNUM * x1, BIGNUM * x2)
{
  return DSA_set0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_set0_key(PyObject *self, PyObject *args)
{
  DSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DSA_set0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_set0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_set0_key _cffi_d_DSA_set0_key
#endif

static int _cffi_d_DSA_set0_pqg(DSA * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return DSA_set0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_set0_pqg(PyObject *self, PyObject *args)
{
  DSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DSA_set0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_set0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_set0_pqg _cffi_d_DSA_set0_pqg
#endif

static int _cffi_d_DSA_sign(int x0, unsigned char const * x1, int x2, unsigned char * x3, unsigned int * x4, DSA * x5)
{
  return DSA_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_sign(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char * x3;
  unsigned int * x4;
  DSA * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "DSA_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1473), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1473), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_sign _cffi_d_DSA_sign
#endif

static int _cffi_d_DSA_size(DSA const * x0)
{
  return DSA_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_size(PyObject *self, PyObject *arg0)
{
  DSA const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1277), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1277), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_size _cffi_d_DSA_size
#endif

static int _cffi_d_DSA_verify(int x0, unsigned char const * x1, int x2, unsigned char const * x3, int x4, DSA * x5)
{
  return DSA_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_verify(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char const * x3;
  int x4;
  DSA * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "DSA_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_verify _cffi_d_DSA_verify
#endif

static DSA * _cffi_d_DSAparams_dup(DSA * x0)
{
  return DSAparams_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSAparams_dup(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSAparams_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(248));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSAparams_dup _cffi_d_DSAparams_dup
#endif

static SSL_METHOD const * _cffi_d_DTLS_client_method(void)
{
  return DTLS_client_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_client_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_client_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3364));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_client_method _cffi_d_DTLS_client_method
#endif

static long _cffi_d_DTLS_get_link_min_mtu(SSL * x0)
{
  return DTLS_get_link_min_mtu(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_get_link_min_mtu(PyObject *self, PyObject *arg0)
{
  SSL * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(163), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(163), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_get_link_min_mtu(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLS_get_link_min_mtu _cffi_d_DTLS_get_link_min_mtu
#endif

static SSL_METHOD const * _cffi_d_DTLS_method(void)
{
  return DTLS_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3364));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_method _cffi_d_DTLS_method
#endif

static SSL_METHOD const * _cffi_d_DTLS_server_method(void)
{
  return DTLS_server_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_server_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_server_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3364));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_server_method _cffi_d_DTLS_server_method
#endif

static long _cffi_d_DTLS_set_link_mtu(SSL * x0, long x1)
{
  return DTLS_set_link_mtu(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_set_link_mtu(PyObject *self, PyObject *args)
{
  SSL * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "DTLS_set_link_mtu", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(163), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(163), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_set_link_mtu(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLS_set_link_mtu _cffi_d_DTLS_set_link_mtu
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_client_method(void)
{
  return DTLSv1_client_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_client_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_client_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3364));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_client_method _cffi_d_DTLSv1_client_method
#endif

static long _cffi_d_DTLSv1_handle_timeout(SSL * x0)
{
  return DTLSv1_handle_timeout(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_handle_timeout(PyObject *self, PyObject *arg0)
{
  SSL * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(163), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(163), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_handle_timeout(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_handle_timeout _cffi_d_DTLSv1_handle_timeout
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_method(void)
{
  return DTLSv1_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3364));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_method _cffi_d_DTLSv1_method
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_server_method(void)
{
  return DTLSv1_server_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_server_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_server_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3364));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_server_method _cffi_d_DTLSv1_server_method
#endif

static int _cffi_d_ECDH_compute_key(void * x0, size_t x1, EC_POINT const * x2, EC_KEY * x3, void *(* x4)(void const *, size_t, void *, size_t *))
{
  return ECDH_compute_key(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDH_compute_key(PyObject *self, PyObject *args)
{
  void * x0;
  size_t x1;
  EC_POINT const * x2;
  EC_KEY * x3;
  void *(* x4)(void const *, size_t, void *, size_t *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "ECDH_compute_key", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = (void *(*)(void const *, size_t, void *, size_t *))_cffi_to_c_pointer(arg4, _cffi_type(2464));
  if (x4 == (void *(*)(void const *, size_t, void *, size_t *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDH_compute_key(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDH_compute_key _cffi_d_ECDH_compute_key
#endif

static void _cffi_d_ECDSA_SIG_free(ECDSA_SIG * x0)
{
  ECDSA_SIG_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_SIG_free(PyObject *self, PyObject *arg0)
{
  ECDSA_SIG * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2860), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ECDSA_SIG *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2860), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ECDSA_SIG_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ECDSA_SIG_free _cffi_d_ECDSA_SIG_free
#endif

static ECDSA_SIG * _cffi_d_ECDSA_SIG_new(void)
{
  return ECDSA_SIG_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_SIG_new(PyObject *self, PyObject *noarg)
{
  ECDSA_SIG * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_SIG_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2860));
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_SIG_new _cffi_d_ECDSA_SIG_new
#endif

static ECDSA_SIG * _cffi_d_ECDSA_do_sign(unsigned char const * x0, int x1, EC_KEY * x2)
{
  return ECDSA_do_sign(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_do_sign(PyObject *self, PyObject *args)
{
  unsigned char const * x0;
  int x1;
  EC_KEY * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ECDSA_SIG * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ECDSA_do_sign", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_do_sign(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2860));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_do_sign _cffi_d_ECDSA_do_sign
#endif

static int _cffi_d_ECDSA_do_verify(unsigned char const * x0, int x1, ECDSA_SIG const * x2, EC_KEY * x3)
{
  return ECDSA_do_verify(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_do_verify(PyObject *self, PyObject *args)
{
  unsigned char const * x0;
  int x1;
  ECDSA_SIG const * x2;
  EC_KEY * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ECDSA_do_verify", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1280), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ECDSA_SIG const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1280), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_do_verify(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_do_verify _cffi_d_ECDSA_do_verify
#endif

static int _cffi_d_ECDSA_sign(int x0, unsigned char const * x1, int x2, unsigned char * x3, unsigned int * x4, EC_KEY * x5)
{
  return ECDSA_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_sign(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char * x3;
  unsigned int * x4;
  EC_KEY * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ECDSA_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1473), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1473), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_sign _cffi_d_ECDSA_sign
#endif

static int _cffi_d_ECDSA_size(EC_KEY const * x0)
{
  return ECDSA_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_size(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(128), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(128), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_size _cffi_d_ECDSA_size
#endif

static int _cffi_d_ECDSA_verify(int x0, unsigned char const * x1, int x2, unsigned char const * x3, int x4, EC_KEY * x5)
{
  return ECDSA_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_verify(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char const * x3;
  int x4;
  EC_KEY * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ECDSA_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_verify _cffi_d_ECDSA_verify
#endif

static void _cffi_d_EC_GROUP_free(EC_GROUP * x0)
{
  EC_GROUP_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_free(PyObject *self, PyObject *arg0)
{
  EC_GROUP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2863), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2863), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_GROUP_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_GROUP_free _cffi_d_EC_GROUP_free
#endif

static EC_POINT const * _cffi_d_EC_GROUP_get0_generator(EC_GROUP const * x0)
{
  return EC_GROUP_get0_generator(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get0_generator(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get0_generator(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(292));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get0_generator _cffi_d_EC_GROUP_get0_generator
#endif

static int _cffi_d_EC_GROUP_get_curve_name(EC_GROUP const * x0)
{
  return EC_GROUP_get_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_curve_name(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_curve_name _cffi_d_EC_GROUP_get_curve_name
#endif

static int _cffi_d_EC_GROUP_get_degree(EC_GROUP const * x0)
{
  return EC_GROUP_get_degree(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_degree(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_degree(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_degree _cffi_d_EC_GROUP_get_degree
#endif

static int _cffi_d_EC_GROUP_get_order(EC_GROUP const * x0, BIGNUM * x1, BN_CTX * x2)
{
  return EC_GROUP_get_order(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_order(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  BIGNUM * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_GROUP_get_order", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_order(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_order _cffi_d_EC_GROUP_get_order
#endif

static EC_METHOD const * _cffi_d_EC_GROUP_method_of(EC_GROUP const * x0)
{
  return EC_GROUP_method_of(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_method_of(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_METHOD const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_method_of(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1384));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_method_of _cffi_d_EC_GROUP_method_of
#endif

static EC_GROUP * _cffi_d_EC_GROUP_new_by_curve_name(int x0)
{
  return EC_GROUP_new_by_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_new_by_curve_name(PyObject *self, PyObject *arg0)
{
  int x0;
  EC_GROUP * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_new_by_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2863));
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_new_by_curve_name _cffi_d_EC_GROUP_new_by_curve_name
#endif

static void _cffi_d_EC_KEY_free(EC_KEY * x0)
{
  EC_KEY_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_free(PyObject *self, PyObject *arg0)
{
  EC_KEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_KEY_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_KEY_free _cffi_d_EC_KEY_free
#endif

static int _cffi_d_EC_KEY_generate_key(EC_KEY * x0)
{
  return EC_KEY_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_generate_key(PyObject *self, PyObject *arg0)
{
  EC_KEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_generate_key _cffi_d_EC_KEY_generate_key
#endif

static EC_GROUP const * _cffi_d_EC_KEY_get0_group(EC_KEY const * x0)
{
  return EC_KEY_get0_group(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_group(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_GROUP const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(128), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(128), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_group(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(286));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_group _cffi_d_EC_KEY_get0_group
#endif

static BIGNUM const * _cffi_d_EC_KEY_get0_private_key(EC_KEY const * x0)
{
  return EC_KEY_get0_private_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_private_key(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(128), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(128), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_private_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(110));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_private_key _cffi_d_EC_KEY_get0_private_key
#endif

static EC_POINT const * _cffi_d_EC_KEY_get0_public_key(EC_KEY const * x0)
{
  return EC_KEY_get0_public_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_public_key(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(128), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(128), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_public_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(292));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_public_key _cffi_d_EC_KEY_get0_public_key
#endif

static EC_KEY * _cffi_d_EC_KEY_new(void)
{
  return EC_KEY_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_new(PyObject *self, PyObject *noarg)
{
  EC_KEY * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(263));
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_new _cffi_d_EC_KEY_new
#endif

static EC_KEY * _cffi_d_EC_KEY_new_by_curve_name(int x0)
{
  return EC_KEY_new_by_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_new_by_curve_name(PyObject *self, PyObject *arg0)
{
  int x0;
  EC_KEY * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_new_by_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(263));
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_new_by_curve_name _cffi_d_EC_KEY_new_by_curve_name
#endif

static void _cffi_d_EC_KEY_set_asn1_flag(EC_KEY * x0, int x1)
{
  EC_KEY_set_asn1_flag(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_asn1_flag(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_asn1_flag", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_KEY_set_asn1_flag(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_KEY_set_asn1_flag _cffi_d_EC_KEY_set_asn1_flag
#endif

static int _cffi_d_EC_KEY_set_group(EC_KEY * x0, EC_GROUP const * x1)
{
  return EC_KEY_set_group(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_group(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  EC_GROUP const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_group", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_group(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_group _cffi_d_EC_KEY_set_group
#endif

static int _cffi_d_EC_KEY_set_private_key(EC_KEY * x0, BIGNUM const * x1)
{
  return EC_KEY_set_private_key(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_private_key(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_private_key", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_private_key(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_private_key _cffi_d_EC_KEY_set_private_key
#endif

static int _cffi_d_EC_KEY_set_public_key(EC_KEY * x0, EC_POINT const * x1)
{
  return EC_KEY_set_public_key(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_public_key(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  EC_POINT const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_public_key", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_public_key(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_public_key _cffi_d_EC_KEY_set_public_key
#endif

static int _cffi_d_EC_KEY_set_public_key_affine_coordinates(EC_KEY * x0, BIGNUM * x1, BIGNUM * x2)
{
  return EC_KEY_set_public_key_affine_coordinates(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_public_key_affine_coordinates(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_public_key_affine_coordinates", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_public_key_affine_coordinates(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_public_key_affine_coordinates _cffi_d_EC_KEY_set_public_key_affine_coordinates
#endif

static int _cffi_d_EC_METHOD_get_field_type(EC_METHOD const * x0)
{
  return EC_METHOD_get_field_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_METHOD_get_field_type(PyObject *self, PyObject *arg0)
{
  EC_METHOD const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1384), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_METHOD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1384), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_METHOD_get_field_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_METHOD_get_field_type _cffi_d_EC_METHOD_get_field_type
#endif

static int _cffi_d_EC_POINT_add(EC_GROUP const * x0, EC_POINT * x1, EC_POINT const * x2, EC_POINT const * x3, BN_CTX * x4)
{
  return EC_POINT_add(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_add(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  EC_POINT const * x2;
  EC_POINT const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_add", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_add(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_add _cffi_d_EC_POINT_add
#endif

static void _cffi_d_EC_POINT_clear_free(EC_POINT * x0)
{
  EC_POINT_clear_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_clear_free(PyObject *self, PyObject *arg0)
{
  EC_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_POINT_clear_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_POINT_clear_free _cffi_d_EC_POINT_clear_free
#endif

static int _cffi_d_EC_POINT_cmp(EC_GROUP const * x0, EC_POINT const * x1, EC_POINT const * x2, BN_CTX * x3)
{
  return EC_POINT_cmp(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_cmp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  EC_POINT const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EC_POINT_cmp", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_cmp(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_cmp _cffi_d_EC_POINT_cmp
#endif

static int _cffi_d_EC_POINT_dbl(EC_GROUP const * x0, EC_POINT * x1, EC_POINT const * x2, BN_CTX * x3)
{
  return EC_POINT_dbl(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_dbl(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  EC_POINT const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EC_POINT_dbl", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_dbl(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_dbl _cffi_d_EC_POINT_dbl
#endif

static EC_POINT * _cffi_d_EC_POINT_dup(EC_POINT const * x0, EC_GROUP const * x1)
{
  return EC_POINT_dup(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_dup(PyObject *self, PyObject *args)
{
  EC_POINT const * x0;
  EC_GROUP const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_POINT_dup", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_dup(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1293));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_dup _cffi_d_EC_POINT_dup
#endif

static void _cffi_d_EC_POINT_free(EC_POINT * x0)
{
  EC_POINT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_free(PyObject *self, PyObject *arg0)
{
  EC_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_POINT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_POINT_free _cffi_d_EC_POINT_free
#endif

static int _cffi_d_EC_POINT_get_affine_coordinates_GF2m(EC_GROUP const * x0, EC_POINT const * x1, BIGNUM * x2, BIGNUM * x3, BN_CTX * x4)
{
  return EC_POINT_get_affine_coordinates_GF2m(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_get_affine_coordinates_GF2m(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_get_affine_coordinates_GF2m", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_get_affine_coordinates_GF2m(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_get_affine_coordinates_GF2m _cffi_d_EC_POINT_get_affine_coordinates_GF2m
#endif

static int _cffi_d_EC_POINT_get_affine_coordinates_GFp(EC_GROUP const * x0, EC_POINT const * x1, BIGNUM * x2, BIGNUM * x3, BN_CTX * x4)
{
  return EC_POINT_get_affine_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_get_affine_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_get_affine_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_get_affine_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_get_affine_coordinates_GFp _cffi_d_EC_POINT_get_affine_coordinates_GFp
#endif

static int _cffi_d_EC_POINT_invert(EC_GROUP const * x0, EC_POINT * x1, BN_CTX * x2)
{
  return EC_POINT_invert(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_invert(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_POINT_invert", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_invert(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_invert _cffi_d_EC_POINT_invert
#endif

static int _cffi_d_EC_POINT_is_at_infinity(EC_GROUP const * x0, EC_POINT const * x1)
{
  return EC_POINT_is_at_infinity(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_is_at_infinity(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_POINT_is_at_infinity", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_is_at_infinity(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_is_at_infinity _cffi_d_EC_POINT_is_at_infinity
#endif

static int _cffi_d_EC_POINT_is_on_curve(EC_GROUP const * x0, EC_POINT const * x1, BN_CTX * x2)
{
  return EC_POINT_is_on_curve(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_is_on_curve(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_POINT_is_on_curve", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_is_on_curve(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_is_on_curve _cffi_d_EC_POINT_is_on_curve
#endif

static int _cffi_d_EC_POINT_mul(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, EC_POINT const * x3, BIGNUM const * x4, BN_CTX * x5)
{
  return EC_POINT_mul(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_mul(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  EC_POINT const * x3;
  BIGNUM const * x4;
  BN_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EC_POINT_mul", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_mul(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_mul _cffi_d_EC_POINT_mul
#endif

static EC_POINT * _cffi_d_EC_POINT_new(EC_GROUP const * x0)
{
  return EC_POINT_new(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_new(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_new(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1293));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_new _cffi_d_EC_POINT_new
#endif

static int _cffi_d_EC_POINT_oct2point(EC_GROUP const * x0, EC_POINT * x1, unsigned char const * x2, size_t x3, BN_CTX * x4)
{
  return EC_POINT_oct2point(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_oct2point(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  unsigned char const * x2;
  size_t x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_oct2point", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_oct2point(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_oct2point _cffi_d_EC_POINT_oct2point
#endif

static size_t _cffi_d_EC_POINT_point2oct(EC_GROUP const * x0, EC_POINT const * x1, point_conversion_form_t x2, unsigned char * x3, size_t x4, BN_CTX * x5)
{
  return EC_POINT_point2oct(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_point2oct(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  point_conversion_form_t x2;
  unsigned char * x3;
  size_t x4;
  BN_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EC_POINT_point2oct", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(292), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(292), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  if (_cffi_to_c((char *)&x2, _cffi_type(2574), arg2) < 0)
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_point2oct(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_point2oct _cffi_d_EC_POINT_point2oct
#endif

static int _cffi_d_EC_POINT_set_affine_coordinates_GF2m(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return EC_POINT_set_affine_coordinates_GF2m(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_affine_coordinates_GF2m(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_affine_coordinates_GF2m", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_affine_coordinates_GF2m(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_affine_coordinates_GF2m _cffi_d_EC_POINT_set_affine_coordinates_GF2m
#endif

static int _cffi_d_EC_POINT_set_affine_coordinates_GFp(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return EC_POINT_set_affine_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_affine_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_affine_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_affine_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_affine_coordinates_GFp _cffi_d_EC_POINT_set_affine_coordinates_GFp
#endif

static int _cffi_d_EC_POINT_set_compressed_coordinates_GF2m(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, int x3, BN_CTX * x4)
{
  return EC_POINT_set_compressed_coordinates_GF2m(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_compressed_coordinates_GF2m(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  int x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_compressed_coordinates_GF2m", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_compressed_coordinates_GF2m(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_compressed_coordinates_GF2m _cffi_d_EC_POINT_set_compressed_coordinates_GF2m
#endif

static int _cffi_d_EC_POINT_set_compressed_coordinates_GFp(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, int x3, BN_CTX * x4)
{
  return EC_POINT_set_compressed_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_compressed_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  int x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_compressed_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(286), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(286), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1293), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1293), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(112), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(112), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_compressed_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_compressed_coordinates_GFp _cffi_d_EC_POINT_set_compressed_coordinates_GFp
#endif

static char const * _cffi_d_EC_curve_nid2nist(int x0)
{
  return EC_curve_nid2nist(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_curve_nid2nist(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_curve_nid2nist(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  return pyresult;
}
#else
#  define _cffi_f_EC_curve_nid2nist _cffi_d_EC_curve_nid2nist
#endif

static size_t _cffi_d_EC_get_builtin_curves(EC_builtin_curve * x0, size_t x1)
{
  return EC_get_builtin_curves(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_get_builtin_curves(PyObject *self, PyObject *args)
{
  EC_builtin_curve * x0;
  size_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_get_builtin_curves", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2580), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_builtin_curve *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2580), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_get_builtin_curves(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_get_builtin_curves _cffi_d_EC_get_builtin_curves
#endif

static ENGINE * _cffi_d_ENGINE_by_id(char const * x0)
{
  return ENGINE_by_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_by_id(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ENGINE * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_by_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(340));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_by_id _cffi_d_ENGINE_by_id
#endif

static int _cffi_d_ENGINE_ctrl_cmd(ENGINE * x0, char const * x1, long x2, void * x3, void(* x4)(void), int x5)
{
  return ENGINE_ctrl_cmd(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_ctrl_cmd(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  long x2;
  void * x3;
  void(* x4)(void);
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ENGINE_ctrl_cmd", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, long);
  if (x2 == (long)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = (void(*)(void))_cffi_to_c_pointer(arg4, _cffi_type(1394));
  if (x4 == (void(*)(void))NULL && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_ctrl_cmd(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_ctrl_cmd _cffi_d_ENGINE_ctrl_cmd
#endif

static int _cffi_d_ENGINE_finish(ENGINE * x0)
{
  return ENGINE_finish(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_finish(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_finish(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_finish _cffi_d_ENGINE_finish
#endif

static int _cffi_d_ENGINE_free(ENGINE * x0)
{
  return ENGINE_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_free(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_free _cffi_d_ENGINE_free
#endif

static ENGINE * _cffi_d_ENGINE_get_default_RAND(void)
{
  return ENGINE_get_default_RAND();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_get_default_RAND(PyObject *self, PyObject *noarg)
{
  ENGINE * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_get_default_RAND(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(340));
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_get_default_RAND _cffi_d_ENGINE_get_default_RAND
#endif

static char const * _cffi_d_ENGINE_get_name(ENGINE const * x0)
{
  return ENGINE_get_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_get_name(PyObject *self, PyObject *arg0)
{
  ENGINE const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(784), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(784), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_get_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_get_name _cffi_d_ENGINE_get_name
#endif

static int _cffi_d_ENGINE_init(ENGINE * x0)
{
  return ENGINE_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_init(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_init _cffi_d_ENGINE_init
#endif

static int _cffi_d_ENGINE_set_default_RAND(ENGINE * x0)
{
  return ENGINE_set_default_RAND(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_set_default_RAND(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_set_default_RAND(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_set_default_RAND _cffi_d_ENGINE_set_default_RAND
#endif

static void _cffi_d_ENGINE_unregister_RAND(ENGINE * x0)
{
  ENGINE_unregister_RAND(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_unregister_RAND(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ENGINE_unregister_RAND(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ENGINE_unregister_RAND _cffi_d_ENGINE_unregister_RAND
#endif

static int _cffi_d_ERR_GET_LIB(unsigned long x0)
{
  return ERR_GET_LIB(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_GET_LIB(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_GET_LIB(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_ERR_GET_LIB _cffi_d_ERR_GET_LIB
#endif

static int _cffi_d_ERR_GET_REASON(unsigned long x0)
{
  return ERR_GET_REASON(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_GET_REASON(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_GET_REASON(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_ERR_GET_REASON _cffi_d_ERR_GET_REASON
#endif

static void _cffi_d_ERR_clear_error(void)
{
  ERR_clear_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_clear_error(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_clear_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_clear_error _cffi_d_ERR_clear_error
#endif

static void _cffi_d_ERR_error_string_n(unsigned long x0, char * x1, size_t x2)
{
  ERR_error_string_n(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_error_string_n(PyObject *self, PyObject *args)
{
  unsigned long x0;
  char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ERR_error_string_n", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(451), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(451), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_error_string_n(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_error_string_n _cffi_d_ERR_error_string_n
#endif

static char const * _cffi_d_ERR_func_error_string(unsigned long x0)
{
  return ERR_func_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_func_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_func_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  return pyresult;
}
#else
#  define _cffi_f_ERR_func_error_string _cffi_d_ERR_func_error_string
#endif

static unsigned long _cffi_d_ERR_get_error(void)
{
  return ERR_get_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_get_error(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_get_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_ERR_get_error _cffi_d_ERR_get_error
#endif

static char const * _cffi_d_ERR_lib_error_string(unsigned long x0)
{
  return ERR_lib_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_lib_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_lib_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  return pyresult;
}
#else
#  define _cffi_f_ERR_lib_error_string _cffi_d_ERR_lib_error_string
#endif

static void _cffi_d_ERR_load_RAND_strings(void)
{
  ERR_load_RAND_strings();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_load_RAND_strings(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_load_RAND_strings(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_load_RAND_strings _cffi_d_ERR_load_RAND_strings
#endif

static unsigned long _cffi_d_ERR_peek_error(void)
{
  return ERR_peek_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_peek_error(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_peek_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_ERR_peek_error _cffi_d_ERR_peek_error
#endif

static unsigned long _cffi_d_ERR_peek_last_error(void)
{
  return ERR_peek_last_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_peek_last_error(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_peek_last_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_ERR_peek_last_error _cffi_d_ERR_peek_last_error
#endif

static char const * _cffi_d_ERR_reason_error_string(unsigned long x0)
{
  return ERR_reason_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_reason_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_reason_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  return pyresult;
}
#else
#  define _cffi_f_ERR_reason_error_string _cffi_d_ERR_reason_error_string
#endif

static EVP_CIPHER const * _cffi_d_EVP_CIPHER_CTX_cipher(EVP_CIPHER_CTX const * x0)
{
  return EVP_CIPHER_CTX_cipher(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_cipher(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_CIPHER const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(307), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(307), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_cipher(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(479));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_cipher _cffi_d_EVP_CIPHER_CTX_cipher
#endif

static int _cffi_d_EVP_CIPHER_CTX_cleanup(EVP_CIPHER_CTX * x0)
{
  return EVP_CIPHER_CTX_cleanup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_cleanup(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_cleanup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_cleanup _cffi_d_EVP_CIPHER_CTX_cleanup
#endif

static int _cffi_d_EVP_CIPHER_CTX_ctrl(EVP_CIPHER_CTX * x0, int x1, int x2, void * x3)
{
  return EVP_CIPHER_CTX_ctrl(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_ctrl(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  int x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_ctrl", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_ctrl(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_ctrl _cffi_d_EVP_CIPHER_CTX_ctrl
#endif

static void _cffi_d_EVP_CIPHER_CTX_free(EVP_CIPHER_CTX * x0)
{
  EVP_CIPHER_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_CIPHER_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_free _cffi_d_EVP_CIPHER_CTX_free
#endif

static EVP_CIPHER_CTX * _cffi_d_EVP_CIPHER_CTX_new(void)
{
  return EVP_CIPHER_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_CIPHER_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1398));
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_new _cffi_d_EVP_CIPHER_CTX_new
#endif

static int _cffi_d_EVP_CIPHER_CTX_set_key_length(EVP_CIPHER_CTX * x0, int x1)
{
  return EVP_CIPHER_CTX_set_key_length(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_set_key_length(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_set_key_length", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_set_key_length(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_set_key_length _cffi_d_EVP_CIPHER_CTX_set_key_length
#endif

static int _cffi_d_EVP_CIPHER_CTX_set_padding(EVP_CIPHER_CTX * x0, int x1)
{
  return EVP_CIPHER_CTX_set_padding(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_set_padding(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_set_padding", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_set_padding(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_set_padding _cffi_d_EVP_CIPHER_CTX_set_padding
#endif

static int _cffi_d_EVP_CipherFinal_ex(EVP_CIPHER_CTX * x0, unsigned char * x1, int * x2)
{
  return EVP_CipherFinal_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherFinal_ex(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  unsigned char * x1;
  int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_CipherFinal_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1251), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1251), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherFinal_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherFinal_ex _cffi_d_EVP_CipherFinal_ex
#endif

static int _cffi_d_EVP_CipherInit_ex(EVP_CIPHER_CTX * x0, EVP_CIPHER const * x1, ENGINE * x2, unsigned char const * x3, unsigned char const * x4, int x5)
{
  return EVP_CipherInit_ex(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherInit_ex(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  EVP_CIPHER const * x1;
  ENGINE * x2;
  unsigned char const * x3;
  unsigned char const * x4;
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EVP_CipherInit_ex", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(479), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(479), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherInit_ex(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherInit_ex _cffi_d_EVP_CipherInit_ex
#endif

static int _cffi_d_EVP_CipherUpdate(EVP_CIPHER_CTX * x0, unsigned char * x1, int * x2, unsigned char const * x3, int x4)
{
  return EVP_CipherUpdate(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherUpdate(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  unsigned char * x1;
  int * x2;
  unsigned char const * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_CipherUpdate", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1398), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1398), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1251), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1251), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherUpdate(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherUpdate _cffi_d_EVP_CipherUpdate
#endif

static int _cffi_d_EVP_DigestFinalXOF(EVP_MD_CTX * x0, unsigned char * x1, size_t x2)
{
  return EVP_DigestFinalXOF(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestFinalXOF(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestFinalXOF", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestFinalXOF(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestFinalXOF _cffi_d_EVP_DigestFinalXOF
#endif

static int _cffi_d_EVP_DigestFinal_ex(EVP_MD_CTX * x0, unsigned char * x1, unsigned int * x2)
{
  return EVP_DigestFinal_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestFinal_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestFinal_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1473), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1473), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestFinal_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestFinal_ex _cffi_d_EVP_DigestFinal_ex
#endif

static int _cffi_d_EVP_DigestInit_ex(EVP_MD_CTX * x0, EVP_MD const * x1, ENGINE * x2)
{
  return EVP_DigestInit_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestInit_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  ENGINE * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestInit_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestInit_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestInit_ex _cffi_d_EVP_DigestInit_ex
#endif

static int _cffi_d_EVP_DigestSign(EVP_MD_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_DigestSign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSign(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSign _cffi_d_EVP_DigestSign
#endif

static int _cffi_d_EVP_DigestSignFinal(EVP_MD_CTX * x0, unsigned char * x1, size_t * x2)
{
  return EVP_DigestSignFinal(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignFinal", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignFinal(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignFinal _cffi_d_EVP_DigestSignFinal
#endif

static int _cffi_d_EVP_DigestSignInit(EVP_MD_CTX * x0, EVP_PKEY_CTX * * x1, EVP_MD const * x2, ENGINE * x3, EVP_PKEY * x4)
{
  return EVP_DigestSignInit(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_PKEY_CTX * * x1;
  EVP_MD const * x2;
  ENGINE * x3;
  EVP_PKEY * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignInit", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1448), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1448), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignInit(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignInit _cffi_d_EVP_DigestSignInit
#endif

static int _cffi_d_EVP_DigestSignUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_DigestSignUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignUpdate _cffi_d_EVP_DigestSignUpdate
#endif

static int _cffi_d_EVP_DigestUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_DigestUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestUpdate _cffi_d_EVP_DigestUpdate
#endif

static int _cffi_d_EVP_DigestVerify(EVP_MD_CTX * x0, unsigned char const * x1, size_t x2, unsigned char const * x3, size_t x4)
{
  return EVP_DigestVerify(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestVerify(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestVerify", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestVerify(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestVerify _cffi_d_EVP_DigestVerify
#endif

static int _cffi_d_EVP_DigestVerifyInit(EVP_MD_CTX * x0, EVP_PKEY_CTX * * x1, EVP_MD const * x2, ENGINE * x3, EVP_PKEY * x4)
{
  return EVP_DigestVerifyInit(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestVerifyInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_PKEY_CTX * * x1;
  EVP_MD const * x2;
  ENGINE * x3;
  EVP_PKEY * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestVerifyInit", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1448), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1448), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestVerifyInit(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestVerifyInit _cffi_d_EVP_DigestVerifyInit
#endif

static int _cffi_d_EVP_MD_CTX_block_size(EVP_MD_CTX const * x0)
{
  return EVP_MD_CTX_block_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_block_size(PyObject *self, PyObject *arg0)
{
  EVP_MD_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1444), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1444), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_MD_CTX_block_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_MD_CTX_block_size _cffi_d_EVP_MD_CTX_block_size
#endif

static int _cffi_d_EVP_MD_CTX_copy_ex(EVP_MD_CTX * x0, EVP_MD_CTX const * x1)
{
  return EVP_MD_CTX_copy_ex(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_copy_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD_CTX const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_MD_CTX_copy_ex", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1444), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1444), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_MD_CTX_copy_ex(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_MD_CTX_copy_ex _cffi_d_EVP_MD_CTX_copy_ex
#endif

static void _cffi_d_EVP_MD_CTX_set_flags(EVP_MD_CTX * x0, int x1)
{
  EVP_MD_CTX_set_flags(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_set_flags(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_MD_CTX_set_flags", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_MD_CTX_set_flags(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_MD_CTX_set_flags _cffi_d_EVP_MD_CTX_set_flags
#endif

static int _cffi_d_EVP_MD_size(EVP_MD const * x0)
{
  return EVP_MD_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_size(PyObject *self, PyObject *arg0)
{
  EVP_MD const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_MD_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_MD_size _cffi_d_EVP_MD_size
#endif

static int _cffi_d_EVP_PBE_scrypt(char const * x0, size_t x1, unsigned char const * x2, size_t x3, uint64_t x4, uint64_t x5, uint64_t x6, uint64_t x7, unsigned char * x8, size_t x9)
{
  return EVP_PBE_scrypt(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PBE_scrypt(PyObject *self, PyObject *args)
{
  char const * x0;
  size_t x1;
  unsigned char const * x2;
  size_t x3;
  uint64_t x4;
  uint64_t x5;
  uint64_t x6;
  uint64_t x7;
  unsigned char * x8;
  size_t x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "EVP_PBE_scrypt", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, uint64_t);
  if (x4 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, uint64_t);
  if (x5 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, uint64_t);
  if (x6 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, uint64_t);
  if (x7 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x9 = _cffi_to_c_int(arg9, size_t);
  if (x9 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PBE_scrypt(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PBE_scrypt _cffi_d_EVP_PBE_scrypt
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_dup(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_CTX_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_dup(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(351));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_dup _cffi_d_EVP_PKEY_CTX_dup
#endif

static void _cffi_d_EVP_PKEY_CTX_free(EVP_PKEY_CTX * x0)
{
  EVP_PKEY_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_PKEY_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_free _cffi_d_EVP_PKEY_CTX_free
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_new(EVP_PKEY * x0, ENGINE * x1)
{
  return EVP_PKEY_CTX_new(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_new(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  ENGINE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_new", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_new(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(351));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_new _cffi_d_EVP_PKEY_CTX_new
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_new_id(int x0, ENGINE * x1)
{
  return EVP_PKEY_CTX_new_id(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_new_id(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_new_id", 2, 2, &arg0, &arg1))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_new_id(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(351));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_new_id _cffi_d_EVP_PKEY_CTX_new_id
#endif

static int _cffi_d_EVP_PKEY_CTX_set0_rsa_oaep_label(EVP_PKEY_CTX * x0, unsigned char * x1, int x2)
{
  return EVP_PKEY_CTX_set0_rsa_oaep_label(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set0_rsa_oaep_label(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set0_rsa_oaep_label", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set0_rsa_oaep_label(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set0_rsa_oaep_label _cffi_d_EVP_PKEY_CTX_set0_rsa_oaep_label
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_mgf1_md(EVP_PKEY_CTX * x0, EVP_MD * x1)
{
  return EVP_PKEY_CTX_set_rsa_mgf1_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_mgf1_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_mgf1_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1544), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1544), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_mgf1_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_mgf1_md _cffi_d_EVP_PKEY_CTX_set_rsa_mgf1_md
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_oaep_md(EVP_PKEY_CTX * x0, EVP_MD * x1)
{
  return EVP_PKEY_CTX_set_rsa_oaep_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_oaep_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_oaep_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1544), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1544), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_oaep_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_oaep_md _cffi_d_EVP_PKEY_CTX_set_rsa_oaep_md
#endif

static int _cffi_d_EVP_PKEY_CTX_set_signature_md(EVP_PKEY_CTX * x0, EVP_MD const * x1)
{
  return EVP_PKEY_CTX_set_signature_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_signature_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_signature_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_signature_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_signature_md _cffi_d_EVP_PKEY_CTX_set_signature_md
#endif

static int _cffi_d_EVP_PKEY_assign_RSA(EVP_PKEY * x0, RSA * x1)
{
  return EVP_PKEY_assign_RSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_assign_RSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  RSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_assign_RSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(517), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(517), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_assign_RSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_assign_RSA _cffi_d_EVP_PKEY_assign_RSA
#endif

static int _cffi_d_EVP_PKEY_cmp(EVP_PKEY const * x0, EVP_PKEY const * x1)
{
  return EVP_PKEY_cmp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_cmp(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  EVP_PKEY const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_cmp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1531), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1531), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1531), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1531), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_cmp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_cmp _cffi_d_EVP_PKEY_cmp
#endif

static int _cffi_d_EVP_PKEY_decrypt(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_decrypt(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_decrypt(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_decrypt", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_decrypt(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_decrypt _cffi_d_EVP_PKEY_decrypt
#endif

static int _cffi_d_EVP_PKEY_decrypt_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_decrypt_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_decrypt_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_decrypt_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_decrypt_init _cffi_d_EVP_PKEY_decrypt_init
#endif

static int _cffi_d_EVP_PKEY_derive(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_derive(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_derive", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive _cffi_d_EVP_PKEY_derive
#endif

static int _cffi_d_EVP_PKEY_derive_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_derive_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive_init _cffi_d_EVP_PKEY_derive_init
#endif

static int _cffi_d_EVP_PKEY_derive_set_peer(EVP_PKEY_CTX * x0, EVP_PKEY * x1)
{
  return EVP_PKEY_derive_set_peer(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive_set_peer(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_derive_set_peer", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive_set_peer(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive_set_peer _cffi_d_EVP_PKEY_derive_set_peer
#endif

static int _cffi_d_EVP_PKEY_encrypt(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_encrypt(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_encrypt(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_encrypt", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_encrypt(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_encrypt _cffi_d_EVP_PKEY_encrypt
#endif

static int _cffi_d_EVP_PKEY_encrypt_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_encrypt_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_encrypt_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_encrypt_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_encrypt_init _cffi_d_EVP_PKEY_encrypt_init
#endif

static void _cffi_d_EVP_PKEY_free(EVP_PKEY * x0)
{
  EVP_PKEY_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_free(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_PKEY_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_PKEY_free _cffi_d_EVP_PKEY_free
#endif

static DH * _cffi_d_EVP_PKEY_get1_DH(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_DH(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_DH(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_DH(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(232));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_DH _cffi_d_EVP_PKEY_get1_DH
#endif

static DSA * _cffi_d_EVP_PKEY_get1_DSA(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_DSA(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_DSA(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_DSA(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(248));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_DSA _cffi_d_EVP_PKEY_get1_DSA
#endif

static EC_KEY * _cffi_d_EVP_PKEY_get1_EC_KEY(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_EC_KEY(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_EC_KEY(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_KEY * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_EC_KEY(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(263));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_EC_KEY _cffi_d_EVP_PKEY_get1_EC_KEY
#endif

static RSA * _cffi_d_EVP_PKEY_get1_RSA(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_RSA(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_RSA(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_RSA(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(517));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_RSA _cffi_d_EVP_PKEY_get1_RSA
#endif

static size_t _cffi_d_EVP_PKEY_get1_tls_encodedpoint(EVP_PKEY * x0, unsigned char * * x1)
{
  return EVP_PKEY_get1_tls_encodedpoint(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_tls_encodedpoint(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  unsigned char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get1_tls_encodedpoint", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(867), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(867), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_tls_encodedpoint(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_tls_encodedpoint _cffi_d_EVP_PKEY_get1_tls_encodedpoint
#endif

static int _cffi_d_EVP_PKEY_get_raw_private_key(EVP_PKEY const * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_get_raw_private_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get_raw_private_key(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get_raw_private_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1531), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1531), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get_raw_private_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get_raw_private_key _cffi_d_EVP_PKEY_get_raw_private_key
#endif

static int _cffi_d_EVP_PKEY_get_raw_public_key(EVP_PKEY const * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_get_raw_public_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get_raw_public_key(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get_raw_public_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1531), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1531), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get_raw_public_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get_raw_public_key _cffi_d_EVP_PKEY_get_raw_public_key
#endif

static int _cffi_d_EVP_PKEY_keygen(EVP_PKEY_CTX * x0, EVP_PKEY * * x1)
{
  return EVP_PKEY_keygen(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_keygen(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_PKEY * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_keygen", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(321), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(321), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_keygen(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_keygen _cffi_d_EVP_PKEY_keygen
#endif

static int _cffi_d_EVP_PKEY_keygen_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_keygen_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_keygen_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_keygen_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_keygen_init _cffi_d_EVP_PKEY_keygen_init
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new(void)
{
  return EVP_PKEY_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new(PyObject *self, PyObject *noarg)
{
  EVP_PKEY * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(235));
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new _cffi_d_EVP_PKEY_new
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new_raw_private_key(int x0, ENGINE * x1, unsigned char const * x2, size_t x3)
{
  return EVP_PKEY_new_raw_private_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new_raw_private_key(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  unsigned char const * x2;
  size_t x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_new_raw_private_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new_raw_private_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(235));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new_raw_private_key _cffi_d_EVP_PKEY_new_raw_private_key
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new_raw_public_key(int x0, ENGINE * x1, unsigned char const * x2, size_t x3)
{
  return EVP_PKEY_new_raw_public_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new_raw_public_key(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  unsigned char const * x2;
  size_t x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_new_raw_public_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new_raw_public_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(235));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new_raw_public_key _cffi_d_EVP_PKEY_new_raw_public_key
#endif

static int _cffi_d_EVP_PKEY_set1_DH(EVP_PKEY * x0, DH * x1)
{
  return EVP_PKEY_set1_DH(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_DH(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_DH", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(232), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(232), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_DH(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_DH _cffi_d_EVP_PKEY_set1_DH
#endif

static int _cffi_d_EVP_PKEY_set1_DSA(EVP_PKEY * x0, DSA * x1)
{
  return EVP_PKEY_set1_DSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_DSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  DSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_DSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(248), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(248), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_DSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_DSA _cffi_d_EVP_PKEY_set1_DSA
#endif

static int _cffi_d_EVP_PKEY_set1_EC_KEY(EVP_PKEY * x0, EC_KEY * x1)
{
  return EVP_PKEY_set1_EC_KEY(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_EC_KEY(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  EC_KEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_EC_KEY", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(263), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(263), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_EC_KEY(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_EC_KEY _cffi_d_EVP_PKEY_set1_EC_KEY
#endif

static int _cffi_d_EVP_PKEY_set1_RSA(EVP_PKEY * x0, RSA * x1)
{
  return EVP_PKEY_set1_RSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_RSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  RSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_RSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(517), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(517), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_RSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_RSA _cffi_d_EVP_PKEY_set1_RSA
#endif

static int _cffi_d_EVP_PKEY_set1_tls_encodedpoint(EVP_PKEY * x0, unsigned char const * x1, size_t x2)
{
  return EVP_PKEY_set1_tls_encodedpoint(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_tls_encodedpoint(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  unsigned char const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_tls_encodedpoint", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_tls_encodedpoint(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_tls_encodedpoint _cffi_d_EVP_PKEY_set1_tls_encodedpoint
#endif

static int _cffi_d_EVP_PKEY_set_type(EVP_PKEY * x0, int x1)
{
  return EVP_PKEY_set_type(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set_type(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set_type", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set_type(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set_type _cffi_d_EVP_PKEY_set_type
#endif

static int _cffi_d_EVP_PKEY_sign(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_sign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_sign(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_sign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1119), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1119), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_sign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_sign _cffi_d_EVP_PKEY_sign
#endif

static int _cffi_d_EVP_PKEY_sign_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_sign_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_sign_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_sign_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_sign_init _cffi_d_EVP_PKEY_sign_init
#endif

static int _cffi_d_EVP_PKEY_size(EVP_PKEY * x0)
{
  return EVP_PKEY_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_size(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_size _cffi_d_EVP_PKEY_size
#endif

static int _cffi_d_EVP_PKEY_type(int x0)
{
  return EVP_PKEY_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_type(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_type _cffi_d_EVP_PKEY_type
#endif

static int _cffi_d_EVP_PKEY_verify(EVP_PKEY_CTX * x0, unsigned char const * x1, size_t x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_verify(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_verify", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify _cffi_d_EVP_PKEY_verify
#endif

static int _cffi_d_EVP_PKEY_verify_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_verify_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify_init _cffi_d_EVP_PKEY_verify_init
#endif

static int _cffi_d_EVP_SignFinal(EVP_MD_CTX * x0, unsigned char * x1, unsigned int * x2, EVP_PKEY * x3)
{
  return EVP_SignFinal(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  EVP_PKEY * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_SignFinal", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1473), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1473), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignFinal(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignFinal _cffi_d_EVP_SignFinal
#endif

static int _cffi_d_EVP_SignInit(EVP_MD_CTX * x0, EVP_MD const * x1)
{
  return EVP_SignInit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_SignInit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignInit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignInit _cffi_d_EVP_SignInit
#endif

static int _cffi_d_EVP_SignUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_SignUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_SignUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignUpdate _cffi_d_EVP_SignUpdate
#endif

static int _cffi_d_EVP_VerifyFinal(EVP_MD_CTX * x0, unsigned char const * x1, unsigned int x2, EVP_PKEY * x3)
{
  return EVP_VerifyFinal(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char const * x1;
  unsigned int x2;
  EVP_PKEY * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyFinal", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, unsigned int);
  if (x2 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyFinal(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyFinal _cffi_d_EVP_VerifyFinal
#endif

static int _cffi_d_EVP_VerifyInit(EVP_MD_CTX * x0, EVP_MD const * x1)
{
  return EVP_VerifyInit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyInit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyInit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyInit _cffi_d_EVP_VerifyInit
#endif

static int _cffi_d_EVP_VerifyUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_VerifyUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1434), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1434), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyUpdate _cffi_d_EVP_VerifyUpdate
#endif

static EVP_CIPHER const * _cffi_d_EVP_get_cipherbyname(char const * x0)
{
  return EVP_get_cipherbyname(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_get_cipherbyname(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_CIPHER const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_get_cipherbyname(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(479));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_get_cipherbyname _cffi_d_EVP_get_cipherbyname
#endif

static EVP_MD const * _cffi_d_EVP_get_digestbyname(char const * x0)
{
  return EVP_get_digestbyname(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_get_digestbyname(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_MD const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_get_digestbyname(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(396));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_get_digestbyname _cffi_d_EVP_get_digestbyname
#endif

static void _cffi_d_GENERAL_NAMES_free(GENERAL_NAMES * x0)
{
  GENERAL_NAMES_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAMES_free(PyObject *self, PyObject *arg0)
{
  GENERAL_NAMES * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(358), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (GENERAL_NAMES *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(358), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { GENERAL_NAMES_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_GENERAL_NAMES_free _cffi_d_GENERAL_NAMES_free
#endif

static GENERAL_NAMES * _cffi_d_GENERAL_NAMES_new(void)
{
  return GENERAL_NAMES_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAMES_new(PyObject *self, PyObject *noarg)
{
  GENERAL_NAMES * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_NAMES_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(358));
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_NAMES_new _cffi_d_GENERAL_NAMES_new
#endif

static void _cffi_d_GENERAL_NAME_free(GENERAL_NAME * x0)
{
  GENERAL_NAME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAME_free(PyObject *self, PyObject *arg0)
{
  GENERAL_NAME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1006), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (GENERAL_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1006), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { GENERAL_NAME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_GENERAL_NAME_free _cffi_d_GENERAL_NAME_free
#endif

static GENERAL_NAME * _cffi_d_GENERAL_NAME_new(void)
{
  return GENERAL_NAME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAME_new(PyObject *self, PyObject *noarg)
{
  GENERAL_NAME * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_NAME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1006));
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_NAME_new _cffi_d_GENERAL_NAME_new
#endif

static int _cffi_d_GENERAL_NAME_print(BIO * x0, GENERAL_NAME * x1)
{
  return GENERAL_NAME_print(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAME_print(PyObject *self, PyObject *args)
{
  BIO * x0;
  GENERAL_NAME * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "GENERAL_NAME_print", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1006), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (GENERAL_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1006), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_NAME_print(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_NAME_print _cffi_d_GENERAL_NAME_print
#endif

static GENERAL_SUBTREE * _cffi_d_GENERAL_SUBTREE_new(void)
{
  return GENERAL_SUBTREE_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_SUBTREE_new(PyObject *self, PyObject *noarg)
{
  GENERAL_SUBTREE * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_SUBTREE_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1166));
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_SUBTREE_new _cffi_d_GENERAL_SUBTREE_new
#endif

static int _cffi_d_HMAC_CTX_copy(HMAC_CTX * x0, HMAC_CTX * x1)
{
  return HMAC_CTX_copy(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_CTX_copy(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  HMAC_CTX * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "HMAC_CTX_copy", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1594), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1594), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_CTX_copy(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_CTX_copy _cffi_d_HMAC_CTX_copy
#endif

static int _cffi_d_HMAC_Final(HMAC_CTX * x0, unsigned char * x1, unsigned int * x2)
{
  return HMAC_Final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Final(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "HMAC_Final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1473), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1473), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Final _cffi_d_HMAC_Final
#endif

static int _cffi_d_HMAC_Init_ex(HMAC_CTX * x0, void const * x1, int x2, EVP_MD const * x3, ENGINE * x4)
{
  return HMAC_Init_ex(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Init_ex(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  void const * x1;
  int x2;
  EVP_MD const * x3;
  ENGINE * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "HMAC_Init_ex", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(151), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(151), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(340), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(340), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Init_ex(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Init_ex _cffi_d_HMAC_Init_ex
#endif

static int _cffi_d_HMAC_Update(HMAC_CTX * x0, unsigned char const * x1, size_t x2)
{
  return HMAC_Update(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Update(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "HMAC_Update", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(121), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(121), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Update(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Update _cffi_d_HMAC_Update
#endif

static void _cffi_d_ISSUING_DIST_POINT_free(ISSUING_DIST_POINT * x0)
{
  ISSUING_DIST_POINT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ISSUING_DIST_POINT_free(PyObject *self, PyObject *arg0)
{
  ISSUING_DIST_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2908), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ISSUING_DIST_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2908), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ISSUING_DIST_POINT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ISSUING_DIST_POINT_free _cffi_d_ISSUING_DIST_POINT_free
#endif

static ISSUING_DIST_POINT * _cffi_d_ISSUING_DIST_POINT_new(void)
{
  return ISSUING_DIST_POINT_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ISSUING_DIST_POINT_new(PyObject *self, PyObject *noarg)
{
  ISSUING_DIST_POINT * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ISSUING_DIST_POINT_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2908));
  return pyresult;
}
#else
#  define _cffi_f_ISSUING_DIST_POINT_new _cffi_d_ISSUING_DIST_POINT_new
#endif

static void _cffi_d_NAME_CONSTRAINTS_free(NAME_CONSTRAINTS * x0)
{
  NAME_CONSTRAINTS_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NAME_CONSTRAINTS_free(PyObject *self, PyObject *arg0)
{
  NAME_CONSTRAINTS * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2911), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NAME_CONSTRAINTS *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2911), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { NAME_CONSTRAINTS_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_NAME_CONSTRAINTS_free _cffi_d_NAME_CONSTRAINTS_free
#endif

static NAME_CONSTRAINTS * _cffi_d_NAME_CONSTRAINTS_new(void)
{
  return NAME_CONSTRAINTS_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NAME_CONSTRAINTS_new(PyObject *self, PyObject *noarg)
{
  NAME_CONSTRAINTS * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NAME_CONSTRAINTS_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2911));
  return pyresult;
}
#else
#  define _cffi_f_NAME_CONSTRAINTS_new _cffi_d_NAME_CONSTRAINTS_new
#endif

static NETSCAPE_SPKI * _cffi_d_NETSCAPE_SPKI_b64_decode(char const * x0, int x1)
{
  return NETSCAPE_SPKI_b64_decode(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_b64_decode(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  NETSCAPE_SPKI * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_b64_decode", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_b64_decode(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(330));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_b64_decode _cffi_d_NETSCAPE_SPKI_b64_decode
#endif

static char * _cffi_d_NETSCAPE_SPKI_b64_encode(NETSCAPE_SPKI * x0)
{
  return NETSCAPE_SPKI_b64_encode(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_b64_encode(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(330), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(330), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_b64_encode(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(451));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_b64_encode _cffi_d_NETSCAPE_SPKI_b64_encode
#endif

static void _cffi_d_NETSCAPE_SPKI_free(NETSCAPE_SPKI * x0)
{
  NETSCAPE_SPKI_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_free(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(330), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(330), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { NETSCAPE_SPKI_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_free _cffi_d_NETSCAPE_SPKI_free
#endif

static EVP_PKEY * _cffi_d_NETSCAPE_SPKI_get_pubkey(NETSCAPE_SPKI * x0)
{
  return NETSCAPE_SPKI_get_pubkey(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_get_pubkey(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(330), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(330), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_get_pubkey(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(235));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_get_pubkey _cffi_d_NETSCAPE_SPKI_get_pubkey
#endif

static NETSCAPE_SPKI * _cffi_d_NETSCAPE_SPKI_new(void)
{
  return NETSCAPE_SPKI_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_new(PyObject *self, PyObject *noarg)
{
  NETSCAPE_SPKI * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(330));
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_new _cffi_d_NETSCAPE_SPKI_new
#endif

static int _cffi_d_NETSCAPE_SPKI_set_pubkey(NETSCAPE_SPKI * x0, EVP_PKEY * x1)
{
  return NETSCAPE_SPKI_set_pubkey(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_set_pubkey(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_set_pubkey", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(330), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(330), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_set_pubkey(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_set_pubkey _cffi_d_NETSCAPE_SPKI_set_pubkey
#endif

static int _cffi_d_NETSCAPE_SPKI_sign(NETSCAPE_SPKI * x0, EVP_PKEY * x1, EVP_MD const * x2)
{
  return NETSCAPE_SPKI_sign(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_sign(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  EVP_MD const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_sign", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(330), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(330), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_sign(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_sign _cffi_d_NETSCAPE_SPKI_sign
#endif

static int _cffi_d_NETSCAPE_SPKI_verify(NETSCAPE_SPKI * x0, EVP_PKEY * x1)
{
  return NETSCAPE_SPKI_verify(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_verify(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_verify", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(330), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(330), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_verify(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_verify _cffi_d_NETSCAPE_SPKI_verify
#endif

static void _cffi_d_NOTICEREF_free(NOTICEREF * x0)
{
  NOTICEREF_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NOTICEREF_free(PyObject *self, PyObject *arg0)
{
  NOTICEREF * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2917), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NOTICEREF *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2917), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { NOTICEREF_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_NOTICEREF_free _cffi_d_NOTICEREF_free
#endif

static NOTICEREF * _cffi_d_NOTICEREF_new(void)
{
  return NOTICEREF_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NOTICEREF_new(PyObject *self, PyObject *noarg)
{
  NOTICEREF * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NOTICEREF_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2917));
  return pyresult;
}
#else
#  define _cffi_f_NOTICEREF_new _cffi_d_NOTICEREF_new
#endif

static void _cffi_d_OBJ_NAME_do_all(int x0, void(* x1)(OBJ_NAME const *, void *), void * x2)
{
  OBJ_NAME_do_all(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_NAME_do_all(PyObject *self, PyObject *args)
{
  int x0;
  void(* x1)(OBJ_NAME const *, void *);
  void * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OBJ_NAME_do_all", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  x1 = (void(*)(OBJ_NAME const *, void *))_cffi_to_c_pointer(arg1, _cffi_type(3219));
  if (x1 == (void(*)(OBJ_NAME const *, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OBJ_NAME_do_all(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OBJ_NAME_do_all _cffi_d_OBJ_NAME_do_all
#endif

static void _cffi_d_OBJ_cleanup(void)
{
  OBJ_cleanup();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_cleanup(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OBJ_cleanup(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OBJ_cleanup _cffi_d_OBJ_cleanup
#endif

static int _cffi_d_OBJ_cmp(ASN1_OBJECT const * x0, ASN1_OBJECT const * x1)
{
  return OBJ_cmp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_cmp(PyObject *self, PyObject *args)
{
  ASN1_OBJECT const * x0;
  ASN1_OBJECT const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OBJ_cmp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(39), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(39), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(39), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(39), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_cmp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_cmp _cffi_d_OBJ_cmp
#endif

static int _cffi_d_OBJ_create(char const * x0, char const * x1, char const * x2)
{
  return OBJ_create(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_create(PyObject *self, PyObject *args)
{
  char const * x0;
  char const * x1;
  char const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OBJ_create", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_create(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_create _cffi_d_OBJ_create
#endif

static ASN1_OBJECT * _cffi_d_OBJ_dup(ASN1_OBJECT const * x0)
{
  return OBJ_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_dup(PyObject *self, PyObject *arg0)
{
  ASN1_OBJECT const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_OBJECT * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(39), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(39), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(657));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_dup _cffi_d_OBJ_dup
#endif

static int _cffi_d_OBJ_ln2nid(char const * x0)
{
  return OBJ_ln2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_ln2nid(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_ln2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_ln2nid _cffi_d_OBJ_ln2nid
#endif

static char const * _cffi_d_OBJ_nid2ln(int x0)
{
  return OBJ_nid2ln(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_nid2ln(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_nid2ln(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  return pyresult;
}
#else
#  define _cffi_f_OBJ_nid2ln _cffi_d_OBJ_nid2ln
#endif

static ASN1_OBJECT * _cffi_d_OBJ_nid2obj(int x0)
{
  return OBJ_nid2obj(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_nid2obj(PyObject *self, PyObject *arg0)
{
  int x0;
  ASN1_OBJECT * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_nid2obj(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(657));
  return pyresult;
}
#else
#  define _cffi_f_OBJ_nid2obj _cffi_d_OBJ_nid2obj
#endif

static char const * _cffi_d_OBJ_nid2sn(int x0)
{
  return OBJ_nid2sn(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_nid2sn(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_nid2sn(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  return pyresult;
}
#else
#  define _cffi_f_OBJ_nid2sn _cffi_d_OBJ_nid2sn
#endif

static int _cffi_d_OBJ_obj2nid(ASN1_OBJECT const * x0)
{
  return OBJ_obj2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_obj2nid(PyObject *self, PyObject *arg0)
{
  ASN1_OBJECT const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(39), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(39), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_obj2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_obj2nid _cffi_d_OBJ_obj2nid
#endif

static int _cffi_d_OBJ_obj2txt(char * x0, int x1, ASN1_OBJECT const * x2, int x3)
{
  return OBJ_obj2txt(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_obj2txt(PyObject *self, PyObject *args)
{
  char * x0;
  int x1;
  ASN1_OBJECT const * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "OBJ_obj2txt", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(451), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(451), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(39), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(39), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_obj2txt(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_obj2txt _cffi_d_OBJ_obj2txt
#endif

static int _cffi_d_OBJ_sn2nid(char const * x0)
{
  return OBJ_sn2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_sn2nid(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_sn2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_sn2nid _cffi_d_OBJ_sn2nid
#endif

static int _cffi_d_OBJ_txt2nid(char const * x0)
{
  return OBJ_txt2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_txt2nid(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_txt2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_txt2nid _cffi_d_OBJ_txt2nid
#endif

static ASN1_OBJECT * _cffi_d_OBJ_txt2obj(char const * x0, int x1)
{
  return OBJ_txt2obj(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_txt2obj(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_OBJECT * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OBJ_txt2obj", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_txt2obj(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(657));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_txt2obj _cffi_d_OBJ_txt2obj
#endif

static int _cffi_d_OCSP_BASICRESP_add_ext(OCSP_BASICRESP * x0, X509_EXTENSION * x1, int x2)
{
  return OCSP_BASICRESP_add_ext(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_add_ext(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  X509_EXTENSION * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_BASICRESP_add_ext", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_EXTENSION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_add_ext(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_add_ext _cffi_d_OCSP_BASICRESP_add_ext
#endif

static void _cffi_d_OCSP_BASICRESP_free(OCSP_BASICRESP * x0)
{
  OCSP_BASICRESP_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_free(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_BASICRESP_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_BASICRESP_free _cffi_d_OCSP_BASICRESP_free
#endif

static X509_EXTENSION * _cffi_d_OCSP_BASICRESP_get_ext(OCSP_BASICRESP * x0, int x1)
{
  return OCSP_BASICRESP_get_ext(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_get_ext(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_EXTENSION * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_BASICRESP_get_ext", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_get_ext(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_get_ext _cffi_d_OCSP_BASICRESP_get_ext
#endif

static int _cffi_d_OCSP_BASICRESP_get_ext_count(OCSP_BASICRESP * x0)
{
  return OCSP_BASICRESP_get_ext_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_get_ext_count(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_get_ext_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_get_ext_count _cffi_d_OCSP_BASICRESP_get_ext_count
#endif

static OCSP_BASICRESP * _cffi_d_OCSP_BASICRESP_new(void)
{
  return OCSP_BASICRESP_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_new(PyObject *self, PyObject *noarg)
{
  OCSP_BASICRESP * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(429));
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_new _cffi_d_OCSP_BASICRESP_new
#endif

static void _cffi_d_OCSP_CERTID_free(OCSP_CERTID * x0)
{
  OCSP_CERTID_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_CERTID_free(PyObject *self, PyObject *arg0)
{
  OCSP_CERTID * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(408), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(408), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_CERTID_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_CERTID_free _cffi_d_OCSP_CERTID_free
#endif

static X509_EXTENSION * _cffi_d_OCSP_ONEREQ_get_ext(OCSP_ONEREQ * x0, int x1)
{
  return OCSP_ONEREQ_get_ext(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_ONEREQ_get_ext(PyObject *self, PyObject *args)
{
  OCSP_ONEREQ * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_EXTENSION * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_ONEREQ_get_ext", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(401), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_ONEREQ *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(401), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_ONEREQ_get_ext(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_ONEREQ_get_ext _cffi_d_OCSP_ONEREQ_get_ext
#endif

static int _cffi_d_OCSP_ONEREQ_get_ext_count(OCSP_ONEREQ * x0)
{
  return OCSP_ONEREQ_get_ext_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_ONEREQ_get_ext_count(PyObject *self, PyObject *arg0)
{
  OCSP_ONEREQ * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(401), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_ONEREQ *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(401), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_ONEREQ_get_ext_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_ONEREQ_get_ext_count _cffi_d_OCSP_ONEREQ_get_ext_count
#endif

static int _cffi_d_OCSP_REQUEST_add_ext(OCSP_REQUEST * x0, X509_EXTENSION * x1, int x2)
{
  return OCSP_REQUEST_add_ext(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_add_ext(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  X509_EXTENSION * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_REQUEST_add_ext", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_EXTENSION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_add_ext(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_add_ext _cffi_d_OCSP_REQUEST_add_ext
#endif

static void _cffi_d_OCSP_REQUEST_free(OCSP_REQUEST * x0)
{
  OCSP_REQUEST_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_free(PyObject *self, PyObject *arg0)
{
  OCSP_REQUEST * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_REQUEST_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_REQUEST_free _cffi_d_OCSP_REQUEST_free
#endif

static X509_EXTENSION * _cffi_d_OCSP_REQUEST_get_ext(OCSP_REQUEST * x0, int x1)
{
  return OCSP_REQUEST_get_ext(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_get_ext(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_EXTENSION * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_REQUEST_get_ext", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_get_ext(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_get_ext _cffi_d_OCSP_REQUEST_get_ext
#endif

static int _cffi_d_OCSP_REQUEST_get_ext_count(OCSP_REQUEST * x0)
{
  return OCSP_REQUEST_get_ext_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_get_ext_count(PyObject *self, PyObject *arg0)
{
  OCSP_REQUEST * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_get_ext_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_get_ext_count _cffi_d_OCSP_REQUEST_get_ext_count
#endif

static OCSP_REQUEST * _cffi_d_OCSP_REQUEST_new(void)
{
  return OCSP_REQUEST_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_new(PyObject *self, PyObject *noarg)
{
  OCSP_REQUEST * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(407));
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_new _cffi_d_OCSP_REQUEST_new
#endif

static void _cffi_d_OCSP_RESPONSE_free(OCSP_RESPONSE * x0)
{
  OCSP_RESPONSE_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_RESPONSE_free(PyObject *self, PyObject *arg0)
{
  OCSP_RESPONSE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(391), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_RESPONSE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(391), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_RESPONSE_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_RESPONSE_free _cffi_d_OCSP_RESPONSE_free
#endif

static OCSP_CERTID const * _cffi_d_OCSP_SINGLERESP_get0_id(OCSP_SINGLERESP const * x0)
{
  return OCSP_SINGLERESP_get0_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_SINGLERESP_get0_id(PyObject *self, PyObject *arg0)
{
  OCSP_SINGLERESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_CERTID const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(404), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(404), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_SINGLERESP_get0_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3330));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_SINGLERESP_get0_id _cffi_d_OCSP_SINGLERESP_get0_id
#endif

static X509_EXTENSION * _cffi_d_OCSP_SINGLERESP_get_ext(OCSP_SINGLERESP * x0, int x1)
{
  return OCSP_SINGLERESP_get_ext(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_SINGLERESP_get_ext(PyObject *self, PyObject *args)
{
  OCSP_SINGLERESP * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_EXTENSION * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_SINGLERESP_get_ext", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(644), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(644), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_SINGLERESP_get_ext(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_SINGLERESP_get_ext _cffi_d_OCSP_SINGLERESP_get_ext
#endif

static int _cffi_d_OCSP_SINGLERESP_get_ext_count(OCSP_SINGLERESP * x0)
{
  return OCSP_SINGLERESP_get_ext_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_SINGLERESP_get_ext_count(PyObject *self, PyObject *arg0)
{
  OCSP_SINGLERESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(644), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(644), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_SINGLERESP_get_ext_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_SINGLERESP_get_ext_count _cffi_d_OCSP_SINGLERESP_get_ext_count
#endif

static int _cffi_d_OCSP_basic_add1_cert(OCSP_BASICRESP * x0, X509 * x1)
{
  return OCSP_basic_add1_cert(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_basic_add1_cert(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  X509 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_basic_add1_cert", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_basic_add1_cert(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_basic_add1_cert _cffi_d_OCSP_basic_add1_cert
#endif

static int _cffi_d_OCSP_basic_add1_nonce(OCSP_BASICRESP * x0, unsigned char * x1, int x2)
{
  return OCSP_basic_add1_nonce(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_basic_add1_nonce(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  unsigned char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_basic_add1_nonce", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_basic_add1_nonce(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_basic_add1_nonce _cffi_d_OCSP_basic_add1_nonce
#endif

static OCSP_SINGLERESP * _cffi_d_OCSP_basic_add1_status(OCSP_BASICRESP * x0, OCSP_CERTID * x1, int x2, int x3, ASN1_OCTET_STRING * x4, ASN1_OCTET_STRING * x5, ASN1_OCTET_STRING * x6)
{
  return OCSP_basic_add1_status(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_basic_add1_status(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  OCSP_CERTID * x1;
  int x2;
  int x3;
  ASN1_OCTET_STRING * x4;
  ASN1_OCTET_STRING * x5;
  ASN1_OCTET_STRING * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_SINGLERESP * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "OCSP_basic_add1_status", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(408), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(408), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_basic_add1_status(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(644));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_basic_add1_status _cffi_d_OCSP_basic_add1_status
#endif

static int _cffi_d_OCSP_basic_sign(OCSP_BASICRESP * x0, X509 * x1, EVP_PKEY * x2, EVP_MD const * x3, Cryptography_STACK_OF_X509 * x4, unsigned long x5)
{
  return OCSP_basic_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_basic_sign(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  X509 * x1;
  EVP_PKEY * x2;
  EVP_MD const * x3;
  Cryptography_STACK_OF_X509 * x4;
  unsigned long x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "OCSP_basic_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(235), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(235), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, unsigned long);
  if (x5 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_basic_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_basic_sign _cffi_d_OCSP_basic_sign
#endif

static OCSP_CERTID * _cffi_d_OCSP_cert_to_id(EVP_MD const * x0, X509 const * x1, X509 const * x2)
{
  return OCSP_cert_to_id(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_cert_to_id(PyObject *self, PyObject *args)
{
  EVP_MD const * x0;
  X509 const * x1;
  X509 const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_CERTID * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_cert_to_id", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(396), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(396), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(397), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(397), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(397), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (X509 const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(397), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_cert_to_id(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(408));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_cert_to_id _cffi_d_OCSP_cert_to_id
#endif

static int _cffi_d_OCSP_id_get0_info(ASN1_OCTET_STRING * * x0, ASN1_OBJECT * * x1, ASN1_OCTET_STRING * * x2, ASN1_INTEGER * * x3, OCSP_CERTID * x4)
{
  return OCSP_id_get0_info(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_id_get0_info(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * * x0;
  ASN1_OBJECT * * x1;
  ASN1_OCTET_STRING * * x2;
  ASN1_INTEGER * * x3;
  OCSP_CERTID * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "OCSP_id_get0_info", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(833), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(833), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(834), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(834), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(833), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(833), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(836), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(836), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(408), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(408), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_id_get0_info(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_id_get0_info _cffi_d_OCSP_id_get0_info
#endif

static OCSP_CERTID * _cffi_d_OCSP_onereq_get0_id(OCSP_ONEREQ * x0)
{
  return OCSP_onereq_get0_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_onereq_get0_id(PyObject *self, PyObject *arg0)
{
  OCSP_ONEREQ * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_CERTID * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(401), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_ONEREQ *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(401), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_onereq_get0_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(408));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_onereq_get0_id _cffi_d_OCSP_onereq_get0_id
#endif

static OCSP_ONEREQ * _cffi_d_OCSP_request_add0_id(OCSP_REQUEST * x0, OCSP_CERTID * x1)
{
  return OCSP_request_add0_id(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_request_add0_id(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  OCSP_CERTID * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_ONEREQ * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_request_add0_id", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(408), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(408), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_request_add0_id(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(401));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_request_add0_id _cffi_d_OCSP_request_add0_id
#endif

static int _cffi_d_OCSP_request_add1_nonce(OCSP_REQUEST * x0, unsigned char * x1, int x2)
{
  return OCSP_request_add1_nonce(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_request_add1_nonce(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  unsigned char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_request_add1_nonce", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(813), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(813), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_request_add1_nonce(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_request_add1_nonce _cffi_d_OCSP_request_add1_nonce
#endif

static int _cffi_d_OCSP_request_onereq_count(OCSP_REQUEST * x0)
{
  return OCSP_request_onereq_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_request_onereq_count(PyObject *self, PyObject *arg0)
{
  OCSP_REQUEST * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_request_onereq_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_request_onereq_count _cffi_d_OCSP_request_onereq_count
#endif

static OCSP_ONEREQ * _cffi_d_OCSP_request_onereq_get0(OCSP_REQUEST * x0, int x1)
{
  return OCSP_request_onereq_get0(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_request_onereq_get0(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_ONEREQ * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_request_onereq_get0", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(407), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(407), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_request_onereq_get0(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(401));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_request_onereq_get0 _cffi_d_OCSP_request_onereq_get0
#endif

static int _cffi_d_OCSP_resp_count(OCSP_BASICRESP * x0)
{
  return OCSP_resp_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_count(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_count _cffi_d_OCSP_resp_count
#endif

static OCSP_SINGLERESP * _cffi_d_OCSP_resp_get0(OCSP_BASICRESP * x0, int x1)
{
  return OCSP_resp_get0(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_SINGLERESP * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_resp_get0", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(644));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0 _cffi_d_OCSP_resp_get0
#endif

static Cryptography_STACK_OF_X509 const * _cffi_d_OCSP_resp_get0_certs(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_certs(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_certs(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  Cryptography_STACK_OF_X509 const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_certs(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3293));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_certs _cffi_d_OCSP_resp_get0_certs
#endif

static int _cffi_d_OCSP_resp_get0_id(OCSP_BASICRESP const * x0, ASN1_OCTET_STRING const * * x1, X509_NAME const * * x2)
{
  return OCSP_resp_get0_id(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_id(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP const * x0;
  ASN1_OCTET_STRING const * * x1;
  X509_NAME const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_resp_get0_id", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1650), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1650), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1651), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (X509_NAME const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1651), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_id(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_id _cffi_d_OCSP_resp_get0_id
#endif

static ASN1_GENERALIZEDTIME const * _cffi_d_OCSP_resp_get0_produced_at(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_produced_at(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_produced_at(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_GENERALIZEDTIME const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_produced_at(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3253));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_produced_at _cffi_d_OCSP_resp_get0_produced_at
#endif

static OCSP_RESPDATA const * _cffi_d_OCSP_resp_get0_respdata(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_respdata(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_respdata(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_RESPDATA const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_respdata(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3334));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_respdata _cffi_d_OCSP_resp_get0_respdata
#endif

static ASN1_OCTET_STRING const * _cffi_d_OCSP_resp_get0_signature(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_signature(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_signature(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_OCTET_STRING const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_signature(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3262));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_signature _cffi_d_OCSP_resp_get0_signature
#endif

static X509_ALGOR const * _cffi_d_OCSP_resp_get0_tbs_sigalg(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_tbs_sigalg(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_tbs_sigalg(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_ALGOR const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_tbs_sigalg(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3372));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_tbs_sigalg _cffi_d_OCSP_resp_get0_tbs_sigalg
#endif

static OCSP_RESPONSE * _cffi_d_OCSP_response_create(int x0, OCSP_BASICRESP * x1)
{
  return OCSP_response_create(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_response_create(PyObject *self, PyObject *args)
{
  int x0;
  OCSP_BASICRESP * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_RESPONSE * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_response_create", 2, 2, &arg0, &arg1))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(429), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(429), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_response_create(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(391));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_response_create _cffi_d_OCSP_response_create
#endif

static OCSP_BASICRESP * _cffi_d_OCSP_response_get1_basic(OCSP_RESPONSE * x0)
{
  return OCSP_response_get1_basic(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_response_get1_basic(PyObject *self, PyObject *arg0)
{
  OCSP_RESPONSE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_BASICRESP * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(391), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_RESPONSE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(391), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_response_get1_basic(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(429));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_response_get1_basic _cffi_d_OCSP_response_get1_basic
#endif

static int _cffi_d_OCSP_response_status(OCSP_RESPONSE * x0)
{
  return OCSP_response_status(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_response_status(PyObject *self, PyObject *arg0)
{
  OCSP_RESPONSE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(391), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_RESPONSE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(391), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_response_status(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_response_status _cffi_d_OCSP_response_status
#endif

static int _cffi_d_OCSP_single_get0_status(OCSP_SINGLERESP * x0, int * x1, ASN1_GENERALIZEDTIME * * x2, ASN1_GENERALIZEDTIME * * x3, ASN1_GENERALIZEDTIME * * x4)
{
  return OCSP_single_get0_status(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_single_get0_status(PyObject *self, PyObject *args)
{
  OCSP_SINGLERESP * x0;
  int * x1;
  ASN1_GENERALIZEDTIME * * x2;
  ASN1_GENERALIZEDTIME * * x3;
  ASN1_GENERALIZEDTIME * * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "OCSP_single_get0_status", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(644), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(644), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1251), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1251), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_single_get0_status(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_single_get0_status _cffi_d_OCSP_single_get0_status
#endif

static void _cffi_d_OPENSSL_cleanup(void)
{
  OPENSSL_cleanup();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_cleanup(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_cleanup(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_cleanup _cffi_d_OPENSSL_cleanup
#endif

static void _cffi_d_OPENSSL_config(char const * x0)
{
  OPENSSL_config(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_config(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(52), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(52), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_config(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_config _cffi_d_OPENSSL_config
#endif

static void _cffi_d_OPENSSL_free(void * x0)
{
  OPENSSL_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_free(PyObject *self, PyObject *arg0)
{
  void * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_free _cffi_d_OPENSSL_free
#endif

static void * _cffi_d_OPENSSL_malloc(size_t x0)
{
  return OPENSSL_malloc(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_malloc(PyObject *self, PyObject *arg0)
{
  size_t x0;
  void * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, size_t);
  if (x0 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OPENSSL_malloc(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(229));
  return pyresult;
}
#else
#  define _cffi_f_OPENSSL_malloc _cffi_d_OPENSSL_malloc
#endif

static void _cffi_d_OPENSSL_no_config(void)
{
  OPENSSL_no_config();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_no_config(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_no_config(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_no_config _cffi_d_OPENSSL_no_config
#endif

static void _cffi_d_OTHERNAME_free(OTHERNAME * x0)
{
  OTHERNAME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OTHERNAME_free(PyObject *self, PyObject *arg0)
{
  OTHERNAME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2936), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OTHERNAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2936), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OTHERNAME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OTHERNAME_free _cffi_d_OTHERNAME_free
#endif

static OTHERNAME * _cffi_d_OTHERNAME_new(void)
{
  return OTHERNAME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OTHERNAME_new(PyObject *self, PyObject *noarg)
{
  OTHERNAME * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OTHERNAME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2936));
  return pyresult;
}
#else
#  define _cffi_f_OTHERNAME_new _cffi_d_OTHERNAME_new
#endif

static void _cffi_d_OpenSSL_add_all_algorithms(void)
{
  OpenSSL_add_all_algorithms();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_add_all_algorithms(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OpenSSL_add_all_algorithms(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OpenSSL_add_all_algorithms _cffi_d_OpenSSL_add_all_algorithms
#endif

static char const * _cffi_d_OpenSSL_version(int x0)
{
  return OpenSSL_version(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_version(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OpenSSL_version(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(52));
  return pyresult;
}
#else
#  define _cffi_f_OpenSSL_version _cffi_d_OpenSSL_version
#endif

static unsigned long _cffi_d_OpenSSL_version_num(void)
{
  return OpenSSL_version_num();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_version_num(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OpenSSL_version_num(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_OpenSSL_version_num _cffi_d_OpenSSL_version_num
#endif

static DH * _cffi_d_PEM_read_bio_DHparams(BIO * x0, DH * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_DHparams(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_DHparams(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_DHparams", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(223), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(223), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(228));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_DHparams(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(232));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_DHparams _cffi_d_PEM_read_bio_DHparams
#endif

static PKCS7 * _cffi_d_PEM_read_bio_PKCS7(BIO * x0, PKCS7 * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PKCS7(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PKCS7(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PKCS7", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(468), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(468), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(228));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PKCS7(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(136));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PKCS7 _cffi_d_PEM_read_bio_PKCS7
#endif

static EVP_PKEY * _cffi_d_PEM_read_bio_PUBKEY(BIO * x0, EVP_PKEY * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PUBKEY(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PUBKEY(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PUBKEY", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(321), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(321), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(228));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PUBKEY(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(235));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PUBKEY _cffi_d_PEM_read_bio_PUBKEY
#endif

static EVP_PKEY * _cffi_d_PEM_read_bio_PrivateKey(BIO * x0, EVP_PKEY * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PrivateKey(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PrivateKey", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(321), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(321), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(228));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PrivateKey(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(235));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PrivateKey _cffi_d_PEM_read_bio_PrivateKey
#endif

static RSA * _cffi_d_PEM_read_bio_RSAPublicKey(BIO * x0, RSA * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_RSAPublicKey(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_RSAPublicKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_RSAPublicKey", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(505), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(505), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(228));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_RSAPublicKey(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(517));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_RSAPublicKey _cffi_d_PEM_read_bio_RSAPublicKey
#endif

static X509 * _cffi_d_PEM_read_bio_X509(BIO * x0, X509 * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509 * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(565), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(565), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(228));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_X509(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(28));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_X509 _cffi_d_PEM_read_bio_X509
#endif

static X509_CRL * _cffi_d_PEM_read_bio_X509_CRL(BIO * x0, X509_CRL * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509_CRL(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509_CRL(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_CRL * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_CRL * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509_CRL", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(137), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(137), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(602), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_CRL * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(602), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(228));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(229), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(229), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }