"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecurityClient = void 0;

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SecurityClient {
  constructor(esClient) {
    this.esClient = esClient;
  }

  async authenticate(request, credentials) {
    const authHeader = Buffer.from(`${credentials.username}:${credentials.password}`).toString('base64');

    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: {
          authorization: `Basic ${authHeader}`
        }
      });
      return {
        username: credentials.username,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant,
        credentials,
        proxyCredentials: credentials
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authenticateWithHeader(request, headerName, headerValue, whitelistedHeadersAndValues = {}, additionalAuthHeaders = {}) {
    try {
      const credentials = {
        headerName,
        headerValue
      };
      const headers = {};

      if (headerValue) {
        headers[headerName] = headerValue;
      } // cannot get config elasticsearch.requestHeadersWhitelist from kibana.yml file in new platfrom
      // meanwhile, do we really need to save all headers in cookie?


      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.teanats,
        selectedTenant: esResponse.user_requested_tenant,
        credentials
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authenticateWithHeaders(request, additionalAuthHeaders = {}) {
    try {
      const esResponse = await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers: additionalAuthHeaders
      });
      return {
        username: esResponse.user_name,
        roles: esResponse.roles,
        backendRoles: esResponse.backend_roles,
        tenants: esResponse.tenants,
        selectedTenant: esResponse.user_requested_tenant
      };
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async authinfo(request, headers = {}) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo', {
        headers
      });
    } catch (error) {
      throw new Error(error.message);
    }
  } // Multi-tenancy APIs


  async getMultitenancyInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.multitenancyinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getTenantInfoWithInternalUser() {
    try {
      return this.esClient.callAsInternalUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getTenantInfo(request) {
    try {
      return await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.tenantinfo');
    } catch (error) {
      throw new Error(error.message);
    }
  }

  async getSamlHeader(request) {
    try {
      // response is expected to be an error
      await this.esClient.asScoped(request).callAsCurrentUser('opensearch_security.authinfo');
    } catch (error) {
      // the error looks like
      // wwwAuthenticateDirective:
      //   '
      //     X-Security-IdP realm="Open Distro Security"
      //     location="https://<your-auth-domain.com>/api/saml2/v1/sso?SAMLRequest=<some-encoded-string>"
      //     requestId="<request_id>"
      //   '
      if (!error.wwwAuthenticateDirective) {
        throw error;
      }

      try {
        const locationRegExp = /location="(.*?)"/;
        const requestIdRegExp = /requestId="(.*?)"/;
        const locationExecArray = locationRegExp.exec(error.wwwAuthenticateDirective);
        const requestExecArray = requestIdRegExp.exec(error.wwwAuthenticateDirective);

        if (locationExecArray && requestExecArray) {
          return {
            location: locationExecArray[1],
            requestId: requestExecArray[1]
          };
        }

        throw Error('failed parsing SAML config');
      } catch (parsingError) {
        console.log(parsingError);
        throw new Error(parsingError);
      }
    }

    throw new Error(`Invalid SAML configuration.`);
  }

  async authToken(requestId, samlResponse, acsEndpoint = undefined) {
    const body = {
      RequestId: requestId,
      SAMLResponse: samlResponse,
      acsEndpoint
    };

    try {
      return await this.esClient.asScoped().callAsCurrentUser('opensearch_security.authtoken', {
        body
      });
    } catch (error) {
      console.log(error);
      throw new Error('failed to get token');
    }
  }

}

exports.SecurityClient = SecurityClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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