"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToCSV = exports.buildRequestBody = void 0;

var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));

var _json2Csv = _interopRequireDefault(require("json-2-csv"));

var _lodash = _interopRequireDefault(require("lodash"));

var _moment = _interopRequireDefault(require("moment"));

var _common = require("../../../../../src/plugins/data/common");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
var metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
}; // Get the selected columns by the user.

exports.metaData = metaData;

const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;

  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }

  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
}; // Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query


exports.getSelectedFields = getSelectedFields;

const buildRequestBody = (report, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();

  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter); // Add time range

  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }

  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  } // Add sorting to the query


  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);

  if (report._source.sorting.length > 0) {
    const sortings = report._source.sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });

    esbSearchQuery.sorts(sortings);
  } // add selected fields to query


  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  } // Add a customizer to merge queries to generate request body


  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });

  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
}; // Fetch the data from OpenSearch


exports.buildRequestBody = buildRequestBody;

const getOpenSearchData = (arrayHits, report, params, dateFormat) => {
  let hits = [];

  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields; // get all the fields of type date and format them to excel format

      for (let dateField of report._source.dateFields) {
        const dateValue = data._source[dateField];

        if (dateValue && dateValue.length !== 0) {
          if (dateValue instanceof Array) {
            // loop through array
            dateValue.forEach((element, index) => {
              data._source[dateField][index] = (0, _moment.default)(fields[dateField][index]).format(dateFormat);
            });
          } else {
            // The fields response always returns an array of values for each field
            // https://www.elastic.co/guide/en/elasticsearch/reference/master/search-fields.html#search-fields-response
            data._source[dateField] = (0, _moment.default)(fields[dateField][0]).format(dateFormat);
          }
        }
      }

      delete data['fields'];

      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      } // Truncate to expected limit size


      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }

  return hits;
}; //Convert the data to Csv format


exports.getOpenSearchData = getOpenSearchData;

const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};

exports.convertToCSV = convertToCSV;

function flattenHits(hits, result = {}, prefix = '') {
  for (const [key, value] of Object.entries(hits)) {
    if (!hits.hasOwnProperty(key)) continue;

    if (value != null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, prefix + key + '.');
    } else {
      result[prefix.replace(/^_source\./, '') + key] = value;
    }
  }

  return result;
} //Return only the selected fields


function traverse(data, keys, result = {}) {
  data = flattenHits(data);
  const sourceKeys = Object.keys(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);

    if (value !== undefined) result[key] = value;else {
      Object.keys(data).filter(sourceKey => sourceKey.startsWith(key + '.')).forEach(sourceKey => result[sourceKey] = data[sourceKey]);
    }
  });
  return result;
}
/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */


function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;

    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }

  return doc;
}

const addDocValueFields = (report, requestBody) => {
  const docValues = [];

  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  } // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.


  requestBody = { ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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