"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAppAnalyticsRouter = registerAppAnalyticsRouter;

var _configSchema = require("@osd/config-schema");

var _application_analytics = require("../../../common/constants/application_analytics");

var _app_analytics_adaptor = require("../../../server/adaptors/application_analytics/app_analytics_adaptor");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/* eslint-disable no-console */
function registerAppAnalyticsRouter(router) {
  const appAnalyticsBackend = new _app_analytics_adaptor.AppAnalyticsAdaptor(); // Fetches all existing applications

  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);
    let applicationsData = [];

    try {
      applicationsData = await appAnalyticsBackend.fetchApps(opensearchClient);
      return response.ok({
        body: {
          data: applicationsData
        }
      });
    } catch (err) {
      console.error('Error occurred while fetching applications', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Fetch application by id

  router.get({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appId}`,
    validate: {
      params: _configSchema.schema.object({
        appId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const appObject = await appAnalyticsBackend.fetchAppById(opensearchClient, request.params.appId);
      return response.ok({
        body: appObject
      });
    } catch (err) {
      console.error('Error occurred while fetching application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Create a new application

  router.post({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.maybe(_configSchema.schema.string()),
        baseQuery: _configSchema.schema.string(),
        servicesEntities: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        traceGroups: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newAppId = await appAnalyticsBackend.createNewApp(opensearchClient, request.body);
      return response.ok({
        body: {
          message: 'Application Created',
          newAppId
        }
      });
    } catch (err) {
      console.error('Error occurred while creating a new application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Renames an existing application

  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/rename`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      await appAnalyticsBackend.renameApp(opensearchClient, request.body.appId, request.body.name);
      return response.ok({
        body: {
          message: 'Application Renamed'
        }
      });
    } catch (err) {
      console.error('Error occurred while renaming an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Updates an existing application

  router.put({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/`,
    validate: {
      body: _configSchema.schema.object({
        appId: _configSchema.schema.string(),
        updateBody: _configSchema.schema.object({
          name: _configSchema.schema.maybe(_configSchema.schema.string()),
          description: _configSchema.schema.maybe(_configSchema.schema.string()),
          servicesEntities: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          traceGroups: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          panelId: _configSchema.schema.maybe(_configSchema.schema.string()),
          availabilityVisId: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const updatedAppId = await appAnalyticsBackend.updateApp(opensearchClient, request.body.appId, request.body.updateBody);
      return response.ok({
        body: {
          message: 'Application Updated',
          updatedAppId
        }
      });
    } catch (err) {
      console.error('Error occurred while updating an existing application', err);
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  }); // Delete applications

  router.delete({
    path: `${_application_analytics.APP_ANALYTICS_API_PREFIX}/{appList}`,
    validate: {
      params: _configSchema.schema.object({
        appList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const delResponse = await appAnalyticsBackend.deleteApp(opensearchClient, request.params.appList);
      return response.ok({
        body: delResponse
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode || 500,
        body: err.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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