const installedApp = {};

const registerExtension = function(i, e, d) {
  installedApp.init = i;
  installedApp.enable = e;
  installedApp.disable = d;
};

function init() {
  if (installedApp.init) {
    installedApp.init();
  }
}
function enable() {
  if (installedApp.enable) {
    installedApp.enable();
  }
}

function disable() {
  if (installedApp.disable) {
    installedApp.disable();
  }
}
 /* Polyfill from Mozilla. */

  // https://tc39.github.io/ecma262/#sec-array.prototype.find
if (!Array.prototype.find && Object.defineProperty) {
  Object.defineProperty(Array.prototype, 'find', {
    value: function(predicate) {
     // 1. Let O be ? ToObject(this value).
      if (this == null) {
        throw new TypeError('"this" is null or not defined');
      }

      var o = Object(this);

      // 2. Let len be ? ToLength(? Get(O, "length")).
      var len = o.length >>> 0;

      // 3. If IsCallable(predicate) is false, throw a TypeError exception.
      if (typeof predicate !== 'function') {
        throw new TypeError('predicate must be a function');
      }

      // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
      var thisArg = arguments[1];

      // 5. Let k be 0.
      var k = 0;

      // 6. Repeat, while k < len
      while (k < len) {
        // a. Let Pk be ! ToString(k).
        // b. Let kValue be ? Get(O, Pk).
        // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
        // d. If testResult is true, return kValue.
        var kValue = o[k];
        if (predicate.call(thisArg, kValue, k, o)) {
          return kValue;
        }
        // e. Increase k by 1.
        k++;
      }

      // 7. Return undefined.
      return undefined;
    }
  });
}

if (!Array.prototype.findIndex && Object.defineProperty) {
  Object.defineProperty(Array.prototype, 'findIndex', {
    value: function(predicate) {
     // 1. Let O be ? ToObject(this value).
      if (this == null) {
        throw new TypeError('"this" is null or not defined');
      }

      var o = Object(this);

      // 2. Let len be ? ToLength(? Get(O, "length")).
      var len = o.length >>> 0;

      // 3. If IsCallable(predicate) is false, throw a TypeError exception.
      if (typeof predicate !== 'function') {
        throw new TypeError('predicate must be a function');
      }

      // 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
      var thisArg = arguments[1];

      // 5. Let k be 0.
      var k = 0;

      // 6. Repeat, while k < len
      while (k < len) {
        // a. Let Pk be ! ToString(k).
        // b. Let kValue be ? Get(O, Pk).
        // c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
        // d. If testResult is true, return k.
        var kValue = o[k];
        if (predicate.call(thisArg, kValue, k, o)) {
          return k;
        }
        // e. Increase k by 1.
        k++;
      }

      // 7. Return -1.
      return -1;
    }
  });
}
/* Logging
 * Written by Sergey
*/
let debug = false;
/**
 * If called with a false argument, log statements are suppressed.
 */
function setLoggingEnabled(enabled) {
    debug = enabled;
}
/**
 * Log logs the given message using the gnome shell logger (global.log) if the
 * debug variable is set to true.
 *
 * Debug messages may be viewed using the bash command `journalctl
 * /usr/bin/gnome-shell` and grepping the results for 'gTile'.
 */
function log(message) {
    if (debug) {
        global.log("gTile " + message);
    }
}

/* Determine if gnome-shell version newer than required
 * Written by Sergey
*/
function getConfig() {
    return imports.misc.config;
}
const VERSION_34 = { major: 3, minor: 34 };
const VERSION_36 = { major: 3, minor: 36 };
/**
 * ShellVersion is used to parse the version string
 */
class ShellVersion {
    constructor(version) {
        const parts = version.split('.').map((part) => Number(part));
        if (parts.length < 2) {
            throw new Error(`invalid version supplied: ${version}`);
        }
        this.major = parts[0];
        this.minor = parts[1];
        // Tolerate "40.alpha.1" for example. See https://github.com/gTile/gTile/issues/187.
        if (isNaN(this.minor)) {
            this.minor = 0;
        }
        if (isNaN(this.major)) {
            throw new Error(`invalid version supplied: ${JSON.stringify(version)}; got major = ${this.major}, minor = ${this.minor}`);
        }
        this.rawVersion = version;
    }
    static defaultVersion() {
        return ShellVersion.parse(getConfig().PACKAGE_VERSION);
    }
    static parse(version) {
        return new ShellVersion(version);
    }
    version_at_least_34() {
        return versionGreaterThanOrEqualTo(this, VERSION_34);
    }
    version_at_least_36() {
        return versionGreaterThanOrEqualTo(this, VERSION_36);
    }
    print_version() {
        log("Init gnome-shell version " + this.rawVersion + " major " + this.major + " minor " + this.minor);
    }
}
/**
 * Returns true if a is >= b.
 */
function versionGreaterThanOrEqualTo(a, b) {
    return a.major > b.major || (a.major === b.major && a.minor >= b.minor);
}

// Library imports
const Main = imports.ui.main;
const Meta = imports.gi.Meta;
const Shell = imports.gi.Shell;
// Extension imports
const Me = imports.misc.extensionUtils.getCurrentExtension();
const Settings = Me.imports.settings;
function bind(keyBindings) {
    log("Binding keys");
    let mySettings = Settings.get();
    keyBindings.forEach((callback, key) => {
        //const key = keyString as KeyBindingSettingName;
        if (Main.wm.addKeybinding && Shell.ActionMode) { // introduced in 3.16
            Main.wm.addKeybinding(key, mySettings, Meta.KeyBindingFlags.NONE, Shell.ActionMode.NORMAL, callback);
        }
        else if (Main.wm.addKeybinding && Shell.KeyBindingMode) { // introduced in 3.7.5
            Main.wm.addKeybinding(key, mySettings, Meta.KeyBindingFlags.NONE, Shell.KeyBindingMode.NORMAL | Shell.KeyBindingMode.MESSAGE_TRAY, callback);
        }
        else {
            global.display.add_keybinding(key, mySettings, Meta.KeyBindingFlags.NONE, callback);
        }
    });
}
function unbind(keyBindings) {
    log("Unbinding keys");
    for (let key of keyBindings.keys()) {
        if (Main.wm.removeKeybinding) { // introduced in 3.7.2
            Main.wm.removeKeybinding(key);
        }
        else {
            global.display.remove_keybinding(key);
        }
    }
}

/* Edited by Sergey after
https://github.com/tpyl/gssnaptoneighbors
 by Timo Pylvanainen <tpyl@iki.fi>
 */
const Meta$1 = imports.gi.Meta;
const WorkspaceManager = global.screen || global.workspace_manager;
const OVERLAP_TOLERANCE = 5;
const SCAN_BOX_SIZE = 50;
/**
 * Return all windows on the currently active workspace
 */
function getWindowsOnActiveWorkspace() {
    let windows = [];
    let windowActors = global.get_window_actors();
    let curWorkSpace = WorkspaceManager.get_active_workspace();
    for (let i = 0; i < windowActors.length; i++) {
        let win = windowActors[i].meta_window;
        if (win.located_on_workspace(curWorkSpace) &&
            !win.minimized &&
            win.get_frame_type() == 0) {
            windows.push(win);
        }
    }
    return windows;
}
/**
 * Find the maximum horzontal expansion from x and
 * returns minx, maxx. The initial maximum x is given
 * as argument, and the expansion is never larger than
 * that.
 *
 * The upper and lower limits define the y coordinate
 * range to check for overlapping windows.
 */
function expandHorizontally(x, upper, lower, minx, maxx, windows) {
    for (let i = 0; i < windows.length; i++) {
        let rect = windows[i].get_frame_rect();
        let wt = rect.y;
        let wb = rect.y + rect.height;
        let wl = rect.x;
        let wr = rect.x + rect.width;
        // Check only  if the window overlaps vertically
        if (wb > upper && wt < lower) {
            if (wr < x && minx < wr) {
                minx = wr;
            }
            if (wl > x && wl < maxx) {
                maxx = wl;
            }
        }
    }
    return { min: minx, max: maxx };
}
/**
 * Find the maximum vertical expansion from  y, and
 * returns miny, maxy. The initial maximum y is given
 * as argument, and the expansion is never larger than
 * that.
 *
 * The left and right limits define the x coordinate
 * range to check for overlapping windows.
 */
function expandVertically(y, left, right, miny, maxy, windows) {
    for (let i = 0; i < windows.length; i++) {
        let rect = windows[i].get_frame_rect();
        let wt = rect.y;
        let wb = rect.y + rect.height;
        let wl = rect.x;
        let wr = rect.x + rect.width;
        // Check only  if the window overlaps horizontally
        if (wr > left && wl < right) {
            if (wb < y && miny < wb) {
                miny = wb;
            }
            if (wt > y && wt < maxy) {
                maxy = wt;
            }
        }
    }
    return { min: miny, max: maxy };
}
/**
 * Resize & move the *window* so it touches adjacent windows or
 * screen edge top, bottom, left and right. The top-left corner
 * of the window defines the expansion point.
 *
 * There is an L-ambiguity where the window could be expanded
 * both vertically and horizontally. The expnasion that results
 * in closer to 1 aspect ratio is selected.
 */
function snapToNeighbors(window) {
    log("snapToNeighbors " + window.get_title());
    // Unmaximize first
    if (window.maximized_horizontally || window.maximized_vertically)
        window.unmaximize(Meta$1.MaximizeFlags.HORIZONTAL | Meta$1.MaximizeFlags.VERTICAL);
    let workArea = window.get_work_area_current_monitor();
    let myrect = window.get_frame_rect();
    let windows = getWindowsOnActiveWorkspace();
    // Scan for overlapping windows in a thin bar around the top of the 
    // window. The vertical height of the window will be adjusted later. 
    let maxHorizw = expandHorizontally(myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2) + SCAN_BOX_SIZE, workArea.x, workArea.x + workArea.width, windows);
    let maxHorizh = expandVertically(myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), maxHorizw.min + OVERLAP_TOLERANCE, maxHorizw.max - OVERLAP_TOLERANCE, workArea.y, workArea.y + workArea.height, windows);
    let maxVerth = expandVertically(myrect.y + Math.min(SCAN_BOX_SIZE, myrect.height / 2), myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2) + SCAN_BOX_SIZE, workArea.y, workArea.y + workArea.height, windows);
    let maxVertw = expandHorizontally(myrect.x + Math.min(SCAN_BOX_SIZE, myrect.width / 2), maxVerth.min + OVERLAP_TOLERANCE, maxVerth.max - OVERLAP_TOLERANCE, workArea.x, workArea.x + workArea.width, windows);
    if ((maxHorizw.max - maxHorizw.min) * (maxHorizh.max - maxHorizh.min) >
        (maxVertw.max - maxVertw.min) * (maxVerth.max - maxVerth.min)) {
        window.move_resize_frame(true, maxHorizw.min, maxHorizh.min, maxHorizw.max - maxHorizw.min, maxHorizh.max - maxHorizh.min);
    }
    else {
        window.move_resize_frame(true, maxVertw.min, maxVerth.min, maxVertw.max - maxVertw.min, maxVerth.max - maxVerth.min);
    }
}

/**
 * TileSpec represents a rectangular area on display by means of specifying a
 * number of evenly spaced tiles and two corners.
 */
class TileSpec {
    constructor(gridWidth, gridHeight, luc, rdc) {
        this.gridWidth = gridWidth;
        this.gridHeight = gridHeight;
        this.luc = luc;
        this.rdc = rdc;
    }
    toString() {
        return ` ${[this.gridWidth, this.gridHeight].join('x')} ${this.luc.toString()} ${this.rdc.toString()}`;
    }
    toFrameRect(workArea) {
        const elemSize = new Size(workArea.size.width / this.gridWidth, workArea.size.height / this.gridHeight);
        let left;
        let top;
        let right;
        let bottom;
        if (this.luc.types.x == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridWidth, this.luc.xy.x);
            left = Math.round(workArea.origin.x + (positiveTileNumber - 1) * elemSize.width);
        }
        else if (this.luc.types.x == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridWidth * this.luc.xy.x);
            left = Math.round(workArea.origin.x + snappedToGrid * elemSize.width);
        }
        else {
            left = Math.round(workArea.origin.x + workArea.size.width * this.luc.xy.x);
        }
        if (this.luc.types.y == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridHeight, this.luc.xy.y);
            top = Math.round(workArea.origin.y + (positiveTileNumber - 1) * elemSize.height);
        }
        else if (this.luc.types.y == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridHeight * this.luc.xy.y);
            top = Math.round(workArea.origin.y + snappedToGrid * elemSize.height);
        }
        else {
            top = Math.round(workArea.origin.y + workArea.size.height * this.luc.xy.y);
        }
        if (this.rdc.types.x == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridWidth, this.rdc.xy.x);
            right = Math.round(workArea.origin.x + positiveTileNumber * elemSize.width);
        }
        else if (this.rdc.types.x == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridWidth * this.rdc.xy.x);
            right = Math.round(workArea.origin.x + snappedToGrid * elemSize.width);
        }
        else {
            right = Math.round(workArea.origin.x + workArea.size.width * this.rdc.xy.x);
        }
        if (this.rdc.types.y == 'tile') {
            const positiveTileNumber = this._convertNegativeCoord(this.gridHeight, this.rdc.xy.y);
            bottom = Math.round(workArea.origin.y + positiveTileNumber * elemSize.height);
        }
        else if (this.rdc.types.y == 'approx_percentage') {
            const snappedToGrid = Math.round(this.gridHeight * this.rdc.xy.y);
            bottom = Math.round(workArea.origin.y + snappedToGrid * elemSize.height);
        }
        else {
            bottom = Math.round(workArea.origin.y + workArea.size.height * this.rdc.xy.y);
        }
        return new Rect(new XY(left, top), new Size(right - left - 1, bottom - top - 1));
    }
    get gridSize() {
        return new GridSize(this.gridWidth, this.gridHeight);
    }
    isFullscreen(workArea) {
        return this.toFrameRect(workArea).equal(workArea, 1);
    }
    /**
     * Converts negative coordinates (e.g. -1:-1) to a positive format on a specified grid.
     * If x or y is a positive number, it is ignored.
     * E.g. -1:-1 on a 3:3 grid is a 3:3, as well as -1:3.
     */
    _convertNegativeCoord(gridEdges, coord) {
        if (coord < 0) {
            return gridEdges + coord + 1;
        }
        else {
            return coord;
        }
    }
}
const MAX_TUPLE_MEMBER_VALUE = Number.MAX_SAFE_INTEGER;
/**
 * Tuple Holder represents a single starting or ending point (x and y coordinates),
 * as well as the type of the coordinate - "tile", "approx_percentage" or "percentage" now.
 *
 * E.g. ~0.75:0.75 is {X:0.75,Y:0.75}, types - 'percentage' & 'percentage'
 * approximate - true.
 */
class TupleHolder {
    constructor(raw) {
        this.raw = raw;
        const gssk = this.raw.split(':');
        this._validateTuple(gssk);
        this.xy = this._parseTuple(gssk);
        this.types = this._parseTypes(gssk);
    }
    toString() {
        return this.raw;
    }
    _parseTuple(tuple) {
        const x = this._parseCoordinate(tuple[0]);
        const y = this._parseCoordinate(tuple[1]);
        return new XY(x, y);
    }
    _parseTypes(tuple) {
        const typeX = this._parseType(tuple[0]);
        const typeY = this._parseType(tuple[1]);
        return new CoordinateTypesHolder(typeX, typeY);
    }
    _parseCoordinate(coord) {
        return Number(coord.replace('~', ''));
    }
    _parseType(coord) {
        if (coord.includes('~')) {
            return 'approx_percentage';
        }
        else if (coord.includes('.')) {
            return 'percentage';
        }
        else {
            return 'tile';
        }
    }
    _validateTuple(gssk) {
        if (gssk.length !== 2) {
            throw new Error(`Failed to split ${this.raw} into two numbers`);
        }
        this._validateCoordinate(gssk[0]);
        this._validateCoordinate(gssk[1]);
    }
    /**
     * Allowed values:
     * 1.0 (exact match)
     * Any float from 0.0 till 0.999..., with or without preceding approx indicator (~)
     * Any positive or negative integer, except 0
     */
    _validateCoordinate(coord) {
        const testRegex = /(~?0\.[0-9]+|1\.0|-?[1-9]+[0-9]*)/;
        if (!testRegex.test(coord)) {
            throw new Error(`Failed to parse ${coord} in tuple ${this.raw}`);
        }
    }
}
/**
 * Holds coordinate types for the tuple.
 * Currently 3 types are supported - tile, approx_percentage and percentage.
 */
class CoordinateTypesHolder {
    constructor(x, y) {
        this.x = x;
        this.y = y;
    }
}
class GridSize {
    constructor(
    // Number of columns.
    width, 
    // Number of rows.
    height) {
        this.width = width;
        this.height = height;
    }
    toString() {
        return `${this.width}x${this.height}`;
    }
    equals(other) {
        return this.width === other.width && this.height == other.height;
    }
}
function parseGridSizesIgnoringErrors(s) {
    return s.split(',').flatMap((part) => {
        const size = parseGridSizeIgnoringErrors(part.trim());
        return size ? [size] : [];
    });
}
function parseGridSizeIgnoringErrors(s) {
    const parts = s.split("x").map(Number);
    if (parts.length !== 2 || !parts.every(x => !isNaN(x))) {
        return null;
    }
    return new GridSize(parts[0], parts[1]);
}
class XY {
    constructor(x, y) {
        this.x = x;
        this.y = y;
    }
    clone() {
        return new XY(this.x, this.y);
    }
    toString() {
        return 'XY(' + [this.x, this.y].join(', ') + ')';
    }
    dot(b) {
        return this.x * b.x + this.y * b.y;
    }
    unit() {
        const norm = this.l2norm();
        return new XY(this.x / norm, this.y / norm);
    }
    l2norm() {
        return Math.sqrt(this.l2normSquared());
    }
    l2normSquared() {
        return this.dot(this);
    }
    scale(s) {
        return new XY(this.x * s, this.y * s);
    }
    project(b) {
        return b.scale(this.dot(b) / b.l2normSquared());
    }
    scalarProjection(b) {
        return this.dot(b.unit());
    }
    minus(b) {
        return new XY(this.x - b.x, this.y - b.y);
    }
    plus(b) {
        return new XY(this.x + b.x, this.y + b.y);
    }
}
const ADJOIN_DOT_PRODUCT_TOL = .02;
class LineSegment {
    constructor(a, b) {
        this.a = a;
        this.b = b;
    }
    static fromTwoPoints(a, b) {
        return new LineSegment(a, b);
    }
    direction() {
        return this.b.minus(this.a).unit();
    }
    adjoins(other, distTol) {
        return this.parallels(other) && this.lineDistance(other) < distTol;
    }
    parallels(other) {
        const unitDot = this.direction().dot(other.direction());
        return withinTol(Math.abs(unitDot), 1, ADJOIN_DOT_PRODUCT_TOL);
    }
    // The distance between the lines of two line segments. If lines are not
    // (close to) parallel, 0 is returned
    lineDistance(other) {
        return this.perpVectorBetweenLines(other).l2norm();
    }
    // The perpendicular vector between the lines of two line segments. If lines
    // are not (close to) parallel, [0, 0] is returned
    perpVectorBetweenLines(other) {
        const otherDir = other.direction();
        const unitDot = this.direction().dot(otherDir);
        if (!withinTol(Math.abs(unitDot), 1, ADJOIN_DOT_PRODUCT_TOL)) {
            return new XY(0, 0);
        }
        // Basically parallel. Now measure the perpendicular distance between
        // this.a->other.a and other.a->other.b.
        const d = other.a.minus(this.a);
        return d.minus(d.project(otherDir));
    }
}
class Size {
    constructor(width, height) {
        this.width = width;
        this.height = height;
    }
    clone() {
        return new Size(this.width, this.height);
    }
    toString() {
        return [this.width, this.height].join('x');
    }
    area() {
        return this.width * this.height;
    }
}
/**
 * A screen rectangle. A (0, 0) origin represents the top left of a display
 * area. Units are typically pixels.
 */
class Rect {
    constructor(origin, size) {
        this.origin = origin;
        this.size = size;
    }
    clone() {
        return new Rect(this.origin.clone(), this.size.clone());
    }
    toString() {
        return [this.origin, this.size].join(' ');
    }
    equal(r, tol) {
        const close = (a, b) => Math.abs(a - b) <= tol;
        return (close(this.origin.x, r.origin.x) &&
            close(this.origin.y, r.origin.y) &&
            close(this.size.width, r.size.width) &&
            close(this.size.height, r.size.height));
    }
    inset(s) {
        return new Rect(new XY(this.origin.x + s.width, this.origin.y + s.height), new Size(this.size.width - 2 * s.width, this.size.height - 2 * s.height));
    }
    edges() {
        const down = new XY(0, this.size.height);
        const right = new XY(this.size.width, 0);
        const seg = (a, b) => LineSegment.fromTwoPoints(a, b);
        // a---b
        // c---d
        const a = this.origin;
        const b = a.plus(right);
        const c = a.plus(down);
        const d = c.plus(right);
        const rv = new Edges({
            top: seg(a, b),
            right: seg(b, d),
            bottom: seg(c, d),
            left: seg(a, c)
        });
        return rv;
    }
    translate(vec) {
        return new Rect(this.origin.plus(vec), this.size);
    }
    // Increases or decreases the size of the rectangle by moving one of its
    // edges d units along the postive x or y axis, where positive x is right
    // and positive y is down.
    translateEdge(side, d) {
        const [w, h] = [this.size.width, this.size.height];
        switch (side) {
            case Side.Top:
                return new Rect(this.origin.plus(new XY(0, d)), new Size(w, h - d));
            case Side.Bottom:
                return new Rect(this.origin, new Size(w, h + d));
            case Side.Right:
                return new Rect(this.origin, new Size(w + d, h));
            case Side.Left:
                return new Rect(this.origin.plus(new XY(d, 0)), new Size(w - d, h));
            default:
                throw TypeError('bad side type ' + side);
        }
    }
    topLeft() {
        return this.origin;
    }
    topRight() {
        return this.origin.plus(new XY(this.size.width, 0));
    }
    bottomRight() {
        return this.origin.plus(new XY(this.size.width, this.size.height));
    }
    bottomLeft() {
        return this.origin.plus(new XY(0, this.size.height));
    }
    intersection(other) {
        // Not optimized, but that's not necessary.
        const origin = new XY(Math.max(this.topLeft().x, other.topLeft().x), Math.max(this.topLeft().y, other.topLeft().y));
        const br = new XY(Math.min(this.bottomRight().x, other.bottomRight().x), Math.min(this.bottomRight().y, other.bottomRight().y));
        const sizeXY = br.minus(origin);
        const size = new Size(sizeXY.x, sizeXY.y);
        if (size.width < 0 || size.height < 0) {
            return new Rect(new XY(0, 0), new Size(0, 0));
        }
        return new Rect(origin, size);
    }
    valid() {
        return this.size.width >= 0 && this.size.height >= 0;
    }
}
var Side;
(function (Side) {
    Side["Top"] = "TOP";
    Side["Right"] = "RIGHT";
    Side["Bottom"] = "BOTTOM";
    Side["Left"] = "LEFT";
})(Side || (Side = {}));
class Edges {
    constructor(obj) {
        this.top = obj.top;
        this.left = obj.left;
        this.bottom = obj.bottom;
        this.right = obj.right;
    }
    getSide(s) {
        switch (s) {
            case Side.Top: return this.top;
            case Side.Right: return this.right;
            case Side.Bottom: return this.bottom;
            case Side.Left: return this.left;
        }
    }
}
function withinTol(a, b, tol) {
    return Math.abs(a - b) <= tol;
}

/**
 * parsePreset parses a sequence of TileSpec objects from a string like "8x8 0:0
 * 0:7, 8x10 0:0 2:7" or "8x8 0:0 0:7, 0:0 2:7"
 *
 * The 8x8 and 8x10 values above are the "grid size." The grid size may be
 * omitted, then fallback grid size will be used.
 */
function parsePreset(preset, fallback) {
    const parts = preset.split(',').map(x => x.trim());
    let mostRecentSpec = null;
    return parts.map((part, index) => {
        if (hasImpliedGridSize(part)) {
            if (mostRecentSpec === null) {
                if (fallback === undefined) {
                    throw new Error(`preset component[${index}] ${part} of ${preset} is missing grid size (e.g., '3x3') and no fallback is specified`);
                }
                else {
                    part = `${fallback.width}x${fallback.height} ${part}`;
                }
            }
            else {
                part = `${mostRecentSpec.gridWidth}x${mostRecentSpec.gridHeight} ${part}`;
            }
        }
        const parsed = parseSinglePreset(part);
        mostRecentSpec = parsed;
        return parsed;
    });
}
function parseSinglePreset(preset) {
    const ps = preset.trim().split(" ");
    if (ps.length != 3) {
        throw new Error(`Bad preset: ${JSON.stringify(preset)}`);
    }
    const gridFormat = parseTuple(ps[0], "x");
    let luc = new TupleHolder(ps[1]);
    let rdc = new TupleHolder(ps[2]);
    if (gridFormat.x < 1 || gridFormat.x > 100
        || gridFormat.y < 1 || gridFormat.y > 100) {
        throw new Error(`Bad preset: ${JSON.stringify(preset)} grid size out of range 1..100`);
    }
    return new TileSpec(gridFormat.x, gridFormat.y, luc, rdc);
}
function hasImpliedGridSize(singlePreset) {
    return singlePreset.trim().split(" ").length === 2;
}
/**
 * Parses a value like like 6x4 or 1:2 into {X: 6, Y: 4} or {X: 1, Y: 2}.
 */
function parseTuple(unparsed, delim) {
    // parsing grid size in unparsed XdelimY, like 6x4 or 1:2
    const gssk = unparsed.split(delim);
    if (gssk.length !== 2) {
        throw new Error("Failed to split " + unparsed + " by delimiter " + delim + " into two numbers");
    }
    const numbers = gssk.map(Number);
    if (numbers.some(n => isNaN(n) || n > MAX_TUPLE_MEMBER_VALUE)) {
        throw new Error(`All elements of tuple must be intgers in [-inf, ${MAX_TUPLE_MEMBER_VALUE}]: ${JSON.stringify(unparsed)}`);
    }
    return new XY(numbers[0], numbers[1]);
}

/**
 * @fileoverview This file contains incomplete typings for gnome shell types.
 *
 * Probably the best source of definitive API documentation is here:
 * https://gjs-docs.gnome.org/
 *
 * However, there are also some ways the GJS works that make the API docs above
 * slightly incomplete.
 * https://wiki.gnome.org/Projects/GnomeShell/Extensions/StepByStepTutorial
 * mentions that constructors can take a property map as an argument. This file
 * does not correctly type the constructors for these types.
 */
/**
 * From https://gjs-docs.gnome.org/meta4~4_api/meta.frametype.
 */
var FrameType;
(function (FrameType) {
    FrameType[FrameType["NORMAL"] = 0] = "NORMAL";
    FrameType[FrameType["DIALOG"] = 1] = "DIALOG";
    FrameType[FrameType["MODAL_DIALOG"] = 2] = "MODAL_DIALOG";
    FrameType[FrameType["UTILITY"] = 3] = "UTILITY";
    FrameType[FrameType["MENU"] = 4] = "MENU";
    FrameType[FrameType["BORDER"] = 5] = "BORDER";
    FrameType[FrameType["ATTACHED"] = 6] = "ATTACHED";
    FrameType[FrameType["LAST"] = 7] = "LAST";
})(FrameType || (FrameType = {}));
var WindowType;
(function (WindowType) {
    WindowType[WindowType["NORMAL"] = 0] = "NORMAL";
    WindowType[WindowType["DESKTOP"] = 1] = "DESKTOP";
    WindowType[WindowType["DOCK"] = 2] = "DOCK";
    WindowType[WindowType["DIALOG"] = 3] = "DIALOG";
    WindowType[WindowType["MODAL_DIALOG"] = 4] = "MODAL_DIALOG";
    WindowType[WindowType["TOOLBAR"] = 5] = "TOOLBAR";
    WindowType[WindowType["MENU"] = 6] = "MENU";
    WindowType[WindowType["UTILITY"] = 7] = "UTILITY";
    WindowType[WindowType["SPLASHSCREEN"] = 8] = "SPLASHSCREEN";
    WindowType[WindowType["DROPDOWN_MENU"] = 9] = "DROPDOWN_MENU";
    WindowType[WindowType["POPUP_MENU"] = 10] = "POPUP_MENU";
    WindowType[WindowType["TOOLTIP"] = 11] = "TOOLTIP";
    WindowType[WindowType["NOTIFICATION"] = 12] = "NOTIFICATION";
    WindowType[WindowType["COMBO"] = 13] = "COMBO";
    WindowType[WindowType["DND"] = 14] = "DND";
    WindowType[WindowType["OVERRIDE_OTHER"] = 15] = "OVERRIDE_OTHER";
})(WindowType || (WindowType = {}));
var MaximizeFlags;
(function (MaximizeFlags) {
    MaximizeFlags[MaximizeFlags["HORIZONTAL"] = 1] = "HORIZONTAL";
    MaximizeFlags[MaximizeFlags["VERTICAL"] = 2] = "VERTICAL";
    MaximizeFlags[MaximizeFlags["BOTH"] = 3] = "BOTH";
})(MaximizeFlags || (MaximizeFlags = {}));

/*****************************************************************

  This extension has been developed by vibou

  With the help of the gnome-shell community

  Edited by Kvis for gnome 3.8
  Edited by Lundal for gnome 3.18
  Edited by Sergey to add keyboard shortcuts and prefs dialog

 ******************************************************************/
/*****************************************************************
  CONST & VARS
 *****************************************************************/
// Library imports
const St = imports.gi.St;
const Main$1 = imports.ui.main;
const Shell$1 = imports.gi.Shell;
const GObject = imports.gi.GObject;
const PanelMenu = imports.ui.panelMenu;
const Meta$2 = imports.gi.Meta;
const Clutter = imports.gi.Clutter;
const Signals = imports.signals;
const Workspace = imports.ui.workspace;
const Mainloop = imports.mainloop;
// Getter for accesing "get_active_workspace" on GNOME <=2.28 and >= 2.30
const WorkspaceManager$1 = (global.screen || global.workspace_manager);
// Extension imports
const Me$1 = imports.misc.extensionUtils.getCurrentExtension();
const Settings$1 = Me$1.imports.settings;
// Globals
const SETTINGS_GRID_SIZES = 'grid-sizes';
const SETTINGS_AUTO_CLOSE = 'auto-close';
const SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT = "auto-close-keyboard-shortcut";
const SETTINGS_ANIMATION = 'animation';
const SETTINGS_SHOW_ICON = 'show-icon';
const SETTINGS_GLOBAL_AUTO_TILING = 'global-auto-tiling';
const SETTINGS_GLOBAL_PRESETS = 'global-presets';
const SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE = "target-presets-to-monitor-of-mouse";
const SETTINGS_MOVERESIZE_ENABLED = 'moveresize-enabled';
const SETTINGS_WINDOW_MARGIN = 'window-margin';
const SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED = 'window-margin-fullscreen-enabled';
const SETTINGS_MAX_TIMEOUT = 'max-timeout';
const SETTINGS_MAIN_WINDOW_SIZES = 'main-window-sizes';
const SETTINGS_SHOW_GRID_LINES = 'show-grid-lines';
const SETTINGS_INSETS_PRIMARY = 'insets-primary';
const SETTINGS_INSETS_PRIMARY_LEFT = 'insets-primary-left';
const SETTINGS_INSETS_PRIMARY_RIGHT = 'insets-primary-right';
const SETTINGS_INSETS_PRIMARY_TOP = 'insets-primary-top';
const SETTINGS_INSETS_PRIMARY_BOTTOM = 'insets-primary-bottom';
const SETTINGS_INSETS_SECONDARY = 'insets-secondary';
const SETTINGS_INSETS_SECONDARY_LEFT = 'insets-secondary-left';
const SETTINGS_INSETS_SECONDARY_RIGHT = 'insets-secondary-right';
const SETTINGS_INSETS_SECONDARY_TOP = 'insets-secondary-top';
const SETTINGS_INSETS_SECONDARY_BOTTOM = 'insets-secondary-bottom';
const SETTINGS_DEBUG = 'debug';
const SETTINGS_THEME = 'theme';
const gridSettings = {
    [SETTINGS_GRID_SIZES]: [],
    [SETTINGS_AUTO_CLOSE]: null,
    [SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT]: null,
    [SETTINGS_ANIMATION]: null,
    [SETTINGS_SHOW_ICON]: null,
    [SETTINGS_GLOBAL_AUTO_TILING]: null,
    [SETTINGS_GLOBAL_PRESETS]: null,
    [SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE]: null,
    [SETTINGS_MOVERESIZE_ENABLED]: null,
    [SETTINGS_WINDOW_MARGIN]: 0,
    [SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED]: false,
    [SETTINGS_MAX_TIMEOUT]: null,
    [SETTINGS_MAIN_WINDOW_SIZES]: [],
    [SETTINGS_SHOW_GRID_LINES]: false,
    [SETTINGS_INSETS_PRIMARY]: 0,
    [SETTINGS_INSETS_PRIMARY_LEFT]: 0,
    [SETTINGS_INSETS_PRIMARY_RIGHT]: 0,
    [SETTINGS_INSETS_PRIMARY_TOP]: 0,
    [SETTINGS_INSETS_PRIMARY_BOTTOM]: 0,
    [SETTINGS_INSETS_SECONDARY]: 0,
    [SETTINGS_INSETS_SECONDARY_LEFT]: 0,
    [SETTINGS_INSETS_SECONDARY_RIGHT]: 0,
    [SETTINGS_INSETS_SECONDARY_TOP]: 0,
    [SETTINGS_INSETS_SECONDARY_BOTTOM]: 0,
    [SETTINGS_DEBUG]: null,
    [SETTINGS_THEME]: null,
};
let launcher;
let tracker;
let nbCols = 0;
let nbRows = 0;
let focusMetaWindow = null;
let focusConnect = false;
let settings = Settings$1.get();
settings.connect('changed', changed_settings);
let keyControlBound = false;
let enabled = false;
let mainWindowSizes = new Array();
let monitorsChangedConnect = false;
const SHELL_VERSION = ShellVersion.defaultVersion();
const lastResizeInfo = {
    variantIndex: 0,
    lastCallTime: new Date(),
    presetName: '',
    windowTitle: '',
};
let theme = getTheme();
// Hangouts workaround
let excludedApplications = new Array("Unknown");
const keyBindings = new Map([
    ['show-toggle-tiling', () => { globalApp.toggleTiling(); }],
    ['show-toggle-tiling-alt', () => { globalApp.toggleTiling(); }],
]);
const key_bindings_tiling = new Map([
    ['move-left', () => { keyMoveResizeEvent('move', 'left'); }],
    ['move-right', () => { keyMoveResizeEvent('move', 'right'); }],
    ['move-up', () => { keyMoveResizeEvent('move', 'up'); }],
    ['move-down', () => { keyMoveResizeEvent('move', 'down'); }],
    ['resize-left', () => { keyMoveResizeEvent('resize', 'left'); }],
    ['resize-right', () => { keyMoveResizeEvent('resize', 'right'); }],
    ['resize-up', () => { keyMoveResizeEvent('resize', 'up'); }],
    ['resize-down', () => { keyMoveResizeEvent('resize', 'down'); }],
    ['move-left-vi', () => { keyMoveResizeEvent('move', 'left'); }],
    ['move-right-vi', () => { keyMoveResizeEvent('move', 'right'); }],
    ['move-up-vi', () => { keyMoveResizeEvent('move', 'up'); }],
    ['move-down-vi', () => { keyMoveResizeEvent('move', 'down'); }],
    ['resize-left-vi', () => { keyMoveResizeEvent('resize', 'left'); }],
    ['resize-right-vi', () => { keyMoveResizeEvent('resize', 'right'); }],
    ['resize-up-vi', () => { keyMoveResizeEvent('resize', 'up'); }],
    ['resize-down-vi', () => { keyMoveResizeEvent('resize', 'down'); }],
    ['cancel-tiling', () => { keyCancelTiling(); }],
    ['set-tiling', () => { keySetTiling(); }],
    ['change-grid-size', () => { keyChangeTiling(); }],
    ['snap-to-neighbors', () => { snapToNeighborsBind(); }],
    ['snap-to-neighbors', () => { snapToNeighborsBind(); }],
]);
const key_bindings_auto_tiling = new Map([
    ['autotile-main', () => { AutoTileMain(); }],
    ['autotile-1', () => { autoTileNCols(1); }],
    ['autotile-2', () => { autoTileNCols(2); }],
    ['autotile-3', () => { autoTileNCols(3); }],
    ['autotile-4', () => { autoTileNCols(4); }],
    ['autotile-5', () => { autoTileNCols(5); }],
    ['autotile-6', () => { autoTileNCols(6); }],
    ['autotile-7', () => { autoTileNCols(7); }],
    ['autotile-8', () => { autoTileNCols(8); }],
    ['autotile-9', () => { autoTileNCols(9); }],
    ['autotile-10', () => { autoTileNCols(10); }],
]);
const key_bindings_presets = new Map([
    ['preset-resize-1', () => { presetResize(1, 'resize1'); }],
    ['preset-resize-2', () => { presetResize(2, 'resize2'); }],
    ['preset-resize-3', () => { presetResize(3, 'resize3'); }],
    ['preset-resize-4', () => { presetResize(4, 'resize4'); }],
    ['preset-resize-5', () => { presetResize(5, 'resize5'); }],
    ['preset-resize-6', () => { presetResize(6, 'resize6'); }],
    ['preset-resize-7', () => { presetResize(7, 'resize7'); }],
    ['preset-resize-8', () => { presetResize(8, 'resize8'); }],
    ['preset-resize-9', () => { presetResize(9, 'resize9'); }],
    ['preset-resize-10', () => { presetResize(10, 'resize10'); }],
    ['preset-resize-11', () => { presetResize(11, 'resize11'); }],
    ['preset-resize-12', () => { presetResize(12, 'resize12'); }],
    ['preset-resize-13', () => { presetResize(13, 'resize13'); }],
    ['preset-resize-14', () => { presetResize(14, 'resize14'); }],
    ['preset-resize-15', () => { presetResize(15, 'resize15'); }],
    ['preset-resize-16', () => { presetResize(16, 'resize16'); }],
    ['preset-resize-17', () => { presetResize(17, 'resize17'); }],
    ['preset-resize-18', () => { presetResize(18, 'resize18'); }],
    ['preset-resize-19', () => { presetResize(19, 'resize19'); }],
    ['preset-resize-20', () => { presetResize(20, 'resize20'); }],
    ['preset-resize-21', () => { presetResize(21, 'resize21'); }],
    ['preset-resize-22', () => { presetResize(22, 'resize22'); }],
    ['preset-resize-23', () => { presetResize(23, 'resize23'); }],
    ['preset-resize-24', () => { presetResize(24, 'resize24'); }],
    ['preset-resize-25', () => { presetResize(25, 'resize25'); }],
    ['preset-resize-26', () => { presetResize(26, 'resize26'); }],
    ['preset-resize-27', () => { presetResize(27, 'resize27'); }],
    ['preset-resize-28', () => { presetResize(28, 'resize28'); }],
    ['preset-resize-29', () => { presetResize(29, 'resize29'); }],
    ['preset-resize-30', () => { presetResize(30, 'resize30'); }],
]);
const keyBindingGlobalResizes = new Map([
    ['action-change-tiling', () => { keyChangeTiling(); }],
    ['action-contract-bottom', () => { keyMoveResizeEvent('contract', 'bottom', true); }],
    ['action-contract-left', () => { keyMoveResizeEvent('contract', 'left', true); }],
    ['action-contract-right', () => { keyMoveResizeEvent('contract', 'right', true); }],
    ['action-contract-top', () => { keyMoveResizeEvent('contract', 'top', true); }],
    ['action-expand-bottom', () => { keyMoveResizeEvent('expand', 'bottom', true); }],
    ['action-expand-left', () => { keyMoveResizeEvent('expand', 'left', true); }],
    ['action-expand-right', () => { keyMoveResizeEvent('expand', 'right', true); }],
    ['action-expand-top', () => { keyMoveResizeEvent('expand', 'top', true); }],
    ['action-move-down', () => { keyMoveResizeEvent('move', 'down', true); }],
    ['action-move-left', () => { keyMoveResizeEvent('move', 'left', true); }],
    ['action-move-right', () => { keyMoveResizeEvent('move', 'right', true); }],
    ['action-move-up', () => { keyMoveResizeEvent('move', 'up', true); }],
    ['action-move-next-monitor', () => { moveWindowToNextMonitor(); }],
]);
class App {
    constructor() {
        this.gridsByMonitorKey = {};
        this.gridShowing = false;
        this.gridWidget = null;
        this.gridLinesTimeout = null;
        this.gridTiles = [];
    }
    enable() {
        this.gridShowing = false;
        tracker = Shell$1.WindowTracker.get_default();
        initSettings();
        const gridWidget = (new St.BoxLayout({ style_class: `${theme}__preview` }));
        this.gridWidget = gridWidget;
        Main$1.uiGroup.add_actor(gridWidget);
        this.initGrids(gridWidget);
        log("Create Button on Panel");
        launcher = new GTileStatusButton(`${theme}__icon`);
        if (gridSettings[SETTINGS_SHOW_ICON]) {
            Main$1.panel.addToStatusArea("GTileStatusButton", launcher);
        }
        bind(keyBindings);
        if (gridSettings[SETTINGS_GLOBAL_AUTO_TILING]) {
            bind(key_bindings_auto_tiling);
        }
        if (gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            bind(key_bindings_presets);
        }
        if (gridSettings[SETTINGS_MOVERESIZE_ENABLED]) {
            bind(keyBindingGlobalResizes);
        }
        if (monitorsChangedConnect) {
            Main$1.layoutManager.disconnect(monitorsChangedConnect);
        }
        log("Connecting monitors-changed");
        monitorsChangedConnect = Main$1.layoutManager.connect('monitors-changed', () => {
            log("Reinitializing grids on monitors-changed");
            this.destroyGrids();
            this.initGrids(gridWidget);
        });
        enabled = true;
        log("Extention enable completed");
    }
    getGrid(monitor) {
        return this.gridsByMonitorKey[getMonitorKey(monitor)];
    }
    initGrids(gridWidget) {
        log("initGrids");
        log("initGrids nobCols " + nbCols + " nbRows " + nbRows);
        const monitors = activeMonitors();
        for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
            log("New Grid for monitor " + monitorIdx);
            let monitor = monitors[monitorIdx];
            let grid = new Grid(gridWidget, monitorIdx, monitor, "gTile", nbCols, nbRows);
            const key = getMonitorKey(monitor);
            this.gridsByMonitorKey[key] = grid;
            log("initGrids adding grid key " + key);
            // TODO: addChrome is poorly documented. I can't find any reference
            // to it in the gjs-docs site.
            Main$1.layoutManager.addChrome(grid.actor, { trackFullscreen: true });
            grid.actor.set_opacity(0);
            grid.hide(true);
            log("Connect hide-tiling for monitor " + monitorIdx);
            grid.connectHideTiling = grid.connect('hide-tiling', () => this.hideTiling());
        }
        log("Init grid done");
    }
    destroyGrids() {
        log("destroyGrids");
        for (let gridKey in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[gridKey];
            grid.hide(true);
            Main$1.layoutManager.removeChrome(grid.actor);
            log("Disconnect hide-tiling for monitor " + grid.monitor_idx);
            grid.disconnect(grid.connectHideTiling);
            delete this.gridsByMonitorKey[gridKey];
        }
        log("destroyGrids done");
        for (let gridKey in this.gridsByMonitorKey) {
            log("ERROR: gridKey still found in this.gridsByMonitorKey: " + gridKey);
        }
    }
    refreshGrids() {
        log("refreshGrids");
        for (let gridIdx in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[gridIdx];
            log("refreshGrids calling refresh on " + gridIdx);
            grid.refresh();
        }
        log("refreshGrids done");
    }
    setGridSize(gridSize) {
        nbCols = gridSize.width;
        nbRows = gridSize.height;
        this.refreshGrids();
        if (gridSettings[SETTINGS_SHOW_GRID_LINES]) {
            this._showGridLines(gridSize);
        }
    }
    moveGrids() {
        log("moveGrids");
        if (!this.gridShowing) {
            return;
        }
        let window = focusMetaWindow;
        if (window) {
            for (let gridKey in this.gridsByMonitorKey) {
                let grid = this.gridsByMonitorKey[gridKey];
                let pos_x;
                let pos_y;
                const monitor = grid.monitor;
                if (!monitor) {
                    return;
                }
                if (window.get_monitor() == grid.monitor_idx) {
                    pos_x = window.get_frame_rect().width / 2 + window.get_frame_rect().x;
                    pos_y = window.get_frame_rect().height / 2 + window.get_frame_rect().y;
                    let [mouse_x, mouse_y, mask] = global.get_pointer();
                    let act_x = pos_x - grid.actor.width / 2;
                    let act_y = pos_y - grid.actor.height / 2;
                    if (mouse_x >= act_x
                        && mouse_x <= act_x + grid.actor.width
                        && mouse_y >= act_y
                        && mouse_y <= act_y + grid.actor.height) {
                        log("Mouse x " + mouse_x + " y " + mouse_y +
                            " is inside grid actor rectangle, changing actor X from " + pos_x + " to " + (mouse_x + grid.actor.width / 2) +
                            ", Y from " + pos_y + " to " + (mouse_y + grid.actor.height / 2));
                        pos_x = mouse_x + grid.actor.width / 2;
                        pos_y = mouse_y + grid.actor.height / 2;
                    }
                }
                else {
                    pos_x = monitor.x + monitor.width / 2;
                    pos_y = monitor.y + monitor.height / 2;
                }
                pos_x = Math.floor(pos_x - grid.actor.width / 2);
                pos_y = Math.floor(pos_y - grid.actor.height / 2);
                if (window.get_monitor() == grid.monitor_idx) {
                    pos_x = (pos_x < monitor.x) ? monitor.x : pos_x;
                    pos_x = ((pos_x + grid.actor.width) > (monitor.width + monitor.x)) ? monitor.x + monitor.width - grid.actor.width : pos_x;
                    pos_y = (pos_y < monitor.y) ? monitor.y : pos_y;
                    pos_y = ((pos_y + grid.actor.height) > (monitor.height + monitor.y)) ? monitor.y + monitor.height - grid.actor.height : pos_y;
                }
                let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0.1;
                grid.actor.ease({
                    time: time,
                    x: pos_x,
                    y: pos_y,
                    transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                });
            }
        }
    }
    updateRegions() {
        var _a;
        /*Main.layoutManager._chrome.updateRegions();*/
        log("updateRegions");
        this.refreshGrids();
        for (let idx in this.gridsByMonitorKey) {
            (_a = this.gridsByMonitorKey[idx].elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
        }
    }
    logState() {
        let count = 0;
        let states = '';
        for (let gridKey in this.gridsByMonitorKey) {
            states += `; ${this.gridsByMonitorKey[gridKey].state()}`;
            count++;
        }
        log(`${count} grids; showing = ${this.gridShowing}: ${states}`);
    }
    showTiling() {
        // TODO(#168): See https://github.com/gTile/gTile/issues/168. Without
        // these two lines, the grid UI does not properly respond to mouseover
        // and other events except for the first time it is shown.
        log("showTiling with fix");
        this.hideTiling();
        this.destroyGrids();
        this.initGrids(this.gridWidget);
        this.logState();
        log("issue#168/showTiling with fix");
        const window = getFocusApp();
        // TODO: remove this global side effect
        focusMetaWindow = window;
        if (!this.gridWidget) {
            return;
        }
        const shouldShowTiling = (() => {
            if (!window) {
                // The tiling UI is for manipulating a particular window.
                return false;
            }
            const wmType = window.get_window_type();
            const layer = window.get_layer();
            return wmType !== WindowType.DESKTOP && layer > 0;
        })();
        if (!shouldShowTiling) {
            return;
        }
        this.gridWidget.visible = true;
        const monitors = activeMonitors();
        for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
            const monitor = monitors[monitorIdx];
            const grid = this.getGrid(monitor);
            if (grid === null) {
                log(`issue#168/showTiling ERROR: did not find grid for monitor ${getMonitorKey(monitor)}`);
                continue;
            }
            let pos_x;
            let pos_y;
            if (window && window.get_monitor() === monitorIdx) {
                // Set pos_x, pos_y to center of the selected window initially.
                pos_x = window.get_frame_rect().width / 2 + window.get_frame_rect().x;
                pos_y = window.get_frame_rect().height / 2 + window.get_frame_rect().y;
                // Old logic with no justifying comments: If the mouse is active
                // and within the rectangle,  set pos_x and pos_y  at mouse
                // position + half the size of the Grid window.
                let [mouse_x, mouse_y, mask] = global.get_pointer();
                let act_x = pos_x - grid.actor.width / 2;
                let act_y = pos_y - grid.actor.height / 2;
                if (mouse_x >= act_x
                    && mouse_x <= act_x + grid.actor.width
                    && mouse_y >= act_y
                    && mouse_y <= act_y + grid.actor.height) {
                    log("Mouse x " + mouse_x + " y " + mouse_y +
                        " is inside grid actor rectangle, changing actor X from " + pos_x + " to " + (mouse_x + grid.actor.width / 2) +
                        ", Y from " + pos_y + " to " + (mouse_y + grid.actor.height / 2));
                    pos_x = mouse_x + grid.actor.width / 2;
                    pos_y = mouse_y + grid.actor.height / 2;
                }
            }
            else {
                pos_x = monitor.x + monitor.width / 2;
                pos_y = monitor.y + monitor.height / 2;
            }
            grid.set_position(Math.floor(pos_x - grid.actor.width / 2), Math.floor(pos_y - grid.actor.height / 2));
            grid.show();
        }
        this.gridShowing = true;
        this.onFocus();
        launcher === null || launcher === void 0 ? void 0 : launcher.activate();
        bindKeyControls();
        this.moveGrids();
    }
    disable() {
        log("Extension disable begin");
        enabled = false;
        // Notice for extension reviewer - this will call Mainloop.RemoveTimeout
        this._hideGridLines();
        if (monitorsChangedConnect) {
            log("Disconnecting monitors-changed");
            Main$1.layoutManager.disconnect(monitorsChangedConnect);
            monitorsChangedConnect = false;
        }
        unbind(keyBindings);
        unbind(key_bindings_auto_tiling);
        unbind(key_bindings_presets);
        unbind(keyBindingGlobalResizes);
        if (keyControlBound) {
            unbind(key_bindings_tiling);
            keyControlBound = false;
        }
        launcher === null || launcher === void 0 ? void 0 : launcher.destroy();
        launcher = null;
        Main$1.uiGroup.remove_actor(this.gridWidget);
        this.destroyGrids();
        resetFocusMetaWindow();
        log("Extention disable completed");
    }
    hideTiling() {
        var _a;
        log("hideTiling");
        for (let key in this.gridsByMonitorKey) {
            const grid = this.gridsByMonitorKey[key];
            (_a = grid.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            grid.hide(false);
        }
        if (this.gridWidget) {
            this.gridWidget.visible = false;
        }
        resetFocusMetaWindow();
        launcher === null || launcher === void 0 ? void 0 : launcher.deactivate();
        this.gridShowing = false;
        unbindKeyControls();
    }
    toggleTiling() {
        if (this.gridShowing) {
            this.hideTiling();
        }
        else {
            this.showTiling();
        }
        return this.gridShowing;
    }
    isGridShowing() {
        return this.gridShowing;
    }
    _hideGridLines(removeTimeout = true) {
        if (this.gridLinesTimeout != null) {
            log("Removing grid lines...");
            if (removeTimeout) {
                Mainloop.timeout_remove(this.gridLinesTimeout);
            }
            this.gridLinesTimeout = null;
            for (let tile of this.gridTiles) {
                Main$1.uiGroup.remove_actor(tile);
            }
        }
        this.gridTiles = [];
    }
    _showGridLines(gridSize) {
        this._hideGridLines();
        log("Started drawing grid lines...");
        for (let monitorIdx = 0; monitorIdx < activeMonitors().length; monitorIdx++) {
            const workArea = workAreaRectByMonitorIndex(monitorIdx);
            const monitor = activeMonitors()[monitorIdx];
            if (!workArea) {
                continue;
            }
            let tileHeight = workArea.size.height / gridSize.height;
            let tileWidth = workArea.size.width / gridSize.width;
            let topOffset = workArea.topLeft().y;
            let leftOffset = workArea.topLeft().x;
            log(`Starting to draw grid lines for monitor ${JSON.stringify(monitor)}`);
            for (let i = 1; i < gridSize.width; i++) {
                const newGridWidget = new St.BoxLayout({ style_class: `${theme}__grid_lines_preview` });
                const posX = leftOffset + tileWidth * i;
                newGridWidget.width = 1;
                newGridWidget.height = workArea.size.height;
                newGridWidget.x = posX;
                newGridWidget.y = 0;
                this.gridTiles.push(newGridWidget);
                Main$1.uiGroup.add_actor(newGridWidget);
                log(`Grid vertical line of size ${tileWidth}:${tileHeight} is drawn at ${posX}:0 (monitor offset ${monitor.x}:${monitor.y})`);
            }
            for (let u = 1; u <= gridSize.height; u++) {
                const newGridWidget = new St.BoxLayout({ style_class: `${theme}__grid_lines_preview` });
                const posY = topOffset + tileHeight * u;
                newGridWidget.width = workArea.size.width;
                newGridWidget.height = 1;
                newGridWidget.x = 0;
                newGridWidget.y = posY;
                this.gridTiles.push(newGridWidget);
                Main$1.uiGroup.add_actor(newGridWidget);
                log(`Grid horizontal line of size ${tileWidth}:${tileHeight} is drawn at 0:${posY} (monitor offset ${monitor.x}:${monitor.y})`);
            }
        }
        this.gridLinesTimeout = Mainloop.timeout_add(1000, () => {
            this._hideGridLines(false);
        });
    }
    /**
     * onFocus is called when the global focus changes.
     */
    onFocus() {
        log("_onFocus ");
        resetFocusMetaWindow();
        const window = getFocusApp();
        if (window && this.gridShowing) {
            log("_onFocus " + window.get_title());
            focusMetaWindow = window;
            let app = tracker.get_window_app(focusMetaWindow);
            let title = focusMetaWindow.get_title();
            const monitors = activeMonitors();
            for (let monitorIdx = 0; monitorIdx < monitors.length; monitorIdx++) {
                let monitor = monitors[monitorIdx];
                let grid = this.getGrid(monitor);
                if (app) {
                    grid === null || grid === void 0 ? void 0 : grid.topbar.setApp(app, title);
                }
                else {
                    grid === null || grid === void 0 ? void 0 : grid.topbar.setTitle(title);
                }
            }
            this.moveGrids();
        }
        else {
            if (this.gridShowing) {
                log("No focus window, hide tiling");
                this.hideTiling();
            }
            else {
                log("tiling window not active");
            }
        }
    }
}
const globalApp = new App();
function changed_settings() {
    log("changed_settings");
    if (enabled) {
        disable();
        enable();
    }
    log("changed_settings complete");
}
const GTileStatusButton = GObject.registerClass({
    GTypeName: 'GTileStatusButton',
}, class GTileStatusButton extends PanelMenu.Button {
    _init(classname) {
        super._init(0.0, "gTile", false);
        //Done by default in PanelMenuButton - Just need to override the method
        if (SHELL_VERSION.version_at_least_34()) {
            const icon = new St.Icon({ style_class: 'system-status-icon' });
            this.add_actor(icon);
            this.add_style_class_name(classname);
            this.connect('button-press-event', this._onButtonPress.bind(this));
        }
        else {
            this.actor.add_style_class_name(classname);
            this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        }
        log("GTileStatusButton _init done");
    }
    reset() {
        this.activated = false;
        if (SHELL_VERSION.version_at_least_34()) {
            this.remove_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    }
    activate() {
        if (SHELL_VERSION.version_at_least_34()) {
            this.add_style_pseudo_class('activate');
        }
        else {
            this.actor.add_style_pseudo_class('activate');
        }
    }
    deactivate() {
        if (SHELL_VERSION.version_at_least_34()) {
            this.remove_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    }
    _onButtonPress(actor, event) {
        log(`_onButtonPress Click Toggle Status on system panel ${this}`);
        globalApp.toggleTiling();
    }
    _destroy() {
        this.activated = null;
    }
});
/*****************************************************************
  SETTINGS
 *****************************************************************/
function initGridSizes(configValue) {
    let gridSizes = parseGridSizesIgnoringErrors(configValue);
    if (gridSizes.length === 0) {
        gridSizes = [
            new GridSize(8, 6),
            new GridSize(6, 4),
            new GridSize(4, 4),
        ];
    }
    gridSettings[SETTINGS_GRID_SIZES] = gridSizes;
}
function getBoolSetting(settingName) {
    const value = settings.get_boolean(settingName);
    if (value === undefined) {
        log("Undefined settings " + settingName);
        gridSettings[settingName] = false;
        return false;
    }
    else {
        gridSettings[settingName] = value;
    }
    return value;
}
function getIntSetting(settingsValue) {
    let iss = settings.get_int(settingsValue);
    if (iss === undefined) {
        log("Undefined settings " + settingsValue);
        return 0;
    }
    else {
        return iss;
    }
}
function initSettings() {
    log("Init settings");
    theme = getTheme();
    const gridSizes = settings.get_string(SETTINGS_GRID_SIZES) || '';
    log(SETTINGS_GRID_SIZES + " set to " + gridSizes);
    initGridSizes(gridSizes);
    getBoolSetting(SETTINGS_AUTO_CLOSE);
    getBoolSetting(SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT);
    getBoolSetting(SETTINGS_ANIMATION);
    getBoolSetting(SETTINGS_SHOW_ICON);
    getBoolSetting(SETTINGS_GLOBAL_AUTO_TILING);
    getBoolSetting(SETTINGS_GLOBAL_PRESETS);
    getBoolSetting(SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE);
    getBoolSetting(SETTINGS_MOVERESIZE_ENABLED);
    getBoolSetting(SETTINGS_SHOW_GRID_LINES);
    gridSettings[SETTINGS_WINDOW_MARGIN] = getIntSetting(SETTINGS_WINDOW_MARGIN);
    gridSettings[SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED] = getBoolSetting(SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED);
    gridSettings[SETTINGS_MAX_TIMEOUT] = getIntSetting(SETTINGS_MAX_TIMEOUT);
    // initialize these from settings, the first set of sizes
    if (nbCols == 0 || nbRows == 0) {
        nbCols = gridSettings[SETTINGS_GRID_SIZES][0].width;
        nbRows = gridSettings[SETTINGS_GRID_SIZES][0].height;
    }
    const mainWindowSizesString = settings.get_string(SETTINGS_MAIN_WINDOW_SIZES);
    log(SETTINGS_MAIN_WINDOW_SIZES + " settings found " + mainWindowSizesString);
    mainWindowSizes = [];
    let mainWindowSizesArray = mainWindowSizesString.split(",");
    for (var i in mainWindowSizesArray) {
        let size = mainWindowSizesArray[i];
        if (size.includes("/")) {
            let fraction = size.split("/");
            let ratio = parseFloat(fraction[0]) / parseFloat(fraction[1]);
            mainWindowSizes.push(ratio);
        }
        else {
            mainWindowSizes.push(parseFloat(size));
        }
    }
    log(SETTINGS_MAIN_WINDOW_SIZES + " set to " + mainWindowSizes);
    log("Init complete, nbCols " + nbCols + " nbRows " + nbRows);
}
function getMonitorTier(monitor) {
    return isPrimaryMonitor(monitor) ? 'primary' : 'secondary';
}
function getMonitorInsets(tier) {
    switch (tier) {
        case 'primary':
            return {
                top: getIntSetting(SETTINGS_INSETS_PRIMARY_TOP),
                bottom: getIntSetting(SETTINGS_INSETS_PRIMARY_BOTTOM),
                left: getIntSetting(SETTINGS_INSETS_PRIMARY_LEFT),
                right: getIntSetting(SETTINGS_INSETS_PRIMARY_RIGHT)
            }; // Insets on primary monitor
        case 'secondary':
            return {
                top: getIntSetting(SETTINGS_INSETS_SECONDARY_TOP),
                bottom: getIntSetting(SETTINGS_INSETS_SECONDARY_BOTTOM),
                left: getIntSetting(SETTINGS_INSETS_SECONDARY_LEFT),
                right: getIntSetting(SETTINGS_INSETS_SECONDARY_RIGHT)
            };
        default:
            throw new Error(`unknown monitor name ${JSON.stringify(tier)}`);
    }
}
function enable() {
    setLoggingEnabled(getBoolSetting(SETTINGS_DEBUG));
    log("Extension enable begin");
    SHELL_VERSION.print_version();
    globalApp.enable();
}
function disable() {
    globalApp.disable();
}
function resetFocusMetaWindow() {
    log("resetFocusMetaWindow");
    focusMetaWindow = null;
}
function reset_window(metaWindow) {
    metaWindow.unmaximize(Meta$2.MaximizeFlags.HORIZONTAL);
    metaWindow.unmaximize(Meta$2.MaximizeFlags.VERTICAL);
    metaWindow.unmaximize(Meta$2.MaximizeFlags.BOTH);
}
function _getInvisibleBorderPadding(metaWindow) {
    let outerRect = metaWindow.get_frame_rect();
    let inputRect = metaWindow.get_buffer_rect();
    let borderX = outerRect.x - inputRect.x;
    let borderY = outerRect.y - inputRect.y;
    return [borderX, borderY];
}
function _getVisibleBorderPadding(metaWindow) {
    let clientRect = metaWindow.get_frame_rect();
    let outerRect = metaWindow.get_frame_rect();
    let borderX = outerRect.width - clientRect.width;
    let borderY = outerRect.height - clientRect.height;
    return [borderX, borderY];
}
function move_maximize_window(metaWindow, x, y) {
    const [borderX, borderY] = _getInvisibleBorderPadding(metaWindow);
    x = x - borderX;
    y = y - borderY;
    metaWindow.move_frame(true, x, y);
    metaWindow.maximize(Meta$2.MaximizeFlags.HORIZONTAL | Meta$2.MaximizeFlags.VERTICAL);
}
/**
 * Resizes window considering margin settings
 * @param metaWindow
 * @param x
 * @param y
 * @param width
 * @param height
 */
function moveResizeWindowWithMargins(metaWindow, x, y, width, height) {
    let [borderX, borderY] = _getInvisibleBorderPadding(metaWindow);
    let [vBorderX, vBorderY] = _getVisibleBorderPadding(metaWindow);
    log("move_resize_window_with_margins " + metaWindow.get_title() + " " + x + ":" + y + " - " + width
        + ":" + height + " margin " + gridSettings[SETTINGS_WINDOW_MARGIN] + " borders invisible " +
        borderX + ":" + borderY + " visible " + vBorderX + ":" + vBorderY);
    x = x + gridSettings[SETTINGS_WINDOW_MARGIN];
    y = y + gridSettings[SETTINGS_WINDOW_MARGIN];
    width = width - gridSettings[SETTINGS_WINDOW_MARGIN] * 2;
    height = height - gridSettings[SETTINGS_WINDOW_MARGIN] * 2;
    x = x + vBorderX;
    y = y + vBorderY;
    width = width - 2 * vBorderX;
    height = height - 2 * vBorderY;
    log("After margins and visible border window is " + x + ":" + y + " - " + width + ":" + height);
    metaWindow.move_frame(true, x, y);
    metaWindow.move_resize_frame(true, x, y, width, height);
}
function getNotFocusedWindowsOfMonitor(monitor) {
    const monitors = activeMonitors();
    let windows = global.get_window_actors().filter(function (w) {
        let app = tracker.get_window_app(w.meta_window);
        if (app == null) {
            return false;
        }
        let appName = app.get_name();
        return !contains(excludedApplications, appName)
            && w.meta_window.get_window_type() == Meta$2.WindowType.NORMAL
            && w.meta_window.get_workspace() == WorkspaceManager$1.get_active_workspace()
            && w.meta_window.showing_on_its_workspace()
            && monitors[w.meta_window.get_monitor()] == monitor
            && focusMetaWindow != w.meta_window;
    });
    return windows;
}
function getMonitorKey(monitor) {
    return monitor.x + ":" + monitor.width + ":" + monitor.y + ":" + monitor.height;
}
function contains(a, obj) {
    var i = a.length;
    while (i--) {
        if (a[i] === obj) {
            return true;
        }
    }
    return false;
}
/**
 * Get focused window by iterating though the windows on the active workspace.
 * @returns {Object} The focussed window object. False if no focussed window was found.
 */
function getFocusApp() {
    if (tracker.focus_app == null) {
        return null;
    }
    let focusedAppName = tracker.focus_app.get_name();
    if (contains(excludedApplications, focusedAppName)) {
        return null;
    }
    return WorkspaceManager$1.get_active_workspace().list_windows().find((window) => window.has_focus()) || null;
}
function getFocusWindow() {
    const focus_app = tracker.focus_app;
    if (!focus_app || excludedApplications[focus_app.get_name()]) {
        return null;
    }
    return WorkspaceManager$1.get_active_workspace().list_windows()
        .find(w => w.has_focus());
}
function activeMonitors() {
    return Main$1.layoutManager.monitors;
}
/**
 * Determine if the given monitor is the primary monitor.
 * @param {Object} monitor The given monitor to evaluate.
 * @returns {boolean} True if the given monitor is the primary monitor.
 * */
function isPrimaryMonitor(monitor) {
    return Main$1.layoutManager.primaryMonitor.x == monitor.x && Main$1.layoutManager.primaryMonitor.y == monitor.y;
}
function getWorkAreaByMonitor(monitor) {
    const monitors = activeMonitors();
    for (let monitor_idx = 0; monitor_idx < monitors.length; monitor_idx++) {
        let mon = monitors[monitor_idx];
        if (mon.x == monitor.x && mon.y == monitor.y) {
            return getWorkArea(monitor, monitor_idx);
        }
    }
    return null;
}
/**
 * @deprecated Use {@link workAreaRectByMonitorIndex} instead.
 */
function getWorkAreaByMonitorIdx(monitor_idx) {
    const monitors = activeMonitors();
    let monitor = monitors[monitor_idx];
    return getWorkArea(monitor, monitor_idx);
}
function workAreaRectByMonitorIndex(monitorIndex) {
    const monitor = activeMonitors()[monitorIndex];
    if (!monitor) {
        return null;
    }
    const waLegacy = getWorkArea(monitor, monitorIndex);
    return (new Rect(new XY(waLegacy.x, waLegacy.y), new Size(waLegacy.width, waLegacy.height)));
}
/**
 * @deprecated Use {@link workAreaRectByMonitorIndex} instead.
 */
function getWorkArea(monitor, monitor_idx) {
    const wkspace = WorkspaceManager$1.get_active_workspace();
    const work_area = wkspace.get_work_area_for_monitor(monitor_idx);
    const insets = getMonitorInsets(getMonitorTier(monitor));
    return {
        x: work_area.x + insets.left,
        y: work_area.y + insets.top,
        width: work_area.width - insets.left - insets.right,
        height: work_area.height - insets.top - insets.bottom
    };
}
function bindKeyControls() {
    if (!keyControlBound) {
        bind(key_bindings_tiling);
        if (focusConnect) {
            global.display.disconnect(focusConnect);
        }
        focusConnect = global.display.connect('notify::focus-window', () => globalApp.onFocus());
        if (!gridSettings[SETTINGS_GLOBAL_AUTO_TILING]) {
            bind(key_bindings_auto_tiling);
        }
        if (!gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            bind(key_bindings_presets);
        }
        keyControlBound = true;
    }
}
function unbindKeyControls() {
    if (keyControlBound) {
        unbind(key_bindings_tiling);
        if (focusConnect) {
            log("Disconnect notify:focus-window");
            global.display.disconnect(focusConnect);
            focusConnect = false;
        }
        if (!gridSettings[SETTINGS_GLOBAL_AUTO_TILING]) {
            unbind(key_bindings_auto_tiling);
        }
        if (!gridSettings[SETTINGS_GLOBAL_PRESETS]) {
            unbind(key_bindings_presets);
        }
        if (!gridSettings[SETTINGS_MOVERESIZE_ENABLED]) {
            unbind(keyBindingGlobalResizes);
        }
        keyControlBound = false;
    }
}
function keyCancelTiling() {
    log("Cancel key event");
    globalApp.hideTiling();
}
function keySetTiling() {
    var _a, _b;
    log("keySetTiling");
    if (focusMetaWindow) {
        const monitors = activeMonitors();
        let mind = focusMetaWindow.get_monitor();
        let monitor = monitors[mind];
        let mkey = getMonitorKey(monitor);
        const grid = globalApp.getGrid(monitor);
        log("In grid " + grid);
        (_b = (_a = grid === null || grid === void 0 ? void 0 : grid.elementsDelegate) === null || _a === void 0 ? void 0 : _a.currentElement) === null || _b === void 0 ? void 0 : _b._onButtonPress();
    }
}
function keyChangeTiling() {
    log("keyChangeTiling. Current nbCols " + nbCols + " nbRos " + nbRows);
    const gridSizes = gridSettings[SETTINGS_GRID_SIZES];
    if (gridSizes.length === 0) {
        return;
    }
    const currentIndex = gridSizes.findIndex((size) => size.width === nbCols && size.height === nbRows);
    const newIndex = currentIndex === -1 ? 0 : (currentIndex + 1) % gridSizes.length;
    const newGridSize = gridSizes[newIndex];
    globalApp.setGridSize(newGridSize);
    setInitialSelection();
}
function setInitialSelection() {
    if (!focusMetaWindow) {
        return;
    }
    let mind = focusMetaWindow.get_monitor();
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    const grid = globalApp.getGrid(monitor);
    if (!grid) {
        log("no grid widget available in setInitialSelection");
        return;
    }
    grid.setInitialSelection(focusMetaWindow);
}
function keyMoveResizeEvent(type, key, is_global = false) {
    if (is_global) {
        focusMetaWindow = getFocusApp();
    }
    log("Got key event " + type + " " + key);
    if (!focusMetaWindow) {
        return;
    }
    log("Going on..");
    let mind = focusMetaWindow.get_monitor();
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    const grid = globalApp.getGrid(monitor);
    if (!grid) {
        return;
    }
    if (!grid.moveResize(focusMetaWindow, type, key)) {
        return;
    }
    if (is_global) {
        keySetTiling();
    }
}
/**
 * Resize window to the given preset.
 * @param  {number}  Identifier of the resize preset (1 - 30)
 */
function presetResize(presetName, settingName) {
    // Check if there's a focusable window
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut");
        return;
    }
    // Lets assume window titles are always unique.
    // Note: window roles 'window.get_role()' would offer a unique identifier.
    // Unfortunately role is often set to NULL.
    log("presetResize window title: " + window.get_title());
    // Ensure that the window is not maximized
    reset_window(window);
    // Fetch, parse and validate the given preset.
    // Expected preset format is "XxY x1:y1 x2:y2[,x1:y1 x2:y2]":
    //  - XxY is grid size like 6x8
    //  - x1:y1 is left upper corner tile coordinates in grid tiles, starting from 0
    //  - x2:y2 is right down corner tile coordinates in grid tiles
    //  - a preset can define multiple variants (e.g. "3x2 0:0 0:1,0:0 1:1,0:0 2:1")
    //  - variants can be activated using the same shortcut consecutively
    const presetString = settings.get_string(settingName) || '';
    log("Preset resize " + presetName + "  is " + presetString);
    let tileSpecs = [];
    try {
        tileSpecs = parsePreset(presetString, currentGridSize());
    }
    catch (err) {
        log(`Bad preset ${presetName} ${JSON.stringify(presetString)}: ${err}`);
        return;
    }
    if (tileSpecs.length === 0) {
        return;
    }
    // parse the preset string (grid size, left-upper-corner, right-down-corner)
    let gridSize = tileSpecs[0].gridSize;
    // handle preset variants (if there are any)
    let variantCount = tileSpecs.length;
    if (variantCount > 1) {
        if ((lastResizeInfo.lastCallTime.getTime() + gridSettings[SETTINGS_MAX_TIMEOUT]) > new Date().getTime() &&
            lastResizeInfo.presetName === presetName.toString() &&
            lastResizeInfo.windowTitle == window.get_title()) {
            // within timeout (default: 2s), same preset & same window:
            // increase variant counter, but consider upper boundary
            lastResizeInfo.variantIndex = (lastResizeInfo.variantIndex + 1) % variantCount;
        }
        else {
            // timeout, new preset or new window:
            // update presetState.last_preset and reset variant counter
            lastResizeInfo.variantIndex = 0;
        }
    }
    else {
        lastResizeInfo.variantIndex = 0;
    }
    // retrieve current preset variant
    const tileSpec = tileSpecs[lastResizeInfo.variantIndex];
    // target monitor of current window
    let monitorIdx = window.get_monitor();
    // optionally target monitor where curser is currently
    if (gridSettings[SETTINGS_TARGET_PRESETS_TO_MONITOR_OF_MOUSE]) {
        const [mouseX, mouseY, mask] = global.get_pointer();
        log(`current mouse position ${mouseX}, ${mouseY}`);
        const monitors = activeMonitors();
        log(`monitors: ${JSON.stringify(monitors)}`);
        for (let idx = 0; idx < monitors.length; idx++) {
            let monitor = monitors[idx];
            if (mouseX >= monitor.x && mouseX <= monitor.x + (monitor.x + monitor.width)
                && mouseY >= monitor.y && mouseY <= (monitor.y + monitor.height)) {
                monitorIdx = idx;
            }
        }
    }
    // do the maths to resize the window
    const workArea = workAreaRectByMonitorIndex(monitorIdx);
    if (!workArea) {
        log(`Failed to get active work area for window while performing preset ${presetName} ${JSON.stringify(presetString)}`);
        return;
    }
    // The rectangle already incorporates insets, but it doesn't incorporate
    // window margin.
    const zeroMargins = tileSpec.isFullscreen(workArea) && !getBoolSetting(SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED);
    const marginSize = new Size(zeroMargins ? 0 : gridSettings[SETTINGS_WINDOW_MARGIN], zeroMargins ? 0 : gridSettings[SETTINGS_WINDOW_MARGIN]);
    const rect = tileSpec.toFrameRect(workArea).inset(marginSize);
    moveWindowToRect(window, rect);
    lastResizeInfo.presetName = presetName.toString();
    lastResizeInfo.windowTitle = window.get_title();
    lastResizeInfo.lastCallTime = new Date();
    if (gridSettings[SETTINGS_AUTO_CLOSE_KEYBOARD_SHORTCUT]) {
        globalApp.hideTiling();
    }
}
// Move the window to the next monitor.
function moveWindowToNextMonitor() {
    log("moveWindowToNextMonitor");
    let window = getFocusWindow();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut to move window");
        return;
    }
    reset_window(window);
    const numMonitors = activeMonitors().length;
    if (numMonitors == 0) {
        return;
    }
    const ts = parsePreset("5x5 1:1 3:3")[0];
    const srcMonitorIndex = window.get_monitor();
    const dstMonitorIndex = (srcMonitorIndex + 1) % numMonitors;
    const workArea = workAreaRectByMonitorIndex(dstMonitorIndex);
    if (!workArea) {
        log(`Failed to get active work area for window while moving it to the next monitor.`);
        return;
    }
    const rect = ts.toFrameRect(workArea);
    moveWindowToRect(window, rect);
}
/**
 * Moves a window to the specified region. This may resize the window as well as
 * move its origin.
 */
function moveWindowToRect(window, rect) {
    // The move_frame line is a workaround for a bug affecting gnome terminal
    // windows. See https://github.com/gTile/gTile/issues/176#issue-751198339.
    window.move_frame(true, rect.origin.x, rect.origin.y);
    window.move_resize_frame(true, rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);
}
// Converts global nbCols & nbRows to GridSize object
function currentGridSize() {
    return new GridSize(nbCols, nbRows);
}
/*****************************************************************
  PROTOTYPES
 *****************************************************************/
class TopBar {
    constructor(_title) {
        this._title = _title;
        this.actor = new St.BoxLayout({
            style_class: `${theme}__title-container`
        });
        this._stlabel = new St.Label({
            style_class: `${theme}__title`,
            text: this._title
        });
        this._closebutton = new St.Button({
            style_class: `${theme}__close`,
        });
        this._closebutton.add_style_class_name(`${theme}__close-container`);
        this._connect_id = this._closebutton.connect('button-press-event', this._onButtonPress);
        this.actor.add_child(this._closebutton);
        this.actor.add_child(this._stlabel);
    }
    setTitle(title) {
        this._title = title;
        this._stlabel.text = this._title;
    }
    setApp(app, title) {
        this._title = app.get_name() + " - " + title;
        log("title: " + this._title);
        this._stlabel.text = this._title;
    }
    _onButtonPress() {
        globalApp.hideTiling();
    }
    destroy() {
        this._closebutton.disconnect(this._connect_id);
    }
}
class ToggleSettingsButtonListener {
    constructor() {
        this.actors = [];
    }
    addActor(actor) {
        log("ToggleSettingsButtonListener Connect update-toggle");
        actor.connect('update-toggle', this._updateToggle.bind(this));
        this.actors.push(actor);
    }
    _updateToggle() {
        log("ToggleSettingsButtonListener _updateToggle");
        for (let actor of this.actors) {
            actor._update();
        }
    }
}
function ToggleSettingsButton(text, property) {
    this._init(text, property);
}
ToggleSettingsButton.prototype = {
    _init: function (text, property) {
        this.text = text;
        this.actor = new St.Button({
            style_class: `${theme}__action-button`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.label = new St.Label({
            style_class: `${theme}__action-label`,
            reactive: true,
            can_focus: true,
            track_hover: true,
            text: this.text
        });
        this.icon = new St.BoxLayout({
            style_class: `${theme}__action-button--${this.text}`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.actor.add_actor(this.icon);
        this.property = property;
        this._update();
        log("ToggleSettingsButton Connect button-press-event");
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        log("ToggleSettingsButton Connect update-toggle");
        this.connect('update-toggle', this._update.bind(this));
    },
    _update: function () {
        log("ToggleSettingsButton _update event " + this.property);
        if (gridSettings[this.property]) {
            this.actor.add_style_pseudo_class('activate');
        }
        else {
            this.actor.remove_style_pseudo_class('activate');
        }
    },
    _onButtonPress: function () {
        gridSettings[this.property] = !gridSettings[this.property];
        log("ToggleSettingsButton _onButtonPress " + this.property + ": " + gridSettings[this.property] + ", emitting signal update-toggle");
        this.emit('update-toggle');
    }
};
Signals.addSignalMethods(ToggleSettingsButton.prototype);
const ACTION_CLASSES = {
    BUTTON: "__action-button",
    MAIN_AND_LIST: "__action-button--main-and-list",
    TWO_LIST: "__action-button--two-list",
};
class ActionButton {
    constructor(grid, classname) {
        this.grid = grid;
        this.grid = grid;
        this.actor = new St.Button({
            style_class: `${getTheme()}${ACTION_CLASSES.BUTTON}`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.icon = new St.BoxLayout({ style_class: classname, reactive: true, can_focus: true, track_hover: true });
        this.actor.add_actor(this.icon);
        log("ActionButton Connect button-press-event");
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
    }
    _onButtonPress() {
        log("ActionButton _onButtonPress Emitting signal button-press-event");
        this.emit('button-press-event');
    }
    /** Functions replaced by Signals.addSignalMethods. */
    connect(eventName, handler) { return 0; }
    disconnect(id) { }
    emit(name, ...args) { }
}
Signals.addSignalMethods(ActionButton.prototype);
class AutoTileMainAndList extends ActionButton {
    constructor(grid) {
        const theme = `${getTheme()}${ACTION_CLASSES.MAIN_AND_LIST}`;
        super(grid, theme);
        this.classname = theme;
        log("AutoTileMainAndList connect button-press-event");
        this.connect('button-press-event', () => this._onButtonPress());
    }
    _onButtonPress() {
        AutoTileMain();
        log("AutoTileMainAndList _onButtonPress Emitting signal resize-done");
        this.emit('resize-done');
    }
}
Signals.addSignalMethods(AutoTileMainAndList.prototype);
function AutoTileMain() {
    let preset = "AutoTileMain";
    log("AutoTileMain");
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut AutoTileMain");
        return;
    }
    reset_window(window);
    let mind = window.get_monitor();
    let work_area = getWorkAreaByMonitorIdx(mind);
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    let workArea = getWorkAreaByMonitor(monitor);
    let notFocusedwindows = getNotFocusedWindowsOfMonitor(monitor);
    if (Object.keys(notFocusedwindows).length === 0) {
        moveResizeWindowWithMargins(focusMetaWindow, workArea.x, workArea.y, workArea.width, workArea.height);
        return;
    }
    log("SETTINGS_MAIN_WINDOW_SIZES:" + mainWindowSizes);
    let ps_variants = mainWindowSizes;
    log("Main window sizes: " + ps_variants);
    // handle preset variants (if there are any)
    let variantCount = ps_variants.length;
    if (variantCount > 1) {
        if ((lastResizeInfo.lastCallTime.getTime() + gridSettings[SETTINGS_MAX_TIMEOUT]) > new Date().getTime() &&
            lastResizeInfo.presetName === preset.toString() &&
            lastResizeInfo.windowTitle == window.get_title()) {
            // within timeout (default: 2s), same preset & same window:
            // increase variant counter, but consider upper boundary
            lastResizeInfo.variantIndex = (lastResizeInfo.variantIndex + 1) % variantCount;
        }
        else {
            // timeout, new preset or new window:
            // update presetState.last_preset and reset variant counter
            lastResizeInfo.variantIndex = 0;
        }
    }
    else {
        lastResizeInfo.variantIndex = 0;
    }
    let mainRatio = ps_variants[lastResizeInfo.variantIndex];
    let mainWidth = workArea.width * mainRatio;
    let minorWidth = workArea.width - mainWidth;
    moveResizeWindowWithMargins(focusMetaWindow, workArea.x, workArea.y, mainWidth, workArea.height);
    let winHeight = workArea.height / notFocusedwindows.length;
    let countWin = 0;
    log("AutoTileMain MonitorHeight: " + monitor.height + ":" + notFocusedwindows.length);
    for (let windowIdx in notFocusedwindows) {
        let metaWindow = notFocusedwindows[windowIdx].meta_window;
        let newOffset = workArea.y + (countWin * winHeight);
        reset_window(metaWindow);
        moveResizeWindowWithMargins(metaWindow, workArea.x + mainWidth, newOffset, minorWidth, winHeight);
        countWin++;
    }
    log("AutoTileMain done");
    lastResizeInfo.presetName = preset;
    lastResizeInfo.windowTitle = window.get_title();
    lastResizeInfo.lastCallTime = new Date();
    log("Resize preset last call: " + lastResizeInfo.lastCallTime);
}
class AutoTileTwoList extends ActionButton {
    constructor(grid) {
        const theme = `${getTheme()}${ACTION_CLASSES.TWO_LIST}`;
        super(grid, theme);
        this.classname = theme;
        log("AutoTileTwoList connect button-press-event");
        this.connect('button-press-event', () => this._onButtonPress());
    }
    _onButtonPress() {
        log("AutotileTwoList");
        autoTileNCols(2);
        log("AutoTileTwoList _onButtonPress Emitting signal resize-done");
        this.emit('resize-done');
        log("Autotile2 done");
    }
}
Signals.addSignalMethods(AutoTileTwoList.prototype);
function autoTileNCols(cols) {
    log("AutoTileNCols " + cols);
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut AutoTileNCols");
        return;
    }
    reset_window(window);
    let mind = window.get_monitor();
    let work_area = getWorkAreaByMonitorIdx(mind);
    const monitors = activeMonitors();
    let monitor = monitors[mind];
    let workArea = getWorkAreaByMonitor(monitor);
    if (!workArea) {
        return;
    }
    let windows = getNotFocusedWindowsOfMonitor(monitor);
    let nbWindowOnEachSide = Math.ceil((windows.length + (globalApp.isGridShowing() ? 1 : 0)) / cols);
    let winHeight = workArea.height / nbWindowOnEachSide;
    let countWin = 0;
    if (globalApp.isGridShowing()) {
        moveResizeWindowWithMargins(window, workArea.x + countWin % cols * workArea.width / cols, workArea.y + (Math.floor(countWin / cols) * winHeight), workArea.width / cols, winHeight);
        countWin++;
    }
    // todo make function
    for (let windowIdx in windows) {
        let metaWindow = windows[windowIdx].meta_window;
        reset_window(metaWindow);
        moveResizeWindowWithMargins(metaWindow, workArea.x + countWin % cols * workArea.width / cols, workArea.y + (Math.floor(countWin / cols) * winHeight), workArea.width / cols, winHeight);
        countWin++;
    }
    log("AutoTileNCols done");
}
function snapToNeighborsBind() {
    log("SnapToNeighbors keybind invoked");
    let window = getFocusApp();
    if (!window) {
        log("No focused window - ignoring keyboard shortcut SnapToNeighbors");
        return;
    }
    snapToNeighbors(window);
}
/**
 * GridSettingsButton is a GUI button that displays text like "8x8" and changes
 * the main grid size used for GUI elements and some presets.
 */
class GridSettingsButton {
    constructor(gridSize, active) {
        this.text = gridSize.toString();
        this.cols = gridSize.width;
        this.rows = gridSize.height;
        this.active = active;
        this.actor = new St.Button({
            style_class: `${theme}__preset-button`,
            reactive: true,
            can_focus: true,
            track_hover: true,
            label: this.text,
        });
        if (this.active) {
            this.activate();
        }
        this.actor.connect('button-press-event', () => this._onButtonPress());
    }
    deactivate() {
        log("Deactivating GridSettingsButton ${cols}:${rows}");
        this.active = false;
        this.actor.remove_style_pseudo_class('activate');
    }
    activate() {
        log("Activating GridSettingsButton ${cols}:${rows}");
        this.active = true;
        this.actor.add_style_pseudo_class('activate');
    }
    _onButtonPress() {
        nbCols = this.cols;
        nbRows = this.rows;
        globalApp.refreshGrids();
    }
}
const GRID_WIDGET_BORDER_SIZE = 1;
class Grid {
    constructor(gridWidget, monitor_idx, monitor, title, cols, rows) {
        this.gridWidget = gridWidget;
        this.tableWidth = 320;
        this.borderwidth = 2;
        this.elements = [];
        this.moveResizeImpl = null;
        let workArea = getWorkArea(monitor, monitor_idx);
        this.tableHeight = (this.tableWidth / workArea.width) * workArea.height;
        this.actor = new St.BoxLayout({
            vertical: true,
            style_class: theme,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        log(`created actor for monitor ${monitor_idx}: ${this.actor} with cols=${cols}, rows=${rows}`);
        log("Grid connect enter-event leave-envent BEGIN");
        // TODO: disconnect these events on teardown.
        this.actor.connect('enter-event', () => this._onMouseEnter());
        this.actor.connect('leave-event', () => this._onMouseLeave());
        log("Grid connect enter-event leave-envent FINISH");
        this.animation_time = gridSettings[SETTINGS_ANIMATION] ? 0.3 : 0;
        this.topbar = new TopBar(title);
        this.bottombarContainer = new St.Bin({
            style_class: `${theme}__preset-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.bottomBarTableLayout = new Clutter.GridLayout();
        this.bottombar = new St.Widget({
            style_class: `${theme}__preset`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            width: this.tableWidth - 20,
            height: 36,
            layout_manager: this.bottomBarTableLayout
        });
        this.bottomBarTableLayout.set_row_homogeneous(true);
        this.bottomBarTableLayout.set_column_homogeneous(true);
        this.bottombarContainer.add_actor(this.bottombar);
        this.veryBottomBarContainer = new St.Bin({
            style_class: `${theme}__action-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.veryBottomBarTableLayout = new Clutter.GridLayout();
        this.veryBottomBar = new St.Widget({
            style_class: `${theme}__action`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            width: this.tableWidth - 20,
            height: 36,
            layout_manager: this.veryBottomBarTableLayout,
        });
        this.bottomBarTableLayout.set_row_homogeneous(true);
        this.veryBottomBarTableLayout.set_column_homogeneous(true);
        this.veryBottomBarContainer.add_actor(this.veryBottomBar);
        let rowNum = 0;
        let colNum = 0;
        let maxPerRow = 4;
        this.rows = rows;
        this.cols = cols;
        this.gridSettingsButtons = [];
        for (let gridSize of gridSettings[SETTINGS_GRID_SIZES]) {
            if (colNum >= maxPerRow) {
                colNum = 0;
                rowNum += 1;
            }
            const isActiveGrid = this.cols == gridSize.width && this.rows == gridSize.height;
            const button = new GridSettingsButton(gridSize, isActiveGrid);
            this.bottomBarTableLayout.attach(button.actor, colNum, rowNum, 1, 1);
            button.actor.connect('notify::hover', () => this._onSettingsButton());
            colNum++;
            this.gridSettingsButtons.push(button);
        }
        this.bottombar.height *= (rowNum + 1);
        this.tableContainer = new St.Bin({
            style_class: `${theme}__tile-container`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.tableLayoutManager = new Clutter.GridLayout();
        this.table = new St.Widget({
            style_class: `${theme}__tile-table`,
            can_focus: true,
            track_hover: true,
            reactive: true,
            height: this.tableHeight,
            width: this.tableWidth - GRID_WIDGET_BORDER_SIZE * 2,
            layout_manager: this.tableLayoutManager,
        });
        this.tableLayoutManager.set_row_homogeneous(true);
        this.tableLayoutManager.set_column_homogeneous(true);
        this.tableContainer.add_actor(this.table);
        this.actor.add_child(this.topbar.actor);
        this.actor.add_child(this.tableContainer);
        this.actor.add_child(this.bottombarContainer);
        this.actor.add_child(this.veryBottomBarContainer);
        this.monitor = monitor;
        this.monitor_idx = monitor_idx;
        this.title = title;
        this.isEntered = false;
        const toggleSettingListener = new ToggleSettingsButtonListener();
        let toggle = new ToggleSettingsButton("animation", SETTINGS_ANIMATION);
        this.veryBottomBarTableLayout.attach(toggle.actor, 0, 0, 1, 1);
        toggleSettingListener.addActor(toggle);
        toggle = new ToggleSettingsButton("auto-close", SETTINGS_AUTO_CLOSE);
        this.veryBottomBarTableLayout.attach(toggle.actor, 1, 0, 1, 1);
        toggleSettingListener.addActor(toggle);
        let action = new AutoTileMainAndList(this);
        this.veryBottomBarTableLayout.attach(action.actor, 2, 0, 1, 1);
        action.connect('resize-done', this._onResize.bind(this));
        action = new AutoTileTwoList(this);
        this.veryBottomBarTableLayout.attach(action.actor, 3, 0, 1, 1);
        action.connect('resize-done', this._onResize.bind(this));
        this.x = 0;
        this.y = 0;
        this.interceptHide = false;
        this._displayElements();
        this.normalScaleY = this.actor.scale_y;
        this.normalScaleX = this.actor.scale_x;
    }
    state() {
        return `grid with actor ${this.actor}`;
    }
    _displayElements() {
        if (this.monitor === null) {
            return;
        }
        log("Grid _displayElements " + this.cols + ":" + this.rows);
        this.elements = new Array();
        let width = (this.tableWidth / this.cols); // - 2*this.borderwidth;
        let height = (this.tableHeight / this.rows); // - 2*this.borderwidth;
        const delegate = new GridElementDelegate(this, this.gridWidget);
        this.elementsDelegate = delegate;
        this.elementsDelegate.connect('resize-done', (actor, event) => this._onResize());
        for (let r = 0; r < this.rows; r++) {
            for (let c = 0; c < this.cols; c++) {
                if (c == 0) {
                    this.elements[r] = new Array();
                }
                const element = new GridElement(this.elementsDelegate, this.monitor, width, height, c, r);
                this.elements[r][c] = element;
                this.tableLayoutManager.attach(element.actor, c, r, 1, 1);
                element.show();
            }
        }
    }
    getElement(row, col) {
        return this.elements[row][col] || null;
    }
    forceGridElementDelegate(x, y, w, h) {
        var _a;
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.forceArea(this.elements[y][x], this.elements[h][w]);
    }
    refresh() {
        log("Grid.refresh from " + this.cols + ":" + this.rows + " to " + nbCols + ":" + nbRows);
        this.elementsDelegate._resetGrid();
        for (let r = 0; r < this.rows; r++) {
            for (let c = 0; c < this.cols; c++) {
                this.elements[r][c]._disconnect();
            }
        }
        this.table.destroy_all_children();
        this.cols = nbCols;
        this.rows = nbRows;
        this._displayElements();
        this._clearMoveResizeState();
        this._updateGridSizeButtons();
    }
    set_position(x, y) {
        this.x = x;
        this.y = y;
        this.actor.set_position(x, y);
    }
    show() {
        this.interceptHide = true;
        this.elementsDelegate.reset();
        let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0;
        Main$1.uiGroup.set_child_above_sibling(this.actor, null);
        Main$1.layoutManager.removeChrome(this.actor);
        Main$1.layoutManager.addChrome(this.actor);
        //this.actor.y = 0 ;
        if (time > 0) {
            this.actor.scale_y = 0;
            this.actor.scale_x = 0;
            this.actor.ease({
                time: this.animation_time,
                opacity: 255,
                visible: true,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                scale_x: this.normalScaleX,
                scale_y: this.normalScaleY,
                onComplete: this._onShowComplete
            });
        }
        else {
            this.actor.scale_x = this.normalScaleX;
            this.actor.scale_y = this.normalScaleY;
            this.actor.opacity = 255;
            this.actor.visible = true;
        }
        this.interceptHide = false;
    }
    hide(immediate) {
        var _a;
        log(`Grid hide immediate=${immediate}`);
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
        let time = (gridSettings[SETTINGS_ANIMATION]) ? 0.3 : 0;
        if (!immediate && time > 0) {
            this.actor.ease({
                time: this.animation_time,
                opacity: 0,
                visible: false,
                scale_x: 0,
                scale_y: 0,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD,
                onComplete: this._onHideComplete
            });
        }
        else {
            this.actor.opacity = 0;
            this.actor.visible = false;
            //this.actor.y = 0;
            this.actor.scale_x = 0;
            this.actor.scale_y = 0;
        }
        this._clearMoveResizeState();
    }
    setInitialSelection(focusMetaWindow) {
        var _a, _b, _c, _d;
        // The window being focused is part of the monitor that this Grid is
        // responsible for manipulating.
        if (focusMetaWindow.get_monitor() !== this.monitor_idx) {
            return;
        }
        if (!this.monitor) {
            return;
        }
        const monitor = this.monitor;
        const workArea = getWorkArea(this.monitor, this.monitor_idx);
        let wx = focusMetaWindow.get_frame_rect().x;
        let wy = focusMetaWindow.get_frame_rect().y;
        let wwidth = focusMetaWindow.get_frame_rect().width;
        let wheight = focusMetaWindow.get_frame_rect().height;
        const grid = this;
        const delegate = grid.elementsDelegate;
        log("Set initial selection");
        log("Focus window position x " + wx + " y " + wy + " width " + wwidth + " height " + wheight);
        log("Focus monitor position x " + monitor.x + " y " + monitor.y + " width " + monitor.width + " height " + monitor.height);
        log("Workarea position x " + workArea.x + " y " + workArea.y + " width " + workArea.width + " height " + workArea.height);
        let wax = Math.max(wx - workArea.x, 0);
        let way = Math.max(wy - workArea.y, 0);
        let grid_element_width = workArea.width / nbCols;
        let grid_element_height = workArea.height / nbRows;
        log("width " + grid_element_width + " height " + grid_element_height);
        let lux = Math.min(Math.max(Math.round(wax / grid_element_width), 0), nbCols - 1);
        log("wx " + (wx - workArea.x) + " el_width " + grid_element_width + " max " + (nbCols - 1) + " res " + lux);
        let luy = Math.min(Math.max(Math.round(way / grid_element_height), 0), grid.rows - 1);
        log("wy " + (wy - workArea.y) + " el_height " + grid_element_height + " max " + (nbRows - 1) + " res " + luy);
        let rdx = Math.min(Math.max(Math.round((wax + wwidth) / grid_element_width) - 1, lux), grid.cols - 1);
        log("wx + wwidth " + (wx + wwidth - workArea.x - 1) + " el_width " + grid_element_width + " max " + (nbCols - 1) + " res " + rdx);
        let rdy = Math.min(Math.max(Math.round((way + wheight) / grid_element_height) - 1, luy), grid.rows - 1);
        log("wy + wheight " + (wy + wheight - workArea.y - 1) + " el_height " + grid_element_height + " max " + (nbRows - 1) + " res " + rdy);
        log("Initial tile selection is " + lux + ":" + luy + " - " + rdx + ":" + rdy);
        grid.forceGridElementDelegate(lux, luy, rdx, rdy);
        grid.elements[luy][lux]._onButtonPress();
        grid.elements[rdy][rdx]._onHoverChanged();
        const cX = (_a = delegate === null || delegate === void 0 ? void 0 : delegate.currentElement) === null || _a === void 0 ? void 0 : _a.coordx;
        const cY = (_b = delegate === null || delegate === void 0 ? void 0 : delegate.currentElement) === null || _b === void 0 ? void 0 : _b.coordy;
        const fX = (_c = delegate === null || delegate === void 0 ? void 0 : delegate.first) === null || _c === void 0 ? void 0 : _c.coordx;
        const fY = (_d = delegate === null || delegate === void 0 ? void 0 : delegate.first) === null || _d === void 0 ? void 0 : _d.coordy;
        log("After initial selection first fX " + fX + " fY " + fY + " current cX " + cX + " cY " + cY);
    }
    moveResize(window, type, key) {
        const delegate = this.elementsDelegate;
        if (!delegate) {
            return false;
        }
        if (!delegate.currentElement) {
            log("Key event while no mouse activation - set current and second element");
            this.setInitialSelection(window);
        }
        if (!delegate.currentElement) {
            log("bug: currentElement must be set in moveResize");
            return;
        }
        if (!delegate.first) {
            log("currentElement is there but no first yet");
            delegate.currentElement._onButtonPress();
        }
        if (!delegate.first) {
            log("bug: first must be set in moveResize");
            return;
        }
        if (this.moveResizeImpl == null) {
            this.moveResizeImpl = new MoveResizeImpl(delegate.currentElement, delegate.first);
        }
        return this.moveResizeImpl.moveResize(this, delegate, type, key);
    }
    _clearMoveResizeState() {
        log("Clear moveResize state");
        this.moveResizeImpl = null;
    }
    _onHideComplete() {
    }
    _onShowComplete() {
    }
    _onResize() {
        log("resize-done");
        globalApp.updateRegions();
        if (gridSettings[SETTINGS_AUTO_CLOSE]) {
            log("Emitting hide-tiling");
            this.emit('hide-tiling');
        }
        this._clearMoveResizeState();
    }
    _onMouseEnter() {
        var _a;
        log("Grid: onMouseEnter");
        if (!this.isEntered) {
            (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            this.isEntered = true;
        }
    }
    _onMouseLeave() {
        var _a;
        log("Grid: onMouseLeave");
        let [x, y, mask] = global.get_pointer();
        if (this.elementsDelegate && (x <= this.actor.x || x >= (this.actor.x + this.actor.width)) || (y <= this.actor.y || y >= (this.actor.y + this.actor.height))) {
            this.isEntered = false;
            (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
            globalApp.refreshGrids();
        }
    }
    _onSettingsButton() {
        var _a;
        log("_onSettingsButton");
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a.reset();
    }
    _destroy() {
        var _a;
        log("Grid _destroy");
        for (let r in this.elements) {
            for (let c in this.elements[r]) {
                this.elements[r][c]._destroy();
            }
        }
        (_a = this.elementsDelegate) === null || _a === void 0 ? void 0 : _a._destroy();
        this.topbar.destroy();
        this.monitor = null;
        this.rows = 0;
        this.title = "";
        this.cols = 0;
        log("Disconnect hide-tiling");
        this.disconnect(this.connectHideTiling);
    }
    _updateGridSizeButtons() {
        for (let button of this.gridSettingsButtons) {
            if (this.cols == button.cols && this.rows == button.rows) {
                button.activate();
            }
            else {
                button.deactivate();
            }
        }
    }
    // Methods replaced by Signals.addSignalMethods.
    connect(name, callback) { return -1; }
    disconnect(id) { }
    emit(name, ...args) { }
}
class MoveResizeImpl {
    constructor(current, first) {
        const cX = current.coordx;
        const cY = current.coordy;
        const fX = first.coordx;
        const fY = first.coordy;
        this.vW = Math.abs(cX - fX) + 1;
        this.vH = Math.abs(cY - fY) + 1;
        this.vX = Math.min(cX, fX);
        this.vY = Math.min(cY, fY);
    }
    moveResize(grid, delegate, type, key) {
        var _a;
        const cols = grid.cols;
        const rows = grid.rows;
        log(`Before move/resize vX = ${this.vX}, vY = ${this.vY}, vW = ${this.vW}, vH = ${this.vH}`);
        if (type === 'move') {
            switch (key) {
                case 'right':
                    if (this.vX < cols - 1) {
                        this.vX += 1;
                    }
                    break;
                case 'left':
                    if (0 < this.vX + this.vW - 1) {
                        this.vX -= 1;
                    }
                    break;
                case 'up':
                    if (0 < this.vY + this.vH - 1) {
                        this.vY -= 1;
                    }
                    break;
                case 'down':
                    if (this.vY < rows - 1) {
                        this.vY += 1;
                    }
                    break;
            }
        }
        else if (type == "resize") {
            switch (key) {
                case 'right':
                    if (this.vW < cols) {
                        this.vW += 1;
                    }
                    break;
                case 'left':
                    if (this.vW > 1) {
                        this.vW -= 1;
                    }
                    break;
                case 'up':
                    if (this.vH > 1) {
                        this.vH -= 1;
                    }
                    break;
                case 'down':
                    if (this.vH < rows) {
                        this.vH += 1;
                    }
                    break;
            }
        }
        else if (type == "contract") {
            switch (key) {
                case 'left':
                    // Contract left edge of current window right one column
                    if (this.vX < cols - 1 && this.vW > 1) {
                        this.vX += 1;
                        this.vW -= 1;
                    }
                    break;
                case 'right':
                    // Contract right edge of current window left one column
                    if (0 < this.vX + this.vW - 1 && this.vW > 1) {
                        this.vW -= 1;
                    }
                    break;
                case 'top':
                    // Contract top edge of current window down one row
                    if (this.vY < rows - 1 && this.vH > 1) {
                        this.vY += 1;
                        this.vH -= 1;
                    }
                    break;
                case 'bottom':
                    // Contract bottom edge of current window up one row
                    if (0 < this.vY + this.vH - 1 && this.vH > 1) {
                        this.vH -= 1;
                    }
                    break;
            }
        }
        else if (type == "expand") {
            switch (key) {
                case 'right':
                    if (this.vW < cols) {
                        this.vW += 1;
                    }
                    break;
                case 'left':
                    if (this.vW < cols) {
                        this.vW += 1;
                        this.vX -= 1;
                    }
                    break;
                case 'top':
                    if (this.vH < rows) {
                        this.vH += 1;
                        this.vY -= 1;
                    }
                    break;
                case 'bottom':
                    if (this.vH < rows) {
                        this.vH += 1;
                    }
                    break;
            }
        }
        var tFX = Math.max(0, this.vX);
        var tCX = Math.min(this.vX + this.vW - 1, cols - 1);
        var tFY = Math.max(0, this.vY);
        var tCY = Math.min(this.vY + this.vH - 1, rows - 1);
        delegate.first = grid.getElement(tFY, tFX);
        (_a = grid.getElement(tCY, tCX)) === null || _a === void 0 ? void 0 : _a._onHoverChanged();
        return true;
    }
}
Signals.addSignalMethods(Grid.prototype);
class GridElementDelegate {
    constructor(grid, gridWidget) {
        this.grid = grid;
        this.gridWidget = gridWidget;
        // The first element clicked in the rectangular selection.
        this.first = null;
        //
        this.currentElement = null;
        // Elements that are in a highlighted state in the UI. The highlighting
        // happens when the grid rectangle selcted includes a particular grid
        // element.
        this.activatedElements = [];
        this.activated = false;
    }
    allSelected() {
        return (this.activatedElements.length === (nbCols * nbRows));
    }
    onButtonPress(gridElement) {
        log("GridElementDelegate _onButtonPress " + gridElement.coordx + ":" + gridElement.coordy);
        //this._logActiveActors("GridElementDelegate _onButtonPress active actors");
        if (!this.currentElement) {
            this.currentElement = gridElement;
        }
        if (!this.activated) {
            log("GridElementDelegate first activation");
            this.activated = true;
            gridElement.active = true;
            this.activatedElements = [gridElement];
            this.first = gridElement;
            return;
        }
        log("GridElementDelegate resize");
        //Check this.activatedActors if equals to nbCols * nbRows
        //before doing anything with the window it must be unmaximized
        //if so move the window then maximize instead of change size
        //if not move the window and change size
        if (!focusMetaWindow) {
            return;
        }
        reset_window(focusMetaWindow);
        if (this.first) {
            const computedSize = this._computeAreaPositionSize(this.first, gridElement);
            if (!computedSize) {
                return;
            }
            const [areaX, areaY, areaWidth, areaHeight] = computedSize;
            if (this.allSelected() && gridSettings[SETTINGS_WINDOW_MARGIN_FULLSCREEN_ENABLED] === false) {
                move_maximize_window(focusMetaWindow, areaX, areaY);
            }
            else {
                moveResizeWindowWithMargins(focusMetaWindow, areaX, areaY, areaWidth, areaHeight);
            }
        }
        this._resizeDone();
    }
    _resizeDone() {
        log("resizeDone, emitting signal resize-done");
        this.emit('resize-done');
    }
    reset() {
        this._resetGrid();
        this.activated = false;
        this.first = null;
        this.currentElement = null;
    }
    _resetGrid() {
        this._hideArea();
        if (this.currentElement) {
            this.currentElement._deactivate();
        }
        for (let act of this.activatedElements) {
            act._deactivate();
        }
        this.activatedElements = new Array();
    }
    _getVarFromGridElement(fromGridElement, toGridElement) {
        let minX = Math.min(fromGridElement.coordx, toGridElement.coordx);
        let maxX = Math.max(fromGridElement.coordx, toGridElement.coordx);
        let minY = Math.min(fromGridElement.coordy, toGridElement.coordy);
        let maxY = Math.max(fromGridElement.coordy, toGridElement.coordy);
        return [minX, maxX, minY, maxY];
    }
    refreshGrid(fromGridElement, toGridElement) {
        this._resetGrid();
        let [minX, maxX, minY, maxY] = this._getVarFromGridElement(fromGridElement, toGridElement);
        if (!fromGridElement.monitor) {
            return;
        }
        for (let r = minY; r <= maxY; r++) {
            for (let c = minX; c <= maxX; c++) {
                const highlightedElement = this.grid.getElement(r, c);
                if (!highlightedElement) {
                    log(`bug: GridElementDelegate refreshGrid is out of sync with Grid: trying to highlight element (row=${r}, col=${c}) and getElement() returned null.`);
                    continue;
                }
                highlightedElement._activate();
                this.activatedElements.push(highlightedElement);
            }
        }
        this._displayArea(fromGridElement, toGridElement);
    }
    _computeAreaPositionSize(fromGridElement, toGridElement) {
        let [minX, maxX, minY, maxY] = this._getVarFromGridElement(fromGridElement, toGridElement);
        let monitor = fromGridElement.monitor;
        const workArea = getWorkAreaByMonitor(monitor);
        if (!workArea) {
            return null;
        }
        let areaWidth = Math.round((workArea.width / nbCols) * ((maxX - minX) + 1));
        let areaHeight = Math.round((workArea.height / nbRows) * ((maxY - minY) + 1));
        let areaX = workArea.x + Math.round((minX * (workArea.width / nbCols)));
        let areaY = workArea.y + Math.round((minY * (workArea.height / nbRows)));
        return [areaX, areaY, areaWidth, areaHeight];
    }
    forceArea(fromGridElement, toGridElement) {
        const computedSize = this._computeAreaPositionSize(fromGridElement, toGridElement);
        if (!computedSize) {
            return;
        }
        const [areaX, areaY, areaWidth, areaHeight] = computedSize;
        this.gridWidget.width = areaWidth;
        this.gridWidget.height = areaHeight;
        this.gridWidget.x = areaX;
        this.gridWidget.y = areaY;
    }
    _displayArea(fromGridElement, toGridElement) {
        const computedSize = this._computeAreaPositionSize(fromGridElement, toGridElement);
        if (!computedSize) {
            return;
        }
        const [areaX, areaY, areaWidth, areaHeight] = computedSize;
        this.gridWidget.add_style_pseudo_class('activate');
        if (gridSettings[SETTINGS_ANIMATION]) {
            this.gridWidget.ease({
                time: 0.2,
                x: areaX,
                y: areaY,
                width: areaWidth,
                height: areaHeight,
                transition: Clutter.AnimationMode.EASE_OUT_QUAD
            });
        }
        else {
            this.gridWidget.width = areaWidth;
            this.gridWidget.height = areaHeight;
            this.gridWidget.x = areaX;
            this.gridWidget.y = areaY;
        }
    }
    _hideArea() {
        this.gridWidget.remove_style_pseudo_class('activate');
    }
    _onHoverChanged(gridElement) {
        log("GridElementDelegate _onHoverChange " + gridElement.coordx + ":" + gridElement.coordy);
        if (this.activated) {
            log("GridElementDelegate _onHoverChange/not active: " + gridElement.coordx + ":" + gridElement.coordy);
            this.refreshGrid(this.first, gridElement);
            this.currentElement = gridElement;
        }
        else if (!this.currentElement || gridElement.id != this.currentElement.id) {
            log("GridElementDelegate _onHoverChange/active: " + gridElement.coordx + ":" + gridElement.coordy);
            if (this.currentElement) {
                this.currentElement._deactivate();
            }
            this.currentElement = gridElement;
            this.currentElement._activate();
            this._displayArea(gridElement, gridElement);
        }
        else {
            log("GridElementDelegate _onHoverChange/else: " + gridElement.coordx + ":" + gridElement.coordy);
        }
    }
    _destroy() {
        this.activated = false;
        this.first = null;
        this.currentElement = null;
        this.activatedElements = [];
    }
    // Methods replaced by Signals.addSignalMethods.
    connect(name, callback) { return -1; }
    disconnect(id) { }
    emit(name, ...args) { }
}
Signals.addSignalMethods(GridElementDelegate.prototype);
class GridElement {
    constructor(delegate, monitor, width, height, coordx, coordy) {
        this.delegate = delegate;
        this.monitor = monitor;
        this.width = width;
        this.height = height;
        this.coordx = coordx;
        this.coordy = coordy;
        this.actor = new St.Button({
            style_class: `${theme}__tile-table-item`,
            reactive: true,
            can_focus: true,
            track_hover: true
        });
        this.actor.visible = false;
        this.actor.opacity = 0;
        this.id = getMonitorKey(monitor) + "-" + coordx + ":" + coordy;
        this.actor.connect('button-press-event', this._onButtonPress.bind(this));
        this.hoverConnect = this.actor.connect('notify::hover', () => this._onHoverChanged());
        this.active = false;
    }
    state() {
        return `id: ${this.id}; visible: ${this.actor.visible}`;
    }
    show() {
        this.actor.opacity = 255;
        this.actor.visible = true;
    }
    hide() {
        this.actor.opacity = 0;
        this.actor.visible = false;
    }
    _onButtonPress() {
        this.delegate.onButtonPress(this);
    }
    _onHoverChanged() {
        this.delegate._onHoverChanged(this);
    }
    _activate() {
        if (!this.active) {
            this.actor.add_style_pseudo_class('activate');
            this.active = true;
        }
    }
    _deactivate() {
        if (this.active) {
            this.actor.remove_style_pseudo_class('activate');
            this.active = false;
        }
    }
    // This logic should probably go into disable().
    // _clean() {
    //     Main.uiGroup.remove_actor(this.gridWidget);
    // }
    _disconnect() {
        this.actor.disconnect(this.hoverConnect);
    }
    _destroy() {
        this.active = false;
    }
}
function getTheme() {
    let themeName = settings.get_string(SETTINGS_THEME) || 'Default';
    themeName = themeName.toLowerCase().replace(/[^A-Za-z0-9]/g, '-');
    const theme = `gtile-${themeName}`;
    log("Theme changed to the" + theme);
    return theme;
}

registerExtension(function () { }, enable, disable);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
