"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.canExpand = canExpand;
exports.getDefaultRegistry = getDefaultRegistry;
exports.getSchemaType = getSchemaType;
exports.getWidget = getWidget;
exports.hasWidget = hasWidget;
exports.getDefaultFormState = getDefaultFormState;
exports.mergeDefaultsWithFormData = mergeDefaultsWithFormData;
exports.getUiOptions = getUiOptions;
exports.getSubmitButtonOptions = getSubmitButtonOptions;
exports.getDisplayLabel = getDisplayLabel;
exports.isObject = isObject;
exports.mergeObjects = mergeObjects;
exports.asNumber = asNumber;
exports.orderProperties = orderProperties;
exports.isConstant = isConstant;
exports.toConstant = toConstant;
exports.isSelect = isSelect;
exports.isMultiSelect = isMultiSelect;
exports.isFilesArray = isFilesArray;
exports.isFixedItems = isFixedItems;
exports.isCustomWidget = isCustomWidget;
exports.allowAdditionalItems = allowAdditionalItems;
exports.optionsList = optionsList;
exports.findSchemaDefinition = findSchemaDefinition;
exports.stubExistingAdditionalProperties = stubExistingAdditionalProperties;
exports.resolveSchema = resolveSchema;
exports.retrieveSchema = retrieveSchema;
exports.mergeSchemas = mergeSchemas;
exports.deepEquals = deepEquals;
exports.shouldRender = shouldRender;
exports.toIdSchema = toIdSchema;
exports.toPathSchema = toPathSchema;
exports.parseDateString = parseDateString;
exports.toDateString = toDateString;
exports.utcToLocal = utcToLocal;
exports.localToUTC = localToUTC;
exports.pad = pad;
exports.dataURItoBlob = dataURItoBlob;
exports.rangeSpec = rangeSpec;
exports.getMatchingOption = getMatchingOption;
exports.schemaRequiresTrueValue = schemaRequiresTrueValue;
exports.guessType = exports.ADDITIONAL_PROPERTY_FLAG = void 0;

var _react = _interopRequireDefault(require("react"));

var ReactIs = _interopRequireWildcard(require("react-is"));

var _jsonSchemaMergeAllof = _interopRequireDefault(require("json-schema-merge-allof"));

var _fill = _interopRequireDefault(require("core-js-pure/features/array/fill"));

var _union = _interopRequireDefault(require("lodash/union"));

var _jsonpointer = _interopRequireDefault(require("jsonpointer"));

var _fields = _interopRequireDefault(require("./components/fields"));

var _widgets = _interopRequireDefault(require("./components/widgets"));

var _validate = _interopRequireWildcard(require("./validate"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return _typeof(key) === "symbol" ? key : String(key); }

function _toPrimitive(input, hint) { if (_typeof(input) !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (_typeof(res) !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); }

function _toConsumableArray(arr) { return _arrayWithoutHoles(arr) || _iterableToArray(arr) || _nonIterableSpread(); }

function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance"); }

function _iterableToArray(iter) { if (Symbol.iterator in Object(iter) || Object.prototype.toString.call(iter) === "[object Arguments]") return Array.from(iter); }

function _arrayWithoutHoles(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = new Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } }

function _typeof(obj) { if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") { _typeof = function _typeof(obj) { return typeof obj; }; } else { _typeof = function _typeof(obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; }; } return _typeof(obj); }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function _objectWithoutProperties(source, excluded) { if (source == null) return {}; var target = _objectWithoutPropertiesLoose(source, excluded); var key, i; if (Object.getOwnPropertySymbols) { var sourceSymbolKeys = Object.getOwnPropertySymbols(source); for (i = 0; i < sourceSymbolKeys.length; i++) { key = sourceSymbolKeys[i]; if (excluded.indexOf(key) >= 0) continue; if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue; target[key] = source[key]; } } return target; }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

var ADDITIONAL_PROPERTY_FLAG = "__additional_property";
exports.ADDITIONAL_PROPERTY_FLAG = ADDITIONAL_PROPERTY_FLAG;
var widgetMap = {
  "boolean": {
    checkbox: "CheckboxWidget",
    radio: "RadioWidget",
    select: "SelectWidget",
    hidden: "HiddenWidget"
  },
  string: {
    text: "TextWidget",
    password: "PasswordWidget",
    email: "EmailWidget",
    hostname: "TextWidget",
    ipv4: "TextWidget",
    ipv6: "TextWidget",
    uri: "URLWidget",
    "data-url": "FileWidget",
    radio: "RadioWidget",
    select: "SelectWidget",
    textarea: "TextareaWidget",
    hidden: "HiddenWidget",
    date: "DateWidget",
    datetime: "DateTimeWidget",
    "date-time": "DateTimeWidget",
    "alt-date": "AltDateWidget",
    "alt-datetime": "AltDateTimeWidget",
    color: "ColorWidget",
    file: "FileWidget"
  },
  number: {
    text: "TextWidget",
    select: "SelectWidget",
    updown: "UpDownWidget",
    range: "RangeWidget",
    radio: "RadioWidget",
    hidden: "HiddenWidget"
  },
  integer: {
    text: "TextWidget",
    select: "SelectWidget",
    updown: "UpDownWidget",
    range: "RangeWidget",
    radio: "RadioWidget",
    hidden: "HiddenWidget"
  },
  array: {
    select: "SelectWidget",
    checkboxes: "CheckboxesWidget",
    files: "FileWidget",
    hidden: "HiddenWidget"
  }
};

function canExpand(schema, uiSchema, formData) {
  if (!schema.additionalProperties) {
    return false;
  }

  var _getUiOptions = getUiOptions(uiSchema),
      expandable = _getUiOptions.expandable;

  if (expandable === false) {
    return expandable;
  } // if ui:options.expandable was not explicitly set to false, we can add
  // another property if we have not exceeded maxProperties yet


  if (schema.maxProperties !== undefined) {
    return Object.keys(formData).length < schema.maxProperties;
  }

  return true;
}

function getDefaultRegistry() {
  return {
    fields: _fields["default"],
    widgets: _widgets["default"],
    definitions: {},
    rootSchema: {},
    formContext: {}
  };
}
/* Gets the type of a given schema. */


function getSchemaType(schema) {
  var type = schema.type;

  if (!type && schema["const"]) {
    return guessType(schema["const"]);
  }

  if (!type && schema["enum"]) {
    return "string";
  }

  if (!type && (schema.properties || schema.additionalProperties)) {
    return "object";
  }

  if (type instanceof Array && type.length === 2 && type.includes("null")) {
    return type.find(function (type) {
      return type !== "null";
    });
  }

  return type;
}

function getWidget(schema, widget) {
  var registeredWidgets = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var type = getSchemaType(schema);

  function mergeOptions(Widget) {
    // cache return value as property of widget for proper react reconciliation
    if (!Widget.MergedWidget) {
      var defaultOptions = Widget.defaultProps && Widget.defaultProps.options || {};

      Widget.MergedWidget = function (_ref) {
        var _ref$options = _ref.options,
            options = _ref$options === void 0 ? {} : _ref$options,
            props = _objectWithoutProperties(_ref, ["options"]);

        return _react["default"].createElement(Widget, _extends({
          options: _objectSpread({}, defaultOptions, options)
        }, props));
      };
    }

    return Widget.MergedWidget;
  }

  if (typeof widget === "function" || ReactIs.isForwardRef(_react["default"].createElement(widget)) || ReactIs.isMemo(widget)) {
    return mergeOptions(widget);
  }

  if (typeof widget !== "string") {
    throw new Error("Unsupported widget definition: ".concat(_typeof(widget)));
  }

  if (registeredWidgets.hasOwnProperty(widget)) {
    var registeredWidget = registeredWidgets[widget];
    return getWidget(schema, registeredWidget, registeredWidgets);
  }

  if (!widgetMap.hasOwnProperty(type)) {
    throw new Error("No widget for type \"".concat(type, "\""));
  }

  if (widgetMap[type].hasOwnProperty(widget)) {
    var _registeredWidget = registeredWidgets[widgetMap[type][widget]];
    return getWidget(schema, _registeredWidget, registeredWidgets);
  }

  throw new Error("No widget \"".concat(widget, "\" for type \"").concat(type, "\""));
}

function hasWidget(schema, widget) {
  var registeredWidgets = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

  try {
    getWidget(schema, widget, registeredWidgets);
    return true;
  } catch (e) {
    if (e.message && (e.message.startsWith("No widget") || e.message.startsWith("Unsupported widget"))) {
      return false;
    }

    throw e;
  }
}

function computeDefaults(_schema, parentDefaults, rootSchema) {
  var rawFormData = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
  var includeUndefinedValues = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : false;
  var schema = isObject(_schema) ? _schema : {};
  var formData = isObject(rawFormData) ? rawFormData : {}; // Compute the defaults recursively: give highest priority to deepest nodes.

  var defaults = parentDefaults;

  if (isObject(defaults) && isObject(schema["default"])) {
    // For object defaults, only override parent defaults that are defined in
    // schema.default.
    defaults = mergeObjects(defaults, schema["default"]);
  } else if ("default" in schema) {
    // Use schema defaults for this node.
    defaults = schema["default"];
  } else if ("$ref" in schema) {
    // Use referenced schema defaults for this node.
    var refSchema = findSchemaDefinition(schema.$ref, rootSchema);
    return computeDefaults(refSchema, defaults, rootSchema, formData, includeUndefinedValues);
  } else if ("dependencies" in schema) {
    var resolvedSchema = resolveDependencies(schema, rootSchema, formData);
    return computeDefaults(resolvedSchema, defaults, rootSchema, formData, includeUndefinedValues);
  } else if (isFixedItems(schema)) {
    defaults = schema.items.map(function (itemSchema, idx) {
      return computeDefaults(itemSchema, Array.isArray(parentDefaults) ? parentDefaults[idx] : undefined, rootSchema, formData, includeUndefinedValues);
    });
  } else if ("oneOf" in schema) {
    schema = schema.oneOf[getMatchingOption(undefined, schema.oneOf, rootSchema)];
  } else if ("anyOf" in schema) {
    schema = schema.anyOf[getMatchingOption(undefined, schema.anyOf, rootSchema)];
  } // Not defaults defined for this node, fallback to generic typed ones.


  if (typeof defaults === "undefined") {
    defaults = schema["default"];
  }

  switch (getSchemaType(schema)) {
    // We need to recur for object schema inner default values.
    case "object":
      return Object.keys(schema.properties || {}).reduce(function (acc, key) {
        // Compute the defaults for this node, with the parent defaults we might
        // have from a previous run: defaults[key].
        var computedDefault = computeDefaults(schema.properties[key], (defaults || {})[key], rootSchema, (formData || {})[key], includeUndefinedValues);

        if (includeUndefinedValues || computedDefault !== undefined) {
          acc[key] = computedDefault;
        }

        return acc;
      }, {});

    case "array":
      // Inject defaults into existing array defaults
      if (Array.isArray(defaults)) {
        defaults = defaults.map(function (item, idx) {
          return computeDefaults(schema.items[idx] || schema.additionalItems || {}, item, rootSchema);
        });
      } // Deeply inject defaults into already existing form data


      if (Array.isArray(rawFormData)) {
        defaults = rawFormData.map(function (item, idx) {
          return computeDefaults(schema.items, (defaults || {})[idx], rootSchema, item);
        });
      }

      if (schema.minItems) {
        if (!isMultiSelect(schema, rootSchema)) {
          var defaultsLength = defaults ? defaults.length : 0;

          if (schema.minItems > defaultsLength) {
            var defaultEntries = defaults || []; // populate the array with the defaults

            var fillerSchema = Array.isArray(schema.items) ? schema.additionalItems : schema.items;
            var fillerEntries = (0, _fill["default"])(new Array(schema.minItems - defaultsLength), computeDefaults(fillerSchema, fillerSchema.defaults, rootSchema)); // then fill up the rest with either the item default or empty, up to minItems

            return defaultEntries.concat(fillerEntries);
          }
        } else {
          return defaults ? defaults : [];
        }
      }

  }

  return defaults;
}

function getDefaultFormState(_schema, formData) {
  var rootSchema = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var includeUndefinedValues = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;

  if (!isObject(_schema)) {
    throw new Error("Invalid schema: " + _schema);
  }

  var schema = retrieveSchema(_schema, rootSchema, formData);
  var defaults = computeDefaults(schema, _schema["default"], rootSchema, formData, includeUndefinedValues);

  if (typeof formData === "undefined") {
    // No form data? Use schema defaults.
    return defaults;
  }

  if (isObject(formData) || Array.isArray(formData)) {
    return mergeDefaultsWithFormData(defaults, formData);
  }

  if (formData === 0 || formData === false || formData === "") {
    return formData;
  }

  return formData || defaults;
}
/**
 * When merging defaults and form data, we want to merge in this specific way:
 * - objects are deeply merged
 * - arrays are merged in such a way that:
 *   - when the array is set in form data, only array entries set in form data
 *     are deeply merged; additional entries from the defaults are ignored
 *   - when the array is not set in form data, the default is copied over
 * - scalars are overwritten/set by form data
 */


function mergeDefaultsWithFormData(defaults, formData) {
  if (Array.isArray(formData)) {
    if (!Array.isArray(defaults)) {
      defaults = [];
    }

    return formData.map(function (value, idx) {
      if (defaults[idx]) {
        return mergeDefaultsWithFormData(defaults[idx], value);
      }

      return value;
    });
  } else if (isObject(formData)) {
    var acc = Object.assign({}, defaults); // Prevent mutation of source object.

    return Object.keys(formData).reduce(function (acc, key) {
      acc[key] = mergeDefaultsWithFormData(defaults ? defaults[key] : {}, formData[key]);
      return acc;
    }, acc);
  } else {
    return formData;
  }
}

function getUiOptions(uiSchema) {
  // get all passed options from ui:widget, ui:options, and ui:<optionName>
  return Object.keys(uiSchema).filter(function (key) {
    return key.indexOf("ui:") === 0;
  }).reduce(function (options, key) {
    var value = uiSchema[key];

    if (key === "ui:widget" && isObject(value)) {
      console.warn("Setting options via ui:widget object is deprecated, use ui:options instead");
      return _objectSpread({}, options, value.options || {}, {
        widget: value.component
      });
    }

    if (key === "ui:options" && isObject(value)) {
      return _objectSpread({}, options, value);
    }

    return _objectSpread({}, options, _defineProperty({}, key.substring(3), value));
  }, {});
}

function getSubmitButtonOptions(uiSchema) {
  var uiOptions = getUiOptions(uiSchema);
  var defaultOptions = {
    props: {
      disabled: false
    },
    submitText: "Submit",
    norender: false
  };

  if (uiOptions && uiOptions["submitButtonOptions"]) {
    return Object.assign({}, defaultOptions, uiOptions["submitButtonOptions"]);
  }

  return defaultOptions;
}

function getDisplayLabel(schema, uiSchema, rootSchema) {
  var uiOptions = getUiOptions(uiSchema);
  var _uiOptions$label = uiOptions.label,
      displayLabel = _uiOptions$label === void 0 ? true : _uiOptions$label;
  var schemaType = getSchemaType(schema);

  if (schemaType === "array") {
    displayLabel = isMultiSelect(schema, rootSchema) || isFilesArray(schema, uiSchema, rootSchema) || isCustomWidget(uiSchema);
  }

  if (schemaType === "object") {
    displayLabel = false;
  }

  if (schemaType === "boolean" && !uiSchema["ui:widget"]) {
    displayLabel = false;
  }

  if (uiSchema["ui:field"]) {
    displayLabel = false;
  }

  return displayLabel;
}

function isObject(thing) {
  if (typeof File !== "undefined" && thing instanceof File) {
    return false;
  }

  return _typeof(thing) === "object" && thing !== null && !Array.isArray(thing);
}

function mergeObjects(obj1, obj2) {
  var concatArrays = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
  // Recursively merge deeply nested objects.
  var acc = Object.assign({}, obj1); // Prevent mutation of source object.

  return Object.keys(obj2).reduce(function (acc, key) {
    var left = obj1 ? obj1[key] : {},
        right = obj2[key];

    if (obj1 && obj1.hasOwnProperty(key) && isObject(right)) {
      acc[key] = mergeObjects(left, right, concatArrays);
    } else if (concatArrays && Array.isArray(left) && Array.isArray(right)) {
      acc[key] = left.concat(right);
    } else {
      acc[key] = right;
    }

    return acc;
  }, acc);
}

function asNumber(value) {
  if (value === "") {
    return undefined;
  }

  if (value === null) {
    return null;
  }

  if (/\.$/.test(value)) {
    // "3." can't really be considered a number even if it parses in js. The
    // user is most likely entering a float.
    return value;
  }

  if (/\.0$/.test(value)) {
    // we need to return this as a string here, to allow for input like 3.07
    return value;
  }

  var n = Number(value);
  var valid = typeof n === "number" && !Number.isNaN(n);

  if (/\.\d*0$/.test(value)) {
    // It's a number, that's cool - but we need it as a string so it doesn't screw
    // with the user when entering dollar amounts or other values (such as those with
    // specific precision or number of significant digits)
    return value;
  }

  return valid ? n : value;
}

function orderProperties(properties, order) {
  if (!Array.isArray(order)) {
    return properties;
  }

  var arrayToHash = function arrayToHash(arr) {
    return arr.reduce(function (prev, curr) {
      prev[curr] = true;
      return prev;
    }, {});
  };

  var errorPropList = function errorPropList(arr) {
    return arr.length > 1 ? "properties '".concat(arr.join("', '"), "'") : "property '".concat(arr[0], "'");
  };

  var propertyHash = arrayToHash(properties);
  var orderFiltered = order.filter(function (prop) {
    return prop === "*" || propertyHash[prop];
  });
  var orderHash = arrayToHash(orderFiltered);
  var rest = properties.filter(function (prop) {
    return !orderHash[prop];
  });
  var restIndex = orderFiltered.indexOf("*");

  if (restIndex === -1) {
    if (rest.length) {
      throw new Error("uiSchema order list does not contain ".concat(errorPropList(rest)));
    }

    return orderFiltered;
  }

  if (restIndex !== orderFiltered.lastIndexOf("*")) {
    throw new Error("uiSchema order list contains more than one wildcard item");
  }

  var complete = _toConsumableArray(orderFiltered);

  complete.splice.apply(complete, [restIndex, 1].concat(_toConsumableArray(rest)));
  return complete;
}
/**
 * This function checks if the given schema matches a single
 * constant value.
 */


function isConstant(schema) {
  return Array.isArray(schema["enum"]) && schema["enum"].length === 1 || schema.hasOwnProperty("const");
}

function toConstant(schema) {
  if (Array.isArray(schema["enum"]) && schema["enum"].length === 1) {
    return schema["enum"][0];
  } else if (schema.hasOwnProperty("const")) {
    return schema["const"];
  } else {
    throw new Error("schema cannot be inferred as a constant");
  }
}

function isSelect(_schema) {
  var rootSchema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var schema = retrieveSchema(_schema, rootSchema);
  var altSchemas = schema.oneOf || schema.anyOf;

  if (Array.isArray(schema["enum"])) {
    return true;
  } else if (Array.isArray(altSchemas)) {
    return altSchemas.every(function (altSchemas) {
      return isConstant(altSchemas);
    });
  }

  return false;
}

function isMultiSelect(schema) {
  var rootSchema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

  if (!schema.uniqueItems || !schema.items) {
    return false;
  }

  return isSelect(schema.items, rootSchema);
}

function isFilesArray(schema, uiSchema) {
  var rootSchema = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

  if (uiSchema["ui:widget"] === "files") {
    return true;
  } else if (schema.items) {
    var itemsSchema = retrieveSchema(schema.items, rootSchema);
    return itemsSchema.type === "string" && itemsSchema.format === "data-url";
  }

  return false;
}

function isFixedItems(schema) {
  return Array.isArray(schema.items) && schema.items.length > 0 && schema.items.every(function (item) {
    return isObject(item);
  });
}

function isCustomWidget(uiSchema) {
  return (// TODO: Remove the `&& uiSchema["ui:widget"] !== "hidden"` once we support hidden widgets for arrays.
    // https://react-jsonschema-form.readthedocs.io/en/latest/usage/widgets/#hidden-widgets
    "widget" in getUiOptions(uiSchema) && getUiOptions(uiSchema)["widget"] !== "hidden"
  );
}

function allowAdditionalItems(schema) {
  if (schema.additionalItems === true) {
    console.warn("additionalItems=true is currently not supported");
  }

  return isObject(schema.additionalItems);
}

function optionsList(schema) {
  if (schema["enum"]) {
    return schema["enum"].map(function (value, i) {
      var label = schema.enumNames && schema.enumNames[i] || String(value);
      return {
        label: label,
        value: value
      };
    });
  } else {
    var altSchemas = schema.oneOf || schema.anyOf;
    return altSchemas.map(function (schema) {
      var value = toConstant(schema);
      var label = schema.title || String(value);
      return {
        schema: schema,
        label: label,
        value: value
      };
    });
  }
}

function findSchemaDefinition($ref) {
  var rootSchema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var origRef = $ref;

  if ($ref.startsWith("#")) {
    // Decode URI fragment representation.
    $ref = decodeURIComponent($ref.substring(1));
  } else {
    throw new Error("Could not find a definition for ".concat(origRef, "."));
  }

  var current = _jsonpointer["default"].get(rootSchema, $ref);

  if (current === undefined) {
    throw new Error("Could not find a definition for ".concat(origRef, "."));
  }

  if (current.hasOwnProperty("$ref")) {
    return findSchemaDefinition(current.$ref, rootSchema);
  }

  return current;
} // In the case where we have to implicitly create a schema, it is useful to know what type to use
//  based on the data we are defining


var guessType = function guessType(value) {
  if (Array.isArray(value)) {
    return "array";
  } else if (typeof value === "string") {
    return "string";
  } else if (value == null) {
    return "null";
  } else if (typeof value === "boolean") {
    return "boolean";
  } else if (!isNaN(value)) {
    return "number";
  } else if (_typeof(value) === "object") {
    return "object";
  } // Default to string if we can't figure it out


  return "string";
}; // This function will create new "properties" items for each key in our formData


exports.guessType = guessType;

function stubExistingAdditionalProperties(schema) {
  var rootSchema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var formData = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  // Clone the schema so we don't ruin the consumer's original
  schema = _objectSpread({}, schema, {
    properties: _objectSpread({}, schema.properties)
  }); // make sure formData is an object

  formData = isObject(formData) ? formData : {};
  Object.keys(formData).forEach(function (key) {
    if (schema.properties.hasOwnProperty(key)) {
      // No need to stub, our schema already has the property
      return;
    }

    var additionalProperties;

    if (schema.additionalProperties.hasOwnProperty("$ref")) {
      additionalProperties = retrieveSchema({
        $ref: schema.additionalProperties["$ref"]
      }, rootSchema, formData);
    } else if (schema.additionalProperties.hasOwnProperty("type")) {
      additionalProperties = _objectSpread({}, schema.additionalProperties);
    } else {
      additionalProperties = {
        type: guessType(formData[key])
      };
    } // The type of our new key should match the additionalProperties value;


    schema.properties[key] = additionalProperties; // Set our additional property flag so we know it was dynamically added

    schema.properties[key][ADDITIONAL_PROPERTY_FLAG] = true;
  });
  return schema;
}
/**
 * Resolves a conditional block (if/else/then) by removing the condition and merging the appropriate conditional branch with the rest of the schema
 */


var resolveCondition = function resolveCondition(schema, rootSchema, formData) {
  var expression = schema["if"],
      then = schema.then,
      otherwise = schema["else"],
      resolvedSchemaLessConditional = _objectWithoutProperties(schema, ["if", "then", "else"]);

  var conditionalSchema = (0, _validate.isValid)(expression, formData, rootSchema) ? then : otherwise;

  if (conditionalSchema) {
    return retrieveSchema(mergeSchemas(resolvedSchemaLessConditional, retrieveSchema(conditionalSchema, rootSchema, formData)), rootSchema, formData);
  } else {
    return retrieveSchema(resolvedSchemaLessConditional, rootSchema, formData);
  }
};
/**
 * Resolves references and dependencies within a schema and its 'allOf' children.
 *
 * Called internally by retrieveSchema.
 */


function resolveSchema(schema) {
  var rootSchema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var formData = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

  if (schema.hasOwnProperty("$ref")) {
    return resolveReference(schema, rootSchema, formData);
  } else if (schema.hasOwnProperty("dependencies")) {
    var resolvedSchema = resolveDependencies(schema, rootSchema, formData);
    return retrieveSchema(resolvedSchema, rootSchema, formData);
  } else if (schema.hasOwnProperty("allOf")) {
    return _objectSpread({}, schema, {
      allOf: schema.allOf.map(function (allOfSubschema) {
        return retrieveSchema(allOfSubschema, rootSchema, formData);
      })
    });
  } else {
    // No $ref or dependencies attribute found, returning the original schema.
    return schema;
  }
}

function resolveReference(schema, rootSchema, formData) {
  // Retrieve the referenced schema definition.
  var $refSchema = findSchemaDefinition(schema.$ref, rootSchema); // Drop the $ref property of the source schema.

  var $ref = schema.$ref,
      localSchema = _objectWithoutProperties(schema, ["$ref"]); // Update referenced schema definition with local schema properties.


  return retrieveSchema(_objectSpread({}, $refSchema, localSchema), rootSchema, formData);
}

function retrieveSchema(schema) {
  var rootSchema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var formData = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

  if (!isObject(schema)) {
    return {};
  }

  var resolvedSchema = resolveSchema(schema, rootSchema, formData);

  if (schema.hasOwnProperty("if")) {
    return resolveCondition(schema, rootSchema, formData);
  } // For each level of the dependency, we need to recursively determine the appropriate resolved schema given the current state of formData.
  // Otherwise, nested allOf subschemas will not be correctly displayed.


  if (resolvedSchema.properties) {
    var properties = {};
    Object.entries(resolvedSchema.properties).forEach(function (entries) {
      var propName = entries[0];
      var propSchema = entries[1];
      var rawPropData = formData && formData[propName];
      var propData = isObject(rawPropData) ? rawPropData : {};
      var resolvedPropSchema = retrieveSchema(propSchema, rootSchema, propData);
      properties[propName] = resolvedPropSchema;

      if (propSchema !== resolvedPropSchema && resolvedSchema.properties !== properties) {
        resolvedSchema = _objectSpread({}, resolvedSchema, {
          properties: properties
        });
      }
    });
  }

  if ("allOf" in schema) {
    try {
      resolvedSchema = (0, _jsonSchemaMergeAllof["default"])(_objectSpread({}, resolvedSchema, {
        allOf: resolvedSchema.allOf
      }));
    } catch (e) {
      console.warn("could not merge subschemas in allOf:\n" + e);

      var _resolvedSchema = resolvedSchema,
          allOf = _resolvedSchema.allOf,
          resolvedSchemaWithoutAllOf = _objectWithoutProperties(_resolvedSchema, ["allOf"]);

      return resolvedSchemaWithoutAllOf;
    }
  }

  var hasAdditionalProperties = resolvedSchema.hasOwnProperty("additionalProperties") && resolvedSchema.additionalProperties !== false;

  if (hasAdditionalProperties) {
    return stubExistingAdditionalProperties(resolvedSchema, rootSchema, formData);
  }

  return resolvedSchema;
}

function resolveDependencies(schema, rootSchema, formData) {
  // Drop the dependencies from the source schema.
  var _schema$dependencies = schema.dependencies,
      dependencies = _schema$dependencies === void 0 ? {} : _schema$dependencies,
      resolvedSchema = _objectWithoutProperties(schema, ["dependencies"]);

  if ("oneOf" in resolvedSchema) {
    resolvedSchema = resolvedSchema.oneOf[getMatchingOption(formData, resolvedSchema.oneOf, rootSchema)];
  } else if ("anyOf" in resolvedSchema) {
    resolvedSchema = resolvedSchema.anyOf[getMatchingOption(formData, resolvedSchema.anyOf, rootSchema)];
  }

  return processDependencies(dependencies, resolvedSchema, rootSchema, formData);
}

function processDependencies(dependencies, resolvedSchema, rootSchema, formData) {
  // Process dependencies updating the local schema properties as appropriate.
  for (var dependencyKey in dependencies) {
    // Skip this dependency if its trigger property is not present.
    if (formData[dependencyKey] === undefined) {
      continue;
    } // Skip this dependency if it is not included in the schema (such as when dependencyKey is itself a hidden dependency.)


    if (resolvedSchema.properties && !(dependencyKey in resolvedSchema.properties)) {
      continue;
    }

    var dependencyValue = dependencies[dependencyKey],
        remainingDependencies = _objectWithoutProperties(dependencies, [dependencyKey].map(_toPropertyKey));

    if (Array.isArray(dependencyValue)) {
      resolvedSchema = withDependentProperties(resolvedSchema, dependencyValue);
    } else if (isObject(dependencyValue)) {
      resolvedSchema = withDependentSchema(resolvedSchema, rootSchema, formData, dependencyKey, dependencyValue);
    }

    return processDependencies(remainingDependencies, resolvedSchema, rootSchema, formData);
  }

  return resolvedSchema;
}

function withDependentProperties(schema, additionallyRequired) {
  if (!additionallyRequired) {
    return schema;
  }

  var required = Array.isArray(schema.required) ? Array.from(new Set([].concat(_toConsumableArray(schema.required), _toConsumableArray(additionallyRequired)))) : additionallyRequired;
  return _objectSpread({}, schema, {
    required: required
  });
}

function withDependentSchema(schema, rootSchema, formData, dependencyKey, dependencyValue) {
  var _retrieveSchema = retrieveSchema(dependencyValue, rootSchema, formData),
      oneOf = _retrieveSchema.oneOf,
      dependentSchema = _objectWithoutProperties(_retrieveSchema, ["oneOf"]);

  schema = mergeSchemas(schema, dependentSchema); // Since it does not contain oneOf, we return the original schema.

  if (oneOf === undefined) {
    return schema;
  } else if (!Array.isArray(oneOf)) {
    throw new Error("invalid: it is some ".concat(_typeof(oneOf), " instead of an array"));
  } // Resolve $refs inside oneOf.


  var resolvedOneOf = oneOf.map(function (subschema) {
    return subschema.hasOwnProperty("$ref") ? resolveReference(subschema, rootSchema, formData) : subschema;
  });
  return withExactlyOneSubschema(schema, rootSchema, formData, dependencyKey, resolvedOneOf);
}

function withExactlyOneSubschema(schema, rootSchema, formData, dependencyKey, oneOf) {
  var validSubschemas = oneOf.filter(function (subschema) {
    if (!subschema.properties) {
      return false;
    }

    var conditionPropertySchema = subschema.properties[dependencyKey];

    if (conditionPropertySchema) {
      var conditionSchema = {
        type: "object",
        properties: _defineProperty({}, dependencyKey, conditionPropertySchema)
      };

      var _validateFormData = (0, _validate["default"])(formData, conditionSchema),
          errors = _validateFormData.errors;

      return errors.length === 0;
    }
  });

  if (validSubschemas.length !== 1) {
    console.warn("ignoring oneOf in dependencies because there isn't exactly one subschema that is valid");
    return schema;
  }

  var subschema = validSubschemas[0];

  var _subschema$properties = subschema.properties,
      conditionPropertySchema = _subschema$properties[dependencyKey],
      dependentSubschema = _objectWithoutProperties(_subschema$properties, [dependencyKey].map(_toPropertyKey));

  var dependentSchema = _objectSpread({}, subschema, {
    properties: dependentSubschema
  });

  return mergeSchemas(schema, retrieveSchema(dependentSchema, rootSchema, formData));
} // Recursively merge deeply nested schemas.
// The difference between mergeSchemas and mergeObjects
// is that mergeSchemas only concats arrays for
// values under the "required" keyword, and when it does,
// it doesn't include duplicate values.


function mergeSchemas(obj1, obj2) {
  var acc = Object.assign({}, obj1); // Prevent mutation of source object.

  return Object.keys(obj2).reduce(function (acc, key) {
    var left = obj1 ? obj1[key] : {},
        right = obj2[key];

    if (obj1 && obj1.hasOwnProperty(key) && isObject(right)) {
      acc[key] = mergeSchemas(left, right);
    } else if (obj1 && obj2 && (getSchemaType(obj1) === "object" || getSchemaType(obj2) === "object") && key === "required" && Array.isArray(left) && Array.isArray(right)) {
      // Don't include duplicate values when merging
      // "required" fields.
      acc[key] = (0, _union["default"])(left, right);
    } else {
      acc[key] = right;
    }

    return acc;
  }, acc);
}

function isArguments(object) {
  return Object.prototype.toString.call(object) === "[object Arguments]";
}

function deepEquals(a, b) {
  var ca = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : [];
  var cb = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : [];

  // Partially extracted from node-deeper and adapted to exclude comparison
  // checks for functions.
  // https://github.com/othiym23/node-deeper
  if (a === b) {
    return true;
  } else if (typeof a === "function" || typeof b === "function") {
    // Assume all functions are equivalent
    // see https://github.com/rjsf-team/react-jsonschema-form/issues/255
    return true;
  } else if (_typeof(a) !== "object" || _typeof(b) !== "object") {
    return false;
  } else if (a === null || b === null) {
    return false;
  } else if (a instanceof Date && b instanceof Date) {
    return a.getTime() === b.getTime();
  } else if (a instanceof RegExp && b instanceof RegExp) {
    return a.source === b.source && a.global === b.global && a.multiline === b.multiline && a.lastIndex === b.lastIndex && a.ignoreCase === b.ignoreCase;
  } else if (isArguments(a) || isArguments(b)) {
    if (!(isArguments(a) && isArguments(b))) {
      return false;
    }

    var slice = Array.prototype.slice;
    return deepEquals(slice.call(a), slice.call(b), ca, cb);
  } else {
    if (a.constructor !== b.constructor) {
      return false;
    }

    var ka = Object.keys(a);
    var kb = Object.keys(b); // don't bother with stack acrobatics if there's nothing there

    if (ka.length === 0 && kb.length === 0) {
      return true;
    }

    if (ka.length !== kb.length) {
      return false;
    }

    var cal = ca.length;

    while (cal--) {
      if (ca[cal] === a) {
        return cb[cal] === b;
      }
    }

    ca.push(a);
    cb.push(b);
    ka.sort();
    kb.sort();

    for (var j = ka.length - 1; j >= 0; j--) {
      if (ka[j] !== kb[j]) {
        return false;
      }
    }

    var _key;

    for (var k = ka.length - 1; k >= 0; k--) {
      _key = ka[k];

      if (!deepEquals(a[_key], b[_key], ca, cb)) {
        return false;
      }
    }

    ca.pop();
    cb.pop();
    return true;
  }
}

function shouldRender(comp, nextProps, nextState) {
  var props = comp.props,
      state = comp.state;
  return !deepEquals(props, nextProps) || !deepEquals(state, nextState);
}

function toIdSchema(schema, id, rootSchema) {
  var formData = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
  var idPrefix = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : "root";
  var idSeparator = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : "_";
  var idSchema = {
    $id: id || idPrefix
  };

  if ("$ref" in schema || "dependencies" in schema || "allOf" in schema) {
    var _schema = retrieveSchema(schema, rootSchema, formData);

    return toIdSchema(_schema, id, rootSchema, formData, idPrefix, idSeparator);
  }

  if ("items" in schema && !schema.items.$ref) {
    return toIdSchema(schema.items, id, rootSchema, formData, idPrefix, idSeparator);
  }

  if (schema.type !== "object") {
    return idSchema;
  }

  for (var name in schema.properties || {}) {
    var field = schema.properties[name];
    var fieldId = idSchema.$id + idSeparator + name;
    idSchema[name] = toIdSchema(isObject(field) ? field : {}, fieldId, rootSchema, // It's possible that formData is not an object -- this can happen if an
    // array item has just been added, but not populated with data yet
    (formData || {})[name], idPrefix, idSeparator);
  }

  return idSchema;
}

function toPathSchema(schema) {
  var name = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : "";
  var rootSchema = arguments.length > 2 ? arguments[2] : undefined;
  var formData = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
  var pathSchema = {
    $name: name.replace(/^\./, "")
  };

  if ("$ref" in schema || "dependencies" in schema || "allOf" in schema) {
    var _schema = retrieveSchema(schema, rootSchema, formData);

    return toPathSchema(_schema, name, rootSchema, formData);
  }

  if (schema.hasOwnProperty("additionalProperties")) {
    pathSchema.__rjsf_additionalProperties = true;
  }

  if (schema.hasOwnProperty("items") && Array.isArray(formData)) {
    formData.forEach(function (element, i) {
      pathSchema[i] = toPathSchema(schema.items, "".concat(name, ".").concat(i), rootSchema, element);
    });
  } else if (schema.hasOwnProperty("properties")) {
    for (var property in schema.properties) {
      pathSchema[property] = toPathSchema(schema.properties[property], "".concat(name, ".").concat(property), rootSchema, // It's possible that formData is not an object -- this can happen if an
      // array item has just been added, but not populated with data yet
      (formData || {})[property]);
    }
  }

  return pathSchema;
}

function parseDateString(dateString) {
  var includeTime = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;

  if (!dateString) {
    return {
      year: -1,
      month: -1,
      day: -1,
      hour: includeTime ? -1 : 0,
      minute: includeTime ? -1 : 0,
      second: includeTime ? -1 : 0
    };
  }

  var date = new Date(dateString);

  if (Number.isNaN(date.getTime())) {
    throw new Error("Unable to parse date " + dateString);
  }

  return {
    year: date.getUTCFullYear(),
    month: date.getUTCMonth() + 1,
    // oh you, javascript.
    day: date.getUTCDate(),
    hour: includeTime ? date.getUTCHours() : 0,
    minute: includeTime ? date.getUTCMinutes() : 0,
    second: includeTime ? date.getUTCSeconds() : 0
  };
}

function toDateString(_ref2) {
  var year = _ref2.year,
      month = _ref2.month,
      day = _ref2.day,
      _ref2$hour = _ref2.hour,
      hour = _ref2$hour === void 0 ? 0 : _ref2$hour,
      _ref2$minute = _ref2.minute,
      minute = _ref2$minute === void 0 ? 0 : _ref2$minute,
      _ref2$second = _ref2.second,
      second = _ref2$second === void 0 ? 0 : _ref2$second;
  var time = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
  var utcTime = Date.UTC(year, month - 1, day, hour, minute, second);
  var datetime = new Date(utcTime).toJSON();
  return time ? datetime : datetime.slice(0, 10);
}

function utcToLocal(jsonDate) {
  if (!jsonDate) {
    return "";
  } // required format of `"yyyy-MM-ddThh:mm" followed by optional ":ss" or ":ss.SSS"
  // https://html.spec.whatwg.org/multipage/input.html#local-date-and-time-state-(type%3Ddatetime-local)
  // > should be a _valid local date and time string_ (not GMT)
  // Note - date constructor passed local ISO-8601 does not correctly
  // change time to UTC in node pre-8


  var date = new Date(jsonDate);
  var yyyy = pad(date.getFullYear(), 4);
  var MM = pad(date.getMonth() + 1, 2);
  var dd = pad(date.getDate(), 2);
  var hh = pad(date.getHours(), 2);
  var mm = pad(date.getMinutes(), 2);
  var ss = pad(date.getSeconds(), 2);
  var SSS = pad(date.getMilliseconds(), 3);
  return "".concat(yyyy, "-").concat(MM, "-").concat(dd, "T").concat(hh, ":").concat(mm, ":").concat(ss, ".").concat(SSS);
}

function localToUTC(dateString) {
  if (dateString) {
    return new Date(dateString).toJSON();
  }
}

function pad(num, size) {
  var s = String(num);

  while (s.length < size) {
    s = "0" + s;
  }

  return s;
}

function dataURItoBlob(dataURI) {
  // Split metadata from data
  var splitted = dataURI.split(","); // Split params

  var params = splitted[0].split(";"); // Get mime-type from params

  var type = params[0].replace("data:", ""); // Filter the name property from params

  var properties = params.filter(function (param) {
    return param.split("=")[0] === "name";
  }); // Look for the name and use unknown if no name property.

  var name;

  if (properties.length !== 1) {
    name = "unknown";
  } else {
    // Because we filtered out the other property,
    // we only have the name case here.
    name = properties[0].split("=")[1];
  } // Built the Uint8Array Blob parameter from the base64 string.


  var binary = atob(splitted[1]);
  var array = [];

  for (var _i = 0; _i < binary.length; _i++) {
    array.push(binary.charCodeAt(_i));
  } // Create the blob object


  var blob = new window.Blob([new Uint8Array(array)], {
    type: type
  });
  return {
    blob: blob,
    name: name
  };
}

function rangeSpec(schema) {
  var spec = {};

  if (schema.multipleOf) {
    spec.step = schema.multipleOf;
  }

  if (schema.minimum || schema.minimum === 0) {
    spec.min = schema.minimum;
  }

  if (schema.maximum || schema.maximum === 0) {
    spec.max = schema.maximum;
  }

  return spec;
}

function getMatchingOption(formData, options, rootSchema) {
  // For performance, skip validating subschemas if formData is undefined. We just
  // want to get the first option in that case.
  if (formData === undefined) {
    return 0;
  }

  for (var _i2 = 0; _i2 < options.length; _i2++) {
    var option = options[_i2]; // If the schema describes an object then we need to add slightly more
    // strict matching to the schema, because unless the schema uses the
    // "requires" keyword, an object will match the schema as long as it
    // doesn't have matching keys with a conflicting type. To do this we use an
    // "anyOf" with an array of requires. This augmentation expresses that the
    // schema should match if any of the keys in the schema are present on the
    // object and pass validation.

    if (option.properties) {
      // Create an "anyOf" schema that requires at least one of the keys in the
      // "properties" object
      var requiresAnyOf = {
        anyOf: Object.keys(option.properties).map(function (key) {
          return {
            required: [key]
          };
        })
      };
      var augmentedSchema = void 0; // If the "anyOf" keyword already exists, wrap the augmentation in an "allOf"

      if (option.anyOf) {
        // Create a shallow clone of the option
        var shallowClone = _extends({}, option);

        if (!shallowClone.allOf) {
          shallowClone.allOf = [];
        } else {
          // If "allOf" already exists, shallow clone the array
          shallowClone.allOf = shallowClone.allOf.slice();
        }

        shallowClone.allOf.push(requiresAnyOf);
        augmentedSchema = shallowClone;
      } else {
        augmentedSchema = Object.assign({}, option, requiresAnyOf);
      } // Remove the "required" field as it's likely that not all fields have
      // been filled in yet, which will mean that the schema is not valid


      delete augmentedSchema.required;

      if ((0, _validate.isValid)(augmentedSchema, formData, rootSchema)) {
        return _i2;
      }
    } else if ((0, _validate.isValid)(option, formData, rootSchema)) {
      return _i2;
    }
  }

  return 0;
} // Check to see if a schema specifies that a value must be true


function schemaRequiresTrueValue(schema) {
  // Check if const is a truthy value
  if (schema["const"]) {
    return true;
  } // Check if an enum has a single value of true


  if (schema["enum"] && schema["enum"].length === 1 && schema["enum"][0] === true) {
    return true;
  } // If anyOf has a single value, evaluate the subschema


  if (schema.anyOf && schema.anyOf.length === 1) {
    return schemaRequiresTrueValue(schema.anyOf[0]);
  } // If oneOf has a single value, evaluate the subschema


  if (schema.oneOf && schema.oneOf.length === 1) {
    return schemaRequiresTrueValue(schema.oneOf[0]);
  } // Evaluate each subschema in allOf, to see if one of them requires a true
  // value


  if (schema.allOf) {
    return schema.allOf.some(schemaRequiresTrueValue);
  }

  return false;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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