"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ZeppelinBackend = void 0;

var _wreck_requests = require("../../common/helpers/notebooks/wreck_requests");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class ZeppelinBackend {
  constructor() {
    _defineProperty(this, "backend", 'Zeppelin Backend');

    _defineProperty(this, "viewNotes", async function (client, wreckOptions) {
      try {
        let output = [];
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/', wreckOptions);
        output = JSON.parse(body.toString()).body;
        return output;
      } catch (error) {
        throw new Error('View Notebooks Error:' + error);
      }
    });

    _defineProperty(this, "fetchNote", async function (client, noteId, wreckOptions) {
      try {
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/' + noteId, wreckOptions);
        return JSON.parse(body.toString()).body.paragraphs;
      } catch (error) {
        throw new Error('Fetching Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addNote", async function (client, params, wreckOptions) {
      wreckOptions.payload = params;

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/', wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });

    _defineProperty(this, "renameNote", async function (client, params, wreckOptions) {
      wreckOptions.payload = {
        name: params.name
      };

      try {
        const body = await (0, _wreck_requests.requestor)('PUT', 'api/notebook/' + params.noteId + '/rename/', wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Renaming Notebook Error:' + error);
      }
    });

    _defineProperty(this, "cloneNote", async function (client, params, wreckOptions) {
      wreckOptions.payload = {
        name: params.name
      };

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/' + params.noteId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Cloning Notebook Error:' + error);
      }
    });

    _defineProperty(this, "deleteNote", async function (client, noteId, wreckOptions) {
      try {
        const body = await (0, _wreck_requests.requestor)('DELETE', 'api/notebook/' + noteId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Deleting Notebook Error:' + error);
      }
    });

    _defineProperty(this, "exportNote", async function (client, noteId, wreckOptions) {
      try {
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/export/' + noteId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Export Notebook Error:' + error);
      }
    });

    _defineProperty(this, "importNote", async function (client, noteObj, wreckOptions) {
      wreckOptions.payload = noteObj;

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/import', wreckOptions);
        const respBody = JSON.parse(body.toString());
        return respBody;
      } catch (error) {
        throw new Error('Import Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addParagraph", async function (wreckOptions, params) {
      const visualizationPrefix = '%sh #vizobject:';
      const observabilityVisualizationPrefix = '%sh #observabilityviz:';
      let paragraphText = params.paragraphInput;

      if (params.inputType === 'VISUALIZATION' && params.paragraphInput.substring(0, 15) !== visualizationPrefix) {
        paragraphText = visualizationPrefix + paragraphText;
      }

      if (params.inputType === 'OBSERVABILITY_VISUALIZATION' && params.paragraphInput.substring(0, 22) !== observabilityVisualizationPrefix) {
        paragraphText = visualizationPrefix + paragraphText;
      }

      if (params.paragraphInput === '') {
        paragraphText = '%md\n';
      }

      wreckOptions.payload = {
        title: params.inputType,
        text: paragraphText,
        index: params.paragraphIndex
      };

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/' + params.noteId + '/paragraph', wreckOptions);
        const respBody = JSON.parse(body.toString());
        return respBody;
      } catch (error) {
        throw new Error('Adding Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateParagraph", async function (wreckOptions, params) {
      wreckOptions.payload = {
        text: params.paragraphInput
      };
      if (params.paragraphType !== undefined) wreckOptions.payload.title = params.paragraphType;

      try {
        const body = await (0, _wreck_requests.requestor)('PUT', 'api/notebook/' + params.noteId + '/paragraph/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Updating Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "runPara", async function (wreckOptions, params) {
      wreckOptions.payload = {};

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/run/' + params.noteId + '/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString()).status;
      } catch (error) {
        throw new Error('Running Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "fetchParagraph", async function (wreckOptions, params) {
      try {
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/' + params.noteId + '/paragraph/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString()).body;
      } catch (error) {
        throw new Error('Fetching Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "deleteParagraph", async function (wreckOptions, params) {
      wreckOptions.payload = {};

      try {
        const body = await (0, _wreck_requests.requestor)('DELETE', 'api/notebook/' + params.noteId + '/paragraph/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString()).status;
      } catch (error) {
        throw new Error('Deleting Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "clearAllParagraphs", async function (wreckOptions, noteid) {
      try {
        const body = await (0, _wreck_requests.requestor)('PUT', 'api/notebook/' + noteid + '/clear', wreckOptions);
        return JSON.parse(body.toString()).status;
      } catch (error) {
        throw new Error('Clearing Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateRunFetchParagraph", async function (client, request, wreckOptions) {
      const params = request.params;

      try {
        const updateInfo = await this.updateParagraph(wreckOptions, params);
        const runInfo = await this.runPara(wreckOptions, params);
        const getInfo = await this.fetchParagraph(wreckOptions, params);
        return getInfo;
      } catch (error) {
        throw new Error('Update/Run Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateFetchParagraph", async function (client, params, wreckOptions) {
      try {
        const updateInfo = await this.updateParagraph(wreckOptions, params);
        const getInfo = await this.fetchParagraph(wreckOptions, params);
        return getInfo;
      } catch (error) {
        throw new Error('Save Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "addFetchNewParagraph", async function (client, params, wreckOptions) {
      try {
        const respBody = await this.addParagraph(wreckOptions, params);
        const payload = { ...params,
          paragraphId: respBody.body
        };
        const getinfo = await this.fetchParagraph(wreckOptions, payload);
        return getinfo;
      } catch (error) {
        throw new Error('add/Fetch Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "deleteFetchParagraphs", async function (client, params, wreckOptions) {
      try {
        const delinfo = await this.deleteParagraph(wreckOptions, params);
        const notebookinfo = await this.fetchNote(client, params.noteId, wreckOptions);
        return {
          paragraphs: notebookinfo
        };
      } catch (error) {
        throw new Error('Delete Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "clearAllFetchParagraphs", async function (client, params, wreckOptions) {
      try {
        const clearinfo = await this.clearAllParagraphs(wreckOptions, params.noteId);
        const notebookinfo = await this.fetchNote(client, params.noteId, wreckOptions);
        return {
          paragraphs: notebookinfo
        };
      } catch (error) {
        throw new Error('Clear Paragraph Error:' + error);
      }
    });
  }

}

exports.ZeppelinBackend = ZeppelinBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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