/*
 * Decompiled with CFR 0.152.
 */
package android.util.apk;

import android.util.apk.ByteBufferDataSource;
import android.util.apk.ByteBufferFactory;
import android.util.apk.DataDigester;
import android.util.apk.DataSource;
import android.util.apk.SignatureInfo;
import android.util.apk.SignatureNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.DigestException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;

public abstract class VerityBuilder {
    private static final int CHUNK_SIZE_BYTES = 4096;
    private static final int DIGEST_SIZE_BYTES = 32;
    private static final int FSVERITY_HEADER_SIZE_BYTES = 64;
    private static final int ZIP_EOCD_CENTRAL_DIR_OFFSET_FIELD_SIZE = 4;
    private static final int ZIP_EOCD_CENTRAL_DIR_OFFSET_FIELD_OFFSET = 16;
    private static final String JCA_DIGEST_ALGORITHM = "SHA-256";
    private static final byte[] DEFAULT_SALT = new byte[8];
    private static final int MMAP_REGION_SIZE_BYTES = 0x100000;

    private VerityBuilder() {
    }

    public static VerityResult generateApkVerityTree(RandomAccessFile apk, SignatureInfo signatureInfo, ByteBufferFactory bufferFactory) throws IOException, SecurityException, NoSuchAlgorithmException, DigestException {
        return VerityBuilder.generateVerityTreeInternal(apk, bufferFactory, signatureInfo);
    }

    private static VerityResult generateVerityTreeInternal(RandomAccessFile apk, ByteBufferFactory bufferFactory, SignatureInfo signatureInfo) throws IOException, SecurityException, NoSuchAlgorithmException, DigestException {
        long signingBlockSize = signatureInfo.centralDirOffset - signatureInfo.apkSigningBlockOffset;
        long dataSize = apk.length() - signingBlockSize;
        int[] levelOffset = VerityBuilder.calculateVerityLevelOffset(dataSize);
        int merkleTreeSize = levelOffset[levelOffset.length - 1];
        ByteBuffer output = bufferFactory.create(merkleTreeSize + 4096);
        output.order(ByteOrder.LITTLE_ENDIAN);
        ByteBuffer tree = VerityBuilder.slice(output, 0, merkleTreeSize);
        byte[] apkRootHash = VerityBuilder.generateVerityTreeInternal(apk, signatureInfo, DEFAULT_SALT, levelOffset, tree);
        return new VerityResult(output, merkleTreeSize, apkRootHash);
    }

    static void generateApkVerityFooter(RandomAccessFile apk, SignatureInfo signatureInfo, ByteBuffer footerOutput) throws IOException {
        footerOutput.order(ByteOrder.LITTLE_ENDIAN);
        VerityBuilder.generateApkVerityHeader(footerOutput, apk.length(), DEFAULT_SALT);
        long signingBlockSize = signatureInfo.centralDirOffset - signatureInfo.apkSigningBlockOffset;
        VerityBuilder.generateApkVerityExtensions(footerOutput, signatureInfo.apkSigningBlockOffset, signingBlockSize, signatureInfo.eocdOffset);
    }

    public static byte[] generateFsVerityRootHash(String apkPath, byte[] salt, ByteBufferFactory bufferFactory) throws IOException, NoSuchAlgorithmException, DigestException {
        try (RandomAccessFile apk = new RandomAccessFile(apkPath, "r");){
            int[] levelOffset = VerityBuilder.calculateVerityLevelOffset(apk.length());
            int merkleTreeSize = levelOffset[levelOffset.length - 1];
            ByteBuffer output = bufferFactory.create(merkleTreeSize + 4096);
            output.order(ByteOrder.LITTLE_ENDIAN);
            ByteBuffer tree = VerityBuilder.slice(output, 0, merkleTreeSize);
            byte[] byArray = VerityBuilder.generateFsVerityTreeInternal(apk, salt, levelOffset, tree);
            return byArray;
        }
    }

    static byte[] generateApkVerityRootHash(RandomAccessFile apk, ByteBuffer apkDigest, SignatureInfo signatureInfo) throws NoSuchAlgorithmException, DigestException, IOException {
        VerityBuilder.assertSigningBlockAlignedAndHasFullPages(signatureInfo);
        ByteBuffer footer = ByteBuffer.allocate(4096).order(ByteOrder.LITTLE_ENDIAN);
        VerityBuilder.generateApkVerityFooter(apk, signatureInfo, footer);
        footer.flip();
        MessageDigest md = MessageDigest.getInstance(JCA_DIGEST_ALGORITHM);
        md.update(footer);
        md.update(apkDigest);
        return md.digest();
    }

    static byte[] generateApkVerity(String apkPath, ByteBufferFactory bufferFactory, SignatureInfo signatureInfo) throws IOException, SignatureNotFoundException, SecurityException, DigestException, NoSuchAlgorithmException {
        try (RandomAccessFile apk = new RandomAccessFile(apkPath, "r");){
            VerityResult result = VerityBuilder.generateVerityTreeInternal(apk, bufferFactory, signatureInfo);
            ByteBuffer footer = VerityBuilder.slice(result.verityData, result.merkleTreeSize, result.verityData.limit());
            VerityBuilder.generateApkVerityFooter(apk, signatureInfo, footer);
            footer.putInt(footer.position() + 4);
            result.verityData.limit(result.merkleTreeSize + footer.position());
            byte[] byArray = result.rootHash;
            return byArray;
        }
    }

    private static void consumeByChunk(DataDigester digester, DataSource source, int chunkSize) throws IOException, DigestException {
        int size;
        long inputOffset = 0L;
        for (long inputRemaining = source.size(); inputRemaining > 0L; inputRemaining -= (long)size) {
            size = (int)Math.min(inputRemaining, (long)chunkSize);
            source.feedIntoDataDigester(digester, inputOffset, size);
            inputOffset += (long)size;
        }
    }

    private static void generateFsVerityDigestAtLeafLevel(RandomAccessFile file2, byte[] salt, ByteBuffer output) throws IOException, NoSuchAlgorithmException, DigestException {
        BufferedDigester digester = new BufferedDigester(salt, output);
        VerityBuilder.consumeByChunk(digester, DataSource.create(file2.getFD(), 0L, file2.length()), 0x100000);
        int lastIncompleteChunkSize = (int)(file2.length() % 4096L);
        if (lastIncompleteChunkSize != 0) {
            digester.consume(ByteBuffer.allocate(4096 - lastIncompleteChunkSize));
        }
        digester.assertEmptyBuffer();
        digester.fillUpLastOutputChunk();
    }

    private static void generateApkVerityDigestAtLeafLevel(RandomAccessFile apk, SignatureInfo signatureInfo, byte[] salt, ByteBuffer output) throws IOException, NoSuchAlgorithmException, DigestException {
        BufferedDigester digester = new BufferedDigester(salt, output);
        VerityBuilder.consumeByChunk(digester, DataSource.create(apk.getFD(), 0L, signatureInfo.apkSigningBlockOffset), 0x100000);
        long eocdCdOffsetFieldPosition = signatureInfo.eocdOffset + 16L;
        VerityBuilder.consumeByChunk(digester, DataSource.create(apk.getFD(), signatureInfo.centralDirOffset, eocdCdOffsetFieldPosition - signatureInfo.centralDirOffset), 0x100000);
        ByteBuffer alternativeCentralDirOffset = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        alternativeCentralDirOffset.putInt(Math.toIntExact(signatureInfo.apkSigningBlockOffset));
        alternativeCentralDirOffset.flip();
        digester.consume(alternativeCentralDirOffset);
        long offsetAfterEocdCdOffsetField = eocdCdOffsetFieldPosition + 4L;
        VerityBuilder.consumeByChunk(digester, DataSource.create(apk.getFD(), offsetAfterEocdCdOffsetField, apk.length() - offsetAfterEocdCdOffsetField), 0x100000);
        int lastIncompleteChunkSize = (int)(apk.length() % 4096L);
        if (lastIncompleteChunkSize != 0) {
            digester.consume(ByteBuffer.allocate(4096 - lastIncompleteChunkSize));
        }
        digester.assertEmptyBuffer();
        digester.fillUpLastOutputChunk();
    }

    private static byte[] generateFsVerityTreeInternal(RandomAccessFile apk, byte[] salt, int[] levelOffset, ByteBuffer output) throws IOException, NoSuchAlgorithmException, DigestException {
        VerityBuilder.generateFsVerityDigestAtLeafLevel(apk, salt, VerityBuilder.slice(output, levelOffset[levelOffset.length - 2], levelOffset[levelOffset.length - 1]));
        for (int level = levelOffset.length - 3; level >= 0; --level) {
            ByteBuffer inputBuffer = VerityBuilder.slice(output, levelOffset[level + 1], levelOffset[level + 2]);
            ByteBuffer outputBuffer = VerityBuilder.slice(output, levelOffset[level], levelOffset[level + 1]);
            ByteBufferDataSource source = new ByteBufferDataSource(inputBuffer);
            BufferedDigester digester = new BufferedDigester(salt, outputBuffer);
            VerityBuilder.consumeByChunk(digester, source, 4096);
            digester.assertEmptyBuffer();
            digester.fillUpLastOutputChunk();
        }
        byte[] rootHash = new byte[32];
        BufferedDigester digester = new BufferedDigester(salt, ByteBuffer.wrap(rootHash));
        digester.consume(VerityBuilder.slice(output, 0, 4096));
        digester.assertEmptyBuffer();
        return rootHash;
    }

    private static byte[] generateVerityTreeInternal(RandomAccessFile apk, SignatureInfo signatureInfo, byte[] salt, int[] levelOffset, ByteBuffer output) throws IOException, NoSuchAlgorithmException, DigestException {
        VerityBuilder.assertSigningBlockAlignedAndHasFullPages(signatureInfo);
        VerityBuilder.generateApkVerityDigestAtLeafLevel(apk, signatureInfo, salt, VerityBuilder.slice(output, levelOffset[levelOffset.length - 2], levelOffset[levelOffset.length - 1]));
        for (int level = levelOffset.length - 3; level >= 0; --level) {
            ByteBuffer inputBuffer = VerityBuilder.slice(output, levelOffset[level + 1], levelOffset[level + 2]);
            ByteBuffer outputBuffer = VerityBuilder.slice(output, levelOffset[level], levelOffset[level + 1]);
            ByteBufferDataSource source = new ByteBufferDataSource(inputBuffer);
            BufferedDigester digester = new BufferedDigester(salt, outputBuffer);
            VerityBuilder.consumeByChunk(digester, source, 4096);
            digester.assertEmptyBuffer();
            digester.fillUpLastOutputChunk();
        }
        byte[] rootHash = new byte[32];
        BufferedDigester digester = new BufferedDigester(salt, ByteBuffer.wrap(rootHash));
        digester.consume(VerityBuilder.slice(output, 0, 4096));
        digester.assertEmptyBuffer();
        return rootHash;
    }

    private static ByteBuffer generateApkVerityHeader(ByteBuffer buffer, long fileSize, byte[] salt) {
        if (salt.length != 8) {
            throw new IllegalArgumentException("salt is not 8 bytes long");
        }
        buffer.put("TrueBrew".getBytes());
        buffer.put((byte)1);
        buffer.put((byte)0);
        buffer.put((byte)12);
        buffer.put((byte)7);
        buffer.putShort((short)1);
        buffer.putShort((short)1);
        buffer.putInt(0);
        buffer.putInt(0);
        buffer.putLong(fileSize);
        buffer.put((byte)2);
        buffer.put((byte)0);
        buffer.put(salt);
        VerityBuilder.skip(buffer, 22);
        return buffer;
    }

    private static ByteBuffer generateApkVerityExtensions(ByteBuffer buffer, long signingBlockOffset, long signingBlockSize, long eocdOffset) {
        int kSizeOfFsverityExtensionHeader = 8;
        int kExtensionSizeAlignment = 8;
        int kSizeOfFsverityElidedExtension = 16;
        buffer.putInt(24);
        buffer.putShort((short)1);
        VerityBuilder.skip(buffer, 2);
        buffer.putLong(signingBlockOffset);
        buffer.putLong(signingBlockSize);
        int kTotalSize = 20;
        buffer.putInt(20);
        buffer.putShort((short)2);
        VerityBuilder.skip(buffer, 2);
        buffer.putLong(eocdOffset + 16L);
        buffer.putInt(Math.toIntExact(signingBlockOffset));
        int kPadding = 4;
        if (kPadding == 8) {
            kPadding = 0;
        }
        VerityBuilder.skip(buffer, kPadding);
        return buffer;
    }

    private static int[] calculateVerityLevelOffset(long fileSize) {
        ArrayList<Long> levelSize = new ArrayList<Long>();
        while (true) {
            long levelDigestSize = VerityBuilder.divideRoundup(fileSize, 4096L) * 32L;
            long chunksSize = 4096L * VerityBuilder.divideRoundup(levelDigestSize, 4096L);
            levelSize.add(chunksSize);
            if (levelDigestSize <= 4096L) break;
            fileSize = levelDigestSize;
        }
        int[] levelOffset = new int[levelSize.size() + 1];
        levelOffset[0] = 0;
        for (int i = 0; i < levelSize.size(); ++i) {
            levelOffset[i + 1] = levelOffset[i] + Math.toIntExact((Long)levelSize.get(levelSize.size() - i - 1));
        }
        return levelOffset;
    }

    private static void assertSigningBlockAlignedAndHasFullPages(SignatureInfo signatureInfo) {
        if (signatureInfo.apkSigningBlockOffset % 4096L != 0L) {
            throw new IllegalArgumentException("APK Signing Block does not start at the page boundary: " + signatureInfo.apkSigningBlockOffset);
        }
        if ((signatureInfo.centralDirOffset - signatureInfo.apkSigningBlockOffset) % 4096L != 0L) {
            throw new IllegalArgumentException("Size of APK Signing Block is not a multiple of 4096: " + (signatureInfo.centralDirOffset - signatureInfo.apkSigningBlockOffset));
        }
    }

    private static ByteBuffer slice(ByteBuffer buffer, int begin, int end) {
        ByteBuffer b = buffer.duplicate();
        b.position(0);
        b.limit(end);
        b.position(begin);
        return b.slice();
    }

    private static void skip(ByteBuffer buffer, int bytes) {
        buffer.position(buffer.position() + bytes);
    }

    private static long divideRoundup(long dividend, long divisor) {
        return (dividend + divisor - 1L) / divisor;
    }

    private static class BufferedDigester
    implements DataDigester {
        private static final int BUFFER_SIZE = 4096;
        private int mBytesDigestedSinceReset;
        private final ByteBuffer mOutput;
        private final MessageDigest mMd;
        private final byte[] mDigestBuffer = new byte[32];
        private final byte[] mSalt;

        private BufferedDigester(byte[] salt, ByteBuffer output) throws NoSuchAlgorithmException {
            this.mSalt = salt;
            this.mOutput = output.slice();
            this.mMd = MessageDigest.getInstance(VerityBuilder.JCA_DIGEST_ALGORITHM);
            if (this.mSalt != null) {
                this.mMd.update(this.mSalt);
            }
            this.mBytesDigestedSinceReset = 0;
        }

        @Override
        public void consume(ByteBuffer buffer) throws DigestException {
            int offset2 = buffer.position();
            int remaining = buffer.remaining();
            while (remaining > 0) {
                int allowance = Math.min(remaining, 4096 - this.mBytesDigestedSinceReset);
                buffer.limit(buffer.position() + allowance);
                this.mMd.update(buffer);
                offset2 += allowance;
                remaining -= allowance;
                this.mBytesDigestedSinceReset += allowance;
                if (this.mBytesDigestedSinceReset != 4096) continue;
                this.mMd.digest(this.mDigestBuffer, 0, this.mDigestBuffer.length);
                this.mOutput.put(this.mDigestBuffer);
                if (this.mSalt != null) {
                    this.mMd.update(this.mSalt);
                }
                this.mBytesDigestedSinceReset = 0;
            }
        }

        public void assertEmptyBuffer() throws DigestException {
            if (this.mBytesDigestedSinceReset != 0) {
                throw new IllegalStateException("Buffer is not empty: " + this.mBytesDigestedSinceReset);
            }
        }

        private void fillUpLastOutputChunk() {
            int lastBlockSize = this.mOutput.position() % 4096;
            if (lastBlockSize == 0) {
                return;
            }
            this.mOutput.put(ByteBuffer.allocate(4096 - lastBlockSize));
        }
    }

    public static class VerityResult {
        public final ByteBuffer verityData;
        public final int merkleTreeSize;
        public final byte[] rootHash;

        private VerityResult(ByteBuffer verityData, int merkleTreeSize, byte[] rootHash) {
            this.verityData = verityData;
            this.merkleTreeSize = merkleTreeSize;
            this.rootHash = rootHash;
        }
    }
}

