!===============================================================================
! Copyright 2021-2022 Intel Corporation.
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!  Content:
!    Construction of natural cubic spline and interpolation for
!    the vector-valued function saved in the non-consecutive memory
!*******************************************************************************


      include 'mkl_df.f90'
      include "errcheck.inc"
      include "generatedata.inc"
      include "rescheck.inc"

      PROGRAM MKL_DF_TEST

      USE MKL_DF_TYPE
      USE MKL_DF
      USE DF_GENERATE_DATA
      USE DF_EXAMPLE_RESCHECK

      ! number of break points
      INTEGER,PARAMETER :: N           = 10
      ! number of functions
      INTEGER,PARAMETER :: NNY         = 3
      ! total number of spline coefficients
      INTEGER,PARAMETER :: NNSCOEFF    = (N-1)*DF_PP_CUBIC
      ! total number of interpolation sites
      INTEGER,PARAMETER :: NNSITE      = 15
      ! size of array describing derivative orders to compute
      INTEGER,PARAMETER :: NNDORDER = 1

      ! left  limit of interpolation interval
      REAL(4),PARAMETER :: LEFT_LIMIT  = -10.0d0
      ! right limit of interpolation interval
      REAL(4),PARAMETER :: RIGHT_LIMIT =  10.0d0

      ! Data Fitting task descriptor
      TYPE (DF_TASK) task
      ! spline order
      INTEGER :: sorder
      ! spline type
      INTEGER :: stype
      ! number of break points
      INTEGER :: nx
      ! additional info about break points
      INTEGER :: xhint
      ! number of functions
      INTEGER :: ny
      ! additional info about function
      INTEGER :: yhint
      ! spline coefficients storage format
      INTEGER :: scoeffhint
      !  size of array describing derivative orders
      INTEGER :: ndorder
      ! boundary conditions type
      INTEGER :: bc_type
      ! internal conditions type
      INTEGER :: ic_type
      ! total number of interpolation sites
      INTEGER :: nsite
      ! additional info about interpolation sites
      INTEGER :: sitehint
      ! interpolation results storage format
      INTEGER :: rhint
      ! left limit of the interpolation interval
      REAL(4) :: left
      ! right limit of the interpolation interval
      REAL(4) :: right
      ! array of break points
      REAL(4) :: x(0:N)
      ! function values
      REAL(4) :: y(0:N-1,0:NNY-1)                        
      ! array of spline coefficients
      REAL(4) :: scoeff(0:NNSCOEFF-1,0:NNY-1)           
      ! array of interpolation sites
      REAL(4) :: site(0:NNSITE-1)
      ! spline evaluation results
      REAL(4) :: r(0:NNY*NNSITE-1)
      ! array of interpolation sites for checking interpolation results
      REAL(4) :: result(0:NNSITE-1,0:NNY-1)
      ! reference interpolation results
      REAL(4) :: ref_r(0:NNY-1,0:NNSITE-1)   
      ! array of derivatives orders
      INTEGER :: dorder(1)  
      ! type of calculations
      INTEGER :: type
      ! method that is used to perform calculations
      INTEGER :: method

      INTEGER :: i,j
      INTEGER(4) :: errcode

      EXTERNAL :: CheckDfError
      INTEGER :: sUniformRandSortedData
      INTEGER :: sSinDataNotUniformGrid

      REAL(4) :: freq               

      errcode = 0

      ! **** Initializing parameters for Data Fitting task ****
      sorder = DF_PP_CUBIC
      stype  = DF_PP_NATURAL

      !***** Parameters describing interpolation interval *****
      nx    = N
      xhint = DF_NO_HINT

      ! Limits of interpolation interval are provided in case
      ! of uniform partition
      left  = LEFT_LIMIT
      right = RIGHT_LIMIT

      !***** Parameters describing function *****
      ny = NNY
      yhint = DF_1ST_COORDINATE

      !***** Parameters describing spline coefficients storage *****
      scoeffhint = DF_1ST_COORDINATE

      !***** Parameters describing boundary conditions type *****
      bc_type = DF_BC_FREE_END

      !***** Parameters describing internal conditions *****
      ic_type = DF_NO_IC

      !***** Parameters describing interpolation sites *****
      nsite      = NNSITE
      sitehint   = DF_SORTED_DATA

      !***** Parameter describing interpolation results storage *****
      rhint = DF_MATRIX_STORAGE_COLS

      !**** Parameter describing array for derivative orders *****
      ndorder = NNDORDER
      dorder(1) = 1

      !***** Generate array of uniformly distributed break points *****
      errcode = sUniformRandSortedData( x, left, right, nx )
      CALL CheckDfError(errcode)

      !***** Generate functions y = sin(2 * Pi * freq * x) *****
      DO i = 0, ny-1
        freq = (i+1) * 1.3d0
        errcode = sSinDataNotUniformGrid( y(:,i), x, freq, nx )
        CALL CheckDfError(errcode)
      END DO

      !***** Generate interpolation sites *****
      errcode = sUniformRandSortedData( site, left, right, nsite )
      CALL CheckDfError(errcode)

      !***** Create Data Fitting task *****
      errcode = dfsNewTask1D( task, nx, x, xhint, ny, y(:,0), yhint )
      CALL CheckDfError(errcode)

      !***** Setting functions *****
      DO i = 1, ny-1
        errcode = dfsEditIdxPtr( task, DF_Y, i, y(:,i))
        CALL CheckDfError(errcode)
      END DO

      !***** Edit task parameters for natural cubic spline construction *****
      errcode = dfsEditPPSpline1D( task, sorder, stype, bc_type,        &        
     & ic_type=ic_type, scoeff=scoeff(:,0), scoeffhint=scoeffhint )
       CALL CheckDfError(errcode)

      !***** Setting coefficients *****
      DO i = 1, ny-1
        errcode = dfsEditIdxPtr( task, DF_PP_SCOEFF, i, scoeff(:,i))
        CALL CheckDfError(errcode)
      END DO

      !***** Construct natural cubic spline using STD method *****
      errcode = dfsConstruct1D( task, DF_PP_SPLINE, DF_METHOD_STD )
       CALL CheckDfError(errcode)
       
      type = DF_INTERP
      method = DF_METHOD_PP

      errcode = dfsInterpolate1D( task, type, method, nsite,            &
     &     site, sitehint, ndorder, dorder, r=r, rhint=rhint)
      CALL CheckDfError(errcode)

      !***** Delete Data Fitting task *****
      errcode = dfDeleteTask( task )
      CALL CheckDfError(errcode)

      !***** Check results of interpolation *****
      DO i = 0, ny-1
        DO j = 0, nsite-1
          result(j,i) = r(i*nsite + j) 
        END DO
      END DO

      DO i = 0, ny-1
        errcode = sCheckCubInterpRes( nx, x, 1, scoeff(:,i), nsite,           &
     & site, ndorder, dorder, r=result(:,i), ref_r=ref_r(i,:) )
        CALL CheckDfError(errcode)
      END DO


      !***** Print results *****
      WRITE (*,901) "Number of break points : ",nx
      
      !***** Print given function *****
      WRITE (*,902) "          X            Y0            Y1            &
     & Y2"
      DO j = 0, nx-1
        WRITE (*,907) " ",x(j),"   ",y(j,0),"   ",y(j,1),"   ",y(j,2)
      END DO

      !***** Print computed spline coefficients *****
      WRITE (*,904) "Coefficients are calculated for a polynomial of the &
     & form:",""
      WRITE (*,905) "Pi(x) = Ai + Bi*(x - x(i)) + Ci*(x - x(i))^2 + &
     &Di*(x - x(i))^3"
      WRITE (*,905) "    where x(i) <= x < x(i+1)"
      DO i = 0, ny-1
        WRITE (*,906) "Spline coefficients for Y",i ," :"
        WRITE (*,911) " i    Ai            Bi            Ci             &
     &Di"
        DO j = 0, nx-2
          WRITE (*,910) " ",j," ", scoeff(sorder*j + 0, i), "   ",        &
     & scoeff(sorder*j + 1, i),"   ", scoeff(sorder*j + 2, i),"   ",      &
     & scoeff(sorder*j + 3, i)
        END DO
      END DO

      !***** Print interpolation results ******
      WRITE (*,902) "Results of interpolation:"
      PRINT  *,     "    Sites         Spline value"
      PRINT  *,     "               Computed    Expected"
      DO i = 0, ny-1
        WRITE (*,906) "    for function Y",i," :"
        DO j = 0, nsite-1 
            WRITE (*,909) " ",site(j), " ",result(j,i)," ",ref_r(i,j) 
        END DO
      END DO

      STOP 0

901   FORMAT (A,I0)
902   FORMAT (/A)
904   FORMAT (/A/A)
905   FORMAT (99A)
906   FORMAT (/A,I0,A)
907   FORMAT (A,SP,F11.6,A,SP,F11.6,A,SP,F11.6,A,SP,F11.6)
909   FORMAT (A,SP,F11.6,A,SP,F11.6,A,SP,F11.6)
910   FORMAT (A,I1,A,SP,F11.6,A,SP,F11.6,A,SP,F11.6,A,SP,F11.6)
911   FORMAT (A)

      END PROGRAM
