/*******************************************************************************
* Copyright 2018-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
 *
 *  Content:
 *            oneapi::mkl::vm::acos example program text (SYCL interface)
 *
 *******************************************************************************/

#include <CL/sycl.hpp>
#include <algorithm>
#include <cstdint>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <map>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl/types.hpp"
#include "oneapi/mkl/vm.hpp"

#include "common_for_examples.hpp"

#include "vml_common.hpp"

static constexpr int VLEN = 4;

static ulp_table_type ulp_table = {
    {MAX_HA_ULP_S, 4.5},   {MAX_LA_ULP_S, 5.0},   {MAX_EP_ULP_S, 5.0E3},
    {MAX_HA_ULP_D, 2.0},   {MAX_LA_ULP_D, 5.0},   {MAX_EP_ULP_D, 7.0E7},
    {MAX_HA_ULP_C, 5.0E3}, {MAX_LA_ULP_C, 5.0E3}, {MAX_EP_ULP_C, 5.0E3},
    {MAX_HA_ULP_Z, 7.0E7}, {MAX_LA_ULP_Z, 7.0E7}, {MAX_EP_ULP_Z, 7.0E7},
};

//!
//! @brief Accuracy test
//!
template <typename A, typename R>
bool vAcosAccuracyLiteTest(const sycl::device &dev) {

    int argtype = ARG1_RES1;
    // *************************************************************
    // Data table declaraion
    // *************************************************************
data_2_t data
{
    .i = 0
,

    .data_f32 = std::vector<data_2_f32_t>

    {

{ UINT32_C(0x3F2E2D16), UINT32_C(0x3F5290C5) }, //  0: vsAcos ( 0.680375457     ) = ( 0.822521508     );
{ UINT32_C(0xBE584DC4), UINT32_C(0x3FE44E1C) }, //  1: vsAcos ( -0.211234152    ) = ( 1.78363371      );
{ UINT32_C(0x3F10F262), UINT32_C(0x3F780A79) }, //  2: vsAcos ( 0.566198468     ) = ( 0.9689098       );
{ UINT32_C(0x3F18CD22), UINT32_C(0x3F6E626E) }, //  3: vsAcos ( 0.596880078     ) = ( 0.931189418     );
 }

,
    .data_f64 = std::vector<data_2_f64_t>

    {

{ UINT64_C(0x3FE5C5A2B3EB8B46), UINT64_C(0x3FEA5218ABED936B) }, //  0: vdAcos ( 0.680375434309419047      ) = ( 0.822521529957709219      );
{ UINT64_C(0xBFCB09B873361370), UINT64_C(0x3FFC89C38B81B281) }, //  1: vdAcos ( -0.211234146361813924     ) = ( 1.78363375181939454       );
{ UINT64_C(0x3FE21E4C34E43C98), UINT64_C(0x3FEF014F38AD46CB) }, //  2: vdAcos ( 0.566198447517211711      ) = ( 0.968909846016879128      );
{ UINT64_C(0x3FE319A439E63348), UINT64_C(0x3FEDCC4DD5287A83) }, //  3: vdAcos ( 0.596880066952146571      ) = ( 0.931189457249118724      );
 }

,

    .data_c32 = std::vector <data_2_c32_t>

    {

{ { UINT32_C(0xBE584DC4), UINT32_C(0x3F2E2D16) }, { UINT32_C(0x3FDF6BAE), UINT32_C(0xBF252AF2) } }, //  0: vcAcos ( -0.211234152    + i * 0.680375457     ) = ( 1.74547362      + i * -0.645186543    );
{ { UINT32_C(0x3F18CD22), UINT32_C(0x3F10F262) }, { UINT32_C(0x3F8618E7), UINT32_C(0xBF1D4034) } }, //  1: vcAcos ( 0.596880078     + i * 0.566198468     ) = ( 1.04763496      + i * -0.614260912    );
{ { UINT32_C(0xBF1ADA8C), UINT32_C(0x3F52C372) }, { UINT32_C(0x400210E5), UINT32_C(0xBF52C236) } }, //  2: vcAcos ( -0.604897261    + i * 0.823294759     ) = ( 2.03228116      + i * -0.823275924    );
{ { UINT32_C(0x3F095564), UINT32_C(0xBEA8BB5E) }, { UINT32_C(0x3F85D851), UINT32_C(0x3EBE9394) } }, //  3: vcAcos ( 0.536459208     + i * -0.329554498    ) = ( 1.04566395      + i * 0.372219682     );
 }

,
    .data_c64 = std::vector <data_2_c64_t>

    {

{ { UINT64_C(0xBFCB09B873361370), UINT64_C(0x3FE5C5A2B3EB8B46) }, { UINT64_C(0x3FFBED75C72580BA), UINT64_C(0xBFE4A55E2B892514) } }, //  0: vzAcos ( -0.211234146361813924     + i * 0.680375434309419047      ) = ( 1.74547364989852705       + i * -0.645186505346972528     );
{ { UINT64_C(0x3FE319A439E63348), UINT64_C(0x3FE21E4C34E43C98) }, { UINT64_C(0x3FF0C31CDB3C5272), UINT64_C(0xBFE3A80667B42868) } }, //  1: vzAcos ( 0.596880066952146571      + i * 0.566198447517211711      ) = ( 1.04763494147223613       + i * -0.614260866686220375     );
{ { UINT64_C(0xBFE35B518066B6A3), UINT64_C(0x3FEA586E28F4B0DC) }, { UINT64_C(0x4000421CA734509A), UINT64_C(0xBFEA5846A28C515F) } }, //  2: vzAcos ( -0.604897261413232079     + i * 0.823294715873568617      ) = ( 2.03228121403124096       + i * -0.823275868870535166     );
{ { UINT64_C(0x3FE12AAC76625558), UINT64_C(0xBFD5176BB5AA2ED8) }, { UINT64_C(0x3FF0BB0A242AB476), UINT64_C(0x3FD7D272742FD332) } }, //  3: vzAcos ( 0.536459189623808008      + i * -0.32955448857022196      ) = ( 1.04566396835005326       + i * 0.372219670737922503      );
 }

};

    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            } catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL exception:\n"
                          << e.what() << std::endl;
            }
        } // for (std::exception_ptr const& e : exceptions)
    };

    // Create execution queue with asynchronous error handling
    sycl::queue main_queue(dev, exception_handler);

    // Get device name
    std::string dev_name =
        main_queue.get_device().get_info<sycl::info::device::name>();

    // *************************************************************
    // Variable declaraions
    // *************************************************************
    // Input arguments
    A arg1;

    A *varg1 = own_malloc<A>(main_queue, VLEN * sizeof(A));

    R ref1;
    // Output results
    R *vres1 = own_malloc<R>(main_queue, VLEN * sizeof(R));
    R *vref1 = own_malloc<R>(main_queue, VLEN * sizeof(R));

    // Output strided results
    R *vresi1 = own_malloc<R>(main_queue, VLEN * sizeof(R));
    R *vrefi1 = own_malloc<R>(main_queue, VLEN * sizeof(R));
    // Number of errors
    int errs = 0;

    // *************************************************************
    // Vector input data initialization
    // *************************************************************
    for (int i = 0; i < VLEN; ++i) {
        // Getting values from reference data table

        data.get_values(arg1, ref1);

        // Pushing values into vectors
        varg1[i] = arg1;
        vref1[i] = ref1;

        // Fill only uneven  results for strided case
        vrefi1[i] = (i & 1) ? ref1 : R(777);
    } // for (int i = 0; i < VLEN; ++i)

    // *************************************************************
    // Loop by all 3 accuracy modes of VM: HA, LA, EP:
    // set computation mode, run VM and check results
    // *************************************************************
    for (int acc = 0; acc < ACCURACY_NUM; ++acc) {
        std::vector<sycl::event> no_deps;

        // Clear result vectors
        for (int i = 0; i < VLEN; ++i) {
            vres1[i] = R(777);

            vresi1[i] = R(777);
        }

        // Run VM function
        oneapi::mkl::vm::acos(main_queue, VLEN, varg1, vres1, no_deps,
                              accuracy_mode[acc]);

        // Strided VM calls for each uneven element
        {
            oneapi::mkl::vm::acos(main_queue, varg1,
                                  oneapi::mkl::slice(1, VLEN / 2, 2), vresi1,
                                  oneapi::mkl::slice(1, VLEN / 2, 2), no_deps,
                                  accuracy_mode[acc]);
        }

        // Catch sycl exceptions
        try {
            main_queue.wait_and_throw();
        } catch (sycl::exception const &e) {
            std::cerr << "SYCL exception during Accuracy Test\n"
                      << e.what() << " code: " << e.code().value() << std::endl;
            return false;
        }

        // *************************************************************
        // Compute ulp between computed and expected (reference)
        // values and check
        // *************************************************************
        for (int i = 0; i < VLEN; ++i) {
            // Check simple indexing function
            errs +=
                check_result<A, R>("", "acos", ",simple", i, argtype, acc,
                                   varg1[i], varg1[i], vres1[i], vres1[i],
                                   vref1[i], vref1[i], ulp_table, false, false);

            // Check strided indexing function
            errs += check_result<A, R>("", "acos", ",strided", i, argtype, acc,
                                       varg1[i], varg1[i], vresi1[i], vresi1[i],
                                       vrefi1[i], vrefi1[i], ulp_table, false,
                                       false);

        } // for (int i = 0; i < VLEN; ++i)
    }     // for (int acc = 0; acc < ACCURACY_NUM; ++acc)

    std::cout << "\tResult: " << ((errs == 0) ? "PASS" : "FAIL") << std::endl;

    own_free<A>(main_queue, varg1);

    own_free<R>(main_queue, vres1);
    own_free<R>(main_queue, vref1);

    own_free<R>(main_queue, vresi1);
    own_free<R>(main_queue, vrefi1);
    return (errs == 0);
} // template <typename A, typename R> bool vFuncAccuracyText (const device
  // &dev)
//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner() {
    std::cout << "" << std::endl;
    std::cout << "#############################################################"
                 "###########"
              << std::endl;
    std::cout << "# General VM "
              << "acos"
              << " Function Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   vm::"
              << "acos" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;

    std::cout << "#   float" << std::endl;

    std::cout << "#   double" << std::endl;

    std::cout << "#   std::complex<float>" << std::endl;

    std::cout << "#   std::complex<double>" << std::endl;

    std::cout << "#############################################################"
                 "###########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_host -- only runs SYCL HOST device
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU device
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU device
// -DSYCL_DEVICES_all (default) -- runs on all: HOST, CPU and GPU devices
//
//  For each device selected and each data type supported, the example
//  runs with all supported data types
//
int main(int argc, char **argv) {
    // Number of errors occured
    int errs = 0;

    // Print standard banner for VM examples
    print_example_banner();

    // List of available devices
    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    // Loop by all available devices
    for (auto dev_type : list_of_devices) {
        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, dev_type);

        // Run tests if the device is available
        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[dev_type]
                      << ".\n";

            std::cout << "\tRunning with single precision real data type:"
                      << std::endl;
            errs += vAcosAccuracyLiteTest<float, float>(my_dev);

            std::cout << "\tRunning with double precision real data type:"
                      << std::endl;
            errs += vAcosAccuracyLiteTest<double, double>(my_dev);

            std::cout << "\tRunning with single precision complex data type:"
                      << std::endl;
            errs +=
                vAcosAccuracyLiteTest<std::complex<float>, std::complex<float>>(
                    my_dev);

            std::cout << "\tRunning with double precision complex data type:"
                      << std::endl;
            errs += vAcosAccuracyLiteTest<std::complex<double>,
                                          std::complex<double>>(my_dev);

        } else {

            std::cout << "No " << sycl_device_names[dev_type]
                      << " devices found; skipping "
                      << sycl_device_names[dev_type] << " tests.\n";
        }
    }

    return 0;
}
