/*******************************************************************************
* Copyright 2021-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (Intel(R) oneMKL) Sparse BLAS
*   C example using Sparse Coordinate (COO) Matrix Format
*
********************************************************************************
*
* Example program for using Intel oneMKL Inspector-Executor Sparse BLAS routines
* for matrices represented in the coordinate (COO) sparse storage format.
*
* The following Inspector Executor Sparse Blas routines are used in the example:
*
*   Initialization/Destruction stage:
*          mkl_sparse_d_create_coo
*          mkl_sparse_destroy
*
*   Inspector stage:
*          mkl_sparse_set_mv_hint  mkl_sparse_set_sv_hint
*          mkl_sparse_set_mm_hint  mkl_sparse_set_sm_hint
*          mkl_sparse_optimize
*
*   Executor stage:
*          mkl_sparse_d_mv         mkl_sparse_d_trsv
*          mkl_sparse_d_mm         mkl_sparse_d_trsm
*
* Consider the matrix A (see Appendix 'Sparse Storage Formats for Sparse Blas
* level 2-3')
*
*                 |   1       -1     -3    0     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*
* decomposed as
*
*                      A = L + D + U,
*
*  where L is the strict  lower triangle of A, U is the strictly  upper triangle
*  of A, D is the main diagonal. Namely
*
*        |   0    0   0    0     0   |       |  0   -1   -3    0   0   |
*        |  -2    0   0    0     0   |       |  0    0    0    0   0   |
*   L  = |   0    0   0    0     0   |,  U=  |  0    0    0    6   4   |
*        |  -4    0   2    0     0   |       |  0    0    0    0   0   |
*        |   0    8   0    0     0   |       |  0    0    0    0   0   |
*
*
*           |   1  0  0   0   0   |
*           |   0  5  0   0   0   |
*   D    =  |   0  0  4   0   0   |.
*           |   0  0  0   7   0   |
*           |   0  0  0   0  -5   |
*
*  The matrix A given above is represented in the coordinate storage scheme with the help of three
*  arrays of length nnz=13 (see Appendix 'Sparse Storage Formats for Sparse Blas level 2-3'
*
*          values  = (1 -1 -3 -2 5 4 6 4 -4 2 7 8 -5)
*          rows    = (0  0  0  1 1 2 2 2  3 3 3 4  4)
*          columns = (0  1  2  0 1 2 3 4  0 2 3 1  4)
*
*  In what follows the symbol ' means transposition of object preceding the symbol.
*
*  The test performs the following operations :
*
*       1. The code computes (L+D)'*S = F using mkl_sparse_d_mm  where S is a known 5 by 2
*          matrix and then the code solves the system (L+D)'*X = F with the help of
*          mkl_sparse_d_trsm. It's evident that X should be equal to S.
*
*       2. The code computes (U+I)*s = F using mkl_sparse_d_mv where s is a vector
*          and then the code calls mkl_sparse_d_trsv which  solves the system
*          (U+I)*x = F with the single right hand side. It's evident that x
*          should be equal to s.
*
*       3. The code computes D*s = f using mkl_sparse_d_mv  where s is a vector
*          and then the code solves the system D*x = f with the single right hand side.
*          It's evident that s should be equal to s.
*
*       4. The next step is the computation (L+D+L') s = f using mkl_sparse_d_mv
*          where s is a vector. It is easy to see that L+D+L' is a symmetric matrix.
*
*       5. The next step is the computation A'* s = f using mkl_sparse_mv where s
*          is a vector.
*
* The code given below uses only one sparse representation for the all operations.
********************************************************************************
*/
#include <stdio.h>
#include "mkl_types.h"
#include "mkl_spblas.h"

#ifdef MKL_ILP64
#define INT_PRINT_FORMAT "%lld"
#else
#define INT_PRINT_FORMAT "%d"
#endif

int main () {
//*******************************************************************************
//     Definition arrays for sparse representation of the matrix A in
//     the coordinate format:
//*******************************************************************************
#define M 5    /* nrows = ncols = M */
#define NNZ 13
#define NRHS 2
    MKL_INT m = M, nrhs = NRHS;

    double  values[NNZ]   = { 1.0, -1.0,     -3.0,
                             -2.0,  5.0,
                                         4.0, 6.0, 4.0,
                             -4.0,       2.0, 7.0,
                                    8.0,          -5.0 };

    MKL_INT  columns[NNZ] = { 0,      1,        3,
                              0,      1,
                                           2,   3,   4,
                              0,           2,   3,
                                      1,             4 };

    MKL_INT  rows[NNZ]    = { 0,      0,        0,
                              1,      1,
                                           2,   2,   2,
                              3,           3,   3,
                                      4,             4 };

//*******************************************************************************
//    Declaration of local variables :
//*******************************************************************************

    double   sol[M*NRHS]   = {1.0, 5.0,
                              1.0, 4.0,
                              1.0, 3.0,
                              1.0, 2.0,
                              1.0, 1.0};

    double   rhs[M*NRHS]   = {0.0, 0.0,
                              0.0, 0.0,
                              0.0, 0.0,
                              0.0, 0.0,
                              0.0, 0.0};

    double   tmp[M*NRHS]   = {0.0, 0.0,
                              0.0, 0.0,
                              0.0, 0.0,
                              0.0, 0.0,
                              0.0, 0.0};

    double      sol_vec[M]  = {1.0, 1.0, 1.0, 1.0, 1.0};
    double      rhs_vec[M]  = {0.0, 0.0, 0.0, 0.0, 0.0};
    double      temp_vec[M] = {0.0, 0.0, 0.0, 0.0, 0.0};

    double      alpha = 1.0, beta = 0.0;
    MKL_INT     i, j;
    struct matrix_descr descrA;
    sparse_matrix_t cooA;

    sparse_status_t status;
    int exit_status = 0;

    printf( "\n EXAMPLE PROGRAM FOR COO format routines from IE Sparse BLAS\n" );
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//   Create COO sparse matrix handle and analyze step
//*******************************************************************************

    status = mkl_sparse_d_create_coo( &cooA,
                                      SPARSE_INDEX_BASE_ZERO,
                                      m,    // number of rows
                                      m,    // number of cols
                                      NNZ,  // number of nonzeros
                                      rows,
                                      columns,
                                      values );

    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_create_coo: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // First we set hints for the different operations before calling the
    // mkl_sparse_optimize() api which actually does the analyze step.  Not all
    // configurations have optimized steps, so the hint apis may return status
    // MKL_SPARSE_STATUS_NOT_SUPPORTED (=6) if no analysis stage is actually available
    // for that configuration.
    //*******************************************************************************

    //*******************************************************************************
    // Set hints for Task 1: Lower triangular transpose MM and SM solve with
    // non-unit diagonal and row-major format
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mm_hint(cooA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_mm_hint: %d \n", status);
    }

    status = mkl_sparse_set_sm_hint(cooA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_sm_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 2: Upper triangular transpose MV and SV solve
    // with unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_set_mv_hint(cooA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 2: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    status = mkl_sparse_set_sv_hint(cooA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 2: mkl_sparse_set_sv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 3: Diagonal MV and SV
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_DIAGONAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(cooA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 3: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    status = mkl_sparse_set_sv_hint(cooA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 3: mkl_sparse_set_sv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 4: Lower symmetric MV with non-unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_SYMMETRIC;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(cooA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 4: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Set hints for Task 5: General transpose MV with non-unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(cooA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 5: mkl_sparse_set_mv_hint: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // Analyze sparse matrix; choose proper kernels and workload balancing strategy
    //*******************************************************************************
    status = mkl_sparse_optimize ( cooA );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_optimize: %d \n", status);
        exit_status = 1;
        goto exit;
    }

//*******************************************************************************
//    Task 1.  Obtain Triangular matrix-matrix multiply (L+D)' *sol --> rhs
//    and solve triangular system   (L+D)' *tmp = rhs with multiple right
//    hand sides. Array tmp must be equal to the array sol
//*******************************************************************************
    printf("                                  \n");
    printf("   TASK 1:                        \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mm \n");
    printf("   WITH LOWER TRIANGULAR MATRIX   \n");
    printf("     m = " INT_PRINT_FORMAT "   NRHS = " INT_PRINT_FORMAT "\n", m, nrhs);
    printf("     ALPHA = %4.1f  BETA = %4.1f  \n", alpha, beta);
    printf("     SPARSE_OPERATION_TRANSPOSE   \n" );
    printf("   Input matrix                   \n");
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", sol[i*nrhs+j]);
        };
        printf("\n");
    };

    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mm( SPARSE_OPERATION_TRANSPOSE, alpha, cooA, descrA,
                              SPARSE_LAYOUT_ROW_MAJOR, sol, nrhs, nrhs, beta,
                              rhs, nrhs);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_mm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mm \n");
    printf("   WITH TRIANGULAR MATRIX          \n");
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", rhs[i*nrhs+j]);
        };
        printf("\n");
    };

    printf("-----------------------------------------------\n");
    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");
    printf("                             \n");

    status =  mkl_sparse_d_trsm( SPARSE_OPERATION_TRANSPOSE, alpha, cooA, descrA, SPARSE_LAYOUT_ROW_MAJOR, rhs, nrhs, nrhs, tmp, nrhs);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_trsm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                     \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_trsm \n");
    printf("   WITH TRIANGULAR MATRIX            \n");
    for (i = 0; i < m; i++) {
        for (j = 0; j < nrhs; j++) {
            printf("%7.1f", tmp[i*nrhs+j]);
        };
        printf("\n");
    };
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 2.    Obtain Triangular matrix-vector multiply (U+I) *sol --> rhs
//    and solve triangular system   (U+I) *tmp = rhs with single right hand sides
//    Array tmp must be equal to the array sol
//*******************************************************************************
    printf("                                     \n");
    printf("   TASK 2:                           \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv    \n");
    printf("   WITH UNIT UPPER TRIANGULAR MATRIX \n");
    printf("     ALPHA = %4.1f  BETA = %4.1f     \n", alpha, beta);
    printf("     SPARSE_OPERATION_NON_TRANSPOSE  \n" );
    printf("   Input vector                      \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", sol_vec[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cooA, descrA, sol_vec, beta, rhs_vec);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH TRIANGULAR MATRIX          \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", rhs_vec[i]);
    };
    printf("-----------------------------------------------\n");
    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");

    status =  mkl_sparse_d_trsv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cooA, descrA, rhs_vec, temp_vec);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                     \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_trsv \n");
    printf("   WITH TRIANGULAR MATRIX            \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", temp_vec[i]);
    };
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 3.  Obtain Diagonal matrix-vector multiply D *sol --> rhs
//    and solve triangular system   D *temp = rhs with single right hand side
//    Array temp must be equal to the array sol
//*******************************************************************************
    printf("                                    \n");
    printf("   TASK 3:                          \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv   \n");
    printf("   WITH DIAGONAL MATRIX             \n");
    printf("     m = " INT_PRINT_FORMAT "       \n", m);
    printf("     SPARSE_OPERATION_NON_TRANSPOSE \n" );
    printf("   Input vector                     \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", sol_vec[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_DIAGONAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cooA, descrA, sol_vec, beta, rhs_vec);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 3 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH DIAGONAL MATRIX            \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", rhs_vec[i]);
    };
    printf("-----------------------------------------------\n");
    printf("   Multiply by inverse      \n");
    printf("   matrix with the help     \n");
    printf("   of MKL_SPARSE_D_TRSV            \n");

    status =  mkl_sparse_d_trsv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cooA, descrA, rhs_vec, temp_vec);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 3  mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                     \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_trsv \n");
    printf("   WITH DIAGONAL MATRIX              \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", temp_vec[i]);
    };
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 4.  Obtain Symmetric matrix-vector multiply (L+D+L')*sol --> rhs
//    with the help of MKL_SPARSE_D_MV
//
//*******************************************************************************
    printf("                                    \n");
    printf("   TASK 4:                          \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv   \n");
    printf("   WITH SYMMETRIC LOWER MATRIX      \n");
    printf("     ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta);
    printf("     SPARSE_OPERATION_NON_TRANSPOSE \n" );
    printf("   Input vector                     \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", sol_vec[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_SYMMETRIC;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_NON_TRANSPOSE, alpha, cooA, descrA, sol_vec, beta, rhs_vec);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 4 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH SYMMETRIC LOWER MATRIX     \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", rhs_vec[i]);
    };
    printf("-----------------------------------------------\n");

//*******************************************************************************
//    Task 5. Obtain General matrix-vector multiply A'*sol --> rhs with
//    the help of MKL_SPARSE_D_MV
//
//*******************************************************************************
    printf("                                 \n");
    printf("   TASK 5:                       \n");
    printf("   INPUT DATA FOR mkl_sparse_d_mv\n");
    printf("   WITH GENERAL MATRIX           \n");
    printf("     ALPHA = %4.1f  BETA = %4.1f \n", alpha, beta);
    printf("     SPARSE_OPERATION_TRANSPOSE  \n" );
    printf("   Input vector                  \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", sol_vec[i]);
    };

    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv( SPARSE_OPERATION_TRANSPOSE, alpha, cooA, descrA, sol_vec, beta, rhs_vec);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 5 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_d_mv \n");
    printf("   WITH GENERAL MATRIX             \n");
    for (i = 0; i < m; i++) {
        printf("%7.1f\n", rhs_vec[i]);
    };
    printf("-----------------------------------------------\n");


exit:
    // Release matrix handle and deallocate matrix
    mkl_sparse_destroy ( cooA );

    return exit_status;
}
