/*******************************************************************************
* Copyright 2021-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Sparse BLAS C OpenMP
*             offload example for mkl_sparse_s_trsv with async execution
*
********************************************************************************
*
* Consider the matrix A (see 'Sparse Storage Formats for Sparse BLAS Level 2
* and Level 3 in the  Intel oneMKL Reference Manual')
*
*                 |   1       -1      0   -3     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*  The matrix A is represented in a zero-based compressed sparse row (CSR) storage
*  scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
*   Intel oneMKL Reference Manual) as follows:
*
*         values  =  ( 1 -1 -3 -2  5  4  6  4 -4  2  7  8 -5 )
*         columns =  ( 0  1  3  0  1  2  3  4  0  2  3  1  4 )
*         rowIndex = ( 0        3     5        8       11    13 )
*
*  The test performs the operation
*
*       U^{-1}*x = y, using mkl_sparse_s_trsv
*
*  Here A is a general sparse matrix
*        U is the upper triangular part of matrix A with unit diagonal
*        w, x, y, and z are vectors
*
********************************************************************************
*/
#include <stdio.h>
#include <assert.h>
#include <math.h>

#include "common_for_sparse_examples.h"
#include "mkl.h"
#include "mkl_omp_offload.h"

int main() {
    //*******************************************************************************
    //     Declaration and initialization of parameters for sparse representation of
    //     the matrix A in the compressed sparse row format:
    //*******************************************************************************
#define M 5
#define N 5
#define NNZ 13
    // Descriptor of main sparse matrix properties
    struct matrix_descr descrA;

    // Structure with sparse matrix stored in CSR format
    sparse_matrix_t csrA;
    //*******************************************************************************
    //    Declaration of local variables:
    //*******************************************************************************
    float alpha = 1.0f;
    MKL_INT i;

    float *values      = (float *)mkl_malloc(sizeof(float) * NNZ, 64);
    MKL_INT *columns   = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * NNZ, 64);
    MKL_INT *row_index = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * (N + 1), 64);
    float *x           = (float *)mkl_malloc(sizeof(float) * N, 64);
    float *w           = (float *)mkl_malloc(sizeof(float) * N, 64);

    float *y  = (float *)mkl_malloc(sizeof(float) * M, 64);
    float *z1 = (float *)mkl_malloc(sizeof(float) * M, 64);
    float *z2 = (float *)mkl_malloc(sizeof(float) * M, 64);

    const int num_pointers = 8;
    void *pointer_array[num_pointers];
    pointer_array[0] = values;
    pointer_array[1] = columns;
    pointer_array[2] = row_index;
    pointer_array[3] = x;
    pointer_array[4] = w;
    pointer_array[5] = y;
    pointer_array[6] = z1;
    pointer_array[7] = z2;

    if (!values || !columns || !row_index || !x || !y || !z1 || !z2 || !w) {
        free_allocated_memories(pointer_array, num_pointers);
        return 1;
    }

    //*******************************************************************************
    //    Sparse representation of the matrix A
    //*******************************************************************************
    float init_values[NNZ] = {1.0f, -1.0f, -3.0f, -2.0f, 5.0f, 4.0f, 6.0f,
                              4.0f, -4.0f, 2.0f,  7.0f,  8.0f, -5.0f};
    MKL_INT init_columns[NNZ]     = {0, 1, 3, 0, 1, 2, 3, 4, 0, 2, 3, 1, 4};
    MKL_INT init_row_index[N + 1] = {0, 3, 5, 8, 11, 13};
    float init_x[N]               = {1.0f, 5.0f, 1.0f, 4.0f, 1.0f};

    for (i = 0; i < NNZ; i++) {
        values[i]  = init_values[i];
        columns[i] = init_columns[i];
    }
    for (i = 0; i < N + 1; i++) {
        row_index[i] = init_row_index[i];
    }

    for (i   = 0; i < N; i++) {
        x[i] = init_x[i];
        w[i] = init_x[i];
    }
    for (i = 0; i < M; i++) {
        y[i]  = 0.0f;
        z1[i] = 0.0f;
        z2[i] = 0.0f;
    }

    printf("\n EXAMPLE PROGRAM FOR mkl_sparse_s_trsv omp_offload \n");
    printf("-----------------------------------------------------\n");
    printf("\n");
    printf("   INPUT DATA FOR mkl_sparse_s_trsv omp offload async \n");
    printf("   UPPER TRIANGULAR SPARSE MATRIX \n");
    printf("   WITH UNIT DIAGONAL             \n");
    printf("   ALPHA = %4.1f \n", alpha);
    printf("   SPARSE_OPERATION_NON_TRANSPOSE \n");
    printf("   Input vector                   \n");
    for ( i = 0; i < N; i++ ) {
        printf("%7.1f\n", x[i]);
    }
    printf("---------------------------------------------------\n");
    fflush(stdout);

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    sparse_status_t ie_status;

    // Create handle with matrix stored in CSR format
    ie_status = mkl_sparse_s_create_csr(&csrA, SPARSE_INDEX_BASE_ZERO,
                                        N, // number of rows
                                        M, // number of cols
                                        row_index, row_index + 1, columns, values);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_create_csr: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    ie_status = mkl_sparse_set_sv_hint(csrA, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1);
    if (ie_status != SPARSE_STATUS_SUCCESS && ie_status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set_sv_hint: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    ie_status = mkl_sparse_optimize(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_optimize: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Compute y = alpha * A^{-1} * x
    ie_status = mkl_sparse_s_trsv(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA, descrA, x, y);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_trsv: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Release matrix handle and deallocate matrix
    ie_status = mkl_sparse_destroy(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_destroy: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    printf( "                                   \n" );
    printf( "   OUTPUT DATA FOR mkl_sparse_s_trsv \n" );

    // y should be equal { 18.0, 5.0, -27.0, 4.0, 1.0 }
    for ( i = 0; i < N; i++ )
    {
        printf( "%7.4f\n", y[i] );
    };
    printf("---------------------------------------------------\n");
    fflush(stdout);

    const int devNum = 0;

    sparse_matrix_t csrA_gpu;

    sparse_status_t status_create;
    sparse_status_t status_hint;
    sparse_status_t status_opt;
    sparse_status_t status_trsv1;
    sparse_status_t status_trsv2;
    sparse_status_t status_destroy;

// call create_csr/set_sv_hint/optimize/trsv/destroy via omp_offload.
#pragma omp target data map(to:row_index[0:N+1],columns[0:NNZ],values[0:NNZ],x[0:N],w[0:N])          \
                        map(tofrom:z1[0:M],z2[0:M]) device(devNum)
    {
        printf("Create CSR matrix via omp_offload\n");

#pragma omp target variant dispatch device(devNum) use_device_ptr(row_index, columns, values)
        status_create = mkl_sparse_s_create_csr(&csrA_gpu, SPARSE_INDEX_BASE_ZERO, N, M,
                                                row_index, row_index + 1, columns, values);

        printf("mkl_sparse_set_sv_hint() ... \n");
#pragma omp target variant dispatch device(devNum) nowait
        status_hint = mkl_sparse_set_sv_hint(csrA_gpu, SPARSE_OPERATION_NON_TRANSPOSE, descrA, 1);
#pragma omp taskwait

        printf("mkl_sparse_optimize() ... \n");
        // NOTE: users can run mkl_sparse_optimize offload functions in async execution,
        // but users should also handle the dependency between the optimize() call and
        // the computation calls, i.e. trsv() here.
#pragma omp target variant dispatch device(devNum) nowait
        status_opt = mkl_sparse_optimize(csrA_gpu);
#pragma omp taskwait
        printf("Compute mkl_sparse_s_trsv via omp_offload\n");

#pragma omp target variant dispatch device(devNum) use_device_ptr(x, z1) nowait
        status_trsv1 = mkl_sparse_s_trsv(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA_gpu, descrA, x, z1);

#pragma omp target variant dispatch device(devNum) use_device_ptr(w, z2) nowait
        status_trsv2 = mkl_sparse_s_trsv(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA_gpu, descrA, w, z2);
#pragma omp taskwait
        printf("Destroy the CSR matrix via omp_offload\n");

#pragma omp target variant dispatch device(devNum)
        status_destroy = mkl_sparse_destroy(csrA_gpu);

        printf("End of mkl_sparse_destroy omp offload call.\n");
    }
    printf("End all offload function calls.\n"); fflush(stdout);

    int flps_per_value = 2 * ((NNZ / N) + 1);
    int status1        = 0;
    int status2        = 0;

    int status_offload = status_create | status_hint | status_opt | status_trsv1 | status_trsv2 | status_destroy;
    if (status_offload != 0) {
        printf("\tERROR: status_create = %d, status_hint = %d, status_opt = %d, "
               "status_trsv1 = %d, status_trsv2 = %d, status_destroy = %d\n",
               status_create, status_hint, status_opt, status_trsv1, status_trsv2, status_destroy);
        fflush(stdout);

        goto cleanup;
    }

    printf("   OUTPUT DATA FOR mkl_sparse_s_trsv_omp_offload async execution\n");
    // z should be equal { 18.0, 5.0, -27.0, 4.0, 1.0 }
    for (i = 0; i < N; i++) {
        printf("%7.4f\n", z1[i]);
    }
    printf("---------------------------------------------------\n");
    fflush(stdout);

    status1 = validation_result_float(y, z1, M, flps_per_value);
    status2 = validation_result_float(y, z2, M, flps_per_value);

cleanup:
    free_allocated_memories(pointer_array, num_pointers);

    const int status_all = status1 | status2 | status_offload;
    printf("Test %s\n", status_all == 0 ? "PASSED" : "FAILED");
    fflush(stdout);

    return status_all;
}
