/*******************************************************************************
* Copyright 2020-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Sparse BLAS C OpenMP
*             offload example for mkl_sparse_s_mv with one-based indexing.
*
********************************************************************************
*
* Consider the matrix A (see 'Sparse Storage Formats for Sparse BLAS Level 2
* and Level 3 in the  Intel oneMKL Reference Manual')
*
*                 |   1       -1      0   -3     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*  The matrix A is represented in a zero-based compressed sparse row (CSR) storage
*  scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
*   Intel oneMKL Reference Manual) as follows:
*
*         values  =  ( 1 -1 -3 -2  5  4  6  4 -4  2  7  8 -5 )
*         columns =  ( 1  2  4  1  2  3  4  5  1  3  4  2  5 )
*         row_index = ( 1        4     6        9       12    14 )
*
*  The test computes the following operations :
*
*       A*x = y using mkl_sparse_s_mv omp offload
*       where A is a general sparse matrix and x and y are vectors
*
********************************************************************************
*/
#include <assert.h>
#include <omp.h>
#include <stdio.h>
#include <stdlib.h>

#include "common_for_sparse_examples.h"
#include "mkl.h"
#include "mkl_omp_offload.h"

int main()
{
//*******************************************************************************
//     Declaration and initialization of parameters for sparse representation of
//     the matrix A in the compressed sparse row format:
//*******************************************************************************
#define M 5
#define N 5
#define NNZ 13
    // Descriptor of main sparse matrix properties
    struct matrix_descr descrA;
    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    // // Structure with sparse matrix stored in CSR format
    sparse_matrix_t csrA;
    //*******************************************************************************
    //    Declaration of local variables:
    //*******************************************************************************
    float alpha = 1.0f, beta = 0.0f;
    MKL_INT i;

    float *values      = (float *)mkl_malloc(sizeof(float) * NNZ, 64);
    MKL_INT *columns   = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * NNZ, 64);
    MKL_INT *row_index = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * (N + 1), 64);
    float *x           = (float *)mkl_malloc(sizeof(float) * M, 64);

    float *y = (float *)mkl_malloc(sizeof(float) * N, 64);
    float *z = (float *)mkl_malloc(sizeof(float) * N, 64);

    const int num_pointers = 6;
    void *pointer_array[num_pointers];
    pointer_array[0] = values;
    pointer_array[1] = columns;
    pointer_array[2] = row_index;
    pointer_array[3] = x;
    pointer_array[4] = y;
    pointer_array[5] = z;

    if (!values || !columns || !row_index || !x || !y || !z) {
        free_allocated_memories(pointer_array, num_pointers);
        return 1;
    }

    //*******************************************************************************
    //    Sparse representation of the matrix A
    //*******************************************************************************
    float init_values[NNZ] = {1.0f, -1.0f, -3.0f, -2.0f, 5.0f, 4.0f, 6.0f,
                              4.0f, -4.0f, 2.0f,  7.0f,  8.0f, -5.0f};
    MKL_INT init_columns[NNZ]     = {1, 2, 4, 1, 2, 3, 4, 5, 1, 3, 4, 2, 5};
    MKL_INT init_row_index[N + 1] = {1, 4, 6, 9, 12, 14};
    float init_x[M]               = {1.0f, 5.0f, 1.0f, 4.0f, 1.0f};

    for (i = 0; i < NNZ; i++) {
        values[i]  = init_values[i];
        columns[i] = init_columns[i];
    }
    for (i = 0; i < N + 1; i++) {
        row_index[i] = init_row_index[i];
    }

    for (i   = 0; i < M; i++)
        x[i] = init_x[i];
    for (i = 0; i < N; i++) {
        y[i] = 0.0f;
        z[i] = 0.0f;
    }

    printf("\n EXAMPLE PROGRAM FOR mkl_sparse_s_mv omp_offload \n");
    printf("---------------------------------------------------\n");
    printf("\n");
    printf("   INPUT DATA FOR mkl_sparse_s_mv omp offload w/ 1-based indexing \n");
    printf("   WITH GENERAL SPARSE MATRIX     \n");
    printf("   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta);
    printf("   SPARSE_OPERATION_NON_TRANSPOSE \n");
    printf("   Input vector                   \n");
    for (i = 0; i < M; i++) {
        printf("%7.1f\n", x[i]);
    }

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    sparse_status_t ie_status;

    // Create handle with matrix stored in CSR format
    ie_status = mkl_sparse_s_create_csr(&csrA, SPARSE_INDEX_BASE_ONE,
                                        N, // number of rows
                                        M, // number of cols
                                        row_index, row_index + 1, columns, values);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_create_csr: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Compute y = alpha * A * x + beta * y
    ie_status = mkl_sparse_s_mv(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA, descrA, x, beta, y);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_mv: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Release matrix handle and deallocate matrix
    ie_status = mkl_sparse_destroy(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_destroy: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    printf("                                   \n");
    printf("   OUTPUT DATA FOR mkl_sparse_s_mv \n");

    // y should be equal { -16.0, 23.0, 32.0, 26.0, 35.0 }
    for (i = 0; i < N; i++) {
        printf("%7.1f\n", y[i]);
    }

    printf("---------------------------------------------------\n");
    fflush(stdout);

    const int devNum = 0;

    sparse_matrix_t csrA_gpu;

    sparse_status_t status_create;
    sparse_status_t status_mv;
    sparse_status_t status_destroy;

// call create_csr/mv/destroy via omp_offload.
#pragma omp target data map(to:row_index[0:N+1],columns[0:NNZ],values[0:NNZ],x[0:M])          \
                        map(tofrom:z[0:N]) device(devNum)
    {
        printf("Create CSR matrix via omp_offload\n");

#pragma omp target variant dispatch device(devNum) use_device_ptr(row_index, columns, values)
        status_create = mkl_sparse_s_create_csr(&csrA_gpu, SPARSE_INDEX_BASE_ONE, N, M,
                                                row_index, row_index + 1, columns, values);

        printf("Compute mkl_sparse_s_mv via omp_offload\n");

#pragma omp target variant dispatch device(devNum) use_device_ptr(x, z)
        status_mv = mkl_sparse_s_mv(SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA_gpu, descrA, x,
                                    beta, z);

        printf("Destroy the CSR matrix via omp_offload\n");

#pragma omp target variant dispatch device(devNum)
        status_destroy = mkl_sparse_destroy(csrA_gpu);
    }

    int flps_per_value = 2 * ((NNZ / N) + 1);
    int status         = 0;

    int status_offload = status_create | status_mv | status_destroy;
    if (status_offload != 0) {
        printf("\tERROR: status_create = %d, status_mv = %d, status_destroy = %d\n", status_create,
               status_mv, status_destroy);
        goto cleanup;
    }

    printf("   OUTPUT DATA FOR mkl_sparse_s_mv_omp_offload \n");
    // z should be equal { -16.0, 23.0, 32.0, 26.0, 35.0 }
    for (i = 0; i < N; i++) {
        printf("%7.1f\n", z[i]);
    }
    printf("---------------------------------------------------\n");
    fflush(stdout);

    status = validation_result_float(y, z, N, flps_per_value);

cleanup:
    free_allocated_memories(pointer_array, num_pointers);

    const int status_all = status | status_offload;
    printf("Test %s\n", status_all == 0 ? "PASSED" : "FAILED");
    fflush(stdout);

    return status_all;
}
