LoadPackage("NormalizInterface");

docstrings := rec(

AffineDim := rec(
  returns := "the affine dimension",
  description := """
The affine dimension of the polyhedron in inhomogeneous computations. Its computation is triggered if necessary.
""",
),

ClassGroup := rec(
  returns := "the class group in a special format",
  description := """
A normal affine monoid $M$ has a well-defined divisor class group.
It is naturally isomorphic to the divisor class group of $K[M]$ where $K$
is a field (or any unique factorization domain).
We represent it as a vector where the first entry is the rank. It is
followed by sequence of pairs of entries <M>n,m</M>. Such two entries
represent a free cyclic summand <M>(\mathbb{Z}/n\mathbb{Z})^m</M>.
Not allowed in inhomogeneous computations.
""",
),

Congruences := rec(
  returns := "a matrix whose rows represent the congruences",
  description := """
The equations, congruences and support hyperplanes together
describe the lattice points of the cone.
""",
),

Deg1Elements := rec(
  returns := "a matrix whose rows are the degree 1 elements",
  description := """
Requires the presence of a grading. Not allowed in inhomogeneous computations.
""",
),

Dehomogenization := rec(
  returns := "the dehomgenization vector",
  description := """
Only for inhomogeneous computations.
""",
),

EmbeddingDim := rec(
  rename := "EmbeddingDimension",
  returns := "the embedding dimension of the cone",
  description := """
The embedding dimension is the dimension of the space in which the
computation is done. It is the number of components of the output vectors.
This value is always known directly after the creation of the cone.
""",
),

Equations := rec(
  returns := "a matrix whose rows represent the equations",
  description := """
The equations cut out the linear space generated by the cone.
The equations, congruences and support hyperplanes together
describe the lattice points of the cone.
""",
),

ExcludedFaces := rec(
  returns := "a matrix whose rows represent the excluded faces",
  description := "",
),

ExtremeRays := rec(
  returns := "a matrix whose rows are the extreme rays",
  description := "",
),

Generators := rec(
  returns := "a matrix whose rows are the generators of <A>cone</A>",
  description := "",
),

GeneratorOfInterior := rec(
  returns := "a vector representing the generator of the interior of <A>cone</A>",
  description := """
If <A>cone</A> is Gorenstein, this function returns the generator of the interior of <A>cone</A>.
If <A>cone</A> is not Gorenstein, an error is raised.
""",
),

Grading := rec(
  returns := "the grading vector",
  description := "",
),

HilbertBasis := rec(
  returns := "a matrix whose rows are the Hilbert basis elements",
  description := "",
),

HilbertQuasiPolynomial := rec(
  returns := "the Hilbert function as a quasipolynomial",
  description := """
The Hilbert function counts the lattice points degreewise. The result is a
quasipolynomial <M>Q</M>, that is, a polynomial with periodic coefficients. It is
given as list of polynomials <M>P_0, \ldots, P_{(p-1)}</M> such that <M>Q(i) = P_{(i \bmod p)} (i)</M>.
""",
),

HilbertSeries := rec(
  returns := "the Hilbert series as rational function",
  description := """
The result consists of a list with two entries. The first is the numerator
polynomial. In inhomogeneous computations this can also be a Laurent
polynomial. The second list entry represents the denominator. It is a list
of pairs <M>[k_i, l_i]</M>. Such a pair represents the factor <M>(1-t^{k_i})^{l_i}</M>.
""",
),

InclusionExclusionData := rec(
  returns := "inclusion-exclusion data",
  description := """
List of faces which are internally have been used in the inclusion-exclusion
scheme. Given as a list pairs. The first pair entry is a key of generators
contained in the face (compare also <Ref Func="NmzTriangulation"/>) and the
multiplicity with which it was considered.
Only available with excluded faces or strict constraints as input.
""",
),

IsDeg1ExtremeRays := rec(
  returns := "<K>true</K> if all extreme rays have degree 1; <K>false</K> otherwise",
  description := "",
),

IsDeg1HilbertBasis := rec(
  returns := "<K>true</K> if all Hilbert basis elements have degree 1; <K>false</K> otherwise",
  description := "",
),

IsGorenstein := rec(
  returns := "whether the cone is Gorenstein",
  description := """
Returns true if <A>cone</A> is Gorenstein, false otherwise.
""",
),

IsInhomogeneous := rec(
  returns := "whether the cone is inhomogeneous",
  description := """
This value is always known directly after the creation of the cone.
""",
),

IsIntegrallyClosed := rec(
  returns := "<K>true</K> if the cone is integrally closed; <K>false</K> otherwise",
  description := """
It is integrally closed when the Hilbert basis is a subset of the original monoid generators. So it is only computable if we have original monoid generators.
""",
),

IsPointed := rec(
  returns := "<K>true</K> if the cone is pointed; <K>false</K> otherwise",
  description := "",
),

IsReesPrimary := rec(
  returns := "<K>true</K> if is the monomial ideal is primary to the irrelevant maximal ideal, <K>false</K> otherwise",
  description := """
Only used with the input type <C>rees_algebra</C>.
""",
),

MaximalSubspace := rec(
  returns := "a matrix whose rows generate the maximale linear subspace",
  description := "",
),

ModuleGenerators := rec(
  returns := "a matrix whose rows are the module generators",
  description := "",
),

ModuleGeneratorsOverOriginalMonoid := rec(
  returns := "a matrix whose rows are the module generators over the original monoid",
  description := """
A minimal system of generators of  the integral closure over the original monoid.
Requires the existence of original monoid generators. Not allowed in inhomogeneous computations.
""",
),

ModuleRank := rec(
  returns := "the rank of the module of lattice points in the polyhedron as a module over the recession monoid",
  description := """
Only for inhomogeneous computations.
""",
),

OriginalMonoidGenerators := rec(
  returns := "a matrix whose rows are the original monoid generators",
  description := "",
),

Rank := rec(
  returns := "the rank of the cone",
  description := """
This value is the rank of the lattice generated by the lattice points of the cone.
""",
),

RecessionRank := rec(
  returns := "the rank of the recession cone",
  description := """
Only for inhomogeneous computations.
""",
),

ReesPrimaryMultiplicity := rec(
  description := """
the multiplicity of a monomial ideal, provided it is primary to the maximal
ideal generated by the indeterminates. Used only with the input type
<C>rees_algebra</C>.
""",
),

SupportHyperplanes := rec(
  returns := "a matrix whose rows represent the support hyperplanes",
  description := """
The equations cut out the linear space generated by the cone.
The equations, congruences and support hyperplanes together
describe the lattice points of the cone.
""",
),

Triangulation := rec(
  returns := "the triangulation",
  description := """
This returns a list of the maximal simplicial cones in a triangulation, i.e., a list of cones dividing the
cone into simplicial cones. Each cone in the list is represented by a pair.
The first entry of such a pair is the key of the simplex, i.e., a list of integers $a_1,\dots,a_n$
referring to the <Ref Func="NmzGenerators"/> (counting from 0) which are used in this simplicial cone.
The second entry of each pair in the list is the absolute value of the determinant of the generator matrix of the simplicial cone.
""",
),

TriangulationDetSum := rec(
  returns := "sum of the absolute values of the determinants of the simplicial cones in the used triangulation",
  description := "",
),

TriangulationSize := rec(
  returns := "the number of simplicial cones in the used triangulation",
  description := "",
),

VerticesFloat := rec(
  returns := "a matrix whose rows are the vertices of the polyhedron <A>cone</A> with float coordinates",
  description := """
The rows of this matrix represent the vertices of <A>cone</A>, printed as floats for better readability.
The result might be inexact, and should therefore not be used for computations.
""",
),

VerticesOfPolyhedron := rec(
  returns := "a matrix whose rows are the vertices of the polyhedron",
  description := "",
),


EhrhartSeries := rec(
  description := """
 Supported in Normaliz >= 3.5.0.
""",
),

EuclideanVolume := rec(
  description := """
 Supported in Normaliz >= 3.5.0.
""",
),

ProjectCone := rec(
  description := """
 Supported in Normaliz >= 3.5.0.
""",
),

SuppHypsFloat := rec(
  description := """
 Supported in Normaliz >= 3.5.2.
""",
),

Volume := rec(
  description := """
 Supported in Normaliz >= 3.5.0.
""",
),

FVector := rec(
  description := """
 Supported in Normaliz >= 3.7.0.
""",
),

FaceLattice := rec(
  description := """
 Supported in Normaliz >= 3.7.0.
""",
),

Incidence := rec(
  description := """
 Supported in Normaliz >= 3.8.0.
""",
),

);

all_props := Set(RecNames(_NmzRecordOfConeProperties));
documented_props := RecNames(docstrings);

for x in Difference(documented_props, all_props) do
    Print("Property ", x, " is documented but not actually available\n");
od;

for x in Difference(all_props, documented_props) do
    Print("Property ", x, " is undocumented\n");
od;

stream := OutputTextFile("lib/cone_property_wrappers.gd", false);
SetPrintFormattingStatus(stream, false);

PrintTo(stream,"""# WARNING: This file was generated by etc/generate_cone_property_wrappers.g,
# DO NOT EDIT IT BY HAND!

#! @Chapter Functions
#! @Section Cone properties

""");
for x in all_props do
    if IsBound(docstrings.(x)) then
        r := docstrings.(x);
    else
        r := rec(description:="");
    fi;
    if IsBound(r.rename) then y := r.rename; else y := x; fi;
    PrintTo(stream, "#! @Arguments cone\n");
    if IsBound(r.returns) then
        PrintTo(stream, "#! @Returns ", r.returns, "\n");
    fi;
    PrintTo(stream, "#! @Description\n");
    lines := SplitString(r.description, "\n");
    for l in lines do
        PrintTo(stream, "#! ", l ,"\n");        
    od;
    if Length(lines) > 0 then
        PrintTo(stream, "#! <P/>\n");
    fi;
    PrintTo(stream, "#! This is an alias for <C>NmzConeProperty( cone, \"", x, "\" );</C> see <Ref Func=\"NmzConeProperty\"/>.\n");
    PrintTo(stream, "DeclareGlobalFunction( \"Nmz", y, "\" );\n");
od;

CloseStream(stream);


stream := OutputTextFile("lib/cone_property_wrappers.gi", false);
SetPrintFormattingStatus(stream, false);
PrintTo(stream,"""# WARNING: This file was generated by etc/generate_cone_property_wrappers.g,
# DO NOT EDIT IT BY HAND!

""");for x in all_props do
    y := x;
    if IsBound(docstrings.(x)) then
        r := docstrings.(x);
        if IsBound(r.rename) then
            y := r.rename;
        fi;
    fi;
    PrintTo(stream, "InstallGlobalFunction( Nmz", y, ", cone -> NmzConeProperty( cone, \"", x,"\" ) );\n");
od;
CloseStream(stream);
