"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.run = void 0;
const globby_1 = __importDefault(require("globby"));
const isString_1 = __importDefault(require("lodash/isString"));
const os_1 = __importDefault(require("os"));
const path_1 = __importDefault(require("path"));
const prompts_ncu_1 = __importDefault(require("prompts-ncu"));
const spawn_please_1 = __importDefault(require("spawn-please"));
const cli_options_1 = require("./cli-options");
const chalk_1 = __importStar(require("./lib/chalk"));
const doctor_1 = __importDefault(require("./lib/doctor"));
const exists_1 = __importDefault(require("./lib/exists"));
const findPackage_1 = __importDefault(require("./lib/findPackage"));
const getNcuRc_1 = __importDefault(require("./lib/getNcuRc"));
const getPackageFileName_1 = __importDefault(require("./lib/getPackageFileName"));
const initOptions_1 = __importDefault(require("./lib/initOptions"));
const logging_1 = require("./lib/logging");
const mergeOptions_1 = __importDefault(require("./lib/mergeOptions"));
const programError_1 = __importDefault(require("./lib/programError"));
const runGlobal_1 = __importDefault(require("./lib/runGlobal"));
const runLocal_1 = __importDefault(require("./lib/runLocal"));
const package_managers_1 = __importDefault(require("./package-managers"));
// allow prompt injection from environment variable for testing purposes
if (process.env.INJECT_PROMPTS) {
    prompts_ncu_1.default.inject(JSON.parse(process.env.INJECT_PROMPTS));
}
// exit with non-zero error code when there is an unhandled promise rejection
process.on('unhandledRejection', err => {
    throw err;
});
/**
 * Volta is a tool for managing JavaScript tooling like Node and npm. Volta has
 * its own system for installing global packages which circumvents npm, so
 * commands like `npm ls -g` do not accurately reflect what is installed.
 *
 * The ability to use `npm ls -g` is tracked in this Volta issue: https://github.com/volta-cli/volta/issues/1012
 */
function checkIfVolta(options) {
    var _a;
    // The first check is for macOS/Linux and the second check is for Windows
    if (options.global && (!!process.env.VOLTA_HOME || ((_a = process.env.PATH) === null || _a === void 0 ? void 0 : _a.includes('\\Volta')))) {
        const message = 'It appears you are using Volta. `npm-check-updates --global` ' +
            'cannot be used with Volta because Volta has its own system for ' +
            'managing global packages which circumvents npm.\n\n' +
            'If you are still receiving this message after uninstalling Volta, ' +
            'ensure your PATH does not contain an entry for Volta and your ' +
            'shell profile does not define VOLTA_HOME. You may need to reboot ' +
            'for changes to your shell profile to take effect.';
        (0, logging_1.print)(options, message, 'error');
        process.exit(1);
    }
}
/** Returns the package manager that should be used to install packages after running "ncu -u". Detects pnpm via pnpm-lock.yarn. This is the one place that pnpm needs to be detected, since otherwise it is backwards compatible with npm. */
const getPackageManagerForInstall = async (options, pkgFile) => {
    var _a;
    if (options.packageManager === 'yarn')
        return 'yarn';
    const cwd = ((_a = options.cwd) !== null && _a !== void 0 ? _a : pkgFile) ? `${pkgFile}/..` : process.cwd();
    const pnpmLockFile = path_1.default.join(cwd, 'pnpm-lock.yaml');
    const pnpm = await (0, exists_1.default)(pnpmLockFile);
    return pnpm ? 'pnpm' : 'npm';
};
/** Either suggest an install command based on the package manager, or in interactive mode, prompt to autoinstall. */
const npmInstallHint = async (pkgs, analysis, options) => {
    // if no packages were upgraded (i.e. all dependendencies deselected in interactive mode), then bail without suggesting an install.
    // normalize the analysis for one or many packages
    const analysisNormalized = pkgs.length === 1 ? { [pkgs[0]]: analysis } : analysis;
    const someUpgraded = Object.values(analysisNormalized).some(upgrades => Object.keys(upgrades).length > 0);
    if (!someUpgraded)
        return;
    let showInstallHint = true;
    // for the purpose of the install hint, just use the package manager used in the first subproject
    // if autoinstalling, the actual package manager in each subproject will be used
    const packageManager = await getPackageManagerForInstall(options, pkgs[0]);
    // by default, show an install hint after upgrading
    // this will be disabled in interactive mode if the user chooses to have npm-check-updates execute the install command
    const installHint = `${chalk_1.default.cyan(packageManager + ' install')}${pkgs.length > 1 ? ' in each project directory' : ''} to install new versions`;
    // prompt the user if they want ncu to run "npm install"
    if (options.interactive) {
        console.info('');
        const response = await (0, prompts_ncu_1.default)({
            type: 'confirm',
            name: 'value',
            message: `${installHint}?`,
            initial: true,
            // allow Ctrl+C to kill the process
            onState: (state) => {
                if (state.aborted) {
                    process.nextTick(() => process.exit(1));
                }
            },
        });
        // autoinstall
        if (response.value) {
            showInstallHint = false;
            pkgs.forEach(async (pkgFile) => {
                const packageManager = await getPackageManagerForInstall(options, pkgFile);
                const cmd = packageManager + (process.platform === 'win32' ? '.cmd' : '');
                const cwd = options.cwd || path_1.default.resolve(pkgFile, '..');
                const stdout = await (0, spawn_please_1.default)(cmd, ['install'], { cwd });
                (0, logging_1.print)(options, stdout, 'verbose');
            });
        }
    }
    // show the install hint unless autoinstall occurred
    if (showInstallHint) {
        (0, logging_1.print)(options, `\nRun ${installHint}.`);
    }
};
/** Main entry point.
 *
 * @returns Promise<
 * PackageFile                    Default returns upgraded package file.
 * | Index<VersionSpec>    --jsonUpgraded returns only upgraded dependencies.
 * | void                         --global upgrade returns void.
 * >
 */
async function run(runOptions = {}, { cli } = {}) {
    const options = await (0, initOptions_1.default)(runOptions, { cli });
    // chalk may already have been intialized in cli.ts, but when imported as a module
    // chalkInit is idempotent
    await (0, chalk_1.chalkInit)(options.color);
    checkIfVolta(options);
    (0, logging_1.print)(options, 'Initializing', 'verbose');
    if (options.packageManager === 'npm' && !options.prefix) {
        options.prefix = await package_managers_1.default.npm.defaultPrefix(options);
    }
    if (options.packageManager === 'yarn' && !options.prefix) {
        options.prefix = await package_managers_1.default.yarn.defaultPrefix(options);
    }
    let timeout;
    let timeoutPromise = new Promise(() => null);
    if (options.timeout) {
        const timeoutMs = (0, isString_1.default)(options.timeout) ? Number.parseInt(options.timeout, 10) : options.timeout;
        timeoutPromise = new Promise((resolve, reject) => {
            timeout = setTimeout(() => {
                // must catch the error and reject explicitly since we are in a setTimeout
                const error = `Exceeded global timeout of ${timeoutMs}ms`;
                reject(error);
                try {
                    (0, programError_1.default)(options, chalk_1.default.red(error));
                }
                catch (e) {
                    /* noop */
                }
            }, timeoutMs);
        });
    }
    /** Runs the dependency upgrades. Loads the ncurc, finds the package file, and handles --deep. */
    async function runUpgrades() {
        const defaultPackageFilename = (0, getPackageFileName_1.default)(options);
        const pkgs = globby_1.default.sync(options.cwd
            ? path_1.default.resolve(options.cwd.replace(/^~/, os_1.default.homedir()), defaultPackageFilename).replace(/\\/g, '/') // convert Windows path to *nix path for globby
            : defaultPackageFilename, {
            ignore: ['**/node_modules/**'],
        });
        // enable --deep if multiple package files are found
        options.deep = options.deep || pkgs.length > 1;
        let analysis;
        if (options.global) {
            const analysis = await (0, runGlobal_1.default)(options);
            clearTimeout(timeout);
            return analysis;
        }
        else if (options.deep) {
            analysis = await pkgs.reduce(async (previousPromise, packageFile) => {
                const packages = await previousPromise;
                // copy object to prevent share .ncurc options between different packageFile, to prevent unpredictable behavior
                const rcResult = await (0, getNcuRc_1.default)({ packageFile, color: options.color });
                let rcConfig = rcResult && rcResult.config ? rcResult.config : {};
                if (options.mergeConfig && Object.keys(rcConfig).length) {
                    // Merge config options.
                    rcConfig = (0, mergeOptions_1.default)(options, rcConfig);
                }
                const pkgOptions = {
                    ...options,
                    ...rcConfig,
                    packageFile,
                };
                const [pkgData, pkgFile] = await (0, findPackage_1.default)(pkgOptions);
                return {
                    ...packages,
                    // index by relative path if cwd was specified
                    [pkgOptions.cwd
                        ? path_1.default
                            .relative(path_1.default.resolve(pkgOptions.cwd), pkgFile)
                            // convert Windows path to *nix path for consistency
                            .replace(/\\/g, '/')
                        : pkgFile]: await (0, runLocal_1.default)(pkgOptions, pkgData, pkgFile),
                };
            }, Promise.resolve({}));
            if (options.json) {
                (0, logging_1.printJson)(options, analysis);
            }
        }
        else {
            // Mutate packageFile when glob patern finds only single package
            if (pkgs.length === 1 && pkgs[0] !== defaultPackageFilename) {
                options.packageFile = pkgs[0];
            }
            const [pkgData, pkgFile] = await (0, findPackage_1.default)(options);
            analysis = await (0, runLocal_1.default)(options, pkgData, pkgFile);
        }
        clearTimeout(timeout);
        // suggest install command or autoinstall
        if (options.upgrade) {
            await npmInstallHint(pkgs, analysis, options);
        }
        return analysis;
    }
    // doctor mode
    if (options.doctor) {
        // execute with -u
        if (options.upgrade) {
            // we have to pass run directly since it would be a circular require if doctor included this file
            return Promise.race([timeoutPromise, (0, doctor_1.default)(run, options)]);
        }
        // print help otherwise
        else {
            (0, logging_1.print)(options, `Usage: ncu --doctor\n\n${cli_options_1.cliOptionsMap.doctor.help()}`, 'warn');
        }
    }
    // normal mode
    else {
        return Promise.race([timeoutPromise, runUpgrades()]);
    }
}
exports.run = run;
exports.default = run;
//# sourceMappingURL=index.js.map