r"""
Free resolutions

Let `R` be a commutative ring. A finite free resolution of an `R`-module `M`
is a chain complex of free `R`-modules

.. MATH::

    0 \xleftarrow{d_0} R^{n_1} \xleftarrow{d_1}  R^{n_1} \xleftarrow{d_2}
    \cdots \xleftarrow{d_k} R^{n_k} \xleftarrow{d_{k+1}} 0

terminating with a zero module at the end that is exact (all homology groups
are zero) such that the image of `d_1` is `M`.

EXAMPLES::

    sage: from sage.homology.free_resolution import FreeResolution
    sage: S.<x,y,z,w> = PolynomialRing(QQ)
    sage: m = matrix(S, 1, [z^2 - y*w, y*z - x*w, y^2 - x*z]).transpose()
    sage: r = FreeResolution(m, name='S')
    sage: r
    S^1 <-- S^3 <-- S^2 <-- 0

    sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
    sage: r = FreeResolution(I)
    sage: r
    S^1 <-- S^3 <-- S^2 <-- 0

::

    sage: from sage.homology.graded_resolution import GradedFreeResolution
    sage: S.<x,y,z,w> = PolynomialRing(QQ)
    sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
    sage: r = GradedFreeResolution(I)
    sage: r
    S(0) <-- S(-2)⊕S(-2)⊕S(-2) <-- S(-3)⊕S(-3) <-- 0

An example of a minimal free resolution from [CLO2005]_::

    sage: R.<x,y,z,w> = QQ[]
    sage: I = R.ideal([y*z - x*w, y^3 - x^2*z, x*z^2 - y^2*w, z^3 - y*w^2])
    sage: r = FreeResolution(I)
    sage: r
    S^1 <-- S^4 <-- S^4 <-- S^1 <-- 0
    sage: len(r)
    3
    sage: r.matrix(2)
    [-z^2 -x*z  y*w -y^2]
    [   y    0   -x    0]
    [  -w    y    z    x]
    [   0    w    0    z]

AUTHORS:

- Kwankyu Lee (2022-05-13): initial version

"""

# ****************************************************************************
#       Copyright (C) 2022 Kwankyu Lee <ekwankyu@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#                  https://www.gnu.org/licenses/
# ****************************************************************************

from sage.libs.singular.singular import si2sa_resolution
from sage.libs.singular.function import singular_function
from sage.misc.lazy_attribute import lazy_attribute
from sage.matrix.matrix_mpolynomial_dense import Matrix_mpolynomial_dense
from sage.modules.free_module_element import vector
from sage.modules.free_module import FreeModule
from sage.modules.free_module import Module_free_ambient
from sage.rings.ideal import Ideal_generic

from sage.structure.sage_object import SageObject


class FreeResolution_generic(SageObject):
    r"""
    Generic base class of finite free resolutions.

    A subclass must provide a ``_maps`` attribute that contains a list of the
    maps defining the resolution.

    The matrix at index `i` in the list defines the differential map from
    `(i + 1)`-th free module to the `i`-th free module over the base ring by
    multiplication on the left. The number of matrices in the list is the
    length of the resolution. The number of rows and columns of the matrices
    define the ranks of the free modules in the resolution.

    Note that the first matrix in the list defines the differential map at
    homological index `1`.

    A subclass can define ``_initial_differential`` attribute that
    contains the `0`-th differential map whose codomain is the target
    of the free resolution.

    EXAMPLES::

        sage: from sage.homology.free_resolution import FreeResolution
        sage: S.<x,y,z,w> = PolynomialRing(QQ)
        sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
        sage: r = FreeResolution(I)
        sage: r.differential(0)
        Coercion map:
          From: Ambient free module of rank 1 over the integral domain
        Multivariate Polynomial Ring in x, y, z, w over Rational Field
          To:   Quotient module by Submodule of Ambient free module of rank 1
        over the integral domain Multivariate Polynomial Ring in x, y, z, w over Rational Field
        Generated by the rows of the matrix:
        [-z^2 + y*w]
        [ y*z - x*w]
        [-y^2 + x*z]
    """
    def __init__(self, base_ring, name='F'):
        """
        Initialize.

        INPUT:

        - ``base_ring`` -- a ring

        TESTS::

            sage: from sage.homology.free_resolution import FreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: m1 = matrix(S, 1, [z^2 - y*w, y*z - x*w, y^2 - x*z])
            sage: r = FreeResolution(m1, name='S')
            sage: TestSuite(r).run(skip=['_test_pickling'])
        """
        self._base_ring = base_ring
        self._name = name

    @lazy_attribute
    def _length(self):
        """
        The length of ``self``.

        TESTS::

            sage: from sage.homology.free_resolution import FreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = FreeResolution(I)
            sage: r._length
            2
        """
        return len(self._maps)

    def _repr_(self):
        """
        Return the string form of this resolution.

        INPUT:

        - ``i`` -- a positive integer

        EXAMPLES::

            sage: from sage.homology.graded_resolution import GradedFreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = GradedFreeResolution(I)
            sage: r
            S(0) <-- S(-2)⊕S(-2)⊕S(-2) <-- S(-3)⊕S(-3) <-- 0
        """
        s = self._repr_module(0)
        for i in range(1, self._length + 1):
            s += ' <-- ' + self._repr_module(i)
        s += ' <-- 0'
        return s

    def _repr_module(self, i):
        r"""
        Return the string form of the `i`-th free module.

        INPUT:

        - ``i`` -- a positive integer

        EXAMPLES::

            sage: from sage.homology.free_resolution import FreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: m = matrix(S, 1, [y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = FreeResolution(m.transpose(), name='S')
            sage: r._repr_module(2)
            'S^2'
            sage: r  # indirect doctest
            S^1 <-- S^3 <-- S^2 <-- 0
        """
        if i == 0:
            r = self._maps[0].nrows()
            s = f'{self._name}^{r}'
            return s
        elif i > self._length:
            s = '0'
        else:
            r = self._maps[i - 1].ncols()
            if r > 0:
                s = f'{self._name}^{r}'
            else:
                s = '0'
        return s

    def __len__(self):
        r"""
        Return the length of this resolution.

        The length of a free resolution is the index of the last nonzero free module.

        EXAMPLES::

            sage: from sage.homology.graded_resolution import GradedFreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = GradedFreeResolution(I)
            sage: r
            S(0) <-- S(-2)⊕S(-2)⊕S(-2) <-- S(-3)⊕S(-3) <-- 0
            sage: len(r)
            2
        """
        return len(self._maps)

    def __getitem__(self, i):
        r"""
        Return the ``i``-th free module of this resolution.

        INPUT:

        - ``i`` -- a positive integer

        EXAMPLES::

            sage: from sage.homology.graded_resolution import GradedFreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = GradedFreeResolution(I)
            sage: r
            S(0) <-- S(-2)⊕S(-2)⊕S(-2) <-- S(-3)⊕S(-3) <-- 0
            sage: r.target()
            Quotient module by Submodule of Ambient free module of rank 1 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            Generated by the rows of the matrix:
            [-z^2 + y*w]
            [ y*z - x*w]
            [-y^2 + x*z]
       """
        if i < 0:
            raise IndexError('invalid index')
        elif i > self._length:
            F = FreeModule(self._base_ring, 0)
        elif i == self._length:
            F = FreeModule(self._base_ring, self._maps[i - 1].ncols())
        else:
            F = FreeModule(self._base_ring, self._maps[i].nrows())
        return F

    def differential(self, i):
        r"""
        Return the matrix representing the ``i``-th differential map.

        INPUT:

        - ``i`` -- a positive integer

        EXAMPLES::

            sage: from sage.homology.graded_resolution import GradedFreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = GradedFreeResolution(I)
            sage: r
            S(0) <-- S(-2)⊕S(-2)⊕S(-2) <-- S(-3)⊕S(-3) <-- 0
            sage: r.differential(3)
            Free module morphism defined by the matrix
            []
            Domain: Ambient free module of rank 0 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            Codomain: Ambient free module of rank 2 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            sage: r.differential(2)
            Free module morphism defined as left-multiplication by the matrix
            [-y  x]
            [ z -y]
            [-w  z]
            Domain: Ambient free module of rank 2 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            Codomain: Ambient free module of rank 3 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            sage: r.differential(1)
            Free module morphism defined as left-multiplication by the matrix
            [z^2 - y*w y*z - x*w y^2 - x*z]
            Domain: Ambient free module of rank 3 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            Codomain: Ambient free module of rank 1 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            sage: r.differential(0)
            Coercion map:
              From: Ambient free module of rank 1 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
              To:   Quotient module by Submodule of Ambient free module of rank 1 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            Generated by the rows of the matrix:
            [-z^2 + y*w]
            [ y*z - x*w]
            [-y^2 + x*z]
        """
        if i < 0:
            raise IndexError('invalid index')
        elif i == 0:
            try:
                return self._initial_differential
            except AttributeError:
                raise ValueError('0th differential map undefined')
        elif i == self._length + 1:
            s = FreeModule(self._base_ring, 0)
            t = FreeModule(self._base_ring, self._maps[i - 2].ncols())
            m = s.hom(0, t)
        elif i > self._length + 1:
            s = FreeModule(self._base_ring, 0)
            t = FreeModule(self._base_ring, 0)
            m = s.hom(0, t)
        else:
            s = FreeModule(self._base_ring, self._maps[i - 1].ncols())
            t = FreeModule(self._base_ring, self._maps[i - 1].nrows())
            m = s.hom(self._maps[i - 1], t, side='right')
        return m

    def target(self):
        r"""
        Return the codomain of the ``0``-th differential map.

        EXAMPLES::

            sage: from sage.homology.graded_resolution import GradedFreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = GradedFreeResolution(I)
            sage: r
            S(0) <-- S(-2)⊕S(-2)⊕S(-2) <-- S(-3)⊕S(-3) <-- 0
            sage: r.target()
            Quotient module by Submodule of Ambient free module of rank 1 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
            Generated by the rows of the matrix:
            [-z^2 + y*w]
            [ y*z - x*w]
            [-y^2 + x*z]
        """
        return self.differential(0).codomain()

    def matrix(self, i):
        r"""
        Return the matrix representing the ``i``-th differential map.

        INPUT:

        - ``i`` -- a positive integer

        EXAMPLES::

            sage: from sage.homology.graded_resolution import GradedFreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = GradedFreeResolution(I)
            sage: r
            S(0) <-- S(-2)⊕S(-2)⊕S(-2) <-- S(-3)⊕S(-3) <-- 0
            sage: r.matrix(3)
            []
            sage: r.matrix(2)
            [-y  x]
            [ z -y]
            [-w  z]
            sage: r.matrix(1)
            [z^2 - y*w y*z - x*w y^2 - x*z]
        """
        if i <= 0:
            raise IndexError('invalid index')
        elif i <= self._length:
            return self._maps[i - 1]
        else:
            return self.differential(i).matrix()

    def chain_complex(self):
        r"""
        Return this resolution as a chain complex.

        A chain complex in Sage has its own useful methods.

        EXAMPLES::

            sage: from sage.homology.graded_resolution import GradedFreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = GradedFreeResolution(I)
            sage: unicode_art(r.chain_complex())
                                                           ⎛-y  x⎞
                                                           ⎜ z -y⎟
                       (z^2 - y*w y*z - x*w y^2 - x*z)     ⎝-w  z⎠
             0 <── C_0 <────────────────────────────── C_1 <────── C_2 <── 0
        """
        from sage.homology.chain_complex import ChainComplex
        mats = {}
        for i in range(self._length, 0, -1):
            mats[i] = self.matrix(i)
        return ChainComplex(mats, degree_of_differential=-1)


class FreeResolution(FreeResolution_generic):
    r"""
    Minimal free resolutions of ideals or submodules of free modules
    of multivariate polynomial rings.

    INPUT:

    - ``module`` -- a submodule of a free module `M` of rank `n` over `S` or
      an ideal of a multi-variate polynomial ring

    - ``name`` -- a string; name of the base ring

    - ``algorithm`` -- Singular algorithm to compute a resolution of ``ideal``

    OUTPUT: a minimal free resolution of the ideal

    If ``module`` is an ideal of `S`, it is considered as a submodule of a
    free module of rank `1` over `S`.

    The available algorithms and the corresponding Singular commands are shown
    below:

        ============= ============================
        algorithm     Singular commands
        ============= ============================
        ``minimal``   ``mres(ideal)``
        ``shreyer``   ``minres(sres(std(ideal)))``
        ``standard``  ``minres(nres(std(ideal)))``
        ``heuristic`` ``minres(res(std(ideal)))``
        ============= ============================

    EXAMPLES::

        sage: from sage.homology.free_resolution import FreeResolution
        sage: S.<x,y,z,w> = PolynomialRing(QQ)
        sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
        sage: r = FreeResolution(I)
        sage: r
        S^1 <-- S^3 <-- S^2 <-- 0
        sage: len(r)
        2

    ::

        sage: FreeResolution(I, algorithm='minimal')
        S^1 <-- S^3 <-- S^2 <-- 0
        sage: FreeResolution(I, algorithm='shreyer')
        S^1 <-- S^3 <-- S^2 <-- 0
        sage: FreeResolution(I, algorithm='standard')
        S^1 <-- S^3 <-- S^2 <-- 0
        sage: FreeResolution(I, algorithm='heuristic')
        S^1 <-- S^3 <-- S^2 <-- 0

    We can also construct a resolution by passing in a matrix defining
    the initial differential::

        sage: m = matrix(S, 1, [z^2 - y*w, y*z - x*w, y^2 - x*z]).transpose()
        sage: r = FreeResolution(m, name='S')
        sage: r
        S^1 <-- S^3 <-- S^2 <-- 0
        sage: r.matrix(1)
        [z^2 - y*w y*z - x*w y^2 - x*z]

    An additional construction is using a submodule of a free module::

        sage: M = m.image()
        sage: r = FreeResolution(M, name='S')
        sage: r
        S^1 <-- S^3 <-- S^2 <-- 0

    A nonhomogeneous ideal::

        sage: I = S.ideal([z^2 - y*w, y*z - x*w, y^2 - x])
        sage: R = FreeResolution(I)
        sage: R
        S^1 <-- S^3 <-- S^3 <-- S^1 <-- 0
        sage: R.matrix(2)
        [ y*z - x*w    y^2 - x          0]
        [-z^2 + y*w          0    y^2 - x]
        [         0 -z^2 + y*w -y*z + x*w]
        sage: R.matrix(3)
        [   y^2 - x]
        [-y*z + x*w]
        [ z^2 - y*w]
    """
    def __init__(self, module, name='S', algorithm='heuristic'):
        """
        Initialize.

        TESTS::

            sage: from sage.homology.free_resolution import FreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = FreeResolution(I)
            sage: TestSuite(r).run(skip=['_test_pickling'])

            sage: m = matrix(S, 1, [z^2 - y*w, y*z - x*w, y^2 - x*z]).transpose()
            sage: r = FreeResolution(m, name='S')
            sage: TestSuite(r).run(skip=['_test_pickling'])

            sage: M = m.image()
            sage: r = FreeResolution(M, name='S')
            sage: TestSuite(r).run(skip=['_test_pickling'])
        """
        if isinstance(module, Ideal_generic):
            S = module.ring()
        elif isinstance(module, Module_free_ambient):
            S = module.base_ring()
        elif isinstance(module, Matrix_mpolynomial_dense):
            S = module.base_ring()
        else:
            raise TypeError('no ideal, module, or matrix')

        self._module = module
        self._algorithm = algorithm
        super().__init__(S, name=name)

    def _m(self):
        r"""
        The defining module of ``self``.

        TESTS::

            sage: from sage.homology.free_resolution import FreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = FreeResolution(I)
            sage: r._m()
            Ideal (-z^2 + y*w, y*z - x*w, -y^2 + x*z) of Multivariate Polynomial Ring in x, y, z, w over Rational Field

            sage: m = matrix(S, 1, [z^2 - y*w, y*z - x*w, y^2 - x*z]).transpose()
            sage: r = FreeResolution(m, name='S')
            sage: r._m()
            [z^2 - y*w y*z - x*w y^2 - x*z]

            sage: M = m.image()
            sage: r = FreeResolution(M, name='S')
            sage: r._m()
            [z^2 - y*w y*z - x*w y^2 - x*z]
        """
        if isinstance(self._module, Ideal_generic):
            return self._module
        if isinstance(self._module, Module_free_ambient):
            return self._module.matrix().transpose()
        if isinstance(self._module, Matrix_mpolynomial_dense):
            return self._module.transpose()

    @lazy_attribute
    def _maps(self):
        r"""
        Return the maps that define ``self``.

        TESTS::

            sage: from sage.homology.free_resolution import FreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = FreeResolution(I)
            sage: r._maps
            [
                                             [-y  x]
                                             [ z -y]
            [z^2 - y*w y*z - x*w y^2 - x*z], [-w  z]
            ]
        """
        # This ensures the first component of the Singular resolution to be a
        # module, like the later components. This is important when the
        # components are converted to Sage modules.
        module = singular_function("module")
        mod = module(self._m())

        if self._algorithm == 'minimal':
            mres = singular_function('mres')  # syzygy method
            r = mres(mod, 0)
        elif self._algorithm == 'shreyer':
            std = singular_function('std')
            sres = singular_function('sres')  # Shreyer method
            minres = singular_function('minres')
            r = minres(sres(std(mod), 0))
        elif self._algorithm == 'standard':
            nres = singular_function('nres')  # standard basis method
            minres = singular_function('minres')
            r = minres(nres(mod, 0))
        elif self._algorithm == 'heuristic':
            std = singular_function('std')
            res = singular_function('res')    # heuristic method
            minres = singular_function('minres')
            r = minres(res(std(mod), 0))

        return si2sa_resolution(r)

    @lazy_attribute
    def _initial_differential(self):
        r"""
        Define the `0`-th differential map of this resolution.

        EXAMPLES::

            sage: from sage.homology.free_resolution import FreeResolution
            sage: S.<x,y,z,w> = PolynomialRing(QQ)
            sage: I = S.ideal([y*w - z^2, -x*w + y*z, x*z - y^2])
            sage: r = FreeResolution(I)
            sage: r._initial_differential
            Coercion map:
              From: Ambient free module of rank 1 over the integral domain
            Multivariate Polynomial Ring in x, y, z, w over Rational Field
              To:   Quotient module by Submodule of Ambient free module of rank 1
            over the integral domain Multivariate Polynomial Ring in x, y, z, w over Rational Field
            Generated by the rows of the matrix:
            [-z^2 + y*w]
            [ y*z - x*w]
            [-y^2 + x*z]
        """
        ideal = self._module
        if isinstance(ideal, Ideal_generic):
            S = ideal.ring()
            M = FreeModule(S, 1)
            N = M.submodule([vector([g]) for g in ideal.gens()])
        elif isinstance(ideal, Module_free_ambient):
            S = ideal.base_ring()
            M = ideal.ambient_module()
            N = ideal
        elif isinstance(ideal, Matrix_mpolynomial_dense):
            S = ideal.base_ring()
            N = ideal.row_space()
            M = N.ambient_module()
        Q = M.quotient(N)
        return Q.coerce_map_from(M)

