"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PostgresQueryRunner = void 0;
var tslib_1 = require("tslib");
var QueryFailedError_1 = require("../../error/QueryFailedError");
var QueryRunnerAlreadyReleasedError_1 = require("../../error/QueryRunnerAlreadyReleasedError");
var TransactionAlreadyStartedError_1 = require("../../error/TransactionAlreadyStartedError");
var TransactionNotStartedError_1 = require("../../error/TransactionNotStartedError");
var BaseQueryRunner_1 = require("../../query-runner/BaseQueryRunner");
var Table_1 = require("../../schema-builder/table/Table");
var TableCheck_1 = require("../../schema-builder/table/TableCheck");
var TableColumn_1 = require("../../schema-builder/table/TableColumn");
var TableExclusion_1 = require("../../schema-builder/table/TableExclusion");
var TableForeignKey_1 = require("../../schema-builder/table/TableForeignKey");
var TableIndex_1 = require("../../schema-builder/table/TableIndex");
var TableUnique_1 = require("../../schema-builder/table/TableUnique");
var View_1 = require("../../schema-builder/view/View");
var Broadcaster_1 = require("../../subscriber/Broadcaster");
var OrmUtils_1 = require("../../util/OrmUtils");
var Query_1 = require("../Query");
var VersionUtils_1 = require("../../util/VersionUtils");
var error_1 = require("../../error");
var QueryResult_1 = require("../../query-runner/QueryResult");
/**
 * Runs queries on a single postgres database connection.
 */
var PostgresQueryRunner = /** @class */ (function (_super) {
    (0, tslib_1.__extends)(PostgresQueryRunner, _super);
    // -------------------------------------------------------------------------
    // Constructor
    // -------------------------------------------------------------------------
    function PostgresQueryRunner(driver, mode) {
        var _this = _super.call(this) || this;
        _this.driver = driver;
        _this.connection = driver.connection;
        _this.mode = mode;
        _this.broadcaster = new Broadcaster_1.Broadcaster(_this);
        return _this;
    }
    // -------------------------------------------------------------------------
    // Public Methods
    // -------------------------------------------------------------------------
    /**
     * Creates/uses database connection from the connection pool to perform further operations.
     * Returns obtained database connection.
     */
    PostgresQueryRunner.prototype.connect = function () {
        var _this = this;
        if (this.databaseConnection)
            return Promise.resolve(this.databaseConnection);
        if (this.databaseConnectionPromise)
            return this.databaseConnectionPromise;
        if (this.mode === "slave" && this.driver.isReplicated) {
            this.databaseConnectionPromise = this.driver.obtainSlaveConnection().then(function (_a) {
                var _b = (0, tslib_1.__read)(_a, 2), connection = _b[0], release = _b[1];
                _this.driver.connectedQueryRunners.push(_this);
                _this.databaseConnection = connection;
                var onErrorCallback = function (err) { return _this.releasePostgresConnection(err); };
                _this.releaseCallback = function () {
                    _this.databaseConnection.removeListener("error", onErrorCallback);
                    release();
                };
                _this.databaseConnection.on("error", onErrorCallback);
                return _this.databaseConnection;
            });
        }
        else { // master
            this.databaseConnectionPromise = this.driver.obtainMasterConnection().then(function (_a) {
                var _b = (0, tslib_1.__read)(_a, 2), connection = _b[0], release = _b[1];
                _this.driver.connectedQueryRunners.push(_this);
                _this.databaseConnection = connection;
                var onErrorCallback = function (err) { return _this.releasePostgresConnection(err); };
                _this.releaseCallback = function () {
                    _this.databaseConnection.removeListener("error", onErrorCallback);
                    release();
                };
                _this.databaseConnection.on("error", onErrorCallback);
                return _this.databaseConnection;
            });
        }
        return this.databaseConnectionPromise;
    };
    /**
     * Release a connection back to the pool, optionally specifying an Error to release with.
     * Per pg-pool documentation this will prevent the pool from re-using the broken connection.
     */
    PostgresQueryRunner.prototype.releasePostgresConnection = function (err) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var index;
            return (0, tslib_1.__generator)(this, function (_a) {
                if (this.isReleased) {
                    return [2 /*return*/];
                }
                this.isReleased = true;
                if (this.releaseCallback) {
                    this.releaseCallback(err);
                    this.releaseCallback = undefined;
                }
                index = this.driver.connectedQueryRunners.indexOf(this);
                if (index !== -1) {
                    this.driver.connectedQueryRunners.splice(index, 1);
                }
                return [2 /*return*/];
            });
        });
    };
    /**
     * Releases used database connection.
     * You cannot use query runner methods once its released.
     */
    PostgresQueryRunner.prototype.release = function () {
        return this.releasePostgresConnection();
    };
    /**
     * Starts transaction.
     */
    PostgresQueryRunner.prototype.startTransaction = function (isolationLevel) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this.isTransactionActive)
                            throw new TransactionAlreadyStartedError_1.TransactionAlreadyStartedError();
                        return [4 /*yield*/, this.broadcaster.broadcast('BeforeTransactionStart')];
                    case 1:
                        _a.sent();
                        this.isTransactionActive = true;
                        return [4 /*yield*/, this.query("START TRANSACTION")];
                    case 2:
                        _a.sent();
                        if (!isolationLevel) return [3 /*break*/, 4];
                        return [4 /*yield*/, this.query("SET TRANSACTION ISOLATION LEVEL " + isolationLevel)];
                    case 3:
                        _a.sent();
                        _a.label = 4;
                    case 4: return [4 /*yield*/, this.broadcaster.broadcast('AfterTransactionStart')];
                    case 5:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Commits transaction.
     * Error will be thrown if transaction was not started.
     */
    PostgresQueryRunner.prototype.commitTransaction = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!this.isTransactionActive)
                            throw new TransactionNotStartedError_1.TransactionNotStartedError();
                        return [4 /*yield*/, this.broadcaster.broadcast('BeforeTransactionCommit')];
                    case 1:
                        _a.sent();
                        return [4 /*yield*/, this.query("COMMIT")];
                    case 2:
                        _a.sent();
                        this.isTransactionActive = false;
                        return [4 /*yield*/, this.broadcaster.broadcast('AfterTransactionCommit')];
                    case 3:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Rollbacks transaction.
     * Error will be thrown if transaction was not started.
     */
    PostgresQueryRunner.prototype.rollbackTransaction = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!this.isTransactionActive)
                            throw new TransactionNotStartedError_1.TransactionNotStartedError();
                        return [4 /*yield*/, this.broadcaster.broadcast('BeforeTransactionRollback')];
                    case 1:
                        _a.sent();
                        return [4 /*yield*/, this.query("ROLLBACK")];
                    case 2:
                        _a.sent();
                        this.isTransactionActive = false;
                        return [4 /*yield*/, this.broadcaster.broadcast('AfterTransactionRollback')];
                    case 3:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Executes a given SQL query.
     */
    PostgresQueryRunner.prototype.query = function (query, parameters, useStructuredResult) {
        if (useStructuredResult === void 0) { useStructuredResult = false; }
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var databaseConnection, queryStartTime, raw, maxQueryExecutionTime, queryEndTime, queryExecutionTime, result, err_1;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this.isReleased)
                            throw new QueryRunnerAlreadyReleasedError_1.QueryRunnerAlreadyReleasedError();
                        return [4 /*yield*/, this.connect()];
                    case 1:
                        databaseConnection = _a.sent();
                        this.driver.connection.logger.logQuery(query, parameters, this);
                        _a.label = 2;
                    case 2:
                        _a.trys.push([2, 4, , 5]);
                        queryStartTime = +new Date();
                        return [4 /*yield*/, databaseConnection.query(query, parameters)];
                    case 3:
                        raw = _a.sent();
                        maxQueryExecutionTime = this.driver.options.maxQueryExecutionTime;
                        queryEndTime = +new Date();
                        queryExecutionTime = queryEndTime - queryStartTime;
                        if (maxQueryExecutionTime && queryExecutionTime > maxQueryExecutionTime)
                            this.driver.connection.logger.logQuerySlow(queryExecutionTime, query, parameters, this);
                        result = new QueryResult_1.QueryResult();
                        if (raw === null || raw === void 0 ? void 0 : raw.hasOwnProperty('rows')) {
                            result.records = raw.rows;
                        }
                        if (raw === null || raw === void 0 ? void 0 : raw.hasOwnProperty('rowCount')) {
                            result.affected = raw.rowCount;
                        }
                        switch (raw.command) {
                            case "DELETE":
                            case "UPDATE":
                                // for UPDATE and DELETE query additionally return number of affected rows
                                result.raw = [raw.rows, raw.rowCount];
                                break;
                            default:
                                result.raw = raw.rows;
                        }
                        if (!useStructuredResult) {
                            return [2 /*return*/, result.raw];
                        }
                        return [2 /*return*/, result];
                    case 4:
                        err_1 = _a.sent();
                        this.driver.connection.logger.logQueryError(err_1, query, parameters, this);
                        throw new QueryFailedError_1.QueryFailedError(query, parameters, err_1);
                    case 5: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Returns raw data stream.
     */
    PostgresQueryRunner.prototype.stream = function (query, parameters, onEnd, onError) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var QueryStream, databaseConnection, stream;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        QueryStream = this.driver.loadStreamDependency();
                        if (this.isReleased)
                            throw new QueryRunnerAlreadyReleasedError_1.QueryRunnerAlreadyReleasedError();
                        return [4 /*yield*/, this.connect()];
                    case 1:
                        databaseConnection = _a.sent();
                        this.driver.connection.logger.logQuery(query, parameters, this);
                        stream = databaseConnection.query(new QueryStream(query, parameters));
                        if (onEnd)
                            stream.on("end", onEnd);
                        if (onError)
                            stream.on("error", onError);
                        return [2 /*return*/, stream];
                }
            });
        });
    };
    /**
     * Returns all available database names including system databases.
     */
    PostgresQueryRunner.prototype.getDatabases = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            return (0, tslib_1.__generator)(this, function (_a) {
                return [2 /*return*/, Promise.resolve([])];
            });
        });
    };
    /**
     * Returns all available schema names including system schemas.
     * If database parameter specified, returns schemas of that database.
     */
    PostgresQueryRunner.prototype.getSchemas = function (database) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            return (0, tslib_1.__generator)(this, function (_a) {
                return [2 /*return*/, Promise.resolve([])];
            });
        });
    };
    /**
     * Checks if database with the given name exist.
     */
    PostgresQueryRunner.prototype.hasDatabase = function (database) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var result;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.query("SELECT * FROM pg_database WHERE datname='" + database + "';")];
                    case 1:
                        result = _a.sent();
                        return [2 /*return*/, result.length ? true : false];
                }
            });
        });
    };
    /**
     * Loads currently using database
     */
    PostgresQueryRunner.prototype.getCurrentDatabase = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var query;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.query("SELECT * FROM current_database()")];
                    case 1:
                        query = _a.sent();
                        return [2 /*return*/, query[0]["current_database"]];
                }
            });
        });
    };
    /**
     * Checks if schema with the given name exist.
     */
    PostgresQueryRunner.prototype.hasSchema = function (schema) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var result;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.query("SELECT * FROM \"information_schema\".\"schemata\" WHERE \"schema_name\" = '" + schema + "'")];
                    case 1:
                        result = _a.sent();
                        return [2 /*return*/, result.length ? true : false];
                }
            });
        });
    };
    /**
     * Loads currently using database schema
     */
    PostgresQueryRunner.prototype.getCurrentSchema = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var query;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.query("SELECT * FROM current_schema()")];
                    case 1:
                        query = _a.sent();
                        return [2 /*return*/, query[0]["current_schema"]];
                }
            });
        });
    };
    /**
     * Checks if table with the given name exist in the database.
     */
    PostgresQueryRunner.prototype.hasTable = function (tableOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var parsedTableName, _a, sql, result;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        parsedTableName = this.driver.parseTableName(tableOrName);
                        if (!!parsedTableName.schema) return [3 /*break*/, 2];
                        _a = parsedTableName;
                        return [4 /*yield*/, this.getCurrentSchema()];
                    case 1:
                        _a.schema = _b.sent();
                        _b.label = 2;
                    case 2:
                        sql = "SELECT * FROM \"information_schema\".\"tables\" WHERE \"table_schema\" = '" + parsedTableName.schema + "' AND \"table_name\" = '" + parsedTableName.tableName + "'";
                        return [4 /*yield*/, this.query(sql)];
                    case 3:
                        result = _b.sent();
                        return [2 /*return*/, result.length ? true : false];
                }
            });
        });
    };
    /**
     * Checks if column with the given name exist in the given table.
     */
    PostgresQueryRunner.prototype.hasColumn = function (tableOrName, columnName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var parsedTableName, _a, sql, result;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        parsedTableName = this.driver.parseTableName(tableOrName);
                        if (!!parsedTableName.schema) return [3 /*break*/, 2];
                        _a = parsedTableName;
                        return [4 /*yield*/, this.getCurrentSchema()];
                    case 1:
                        _a.schema = _b.sent();
                        _b.label = 2;
                    case 2:
                        sql = "SELECT * FROM \"information_schema\".\"columns\" WHERE \"table_schema\" = '" + parsedTableName.schema + "' AND \"table_name\" = '" + parsedTableName.tableName + "' AND \"column_name\" = '" + columnName + "'";
                        return [4 /*yield*/, this.query(sql)];
                    case 3:
                        result = _b.sent();
                        return [2 /*return*/, result.length ? true : false];
                }
            });
        });
    };
    /**
     * Creates a new database.
     * Note: Postgres does not support database creation inside a transaction block.
     */
    PostgresQueryRunner.prototype.createDatabase = function (database, ifNotExist) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var databaseAlreadyExists, up, down;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!ifNotExist) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.hasDatabase(database)];
                    case 1:
                        databaseAlreadyExists = _a.sent();
                        if (databaseAlreadyExists)
                            return [2 /*return*/, Promise.resolve()];
                        _a.label = 2;
                    case 2:
                        up = "CREATE DATABASE \"" + database + "\"";
                        down = "DROP DATABASE \"" + database + "\"";
                        return [4 /*yield*/, this.executeQueries(new Query_1.Query(up), new Query_1.Query(down))];
                    case 3:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops database.
     * Note: Postgres does not support database dropping inside a transaction block.
     */
    PostgresQueryRunner.prototype.dropDatabase = function (database, ifExist) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var up, down;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        up = ifExist ? "DROP DATABASE IF EXISTS \"" + database + "\"" : "DROP DATABASE \"" + database + "\"";
                        down = "CREATE DATABASE \"" + database + "\"";
                        return [4 /*yield*/, this.executeQueries(new Query_1.Query(up), new Query_1.Query(down))];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new table schema.
     */
    PostgresQueryRunner.prototype.createSchema = function (schemaPath, ifNotExist) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var schema, up, down;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        schema = schemaPath.indexOf(".") === -1 ? schemaPath : schemaPath.split(".")[1];
                        up = ifNotExist ? "CREATE SCHEMA IF NOT EXISTS \"" + schema + "\"" : "CREATE SCHEMA \"" + schema + "\"";
                        down = "DROP SCHEMA \"" + schema + "\" CASCADE";
                        return [4 /*yield*/, this.executeQueries(new Query_1.Query(up), new Query_1.Query(down))];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops table schema.
     */
    PostgresQueryRunner.prototype.dropSchema = function (schemaPath, ifExist, isCascade) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var schema, up, down;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        schema = schemaPath.indexOf(".") === -1 ? schemaPath : schemaPath.split(".")[1];
                        up = ifExist ? "DROP SCHEMA IF EXISTS \"" + schema + "\" " + (isCascade ? "CASCADE" : "") : "DROP SCHEMA \"" + schema + "\" " + (isCascade ? "CASCADE" : "");
                        down = "CREATE SCHEMA \"" + schema + "\"";
                        return [4 /*yield*/, this.executeQueries(new Query_1.Query(up), new Query_1.Query(down))];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new table.
     */
    PostgresQueryRunner.prototype.createTable = function (table, ifNotExist, createForeignKeys, createIndices) {
        if (ifNotExist === void 0) { ifNotExist = false; }
        if (createForeignKeys === void 0) { createForeignKeys = true; }
        if (createIndices === void 0) { createIndices = true; }
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var isTableExist, upQueries, downQueries, enumColumns, createdEnumTypes, enumColumns_1, enumColumns_1_1, column, hasEnum, enumName, e_1_1;
            var e_1, _a;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!ifNotExist) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.hasTable(table)];
                    case 1:
                        isTableExist = _b.sent();
                        if (isTableExist)
                            return [2 /*return*/, Promise.resolve()];
                        _b.label = 2;
                    case 2:
                        upQueries = [];
                        downQueries = [];
                        enumColumns = table.columns.filter(function (column) { return column.type === "enum" || column.type === "simple-enum"; });
                        createdEnumTypes = [];
                        _b.label = 3;
                    case 3:
                        _b.trys.push([3, 8, 9, 10]);
                        enumColumns_1 = (0, tslib_1.__values)(enumColumns), enumColumns_1_1 = enumColumns_1.next();
                        _b.label = 4;
                    case 4:
                        if (!!enumColumns_1_1.done) return [3 /*break*/, 7];
                        column = enumColumns_1_1.value;
                        return [4 /*yield*/, this.hasEnumType(table, column)];
                    case 5:
                        hasEnum = _b.sent();
                        enumName = this.buildEnumName(table, column);
                        // if enum with the same "enumName" is defined more then once, me must prevent double creation
                        if (!hasEnum && createdEnumTypes.indexOf(enumName) === -1) {
                            createdEnumTypes.push(enumName);
                            upQueries.push(this.createEnumTypeSql(table, column, enumName));
                            downQueries.push(this.dropEnumTypeSql(table, column, enumName));
                        }
                        _b.label = 6;
                    case 6:
                        enumColumns_1_1 = enumColumns_1.next();
                        return [3 /*break*/, 4];
                    case 7: return [3 /*break*/, 10];
                    case 8:
                        e_1_1 = _b.sent();
                        e_1 = { error: e_1_1 };
                        return [3 /*break*/, 10];
                    case 9:
                        try {
                            if (enumColumns_1_1 && !enumColumns_1_1.done && (_a = enumColumns_1.return)) _a.call(enumColumns_1);
                        }
                        finally { if (e_1) throw e_1.error; }
                        return [7 /*endfinally*/];
                    case 10:
                        upQueries.push(this.createTableSql(table, createForeignKeys));
                        downQueries.push(this.dropTableSql(table));
                        // if createForeignKeys is true, we must drop created foreign keys in down query.
                        // createTable does not need separate method to create foreign keys, because it create fk's in the same query with table creation.
                        if (createForeignKeys)
                            table.foreignKeys.forEach(function (foreignKey) { return downQueries.push(_this.dropForeignKeySql(table, foreignKey)); });
                        if (createIndices) {
                            table.indices.forEach(function (index) {
                                // new index may be passed without name. In this case we generate index name manually.
                                if (!index.name)
                                    index.name = _this.connection.namingStrategy.indexName(table, index.columnNames, index.where);
                                upQueries.push(_this.createIndexSql(table, index));
                                downQueries.push(_this.dropIndexSql(table, index));
                            });
                        }
                        return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 11:
                        _b.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops the table.
     */
    PostgresQueryRunner.prototype.dropTable = function (target, ifExist, dropForeignKeys, dropIndices) {
        if (dropForeignKeys === void 0) { dropForeignKeys = true; }
        if (dropIndices === void 0) { dropIndices = true; }
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var isTableExist, createForeignKeys, tablePath, table, upQueries, downQueries;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!ifExist) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.hasTable(target)];
                    case 1:
                        isTableExist = _a.sent();
                        if (!isTableExist)
                            return [2 /*return*/, Promise.resolve()];
                        _a.label = 2;
                    case 2:
                        createForeignKeys = dropForeignKeys;
                        tablePath = this.getTablePath(target);
                        return [4 /*yield*/, this.getCachedTable(tablePath)];
                    case 3:
                        table = _a.sent();
                        upQueries = [];
                        downQueries = [];
                        if (dropIndices) {
                            table.indices.forEach(function (index) {
                                upQueries.push(_this.dropIndexSql(table, index));
                                downQueries.push(_this.createIndexSql(table, index));
                            });
                        }
                        if (dropForeignKeys)
                            table.foreignKeys.forEach(function (foreignKey) { return upQueries.push(_this.dropForeignKeySql(table, foreignKey)); });
                        upQueries.push(this.dropTableSql(table));
                        downQueries.push(this.createTableSql(table, createForeignKeys));
                        return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 4:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new view.
     */
    PostgresQueryRunner.prototype.createView = function (view) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var upQueries, downQueries, _a, _b, _c, _d;
            return (0, tslib_1.__generator)(this, function (_e) {
                switch (_e.label) {
                    case 0:
                        upQueries = [];
                        downQueries = [];
                        upQueries.push(this.createViewSql(view));
                        _b = (_a = upQueries).push;
                        return [4 /*yield*/, this.insertViewDefinitionSql(view)];
                    case 1:
                        _b.apply(_a, [_e.sent()]);
                        downQueries.push(this.dropViewSql(view));
                        _d = (_c = downQueries).push;
                        return [4 /*yield*/, this.deleteViewDefinitionSql(view)];
                    case 2:
                        _d.apply(_c, [_e.sent()]);
                        return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 3:
                        _e.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops the view.
     */
    PostgresQueryRunner.prototype.dropView = function (target) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var viewName, view, upQueries, downQueries, _a, _b, _c, _d;
            return (0, tslib_1.__generator)(this, function (_e) {
                switch (_e.label) {
                    case 0:
                        viewName = target instanceof View_1.View ? target.name : target;
                        return [4 /*yield*/, this.getCachedView(viewName)];
                    case 1:
                        view = _e.sent();
                        upQueries = [];
                        downQueries = [];
                        _b = (_a = upQueries).push;
                        return [4 /*yield*/, this.deleteViewDefinitionSql(view)];
                    case 2:
                        _b.apply(_a, [_e.sent()]);
                        upQueries.push(this.dropViewSql(view));
                        _d = (_c = downQueries).push;
                        return [4 /*yield*/, this.insertViewDefinitionSql(view)];
                    case 3:
                        _d.apply(_c, [_e.sent()]);
                        downQueries.push(this.createViewSql(view));
                        return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 4:
                        _e.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Renames the given table.
     */
    PostgresQueryRunner.prototype.renameTable = function (oldTableOrName, newTableName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var upQueries, downQueries, oldTable, _a, newTable, _b, schemaName, oldTableName, columnNames, oldPkName, newPkName, enumColumns, enumColumns_2, enumColumns_2_1, column, oldEnumType, e_2_1;
            var e_2, _c;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_d) {
                switch (_d.label) {
                    case 0:
                        upQueries = [];
                        downQueries = [];
                        if (!(oldTableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = oldTableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(oldTableOrName)];
                    case 2:
                        _a = _d.sent();
                        _d.label = 3;
                    case 3:
                        oldTable = _a;
                        newTable = oldTable.clone();
                        _b = this.driver.parseTableName(oldTable), schemaName = _b.schema, oldTableName = _b.tableName;
                        newTable.name = schemaName ? schemaName + "." + newTableName : newTableName;
                        upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(oldTable) + " RENAME TO \"" + newTableName + "\""));
                        downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(newTable) + " RENAME TO \"" + oldTableName + "\""));
                        // rename column primary key constraint
                        if (newTable.primaryColumns.length > 0) {
                            columnNames = newTable.primaryColumns.map(function (column) { return column.name; });
                            oldPkName = this.connection.namingStrategy.primaryKeyName(oldTable, columnNames);
                            newPkName = this.connection.namingStrategy.primaryKeyName(newTable, columnNames);
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(newTable) + " RENAME CONSTRAINT \"" + oldPkName + "\" TO \"" + newPkName + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(newTable) + " RENAME CONSTRAINT \"" + newPkName + "\" TO \"" + oldPkName + "\""));
                        }
                        // rename sequences
                        newTable.columns.map(function (col) {
                            if (col.isGenerated && col.generationStrategy === "increment") {
                                var sequencePath = _this.buildSequencePath(oldTable, col.name);
                                var sequenceName = _this.buildSequenceName(oldTable, col.name);
                                var newSequencePath = _this.buildSequencePath(newTable, col.name);
                                var newSequenceName = _this.buildSequenceName(newTable, col.name);
                                var up = "ALTER SEQUENCE " + _this.escapePath(sequencePath) + " RENAME TO \"" + newSequenceName + "\"";
                                var down = "ALTER SEQUENCE " + _this.escapePath(newSequencePath) + " RENAME TO \"" + sequenceName + "\"";
                                upQueries.push(new Query_1.Query(up));
                                downQueries.push(new Query_1.Query(down));
                            }
                        });
                        // rename unique constraints
                        newTable.uniques.forEach(function (unique) {
                            // build new constraint name
                            var newUniqueName = _this.connection.namingStrategy.uniqueConstraintName(newTable, unique.columnNames);
                            // build queries
                            upQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(newTable) + " RENAME CONSTRAINT \"" + unique.name + "\" TO \"" + newUniqueName + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(newTable) + " RENAME CONSTRAINT \"" + newUniqueName + "\" TO \"" + unique.name + "\""));
                            // replace constraint name
                            unique.name = newUniqueName;
                        });
                        // rename index constraints
                        newTable.indices.forEach(function (index) {
                            // build new constraint name
                            var schema = _this.driver.parseTableName(newTable).schema;
                            var newIndexName = _this.connection.namingStrategy.indexName(newTable, index.columnNames, index.where);
                            // build queries
                            var up = schema ? "ALTER INDEX \"" + schema + "\".\"" + index.name + "\" RENAME TO \"" + newIndexName + "\"" : "ALTER INDEX \"" + index.name + "\" RENAME TO \"" + newIndexName + "\"";
                            var down = schema ? "ALTER INDEX \"" + schema + "\".\"" + newIndexName + "\" RENAME TO \"" + index.name + "\"" : "ALTER INDEX \"" + newIndexName + "\" RENAME TO \"" + index.name + "\"";
                            upQueries.push(new Query_1.Query(up));
                            downQueries.push(new Query_1.Query(down));
                            // replace constraint name
                            index.name = newIndexName;
                        });
                        // rename foreign key constraints
                        newTable.foreignKeys.forEach(function (foreignKey) {
                            // build new constraint name
                            var newForeignKeyName = _this.connection.namingStrategy.foreignKeyName(newTable, foreignKey.columnNames, _this.getTablePath(foreignKey), foreignKey.referencedColumnNames);
                            // build queries
                            upQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(newTable) + " RENAME CONSTRAINT \"" + foreignKey.name + "\" TO \"" + newForeignKeyName + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(newTable) + " RENAME CONSTRAINT \"" + newForeignKeyName + "\" TO \"" + foreignKey.name + "\""));
                            // replace constraint name
                            foreignKey.name = newForeignKeyName;
                        });
                        enumColumns = newTable.columns.filter(function (column) { return column.type === "enum" || column.type === "simple-enum"; });
                        _d.label = 4;
                    case 4:
                        _d.trys.push([4, 9, 10, 11]);
                        enumColumns_2 = (0, tslib_1.__values)(enumColumns), enumColumns_2_1 = enumColumns_2.next();
                        _d.label = 5;
                    case 5:
                        if (!!enumColumns_2_1.done) return [3 /*break*/, 8];
                        column = enumColumns_2_1.value;
                        // skip renaming for user-defined enum name
                        if (column.enumName)
                            return [3 /*break*/, 7];
                        return [4 /*yield*/, this.getUserDefinedTypeName(oldTable, column)];
                    case 6:
                        oldEnumType = _d.sent();
                        upQueries.push(new Query_1.Query("ALTER TYPE \"" + oldEnumType.schema + "\".\"" + oldEnumType.name + "\" RENAME TO " + this.buildEnumName(newTable, column, false)));
                        downQueries.push(new Query_1.Query("ALTER TYPE " + this.buildEnumName(newTable, column) + " RENAME TO \"" + oldEnumType.name + "\""));
                        _d.label = 7;
                    case 7:
                        enumColumns_2_1 = enumColumns_2.next();
                        return [3 /*break*/, 5];
                    case 8: return [3 /*break*/, 11];
                    case 9:
                        e_2_1 = _d.sent();
                        e_2 = { error: e_2_1 };
                        return [3 /*break*/, 11];
                    case 10:
                        try {
                            if (enumColumns_2_1 && !enumColumns_2_1.done && (_c = enumColumns_2.return)) _c.call(enumColumns_2);
                        }
                        finally { if (e_2) throw e_2.error; }
                        return [7 /*endfinally*/];
                    case 11: return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 12:
                        _d.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new column from the column in the table.
     */
    PostgresQueryRunner.prototype.addColumn = function (tableOrName, column) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, clonedTable, upQueries, downQueries, hasEnum, primaryColumns, pkName_1, columnNames_1, pkName, columnNames, columnIndex, uniqueConstraint;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        clonedTable = table.clone();
                        upQueries = [];
                        downQueries = [];
                        if (!(column.type === "enum" || column.type === "simple-enum")) return [3 /*break*/, 5];
                        return [4 /*yield*/, this.hasEnumType(table, column)];
                    case 4:
                        hasEnum = _b.sent();
                        if (!hasEnum) {
                            upQueries.push(this.createEnumTypeSql(table, column));
                            downQueries.push(this.dropEnumTypeSql(table, column));
                        }
                        _b.label = 5;
                    case 5:
                        upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD " + this.buildCreateColumnSql(table, column)));
                        downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP COLUMN \"" + column.name + "\""));
                        // create or update primary key constraint
                        if (column.isPrimary) {
                            primaryColumns = clonedTable.primaryColumns;
                            // if table already have primary key, me must drop it and recreate again
                            if (primaryColumns.length > 0) {
                                pkName_1 = this.connection.namingStrategy.primaryKeyName(clonedTable, primaryColumns.map(function (column) { return column.name; }));
                                columnNames_1 = primaryColumns.map(function (column) { return "\"" + column.name + "\""; }).join(", ");
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + pkName_1 + "\""));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + pkName_1 + "\" PRIMARY KEY (" + columnNames_1 + ")"));
                            }
                            primaryColumns.push(column);
                            pkName = this.connection.namingStrategy.primaryKeyName(clonedTable, primaryColumns.map(function (column) { return column.name; }));
                            columnNames = primaryColumns.map(function (column) { return "\"" + column.name + "\""; }).join(", ");
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + pkName + "\" PRIMARY KEY (" + columnNames + ")"));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + pkName + "\""));
                        }
                        columnIndex = clonedTable.indices.find(function (index) { return index.columnNames.length === 1 && index.columnNames[0] === column.name; });
                        if (columnIndex) {
                            upQueries.push(this.createIndexSql(table, columnIndex));
                            downQueries.push(this.dropIndexSql(table, columnIndex));
                        }
                        // create unique constraint
                        if (column.isUnique) {
                            uniqueConstraint = new TableUnique_1.TableUnique({
                                name: this.connection.namingStrategy.uniqueConstraintName(table, [column.name]),
                                columnNames: [column.name]
                            });
                            clonedTable.uniques.push(uniqueConstraint);
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + uniqueConstraint.name + "\" UNIQUE (\"" + column.name + "\")"));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + uniqueConstraint.name + "\""));
                        }
                        // create column's comment
                        if (column.comment) {
                            upQueries.push(new Query_1.Query("COMMENT ON COLUMN " + this.escapePath(table) + ".\"" + column.name + "\" IS " + this.escapeComment(column.comment)));
                            downQueries.push(new Query_1.Query("COMMENT ON COLUMN " + this.escapePath(table) + ".\"" + column.name + "\" IS " + this.escapeComment(column.comment)));
                        }
                        return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 6:
                        _b.sent();
                        clonedTable.addColumn(column);
                        this.replaceCachedTable(table, clonedTable);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new columns from the column in the table.
     */
    PostgresQueryRunner.prototype.addColumns = function (tableOrName, columns) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var columns_1, columns_1_1, column, e_3_1;
            var e_3, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        columns_1 = (0, tslib_1.__values)(columns), columns_1_1 = columns_1.next();
                        _b.label = 1;
                    case 1:
                        if (!!columns_1_1.done) return [3 /*break*/, 4];
                        column = columns_1_1.value;
                        return [4 /*yield*/, this.addColumn(tableOrName, column)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        columns_1_1 = columns_1.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_3_1 = _b.sent();
                        e_3 = { error: e_3_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (columns_1_1 && !columns_1_1.done && (_a = columns_1.return)) _a.call(columns_1);
                        }
                        finally { if (e_3) throw e_3.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Renames column in the given table.
     */
    PostgresQueryRunner.prototype.renameColumn = function (tableOrName, oldTableColumnOrName, newTableColumnOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, oldColumn, newColumn;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        oldColumn = oldTableColumnOrName instanceof TableColumn_1.TableColumn ? oldTableColumnOrName : table.columns.find(function (c) { return c.name === oldTableColumnOrName; });
                        if (!oldColumn)
                            throw new error_1.TypeORMError("Column \"" + oldTableColumnOrName + "\" was not found in the \"" + table.name + "\" table.");
                        if (newTableColumnOrName instanceof TableColumn_1.TableColumn) {
                            newColumn = newTableColumnOrName;
                        }
                        else {
                            newColumn = oldColumn.clone();
                            newColumn.name = newTableColumnOrName;
                        }
                        return [2 /*return*/, this.changeColumn(table, oldColumn, newColumn)];
                }
            });
        });
    };
    /**
     * Changes a column in the table.
     */
    PostgresQueryRunner.prototype.changeColumn = function (tableOrName, oldTableColumnOrName, newColumn) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, clonedTable, upQueries, downQueries, defaultValueChanged, oldColumn, oldEnumType, primaryColumns, columnNames, oldPkName, newPkName, sequencePath, sequenceName, newSequencePath, newSequenceName, up, down, oldTableColumn, arraySuffix, newEnumName, oldEnumName, oldEnumNameWithoutSchema, oldEnumNameWithSchema_old, oldEnumNameWithoutSchema_old, upType, downType, primaryColumns, pkName, columnNames, column, pkName, columnNames, primaryColumn, column, pkName, columnNames, uniqueConstraint, uniqueConstraint;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        clonedTable = table.clone();
                        upQueries = [];
                        downQueries = [];
                        defaultValueChanged = false;
                        oldColumn = oldTableColumnOrName instanceof TableColumn_1.TableColumn
                            ? oldTableColumnOrName
                            : table.columns.find(function (column) { return column.name === oldTableColumnOrName; });
                        if (!oldColumn)
                            throw new error_1.TypeORMError("Column \"" + oldTableColumnOrName + "\" was not found in the \"" + table.name + "\" table.");
                        if (!(oldColumn.type !== newColumn.type || oldColumn.length !== newColumn.length || newColumn.isArray !== oldColumn.isArray)) return [3 /*break*/, 6];
                        // To avoid data conversion, we just recreate column
                        return [4 /*yield*/, this.dropColumn(table, oldColumn)];
                    case 4:
                        // To avoid data conversion, we just recreate column
                        _b.sent();
                        return [4 /*yield*/, this.addColumn(table, newColumn)];
                    case 5:
                        _b.sent();
                        // update cloned table
                        clonedTable = table.clone();
                        return [3 /*break*/, 10];
                    case 6:
                        if (!(oldColumn.name !== newColumn.name)) return [3 /*break*/, 9];
                        // rename column
                        upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " RENAME COLUMN \"" + oldColumn.name + "\" TO \"" + newColumn.name + "\""));
                        downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " RENAME COLUMN \"" + newColumn.name + "\" TO \"" + oldColumn.name + "\""));
                        if (!(oldColumn.type === "enum" || oldColumn.type === "simple-enum")) return [3 /*break*/, 8];
                        return [4 /*yield*/, this.getUserDefinedTypeName(table, oldColumn)];
                    case 7:
                        oldEnumType = _b.sent();
                        upQueries.push(new Query_1.Query("ALTER TYPE \"" + oldEnumType.schema + "\".\"" + oldEnumType.name + "\" RENAME TO " + this.buildEnumName(table, newColumn, false)));
                        downQueries.push(new Query_1.Query("ALTER TYPE " + this.buildEnumName(table, newColumn) + " RENAME TO \"" + oldEnumType.name + "\""));
                        _b.label = 8;
                    case 8:
                        // rename column primary key constraint
                        if (oldColumn.isPrimary === true) {
                            primaryColumns = clonedTable.primaryColumns;
                            columnNames = primaryColumns.map(function (column) { return column.name; });
                            oldPkName = this.connection.namingStrategy.primaryKeyName(clonedTable, columnNames);
                            // replace old column name with new column name
                            columnNames.splice(columnNames.indexOf(oldColumn.name), 1);
                            columnNames.push(newColumn.name);
                            newPkName = this.connection.namingStrategy.primaryKeyName(clonedTable, columnNames);
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " RENAME CONSTRAINT \"" + oldPkName + "\" TO \"" + newPkName + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " RENAME CONSTRAINT \"" + newPkName + "\" TO \"" + oldPkName + "\""));
                        }
                        // rename column sequence
                        if (oldColumn.isGenerated === true && newColumn.generationStrategy === "increment") {
                            sequencePath = this.buildSequencePath(table, oldColumn.name);
                            sequenceName = this.buildSequenceName(table, oldColumn.name);
                            newSequencePath = this.buildSequencePath(table, newColumn.name);
                            newSequenceName = this.buildSequenceName(table, newColumn.name);
                            up = "ALTER SEQUENCE " + this.escapePath(sequencePath) + " RENAME TO \"" + newSequenceName + "\"";
                            down = "ALTER SEQUENCE " + this.escapePath(newSequencePath) + " RENAME TO \"" + sequenceName + "\"";
                            upQueries.push(new Query_1.Query(up));
                            downQueries.push(new Query_1.Query(down));
                        }
                        // rename unique constraints
                        clonedTable.findColumnUniques(oldColumn).forEach(function (unique) {
                            // build new constraint name
                            unique.columnNames.splice(unique.columnNames.indexOf(oldColumn.name), 1);
                            unique.columnNames.push(newColumn.name);
                            var newUniqueName = _this.connection.namingStrategy.uniqueConstraintName(clonedTable, unique.columnNames);
                            // build queries
                            upQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(table) + " RENAME CONSTRAINT \"" + unique.name + "\" TO \"" + newUniqueName + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(table) + " RENAME CONSTRAINT \"" + newUniqueName + "\" TO \"" + unique.name + "\""));
                            // replace constraint name
                            unique.name = newUniqueName;
                        });
                        // rename index constraints
                        clonedTable.findColumnIndices(oldColumn).forEach(function (index) {
                            // build new constraint name
                            index.columnNames.splice(index.columnNames.indexOf(oldColumn.name), 1);
                            index.columnNames.push(newColumn.name);
                            var schema = _this.driver.parseTableName(table).schema;
                            var newIndexName = _this.connection.namingStrategy.indexName(clonedTable, index.columnNames, index.where);
                            // build queries
                            var up = schema ? "ALTER INDEX \"" + schema + "\".\"" + index.name + "\" RENAME TO \"" + newIndexName + "\"" : "ALTER INDEX \"" + index.name + "\" RENAME TO \"" + newIndexName + "\"";
                            var down = schema ? "ALTER INDEX \"" + schema + "\".\"" + newIndexName + "\" RENAME TO \"" + index.name + "\"" : "ALTER INDEX \"" + newIndexName + "\" RENAME TO \"" + index.name + "\"";
                            upQueries.push(new Query_1.Query(up));
                            downQueries.push(new Query_1.Query(down));
                            // replace constraint name
                            index.name = newIndexName;
                        });
                        // rename foreign key constraints
                        clonedTable.findColumnForeignKeys(oldColumn).forEach(function (foreignKey) {
                            // build new constraint name
                            foreignKey.columnNames.splice(foreignKey.columnNames.indexOf(oldColumn.name), 1);
                            foreignKey.columnNames.push(newColumn.name);
                            var newForeignKeyName = _this.connection.namingStrategy.foreignKeyName(clonedTable, foreignKey.columnNames, _this.getTablePath(foreignKey), foreignKey.referencedColumnNames);
                            // build queries
                            upQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(table) + " RENAME CONSTRAINT \"" + foreignKey.name + "\" TO \"" + newForeignKeyName + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + _this.escapePath(table) + " RENAME CONSTRAINT \"" + newForeignKeyName + "\" TO \"" + foreignKey.name + "\""));
                            // replace constraint name
                            foreignKey.name = newForeignKeyName;
                        });
                        oldTableColumn = clonedTable.columns.find(function (column) { return column.name === oldColumn.name; });
                        clonedTable.columns[clonedTable.columns.indexOf(oldTableColumn)].name = newColumn.name;
                        oldColumn.name = newColumn.name;
                        _b.label = 9;
                    case 9:
                        if (newColumn.precision !== oldColumn.precision || newColumn.scale !== oldColumn.scale) {
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" TYPE " + this.driver.createFullType(newColumn)));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" TYPE " + this.driver.createFullType(oldColumn)));
                        }
                        if ((newColumn.type === "enum" || newColumn.type === "simple-enum")
                            && (oldColumn.type === "enum" || oldColumn.type === "simple-enum")
                            && (!OrmUtils_1.OrmUtils.isArraysEqual(newColumn.enum, oldColumn.enum) || newColumn.enumName !== oldColumn.enumName)) {
                            arraySuffix = newColumn.isArray ? "[]" : "";
                            newEnumName = this.buildEnumName(table, newColumn);
                            oldEnumName = this.buildEnumName(table, oldColumn);
                            oldEnumNameWithoutSchema = this.buildEnumName(table, oldColumn, false);
                            oldEnumNameWithSchema_old = this.buildEnumName(table, oldColumn, true, false, true);
                            oldEnumNameWithoutSchema_old = this.buildEnumName(table, oldColumn, false, false, true);
                            // rename old ENUM
                            upQueries.push(new Query_1.Query("ALTER TYPE " + oldEnumName + " RENAME TO " + oldEnumNameWithoutSchema_old));
                            downQueries.push(new Query_1.Query("ALTER TYPE " + oldEnumNameWithSchema_old + " RENAME TO " + oldEnumNameWithoutSchema));
                            // create new ENUM
                            upQueries.push(this.createEnumTypeSql(table, newColumn, newEnumName));
                            downQueries.push(this.dropEnumTypeSql(table, newColumn, newEnumName));
                            // if column have default value, we must drop it to avoid issues with type casting
                            if (oldColumn.default !== null && oldColumn.default !== undefined) {
                                // mark default as changed to prevent double update
                                defaultValueChanged = true;
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + oldColumn.name + "\" DROP DEFAULT"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + oldColumn.name + "\" SET DEFAULT " + oldColumn.default));
                            }
                            upType = "" + newEnumName + arraySuffix + " USING \"" + newColumn.name + "\"::\"text\"::" + newEnumName + arraySuffix;
                            downType = "" + oldEnumNameWithSchema_old + arraySuffix + " USING \"" + newColumn.name + "\"::\"text\"::" + oldEnumNameWithSchema_old + arraySuffix;
                            // update column to use new type
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" TYPE " + upType));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" TYPE " + downType));
                            // restore column default or create new one
                            if (newColumn.default !== null && newColumn.default !== undefined) {
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" SET DEFAULT " + newColumn.default));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" DROP DEFAULT"));
                            }
                            // remove old ENUM
                            upQueries.push(this.dropEnumTypeSql(table, oldColumn, oldEnumNameWithSchema_old));
                            downQueries.push(this.createEnumTypeSql(table, oldColumn, oldEnumNameWithSchema_old));
                        }
                        if (oldColumn.isNullable !== newColumn.isNullable) {
                            if (newColumn.isNullable) {
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + oldColumn.name + "\" DROP NOT NULL"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + oldColumn.name + "\" SET NOT NULL"));
                            }
                            else {
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + oldColumn.name + "\" SET NOT NULL"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + oldColumn.name + "\" DROP NOT NULL"));
                            }
                        }
                        if (oldColumn.comment !== newColumn.comment) {
                            upQueries.push(new Query_1.Query("COMMENT ON COLUMN " + this.escapePath(table) + ".\"" + oldColumn.name + "\" IS " + this.escapeComment(newColumn.comment)));
                            downQueries.push(new Query_1.Query("COMMENT ON COLUMN " + this.escapePath(table) + ".\"" + newColumn.name + "\" IS " + this.escapeComment(oldColumn.comment)));
                        }
                        if (newColumn.isPrimary !== oldColumn.isPrimary) {
                            primaryColumns = clonedTable.primaryColumns;
                            // if primary column state changed, we must always drop existed constraint.
                            if (primaryColumns.length > 0) {
                                pkName = this.connection.namingStrategy.primaryKeyName(clonedTable, primaryColumns.map(function (column) { return column.name; }));
                                columnNames = primaryColumns.map(function (column) { return "\"" + column.name + "\""; }).join(", ");
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + pkName + "\""));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + pkName + "\" PRIMARY KEY (" + columnNames + ")"));
                            }
                            if (newColumn.isPrimary === true) {
                                primaryColumns.push(newColumn);
                                column = clonedTable.columns.find(function (column) { return column.name === newColumn.name; });
                                column.isPrimary = true;
                                pkName = this.connection.namingStrategy.primaryKeyName(clonedTable, primaryColumns.map(function (column) { return column.name; }));
                                columnNames = primaryColumns.map(function (column) { return "\"" + column.name + "\""; }).join(", ");
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + pkName + "\" PRIMARY KEY (" + columnNames + ")"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + pkName + "\""));
                            }
                            else {
                                primaryColumn = primaryColumns.find(function (c) { return c.name === newColumn.name; });
                                primaryColumns.splice(primaryColumns.indexOf(primaryColumn), 1);
                                column = clonedTable.columns.find(function (column) { return column.name === newColumn.name; });
                                column.isPrimary = false;
                                // if we have another primary keys, we must recreate constraint.
                                if (primaryColumns.length > 0) {
                                    pkName = this.connection.namingStrategy.primaryKeyName(clonedTable, primaryColumns.map(function (column) { return column.name; }));
                                    columnNames = primaryColumns.map(function (column) { return "\"" + column.name + "\""; }).join(", ");
                                    upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + pkName + "\" PRIMARY KEY (" + columnNames + ")"));
                                    downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + pkName + "\""));
                                }
                            }
                        }
                        if (newColumn.isUnique !== oldColumn.isUnique) {
                            if (newColumn.isUnique === true) {
                                uniqueConstraint = new TableUnique_1.TableUnique({
                                    name: this.connection.namingStrategy.uniqueConstraintName(table, [newColumn.name]),
                                    columnNames: [newColumn.name]
                                });
                                clonedTable.uniques.push(uniqueConstraint);
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + uniqueConstraint.name + "\" UNIQUE (\"" + newColumn.name + "\")"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + uniqueConstraint.name + "\""));
                            }
                            else {
                                uniqueConstraint = clonedTable.uniques.find(function (unique) {
                                    return unique.columnNames.length === 1 && !!unique.columnNames.find(function (columnName) { return columnName === newColumn.name; });
                                });
                                clonedTable.uniques.splice(clonedTable.uniques.indexOf(uniqueConstraint), 1);
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + uniqueConstraint.name + "\""));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + uniqueConstraint.name + "\" UNIQUE (\"" + newColumn.name + "\")"));
                            }
                        }
                        if (oldColumn.isGenerated !== newColumn.isGenerated && newColumn.generationStrategy !== "uuid") {
                            if (newColumn.isGenerated === true) {
                                upQueries.push(new Query_1.Query("CREATE SEQUENCE IF NOT EXISTS " + this.escapePath(this.buildSequencePath(table, newColumn)) + " OWNED BY " + this.escapePath(table) + ".\"" + newColumn.name + "\""));
                                downQueries.push(new Query_1.Query("DROP SEQUENCE " + this.escapePath(this.buildSequencePath(table, newColumn))));
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" SET DEFAULT nextval('" + this.escapePath(this.buildSequencePath(table, newColumn)) + "')"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" DROP DEFAULT"));
                            }
                            else {
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" DROP DEFAULT"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" SET DEFAULT nextval('" + this.escapePath(this.buildSequencePath(table, newColumn)) + "')"));
                                upQueries.push(new Query_1.Query("DROP SEQUENCE " + this.escapePath(this.buildSequencePath(table, newColumn))));
                                downQueries.push(new Query_1.Query("CREATE SEQUENCE IF NOT EXISTS " + this.escapePath(this.buildSequencePath(table, newColumn)) + " OWNED BY " + this.escapePath(table) + ".\"" + newColumn.name + "\""));
                            }
                        }
                        // the default might have changed when the enum changed
                        if (newColumn.default !== oldColumn.default && !defaultValueChanged) {
                            if (newColumn.default !== null && newColumn.default !== undefined) {
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" SET DEFAULT " + newColumn.default));
                                if (oldColumn.default !== null && oldColumn.default !== undefined) {
                                    downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" SET DEFAULT " + oldColumn.default));
                                }
                                else {
                                    downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" DROP DEFAULT"));
                                }
                            }
                            else if (oldColumn.default !== null && oldColumn.default !== undefined) {
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" DROP DEFAULT"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" SET DEFAULT " + oldColumn.default));
                            }
                        }
                        if ((newColumn.spatialFeatureType || "").toLowerCase() !== (oldColumn.spatialFeatureType || "").toLowerCase() || newColumn.srid !== oldColumn.srid) {
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" TYPE " + this.driver.createFullType(newColumn)));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ALTER COLUMN \"" + newColumn.name + "\" TYPE " + this.driver.createFullType(oldColumn)));
                        }
                        _b.label = 10;
                    case 10: return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 11:
                        _b.sent();
                        this.replaceCachedTable(table, clonedTable);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Changes a column in the table.
     */
    PostgresQueryRunner.prototype.changeColumns = function (tableOrName, changedColumns) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var changedColumns_1, changedColumns_1_1, _a, oldColumn, newColumn, e_4_1;
            var e_4, _b;
            return (0, tslib_1.__generator)(this, function (_c) {
                switch (_c.label) {
                    case 0:
                        _c.trys.push([0, 5, 6, 7]);
                        changedColumns_1 = (0, tslib_1.__values)(changedColumns), changedColumns_1_1 = changedColumns_1.next();
                        _c.label = 1;
                    case 1:
                        if (!!changedColumns_1_1.done) return [3 /*break*/, 4];
                        _a = changedColumns_1_1.value, oldColumn = _a.oldColumn, newColumn = _a.newColumn;
                        return [4 /*yield*/, this.changeColumn(tableOrName, oldColumn, newColumn)];
                    case 2:
                        _c.sent();
                        _c.label = 3;
                    case 3:
                        changedColumns_1_1 = changedColumns_1.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_4_1 = _c.sent();
                        e_4 = { error: e_4_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (changedColumns_1_1 && !changedColumns_1_1.done && (_b = changedColumns_1.return)) _b.call(changedColumns_1);
                        }
                        finally { if (e_4) throw e_4.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops column in the table.
     */
    PostgresQueryRunner.prototype.dropColumn = function (tableOrName, columnOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, column, clonedTable, upQueries, downQueries, pkName, columnNames, tableColumn, pkName_2, columnNames_2, columnIndex, columnCheck, columnUnique, hasEnum, enumType, escapedEnumName;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        column = columnOrName instanceof TableColumn_1.TableColumn ? columnOrName : table.findColumnByName(columnOrName);
                        if (!column)
                            throw new error_1.TypeORMError("Column \"" + columnOrName + "\" was not found in table \"" + table.name + "\"");
                        clonedTable = table.clone();
                        upQueries = [];
                        downQueries = [];
                        // drop primary key constraint
                        if (column.isPrimary) {
                            pkName = this.connection.namingStrategy.primaryKeyName(clonedTable, clonedTable.primaryColumns.map(function (column) { return column.name; }));
                            columnNames = clonedTable.primaryColumns.map(function (primaryColumn) { return "\"" + primaryColumn.name + "\""; }).join(", ");
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(clonedTable) + " DROP CONSTRAINT \"" + pkName + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(clonedTable) + " ADD CONSTRAINT \"" + pkName + "\" PRIMARY KEY (" + columnNames + ")"));
                            tableColumn = clonedTable.findColumnByName(column.name);
                            tableColumn.isPrimary = false;
                            // if primary key have multiple columns, we must recreate it without dropped column
                            if (clonedTable.primaryColumns.length > 0) {
                                pkName_2 = this.connection.namingStrategy.primaryKeyName(clonedTable, clonedTable.primaryColumns.map(function (column) { return column.name; }));
                                columnNames_2 = clonedTable.primaryColumns.map(function (primaryColumn) { return "\"" + primaryColumn.name + "\""; }).join(", ");
                                upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(clonedTable) + " ADD CONSTRAINT \"" + pkName_2 + "\" PRIMARY KEY (" + columnNames_2 + ")"));
                                downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(clonedTable) + " DROP CONSTRAINT \"" + pkName_2 + "\""));
                            }
                        }
                        columnIndex = clonedTable.indices.find(function (index) { return index.columnNames.length === 1 && index.columnNames[0] === column.name; });
                        if (columnIndex) {
                            clonedTable.indices.splice(clonedTable.indices.indexOf(columnIndex), 1);
                            upQueries.push(this.dropIndexSql(table, columnIndex));
                            downQueries.push(this.createIndexSql(table, columnIndex));
                        }
                        columnCheck = clonedTable.checks.find(function (check) { return !!check.columnNames && check.columnNames.length === 1 && check.columnNames[0] === column.name; });
                        if (columnCheck) {
                            clonedTable.checks.splice(clonedTable.checks.indexOf(columnCheck), 1);
                            upQueries.push(this.dropCheckConstraintSql(table, columnCheck));
                            downQueries.push(this.createCheckConstraintSql(table, columnCheck));
                        }
                        columnUnique = clonedTable.uniques.find(function (unique) { return unique.columnNames.length === 1 && unique.columnNames[0] === column.name; });
                        if (columnUnique) {
                            clonedTable.uniques.splice(clonedTable.uniques.indexOf(columnUnique), 1);
                            upQueries.push(this.dropUniqueConstraintSql(table, columnUnique));
                            downQueries.push(this.createUniqueConstraintSql(table, columnUnique));
                        }
                        upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP COLUMN \"" + column.name + "\""));
                        downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD " + this.buildCreateColumnSql(table, column)));
                        if (!(column.type === "enum" || column.type === "simple-enum")) return [3 /*break*/, 6];
                        return [4 /*yield*/, this.hasEnumType(table, column)];
                    case 4:
                        hasEnum = _b.sent();
                        if (!hasEnum) return [3 /*break*/, 6];
                        return [4 /*yield*/, this.getUserDefinedTypeName(table, column)];
                    case 5:
                        enumType = _b.sent();
                        escapedEnumName = "\"" + enumType.schema + "\".\"" + enumType.name + "\"";
                        upQueries.push(this.dropEnumTypeSql(table, column, escapedEnumName));
                        downQueries.push(this.createEnumTypeSql(table, column, escapedEnumName));
                        _b.label = 6;
                    case 6: return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 7:
                        _b.sent();
                        clonedTable.removeColumn(column);
                        this.replaceCachedTable(table, clonedTable);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops the columns in the table.
     */
    PostgresQueryRunner.prototype.dropColumns = function (tableOrName, columns) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var columns_2, columns_2_1, column, e_5_1;
            var e_5, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        columns_2 = (0, tslib_1.__values)(columns), columns_2_1 = columns_2.next();
                        _b.label = 1;
                    case 1:
                        if (!!columns_2_1.done) return [3 /*break*/, 4];
                        column = columns_2_1.value;
                        return [4 /*yield*/, this.dropColumn(tableOrName, column)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        columns_2_1 = columns_2.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_5_1 = _b.sent();
                        e_5 = { error: e_5_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (columns_2_1 && !columns_2_1.done && (_a = columns_2.return)) _a.call(columns_2);
                        }
                        finally { if (e_5) throw e_5.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new primary key.
     */
    PostgresQueryRunner.prototype.createPrimaryKey = function (tableOrName, columnNames) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, clonedTable, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        clonedTable = table.clone();
                        up = this.createPrimaryKeySql(table, columnNames);
                        // mark columns as primary, because dropPrimaryKeySql build constraint name from table primary column names.
                        clonedTable.columns.forEach(function (column) {
                            if (columnNames.find(function (columnName) { return columnName === column.name; }))
                                column.isPrimary = true;
                        });
                        down = this.dropPrimaryKeySql(clonedTable);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        this.replaceCachedTable(table, clonedTable);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Updates composite primary keys.
     */
    PostgresQueryRunner.prototype.updatePrimaryKeys = function (tableOrName, columns) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, clonedTable, columnNames, upQueries, downQueries, primaryColumns, pkName_3, columnNamesString_1, pkName, columnNamesString;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        clonedTable = table.clone();
                        columnNames = columns.map(function (column) { return column.name; });
                        upQueries = [];
                        downQueries = [];
                        primaryColumns = clonedTable.primaryColumns;
                        if (primaryColumns.length > 0) {
                            pkName_3 = this.connection.namingStrategy.primaryKeyName(clonedTable, primaryColumns.map(function (column) { return column.name; }));
                            columnNamesString_1 = primaryColumns.map(function (column) { return "\"" + column.name + "\""; }).join(", ");
                            upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + pkName_3 + "\""));
                            downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + pkName_3 + "\" PRIMARY KEY (" + columnNamesString_1 + ")"));
                        }
                        // update columns in table.
                        clonedTable.columns
                            .filter(function (column) { return columnNames.indexOf(column.name) !== -1; })
                            .forEach(function (column) { return column.isPrimary = true; });
                        pkName = this.connection.namingStrategy.primaryKeyName(clonedTable, columnNames);
                        columnNamesString = columnNames.map(function (columnName) { return "\"" + columnName + "\""; }).join(", ");
                        upQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + pkName + "\" PRIMARY KEY (" + columnNamesString + ")"));
                        downQueries.push(new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + pkName + "\""));
                        return [4 /*yield*/, this.executeQueries(upQueries, downQueries)];
                    case 4:
                        _b.sent();
                        this.replaceCachedTable(table, clonedTable);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops a primary key.
     */
    PostgresQueryRunner.prototype.dropPrimaryKey = function (tableOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        up = this.dropPrimaryKeySql(table);
                        down = this.createPrimaryKeySql(table, table.primaryColumns.map(function (column) { return column.name; }));
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.primaryColumns.forEach(function (column) {
                            column.isPrimary = false;
                        });
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates new unique constraint.
     */
    PostgresQueryRunner.prototype.createUniqueConstraint = function (tableOrName, uniqueConstraint) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        // new unique constraint may be passed without name. In this case we generate unique name manually.
                        if (!uniqueConstraint.name)
                            uniqueConstraint.name = this.connection.namingStrategy.uniqueConstraintName(table, uniqueConstraint.columnNames);
                        up = this.createUniqueConstraintSql(table, uniqueConstraint);
                        down = this.dropUniqueConstraintSql(table, uniqueConstraint);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.addUniqueConstraint(uniqueConstraint);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates new unique constraints.
     */
    PostgresQueryRunner.prototype.createUniqueConstraints = function (tableOrName, uniqueConstraints) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var uniqueConstraints_1, uniqueConstraints_1_1, uniqueConstraint, e_6_1;
            var e_6, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        uniqueConstraints_1 = (0, tslib_1.__values)(uniqueConstraints), uniqueConstraints_1_1 = uniqueConstraints_1.next();
                        _b.label = 1;
                    case 1:
                        if (!!uniqueConstraints_1_1.done) return [3 /*break*/, 4];
                        uniqueConstraint = uniqueConstraints_1_1.value;
                        return [4 /*yield*/, this.createUniqueConstraint(tableOrName, uniqueConstraint)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        uniqueConstraints_1_1 = uniqueConstraints_1.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_6_1 = _b.sent();
                        e_6 = { error: e_6_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (uniqueConstraints_1_1 && !uniqueConstraints_1_1.done && (_a = uniqueConstraints_1.return)) _a.call(uniqueConstraints_1);
                        }
                        finally { if (e_6) throw e_6.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops unique constraint.
     */
    PostgresQueryRunner.prototype.dropUniqueConstraint = function (tableOrName, uniqueOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, uniqueConstraint, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        uniqueConstraint = uniqueOrName instanceof TableUnique_1.TableUnique ? uniqueOrName : table.uniques.find(function (u) { return u.name === uniqueOrName; });
                        if (!uniqueConstraint)
                            throw new error_1.TypeORMError("Supplied unique constraint was not found in table " + table.name);
                        up = this.dropUniqueConstraintSql(table, uniqueConstraint);
                        down = this.createUniqueConstraintSql(table, uniqueConstraint);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.removeUniqueConstraint(uniqueConstraint);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops unique constraints.
     */
    PostgresQueryRunner.prototype.dropUniqueConstraints = function (tableOrName, uniqueConstraints) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var uniqueConstraints_2, uniqueConstraints_2_1, uniqueConstraint, e_7_1;
            var e_7, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        uniqueConstraints_2 = (0, tslib_1.__values)(uniqueConstraints), uniqueConstraints_2_1 = uniqueConstraints_2.next();
                        _b.label = 1;
                    case 1:
                        if (!!uniqueConstraints_2_1.done) return [3 /*break*/, 4];
                        uniqueConstraint = uniqueConstraints_2_1.value;
                        return [4 /*yield*/, this.dropUniqueConstraint(tableOrName, uniqueConstraint)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        uniqueConstraints_2_1 = uniqueConstraints_2.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_7_1 = _b.sent();
                        e_7 = { error: e_7_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (uniqueConstraints_2_1 && !uniqueConstraints_2_1.done && (_a = uniqueConstraints_2.return)) _a.call(uniqueConstraints_2);
                        }
                        finally { if (e_7) throw e_7.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates new check constraint.
     */
    PostgresQueryRunner.prototype.createCheckConstraint = function (tableOrName, checkConstraint) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        // new unique constraint may be passed without name. In this case we generate unique name manually.
                        if (!checkConstraint.name)
                            checkConstraint.name = this.connection.namingStrategy.checkConstraintName(table, checkConstraint.expression);
                        up = this.createCheckConstraintSql(table, checkConstraint);
                        down = this.dropCheckConstraintSql(table, checkConstraint);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.addCheckConstraint(checkConstraint);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates new check constraints.
     */
    PostgresQueryRunner.prototype.createCheckConstraints = function (tableOrName, checkConstraints) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var promises;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        promises = checkConstraints.map(function (checkConstraint) { return _this.createCheckConstraint(tableOrName, checkConstraint); });
                        return [4 /*yield*/, Promise.all(promises)];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops check constraint.
     */
    PostgresQueryRunner.prototype.dropCheckConstraint = function (tableOrName, checkOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, checkConstraint, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        checkConstraint = checkOrName instanceof TableCheck_1.TableCheck ? checkOrName : table.checks.find(function (c) { return c.name === checkOrName; });
                        if (!checkConstraint)
                            throw new error_1.TypeORMError("Supplied check constraint was not found in table " + table.name);
                        up = this.dropCheckConstraintSql(table, checkConstraint);
                        down = this.createCheckConstraintSql(table, checkConstraint);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.removeCheckConstraint(checkConstraint);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops check constraints.
     */
    PostgresQueryRunner.prototype.dropCheckConstraints = function (tableOrName, checkConstraints) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var promises;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        promises = checkConstraints.map(function (checkConstraint) { return _this.dropCheckConstraint(tableOrName, checkConstraint); });
                        return [4 /*yield*/, Promise.all(promises)];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates new exclusion constraint.
     */
    PostgresQueryRunner.prototype.createExclusionConstraint = function (tableOrName, exclusionConstraint) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        // new unique constraint may be passed without name. In this case we generate unique name manually.
                        if (!exclusionConstraint.name)
                            exclusionConstraint.name = this.connection.namingStrategy.exclusionConstraintName(table, exclusionConstraint.expression);
                        up = this.createExclusionConstraintSql(table, exclusionConstraint);
                        down = this.dropExclusionConstraintSql(table, exclusionConstraint);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.addExclusionConstraint(exclusionConstraint);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates new exclusion constraints.
     */
    PostgresQueryRunner.prototype.createExclusionConstraints = function (tableOrName, exclusionConstraints) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var promises;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        promises = exclusionConstraints.map(function (exclusionConstraint) { return _this.createExclusionConstraint(tableOrName, exclusionConstraint); });
                        return [4 /*yield*/, Promise.all(promises)];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops exclusion constraint.
     */
    PostgresQueryRunner.prototype.dropExclusionConstraint = function (tableOrName, exclusionOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, exclusionConstraint, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        exclusionConstraint = exclusionOrName instanceof TableExclusion_1.TableExclusion ? exclusionOrName : table.exclusions.find(function (c) { return c.name === exclusionOrName; });
                        if (!exclusionConstraint)
                            throw new error_1.TypeORMError("Supplied exclusion constraint was not found in table " + table.name);
                        up = this.dropExclusionConstraintSql(table, exclusionConstraint);
                        down = this.createExclusionConstraintSql(table, exclusionConstraint);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.removeExclusionConstraint(exclusionConstraint);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops exclusion constraints.
     */
    PostgresQueryRunner.prototype.dropExclusionConstraints = function (tableOrName, exclusionConstraints) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var promises;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        promises = exclusionConstraints.map(function (exclusionConstraint) { return _this.dropExclusionConstraint(tableOrName, exclusionConstraint); });
                        return [4 /*yield*/, Promise.all(promises)];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new foreign key.
     */
    PostgresQueryRunner.prototype.createForeignKey = function (tableOrName, foreignKey) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        // new FK may be passed without name. In this case we generate FK name manually.
                        if (!foreignKey.name)
                            foreignKey.name = this.connection.namingStrategy.foreignKeyName(table, foreignKey.columnNames, this.getTablePath(foreignKey), foreignKey.referencedColumnNames);
                        up = this.createForeignKeySql(table, foreignKey);
                        down = this.dropForeignKeySql(table, foreignKey);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.addForeignKey(foreignKey);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new foreign keys.
     */
    PostgresQueryRunner.prototype.createForeignKeys = function (tableOrName, foreignKeys) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var foreignKeys_1, foreignKeys_1_1, foreignKey, e_8_1;
            var e_8, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        foreignKeys_1 = (0, tslib_1.__values)(foreignKeys), foreignKeys_1_1 = foreignKeys_1.next();
                        _b.label = 1;
                    case 1:
                        if (!!foreignKeys_1_1.done) return [3 /*break*/, 4];
                        foreignKey = foreignKeys_1_1.value;
                        return [4 /*yield*/, this.createForeignKey(tableOrName, foreignKey)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        foreignKeys_1_1 = foreignKeys_1.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_8_1 = _b.sent();
                        e_8 = { error: e_8_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (foreignKeys_1_1 && !foreignKeys_1_1.done && (_a = foreignKeys_1.return)) _a.call(foreignKeys_1);
                        }
                        finally { if (e_8) throw e_8.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops a foreign key from the table.
     */
    PostgresQueryRunner.prototype.dropForeignKey = function (tableOrName, foreignKeyOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, foreignKey, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        foreignKey = foreignKeyOrName instanceof TableForeignKey_1.TableForeignKey ? foreignKeyOrName : table.foreignKeys.find(function (fk) { return fk.name === foreignKeyOrName; });
                        if (!foreignKey)
                            throw new error_1.TypeORMError("Supplied foreign key was not found in table " + table.name);
                        up = this.dropForeignKeySql(table, foreignKey);
                        down = this.createForeignKeySql(table, foreignKey);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.removeForeignKey(foreignKey);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops a foreign keys from the table.
     */
    PostgresQueryRunner.prototype.dropForeignKeys = function (tableOrName, foreignKeys) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var foreignKeys_2, foreignKeys_2_1, foreignKey, e_9_1;
            var e_9, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        foreignKeys_2 = (0, tslib_1.__values)(foreignKeys), foreignKeys_2_1 = foreignKeys_2.next();
                        _b.label = 1;
                    case 1:
                        if (!!foreignKeys_2_1.done) return [3 /*break*/, 4];
                        foreignKey = foreignKeys_2_1.value;
                        return [4 /*yield*/, this.dropForeignKey(tableOrName, foreignKey)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        foreignKeys_2_1 = foreignKeys_2.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_9_1 = _b.sent();
                        e_9 = { error: e_9_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (foreignKeys_2_1 && !foreignKeys_2_1.done && (_a = foreignKeys_2.return)) _a.call(foreignKeys_2);
                        }
                        finally { if (e_9) throw e_9.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new index.
     */
    PostgresQueryRunner.prototype.createIndex = function (tableOrName, index) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        // new index may be passed without name. In this case we generate index name manually.
                        if (!index.name)
                            index.name = this.connection.namingStrategy.indexName(table, index.columnNames, index.where);
                        up = this.createIndexSql(table, index);
                        down = this.dropIndexSql(table, index);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.addIndex(index);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Creates a new indices
     */
    PostgresQueryRunner.prototype.createIndices = function (tableOrName, indices) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var indices_1, indices_1_1, index, e_10_1;
            var e_10, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        indices_1 = (0, tslib_1.__values)(indices), indices_1_1 = indices_1.next();
                        _b.label = 1;
                    case 1:
                        if (!!indices_1_1.done) return [3 /*break*/, 4];
                        index = indices_1_1.value;
                        return [4 /*yield*/, this.createIndex(tableOrName, index)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        indices_1_1 = indices_1.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_10_1 = _b.sent();
                        e_10 = { error: e_10_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (indices_1_1 && !indices_1_1.done && (_a = indices_1.return)) _a.call(indices_1);
                        }
                        finally { if (e_10) throw e_10.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops an index from the table.
     */
    PostgresQueryRunner.prototype.dropIndex = function (tableOrName, indexOrName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var table, _a, index, up, down;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!(tableOrName instanceof Table_1.Table)) return [3 /*break*/, 1];
                        _a = tableOrName;
                        return [3 /*break*/, 3];
                    case 1: return [4 /*yield*/, this.getCachedTable(tableOrName)];
                    case 2:
                        _a = _b.sent();
                        _b.label = 3;
                    case 3:
                        table = _a;
                        index = indexOrName instanceof TableIndex_1.TableIndex ? indexOrName : table.indices.find(function (i) { return i.name === indexOrName; });
                        if (!index)
                            throw new error_1.TypeORMError("Supplied index was not found in table " + table.name);
                        up = this.dropIndexSql(table, index);
                        down = this.createIndexSql(table, index);
                        return [4 /*yield*/, this.executeQueries(up, down)];
                    case 4:
                        _b.sent();
                        table.removeIndex(index);
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Drops an indices from the table.
     */
    PostgresQueryRunner.prototype.dropIndices = function (tableOrName, indices) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var indices_2, indices_2_1, index, e_11_1;
            var e_11, _a;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _b.trys.push([0, 5, 6, 7]);
                        indices_2 = (0, tslib_1.__values)(indices), indices_2_1 = indices_2.next();
                        _b.label = 1;
                    case 1:
                        if (!!indices_2_1.done) return [3 /*break*/, 4];
                        index = indices_2_1.value;
                        return [4 /*yield*/, this.dropIndex(tableOrName, index)];
                    case 2:
                        _b.sent();
                        _b.label = 3;
                    case 3:
                        indices_2_1 = indices_2.next();
                        return [3 /*break*/, 1];
                    case 4: return [3 /*break*/, 7];
                    case 5:
                        e_11_1 = _b.sent();
                        e_11 = { error: e_11_1 };
                        return [3 /*break*/, 7];
                    case 6:
                        try {
                            if (indices_2_1 && !indices_2_1.done && (_a = indices_2.return)) _a.call(indices_2);
                        }
                        finally { if (e_11) throw e_11.error; }
                        return [7 /*endfinally*/];
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Clears all table contents.
     * Note: this operation uses SQL's TRUNCATE query which cannot be reverted in transactions.
     */
    PostgresQueryRunner.prototype.clearTable = function (tableName) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.query("TRUNCATE TABLE " + this.escapePath(tableName))];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Removes all tables from the currently connected database.
     */
    PostgresQueryRunner.prototype.clearDatabase = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var schemas, schemaNamesString, version, selectViewDropsQuery, dropViewQueries, selectMatViewDropsQuery, dropMatViewQueries, selectTableDropsQuery, dropTableQueries, error_2, rollbackError_1;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        schemas = [];
                        this.connection.entityMetadatas
                            .filter(function (metadata) { return metadata.schema; })
                            .forEach(function (metadata) {
                            var isSchemaExist = !!schemas.find(function (schema) { return schema === metadata.schema; });
                            if (!isSchemaExist)
                                schemas.push(metadata.schema);
                        });
                        schemas.push(this.driver.options.schema || "current_schema()");
                        schemaNamesString = schemas.map(function (name) {
                            return name === "current_schema()" ? name : "'" + name + "'";
                        }).join(", ");
                        return [4 /*yield*/, this.startTransaction()];
                    case 1:
                        _a.sent();
                        _a.label = 2;
                    case 2:
                        _a.trys.push([2, 13, , 18]);
                        return [4 /*yield*/, this.getVersion()
                            // drop views
                        ];
                    case 3:
                        version = _a.sent();
                        selectViewDropsQuery = "SELECT 'DROP VIEW IF EXISTS \"' || schemaname || '\".\"' || viewname || '\" CASCADE;' as \"query\" " +
                            ("FROM \"pg_views\" WHERE \"schemaname\" IN (" + schemaNamesString + ") AND \"viewname\" NOT IN ('geography_columns', 'geometry_columns', 'raster_columns', 'raster_overviews')");
                        return [4 /*yield*/, this.query(selectViewDropsQuery)];
                    case 4:
                        dropViewQueries = _a.sent();
                        return [4 /*yield*/, Promise.all(dropViewQueries.map(function (q) { return _this.query(q["query"]); }))];
                    case 5:
                        _a.sent();
                        if (!VersionUtils_1.VersionUtils.isGreaterOrEqual(version, "9.3")) return [3 /*break*/, 8];
                        selectMatViewDropsQuery = "SELECT 'DROP MATERIALIZED VIEW IF EXISTS \"' || schemaname || '\".\"' || matviewname || '\" CASCADE;' as \"query\" " +
                            ("FROM \"pg_matviews\" WHERE \"schemaname\" IN (" + schemaNamesString + ")");
                        return [4 /*yield*/, this.query(selectMatViewDropsQuery)];
                    case 6:
                        dropMatViewQueries = _a.sent();
                        return [4 /*yield*/, Promise.all(dropMatViewQueries.map(function (q) { return _this.query(q["query"]); }))];
                    case 7:
                        _a.sent();
                        _a.label = 8;
                    case 8:
                        selectTableDropsQuery = "SELECT 'DROP TABLE IF EXISTS \"' || schemaname || '\".\"' || tablename || '\" CASCADE;' as \"query\" FROM \"pg_tables\" WHERE \"schemaname\" IN (" + schemaNamesString + ") AND \"tablename\" NOT IN ('spatial_ref_sys')";
                        return [4 /*yield*/, this.query(selectTableDropsQuery)];
                    case 9:
                        dropTableQueries = _a.sent();
                        return [4 /*yield*/, Promise.all(dropTableQueries.map(function (q) { return _this.query(q["query"]); }))];
                    case 10:
                        _a.sent();
                        // drop enum types
                        return [4 /*yield*/, this.dropEnumTypes(schemaNamesString)];
                    case 11:
                        // drop enum types
                        _a.sent();
                        return [4 /*yield*/, this.commitTransaction()];
                    case 12:
                        _a.sent();
                        return [3 /*break*/, 18];
                    case 13:
                        error_2 = _a.sent();
                        _a.label = 14;
                    case 14:
                        _a.trys.push([14, 16, , 17]);
                        return [4 /*yield*/, this.rollbackTransaction()];
                    case 15:
                        _a.sent();
                        return [3 /*break*/, 17];
                    case 16:
                        rollbackError_1 = _a.sent();
                        return [3 /*break*/, 17];
                    case 17: throw error_2;
                    case 18: return [2 /*return*/];
                }
            });
        });
    };
    // -------------------------------------------------------------------------
    // Protected Methods
    // -------------------------------------------------------------------------
    PostgresQueryRunner.prototype.loadViews = function (viewNames) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var hasTable, currentDatabase, currentSchema, viewsCondition, query, dbViews;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.hasTable(this.getTypeormMetadataTableName())];
                    case 1:
                        hasTable = _a.sent();
                        if (!hasTable)
                            return [2 /*return*/, []];
                        if (!viewNames) {
                            viewNames = [];
                        }
                        return [4 /*yield*/, this.getCurrentDatabase()];
                    case 2:
                        currentDatabase = _a.sent();
                        return [4 /*yield*/, this.getCurrentSchema()];
                    case 3:
                        currentSchema = _a.sent();
                        viewsCondition = viewNames.length === 0 ? "1=1" : viewNames
                            .map(function (tableName) { return _this.driver.parseTableName(tableName); })
                            .map(function (_a) {
                            var schema = _a.schema, tableName = _a.tableName;
                            if (!schema) {
                                schema = _this.driver.options.schema || currentSchema;
                            }
                            return "(\"t\".\"schema\" = '" + schema + "' AND \"t\".\"name\" = '" + tableName + "')";
                        }).join(" OR ");
                        query = "SELECT \"t\".* FROM " + this.escapePath(this.getTypeormMetadataTableName()) + " \"t\" " +
                            "INNER JOIN \"pg_catalog\".\"pg_class\" \"c\" ON \"c\".\"relname\" = \"t\".\"name\" " +
                            "INNER JOIN \"pg_namespace\" \"n\" ON \"n\".\"oid\" = \"c\".\"relnamespace\" AND \"n\".\"nspname\" = \"t\".\"schema\" " +
                            ("WHERE \"t\".\"type\" IN ('VIEW', 'MATERIALIZED_VIEW') " + (viewsCondition ? "AND (" + viewsCondition + ")" : ""));
                        return [4 /*yield*/, this.query(query)];
                    case 4:
                        dbViews = _a.sent();
                        return [2 /*return*/, dbViews.map(function (dbView) {
                                var view = new View_1.View();
                                var schema = dbView["schema"] === currentSchema && !_this.driver.options.schema ? undefined : dbView["schema"];
                                view.database = currentDatabase;
                                view.schema = dbView["schema"];
                                view.name = _this.driver.buildTableName(dbView["name"], schema);
                                view.expression = dbView["value"];
                                view.materialized = dbView["type"] === "MATERIALIZED_VIEW";
                                return view;
                            })];
                }
            });
        });
    };
    /**
     * Loads all tables (with given names) from the database and creates a Table from them.
     */
    PostgresQueryRunner.prototype.loadTables = function (tableNames) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var currentSchema, currentDatabase, dbTables, tablesSql, _a, _b, _c, _d, tablesCondition, tablesSql, _e, _f, _g, _h, columnsCondition, columnsSql, constraintsCondition, constraintsSql, indicesSql, foreignKeysCondition, hasRelispartitionColumn, isPartitionCondition, foreignKeysSql, _j, dbColumns, dbConstraints, dbIndices, dbForeignKeys;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_k) {
                switch (_k.label) {
                    case 0:
                        // if no tables given then no need to proceed
                        if (tableNames && tableNames.length === 0) {
                            return [2 /*return*/, []];
                        }
                        return [4 /*yield*/, this.getCurrentSchema()];
                    case 1:
                        currentSchema = _k.sent();
                        return [4 /*yield*/, this.getCurrentDatabase()];
                    case 2:
                        currentDatabase = _k.sent();
                        dbTables = [];
                        if (!!tableNames) return [3 /*break*/, 4];
                        tablesSql = "SELECT \"table_schema\", \"table_name\" FROM \"information_schema\".\"tables\"";
                        _b = (_a = dbTables.push).apply;
                        _c = [dbTables];
                        _d = [[]];
                        return [4 /*yield*/, this.query(tablesSql)];
                    case 3:
                        _b.apply(_a, _c.concat([tslib_1.__spreadArray.apply(void 0, _d.concat([tslib_1.__read.apply(void 0, [_k.sent()]), false]))]));
                        return [3 /*break*/, 6];
                    case 4:
                        tablesCondition = tableNames
                            .map(function (tableName) { return _this.driver.parseTableName(tableName); })
                            .map(function (_a) {
                            var schema = _a.schema, tableName = _a.tableName;
                            return "(\"table_schema\" = '" + (schema || currentSchema) + "' AND \"table_name\" = '" + tableName + "')";
                        }).join(" OR ");
                        tablesSql = "SELECT \"table_schema\", \"table_name\" FROM \"information_schema\".\"tables\" WHERE " + tablesCondition;
                        _f = (_e = dbTables.push).apply;
                        _g = [dbTables];
                        _h = [[]];
                        return [4 /*yield*/, this.query(tablesSql)];
                    case 5:
                        _f.apply(_e, _g.concat([tslib_1.__spreadArray.apply(void 0, _h.concat([tslib_1.__read.apply(void 0, [_k.sent()]), false]))]));
                        _k.label = 6;
                    case 6:
                        // if tables were not found in the db, no need to proceed
                        if (dbTables.length === 0) {
                            return [2 /*return*/, []];
                        }
                        columnsCondition = dbTables.map(function (_a) {
                            var table_schema = _a.table_schema, table_name = _a.table_name;
                            return "(\"table_schema\" = '" + table_schema + "' AND \"table_name\" = '" + table_name + "')";
                        }).join(" OR ");
                        columnsSql = "SELECT columns.*, pg_catalog.col_description(('\"' || table_catalog || '\".\"' || table_schema || '\".\"' || table_name || '\"')::regclass::oid, ordinal_position) AS description, " +
                            "('\"' || \"udt_schema\" || '\".\"' || \"udt_name\" || '\"')::\"regtype\" AS \"regtype\", pg_catalog.format_type(\"col_attr\".\"atttypid\", \"col_attr\".\"atttypmod\") AS \"format_type\" " +
                            "FROM \"information_schema\".\"columns\" " +
                            "LEFT JOIN \"pg_catalog\".\"pg_attribute\" AS \"col_attr\" ON \"col_attr\".\"attname\" = \"columns\".\"column_name\" " +
                            "AND \"col_attr\".\"attrelid\" = ( " +
                            "SELECT \"cls\".\"oid\" FROM \"pg_catalog\".\"pg_class\" AS \"cls\" " +
                            "LEFT JOIN \"pg_catalog\".\"pg_namespace\" AS \"ns\" ON \"ns\".\"oid\" = \"cls\".\"relnamespace\" " +
                            "WHERE \"cls\".\"relname\" = \"columns\".\"table_name\" " +
                            "AND \"ns\".\"nspname\" = \"columns\".\"table_schema\" " +
                            ") " +
                            "WHERE " + columnsCondition;
                        constraintsCondition = dbTables.map(function (_a) {
                            var table_schema = _a.table_schema, table_name = _a.table_name;
                            return "(\"ns\".\"nspname\" = '" + table_schema + "' AND \"t\".\"relname\" = '" + table_name + "')";
                        }).join(" OR ");
                        constraintsSql = "SELECT \"ns\".\"nspname\" AS \"table_schema\", \"t\".\"relname\" AS \"table_name\", \"cnst\".\"conname\" AS \"constraint_name\", " +
                            "pg_get_constraintdef(\"cnst\".\"oid\") AS \"expression\", " +
                            "CASE \"cnst\".\"contype\" WHEN 'p' THEN 'PRIMARY' WHEN 'u' THEN 'UNIQUE' WHEN 'c' THEN 'CHECK' WHEN 'x' THEN 'EXCLUDE' END AS \"constraint_type\", \"a\".\"attname\" AS \"column_name\" " +
                            "FROM \"pg_constraint\" \"cnst\" " +
                            "INNER JOIN \"pg_class\" \"t\" ON \"t\".\"oid\" = \"cnst\".\"conrelid\" " +
                            "INNER JOIN \"pg_namespace\" \"ns\" ON \"ns\".\"oid\" = \"cnst\".\"connamespace\" " +
                            "LEFT JOIN \"pg_attribute\" \"a\" ON \"a\".\"attrelid\" = \"cnst\".\"conrelid\" AND \"a\".\"attnum\" = ANY (\"cnst\".\"conkey\") " +
                            ("WHERE \"t\".\"relkind\" IN ('r', 'p') AND (" + constraintsCondition + ")");
                        indicesSql = "SELECT \"ns\".\"nspname\" AS \"table_schema\", \"t\".\"relname\" AS \"table_name\", \"i\".\"relname\" AS \"constraint_name\", \"a\".\"attname\" AS \"column_name\", " +
                            "CASE \"ix\".\"indisunique\" WHEN 't' THEN 'TRUE' ELSE'FALSE' END AS \"is_unique\", pg_get_expr(\"ix\".\"indpred\", \"ix\".\"indrelid\") AS \"condition\", " +
                            "\"types\".\"typname\" AS \"type_name\" " +
                            "FROM \"pg_class\" \"t\" " +
                            "INNER JOIN \"pg_index\" \"ix\" ON \"ix\".\"indrelid\" = \"t\".\"oid\" " +
                            "INNER JOIN \"pg_attribute\" \"a\" ON \"a\".\"attrelid\" = \"t\".\"oid\"  AND \"a\".\"attnum\" = ANY (\"ix\".\"indkey\") " +
                            "INNER JOIN \"pg_namespace\" \"ns\" ON \"ns\".\"oid\" = \"t\".\"relnamespace\" " +
                            "INNER JOIN \"pg_class\" \"i\" ON \"i\".\"oid\" = \"ix\".\"indexrelid\" " +
                            "INNER JOIN \"pg_type\" \"types\" ON \"types\".\"oid\" = \"a\".\"atttypid\" " +
                            "LEFT JOIN \"pg_constraint\" \"cnst\" ON \"cnst\".\"conname\" = \"i\".\"relname\" " +
                            ("WHERE \"t\".\"relkind\" IN ('r', 'p') AND \"cnst\".\"contype\" IS NULL AND (" + constraintsCondition + ")");
                        foreignKeysCondition = dbTables.map(function (_a) {
                            var table_schema = _a.table_schema, table_name = _a.table_name;
                            return "(\"ns\".\"nspname\" = '" + table_schema + "' AND \"cl\".\"relname\" = '" + table_name + "')";
                        }).join(" OR ");
                        return [4 /*yield*/, this.hasSupportForPartitionedTables()];
                    case 7:
                        hasRelispartitionColumn = _k.sent();
                        isPartitionCondition = hasRelispartitionColumn ? " AND \"cl\".\"relispartition\" = 'f'" : "";
                        foreignKeysSql = "SELECT \"con\".\"conname\" AS \"constraint_name\", \"con\".\"nspname\" AS \"table_schema\", \"con\".\"relname\" AS \"table_name\", \"att2\".\"attname\" AS \"column_name\", " +
                            "\"ns\".\"nspname\" AS \"referenced_table_schema\", \"cl\".\"relname\" AS \"referenced_table_name\", \"att\".\"attname\" AS \"referenced_column_name\", \"con\".\"confdeltype\" AS \"on_delete\", " +
                            "\"con\".\"confupdtype\" AS \"on_update\", \"con\".\"condeferrable\" AS \"deferrable\", \"con\".\"condeferred\" AS \"deferred\" " +
                            "FROM ( " +
                            "SELECT UNNEST (\"con1\".\"conkey\") AS \"parent\", UNNEST (\"con1\".\"confkey\") AS \"child\", \"con1\".\"confrelid\", \"con1\".\"conrelid\", \"con1\".\"conname\", \"con1\".\"contype\", \"ns\".\"nspname\", " +
                            "\"cl\".\"relname\", \"con1\".\"condeferrable\", " +
                            "CASE WHEN \"con1\".\"condeferred\" THEN 'INITIALLY DEFERRED' ELSE 'INITIALLY IMMEDIATE' END as condeferred, " +
                            "CASE \"con1\".\"confdeltype\" WHEN 'a' THEN 'NO ACTION' WHEN 'r' THEN 'RESTRICT' WHEN 'c' THEN 'CASCADE' WHEN 'n' THEN 'SET NULL' WHEN 'd' THEN 'SET DEFAULT' END as \"confdeltype\", " +
                            "CASE \"con1\".\"confupdtype\" WHEN 'a' THEN 'NO ACTION' WHEN 'r' THEN 'RESTRICT' WHEN 'c' THEN 'CASCADE' WHEN 'n' THEN 'SET NULL' WHEN 'd' THEN 'SET DEFAULT' END as \"confupdtype\" " +
                            "FROM \"pg_class\" \"cl\" " +
                            "INNER JOIN \"pg_namespace\" \"ns\" ON \"cl\".\"relnamespace\" = \"ns\".\"oid\" " +
                            "INNER JOIN \"pg_constraint\" \"con1\" ON \"con1\".\"conrelid\" = \"cl\".\"oid\" " +
                            ("WHERE \"con1\".\"contype\" = 'f' AND (" + foreignKeysCondition + ") ") +
                            ") \"con\" " +
                            "INNER JOIN \"pg_attribute\" \"att\" ON \"att\".\"attrelid\" = \"con\".\"confrelid\" AND \"att\".\"attnum\" = \"con\".\"child\" " +
                            ("INNER JOIN \"pg_class\" \"cl\" ON \"cl\".\"oid\" = \"con\".\"confrelid\" " + isPartitionCondition) +
                            "INNER JOIN \"pg_namespace\" \"ns\" ON \"cl\".\"relnamespace\" = \"ns\".\"oid\" " +
                            "INNER JOIN \"pg_attribute\" \"att2\" ON \"att2\".\"attrelid\" = \"con\".\"conrelid\" AND \"att2\".\"attnum\" = \"con\".\"parent\"";
                        return [4 /*yield*/, Promise.all([
                                this.query(columnsSql),
                                this.query(constraintsSql),
                                this.query(indicesSql),
                                this.query(foreignKeysSql),
                            ])];
                    case 8:
                        _j = tslib_1.__read.apply(void 0, [_k.sent(), 4]), dbColumns = _j[0], dbConstraints = _j[1], dbIndices = _j[2], dbForeignKeys = _j[3];
                        // create tables for loaded tables
                        return [2 /*return*/, Promise.all(dbTables.map(function (dbTable) { return (0, tslib_1.__awaiter)(_this, void 0, void 0, function () {
                                var table, getSchemaFromKey, schema, _a, tableUniqueConstraints, tableCheckConstraints, tableExclusionConstraints, tableForeignKeyConstraints, tableIndexConstraints;
                                var _this = this;
                                return (0, tslib_1.__generator)(this, function (_b) {
                                    switch (_b.label) {
                                        case 0:
                                            table = new Table_1.Table();
                                            getSchemaFromKey = function (dbObject, key) {
                                                return dbObject[key] === currentSchema && (!_this.driver.options.schema || _this.driver.options.schema === currentSchema)
                                                    ? undefined
                                                    : dbObject[key];
                                            };
                                            schema = getSchemaFromKey(dbTable, "table_schema");
                                            table.database = currentDatabase;
                                            table.schema = dbTable["table_schema"];
                                            table.name = this.driver.buildTableName(dbTable["table_name"], schema);
                                            // create columns from the loaded columns
                                            _a = table;
                                            return [4 /*yield*/, Promise.all(dbColumns
                                                    .filter(function (dbColumn) { return (dbColumn["table_name"] === dbTable["table_name"] && dbColumn["table_schema"] === dbTable["table_schema"]); })
                                                    .map(function (dbColumn) { return (0, tslib_1.__awaiter)(_this, void 0, void 0, function () {
                                                    var columnConstraints, tableColumn, name, builtEnumName, enumName, sql, results, type, geometryColumnSql, results, geographyColumnSql, results, length, match, uniqueConstraint, isConstraintComposite, serialDefaultName, serialDefaultPath, defaultWithoutQuotes;
                                                    return (0, tslib_1.__generator)(this, function (_a) {
                                                        switch (_a.label) {
                                                            case 0:
                                                                columnConstraints = dbConstraints.filter(function (dbConstraint) {
                                                                    return (dbConstraint["table_name"] === dbColumn["table_name"] &&
                                                                        dbConstraint["table_schema"] === dbColumn["table_schema"] &&
                                                                        dbConstraint["column_name"] === dbColumn["column_name"]);
                                                                });
                                                                tableColumn = new TableColumn_1.TableColumn();
                                                                tableColumn.name = dbColumn["column_name"];
                                                                tableColumn.type = dbColumn["regtype"].toLowerCase();
                                                                if (tableColumn.type === "numeric" || tableColumn.type === "decimal" || tableColumn.type === "float") {
                                                                    // If one of these properties was set, and another was not, Postgres sets '0' in to unspecified property
                                                                    // we set 'undefined' in to unspecified property to avoid changing column on sync
                                                                    if (dbColumn["numeric_precision"] !== null && !this.isDefaultColumnPrecision(table, tableColumn, dbColumn["numeric_precision"])) {
                                                                        tableColumn.precision = dbColumn["numeric_precision"];
                                                                    }
                                                                    else if (dbColumn["numeric_scale"] !== null && !this.isDefaultColumnScale(table, tableColumn, dbColumn["numeric_scale"])) {
                                                                        tableColumn.precision = undefined;
                                                                    }
                                                                    if (dbColumn["numeric_scale"] !== null && !this.isDefaultColumnScale(table, tableColumn, dbColumn["numeric_scale"])) {
                                                                        tableColumn.scale = dbColumn["numeric_scale"];
                                                                    }
                                                                    else if (dbColumn["numeric_precision"] !== null && !this.isDefaultColumnPrecision(table, tableColumn, dbColumn["numeric_precision"])) {
                                                                        tableColumn.scale = undefined;
                                                                    }
                                                                }
                                                                if (tableColumn.type === "interval"
                                                                    || tableColumn.type === "time without time zone"
                                                                    || tableColumn.type === "time with time zone"
                                                                    || tableColumn.type === "timestamp without time zone"
                                                                    || tableColumn.type === "timestamp with time zone") {
                                                                    tableColumn.precision = !this.isDefaultColumnPrecision(table, tableColumn, dbColumn["datetime_precision"]) ? dbColumn["datetime_precision"] : undefined;
                                                                }
                                                                if (!(dbColumn["data_type"] === "USER-DEFINED" || dbColumn["data_type"] === "ARRAY")) return [3 /*break*/, 3];
                                                                return [4 /*yield*/, this.getUserDefinedTypeName(table, tableColumn)
                                                                    // check if `enumName` is specified by user
                                                                ];
                                                            case 1:
                                                                name = (_a.sent()).name;
                                                                builtEnumName = this.buildEnumName(table, tableColumn, false, true);
                                                                enumName = builtEnumName !== name ? name : undefined;
                                                                sql = "SELECT \"e\".\"enumlabel\" AS \"value\" FROM \"pg_enum\" \"e\" " +
                                                                    "INNER JOIN \"pg_type\" \"t\" ON \"t\".\"oid\" = \"e\".\"enumtypid\" " +
                                                                    "INNER JOIN \"pg_namespace\" \"n\" ON \"n\".\"oid\" = \"t\".\"typnamespace\" " +
                                                                    ("WHERE \"n\".\"nspname\" = '" + dbTable["table_schema"] + "' AND \"t\".\"typname\" = '" + (enumName || name) + "'");
                                                                return [4 /*yield*/, this.query(sql)];
                                                            case 2:
                                                                results = _a.sent();
                                                                if (results.length) {
                                                                    tableColumn.type = "enum";
                                                                    tableColumn.enum = results.map(function (result) { return result["value"]; });
                                                                    tableColumn.enumName = enumName;
                                                                }
                                                                if (dbColumn["data_type"] === "ARRAY") {
                                                                    tableColumn.isArray = true;
                                                                    type = tableColumn.type.replace("[]", "");
                                                                    tableColumn.type = this.connection.driver.normalizeType({ type: type });
                                                                }
                                                                _a.label = 3;
                                                            case 3:
                                                                if (!(tableColumn.type === "geometry")) return [3 /*break*/, 5];
                                                                geometryColumnSql = "SELECT * FROM (\n                        SELECT\n                          \"f_table_schema\" \"table_schema\",\n                          \"f_table_name\" \"table_name\",\n                          \"f_geometry_column\" \"column_name\",\n                          \"srid\",\n                          \"type\"\n                        FROM \"geometry_columns\"\n                      ) AS _\n                      WHERE\n                          \"column_name\" = '" + dbColumn["column_name"] + "' AND\n                          \"table_schema\" = '" + dbColumn["table_schema"] + "' AND\n                          \"table_name\" = '" + dbColumn["table_name"] + "'";
                                                                return [4 /*yield*/, this.query(geometryColumnSql)];
                                                            case 4:
                                                                results = _a.sent();
                                                                if (results.length > 0) {
                                                                    tableColumn.spatialFeatureType = results[0].type;
                                                                    tableColumn.srid = results[0].srid;
                                                                }
                                                                _a.label = 5;
                                                            case 5:
                                                                if (!(tableColumn.type === "geography")) return [3 /*break*/, 7];
                                                                geographyColumnSql = "SELECT * FROM (\n                        SELECT\n                          \"f_table_schema\" \"table_schema\",\n                          \"f_table_name\" \"table_name\",\n                          \"f_geography_column\" \"column_name\",\n                          \"srid\",\n                          \"type\"\n                        FROM \"geography_columns\"\n                      ) AS _\n                      WHERE\n                          \"column_name\" = '" + dbColumn["column_name"] + "' AND\n                          \"table_schema\" = '" + dbColumn["table_schema"] + "' AND\n                          \"table_name\" = '" + dbColumn["table_name"] + "'";
                                                                return [4 /*yield*/, this.query(geographyColumnSql)];
                                                            case 6:
                                                                results = _a.sent();
                                                                if (results.length > 0) {
                                                                    tableColumn.spatialFeatureType = results[0].type;
                                                                    tableColumn.srid = results[0].srid;
                                                                }
                                                                _a.label = 7;
                                                            case 7:
                                                                // check only columns that have length property
                                                                if (this.driver.withLengthColumnTypes.indexOf(tableColumn.type) !== -1) {
                                                                    length = void 0;
                                                                    if (tableColumn.isArray) {
                                                                        match = /\((\d+)\)/.exec(dbColumn["format_type"]);
                                                                        length = match ? match[1] : undefined;
                                                                    }
                                                                    else if (dbColumn["character_maximum_length"]) {
                                                                        length = dbColumn["character_maximum_length"].toString();
                                                                    }
                                                                    if (length) {
                                                                        tableColumn.length = !this.isDefaultColumnLength(table, tableColumn, length) ? length : "";
                                                                    }
                                                                }
                                                                tableColumn.isNullable = dbColumn["is_nullable"] === "YES";
                                                                tableColumn.isPrimary = !!columnConstraints.find(function (constraint) { return constraint["constraint_type"] === "PRIMARY"; });
                                                                uniqueConstraint = columnConstraints.find(function (constraint) { return constraint["constraint_type"] === "UNIQUE"; });
                                                                isConstraintComposite = uniqueConstraint
                                                                    ? !!dbConstraints.find(function (dbConstraint) { return dbConstraint["constraint_type"] === "UNIQUE"
                                                                        && dbConstraint["constraint_name"] === uniqueConstraint["constraint_name"]
                                                                        && dbConstraint["column_name"] !== dbColumn["column_name"]; })
                                                                    : false;
                                                                tableColumn.isUnique = !!uniqueConstraint && !isConstraintComposite;
                                                                if (dbColumn["column_default"] !== null && dbColumn["column_default"] !== undefined) {
                                                                    serialDefaultName = "nextval('" + this.buildSequenceName(table, dbColumn["column_name"]) + "'::regclass)";
                                                                    serialDefaultPath = "nextval('" + this.buildSequencePath(table, dbColumn["column_name"]) + "'::regclass)";
                                                                    defaultWithoutQuotes = dbColumn["column_default"].replace(/"/g, "");
                                                                    if (defaultWithoutQuotes === serialDefaultName || defaultWithoutQuotes === serialDefaultPath) {
                                                                        tableColumn.isGenerated = true;
                                                                        tableColumn.generationStrategy = "increment";
                                                                    }
                                                                    else if (dbColumn["column_default"] === "gen_random_uuid()" || /^uuid_generate_v\d\(\)/.test(dbColumn["column_default"])) {
                                                                        tableColumn.isGenerated = true;
                                                                        tableColumn.generationStrategy = "uuid";
                                                                    }
                                                                    else if (dbColumn["column_default"] === "now()" || dbColumn["column_default"].indexOf("'now'::text") !== -1) {
                                                                        tableColumn.default = dbColumn["column_default"];
                                                                    }
                                                                    else {
                                                                        tableColumn.default = dbColumn["column_default"].replace(/::[\w\s\[\]\"]+/g, "");
                                                                        tableColumn.default = tableColumn.default.replace(/^(-?\d+)$/, "'$1'");
                                                                    }
                                                                }
                                                                tableColumn.comment = dbColumn["description"] ? dbColumn["description"] : undefined;
                                                                if (dbColumn["character_set_name"])
                                                                    tableColumn.charset = dbColumn["character_set_name"];
                                                                if (dbColumn["collation_name"])
                                                                    tableColumn.collation = dbColumn["collation_name"];
                                                                return [2 /*return*/, tableColumn];
                                                        }
                                                    });
                                                }); }))];
                                        case 1:
                                            // create columns from the loaded columns
                                            _a.columns = _b.sent();
                                            tableUniqueConstraints = OrmUtils_1.OrmUtils.uniq(dbConstraints.filter(function (dbConstraint) {
                                                return (dbConstraint["table_name"] === dbTable["table_name"] &&
                                                    dbConstraint["table_schema"] === dbTable["table_schema"] &&
                                                    dbConstraint["constraint_type"] === "UNIQUE");
                                            }), function (dbConstraint) { return dbConstraint["constraint_name"]; });
                                            table.uniques = tableUniqueConstraints.map(function (constraint) {
                                                var uniques = dbConstraints.filter(function (dbC) { return dbC["constraint_name"] === constraint["constraint_name"]; });
                                                return new TableUnique_1.TableUnique({
                                                    name: constraint["constraint_name"],
                                                    columnNames: uniques.map(function (u) { return u["column_name"]; })
                                                });
                                            });
                                            tableCheckConstraints = OrmUtils_1.OrmUtils.uniq(dbConstraints.filter(function (dbConstraint) {
                                                return (dbConstraint["table_name"] === dbTable["table_name"] &&
                                                    dbConstraint["table_schema"] === dbTable["table_schema"] &&
                                                    dbConstraint["constraint_type"] === "CHECK");
                                            }), function (dbConstraint) { return dbConstraint["constraint_name"]; });
                                            table.checks = tableCheckConstraints.map(function (constraint) {
                                                var checks = dbConstraints.filter(function (dbC) { return dbC["constraint_name"] === constraint["constraint_name"]; });
                                                return new TableCheck_1.TableCheck({
                                                    name: constraint["constraint_name"],
                                                    columnNames: checks.map(function (c) { return c["column_name"]; }),
                                                    expression: constraint["expression"].replace(/^\s*CHECK\s*\((.*)\)\s*$/i, "$1")
                                                });
                                            });
                                            tableExclusionConstraints = OrmUtils_1.OrmUtils.uniq(dbConstraints.filter(function (dbConstraint) {
                                                return (dbConstraint["table_name"] === dbTable["table_name"] &&
                                                    dbConstraint["table_schema"] === dbTable["table_schema"] &&
                                                    dbConstraint["constraint_type"] === "EXCLUDE");
                                            }), function (dbConstraint) { return dbConstraint["constraint_name"]; });
                                            table.exclusions = tableExclusionConstraints.map(function (constraint) {
                                                return new TableExclusion_1.TableExclusion({
                                                    name: constraint["constraint_name"],
                                                    expression: constraint["expression"].substring(8) // trim EXCLUDE from start of expression
                                                });
                                            });
                                            tableForeignKeyConstraints = OrmUtils_1.OrmUtils.uniq(dbForeignKeys.filter(function (dbForeignKey) {
                                                return (dbForeignKey["table_name"] === dbTable["table_name"] &&
                                                    dbForeignKey["table_schema"] === dbTable["table_schema"]);
                                            }), function (dbForeignKey) { return dbForeignKey["constraint_name"]; });
                                            table.foreignKeys = tableForeignKeyConstraints.map(function (dbForeignKey) {
                                                var foreignKeys = dbForeignKeys.filter(function (dbFk) { return dbFk["constraint_name"] === dbForeignKey["constraint_name"]; });
                                                // if referenced table located in currently used schema, we don't need to concat schema name to table name.
                                                var schema = getSchemaFromKey(dbForeignKey, "referenced_table_schema");
                                                var referencedTableName = _this.driver.buildTableName(dbForeignKey["referenced_table_name"], schema);
                                                return new TableForeignKey_1.TableForeignKey({
                                                    name: dbForeignKey["constraint_name"],
                                                    columnNames: foreignKeys.map(function (dbFk) { return dbFk["column_name"]; }),
                                                    referencedTableName: referencedTableName,
                                                    referencedColumnNames: foreignKeys.map(function (dbFk) { return dbFk["referenced_column_name"]; }),
                                                    onDelete: dbForeignKey["on_delete"],
                                                    onUpdate: dbForeignKey["on_update"],
                                                    deferrable: dbForeignKey["deferrable"] ? dbForeignKey["deferred"] : undefined,
                                                });
                                            });
                                            tableIndexConstraints = OrmUtils_1.OrmUtils.uniq(dbIndices.filter(function (dbIndex) {
                                                return (dbIndex["table_name"] === dbTable["table_name"] &&
                                                    dbIndex["table_schema"] === dbTable["table_schema"]);
                                            }), function (dbIndex) { return dbIndex["constraint_name"]; });
                                            table.indices = tableIndexConstraints.map(function (constraint) {
                                                var indices = dbIndices.filter(function (index) {
                                                    return index["table_schema"] === constraint["table_schema"]
                                                        && index["table_name"] === constraint["table_name"]
                                                        && index["constraint_name"] === constraint["constraint_name"];
                                                });
                                                return new TableIndex_1.TableIndex({
                                                    table: table,
                                                    name: constraint["constraint_name"],
                                                    columnNames: indices.map(function (i) { return i["column_name"]; }),
                                                    isUnique: constraint["is_unique"] === "TRUE",
                                                    where: constraint["condition"],
                                                    isSpatial: indices.every(function (i) { return _this.driver.spatialTypes.indexOf(i["type_name"]) >= 0; }),
                                                    isFulltext: false
                                                });
                                            });
                                            return [2 /*return*/, table];
                                    }
                                });
                            }); }))];
                }
            });
        });
    };
    /**
     * Builds create table sql.
     */
    PostgresQueryRunner.prototype.createTableSql = function (table, createForeignKeys) {
        var _this = this;
        var columnDefinitions = table.columns.map(function (column) { return _this.buildCreateColumnSql(table, column); }).join(", ");
        var sql = "CREATE TABLE " + this.escapePath(table) + " (" + columnDefinitions;
        table.columns
            .filter(function (column) { return column.isUnique; })
            .forEach(function (column) {
            var isUniqueExist = table.uniques.some(function (unique) { return unique.columnNames.length === 1 && unique.columnNames[0] === column.name; });
            if (!isUniqueExist)
                table.uniques.push(new TableUnique_1.TableUnique({
                    name: _this.connection.namingStrategy.uniqueConstraintName(table, [column.name]),
                    columnNames: [column.name]
                }));
        });
        if (table.uniques.length > 0) {
            var uniquesSql = table.uniques.map(function (unique) {
                var uniqueName = unique.name ? unique.name : _this.connection.namingStrategy.uniqueConstraintName(table, unique.columnNames);
                var columnNames = unique.columnNames.map(function (columnName) { return "\"" + columnName + "\""; }).join(", ");
                return "CONSTRAINT \"" + uniqueName + "\" UNIQUE (" + columnNames + ")";
            }).join(", ");
            sql += ", " + uniquesSql;
        }
        if (table.checks.length > 0) {
            var checksSql = table.checks.map(function (check) {
                var checkName = check.name ? check.name : _this.connection.namingStrategy.checkConstraintName(table, check.expression);
                return "CONSTRAINT \"" + checkName + "\" CHECK (" + check.expression + ")";
            }).join(", ");
            sql += ", " + checksSql;
        }
        if (table.exclusions.length > 0) {
            var exclusionsSql = table.exclusions.map(function (exclusion) {
                var exclusionName = exclusion.name ? exclusion.name : _this.connection.namingStrategy.exclusionConstraintName(table, exclusion.expression);
                return "CONSTRAINT \"" + exclusionName + "\" EXCLUDE " + exclusion.expression;
            }).join(", ");
            sql += ", " + exclusionsSql;
        }
        if (table.foreignKeys.length > 0 && createForeignKeys) {
            var foreignKeysSql = table.foreignKeys.map(function (fk) {
                var columnNames = fk.columnNames.map(function (columnName) { return "\"" + columnName + "\""; }).join(", ");
                if (!fk.name)
                    fk.name = _this.connection.namingStrategy.foreignKeyName(table, fk.columnNames, _this.getTablePath(fk), fk.referencedColumnNames);
                var referencedColumnNames = fk.referencedColumnNames.map(function (columnName) { return "\"" + columnName + "\""; }).join(", ");
                var constraint = "CONSTRAINT \"" + fk.name + "\" FOREIGN KEY (" + columnNames + ") REFERENCES " + _this.escapePath(_this.getTablePath(fk)) + " (" + referencedColumnNames + ")";
                if (fk.onDelete)
                    constraint += " ON DELETE " + fk.onDelete;
                if (fk.onUpdate)
                    constraint += " ON UPDATE " + fk.onUpdate;
                if (fk.deferrable)
                    constraint += " DEFERRABLE " + fk.deferrable;
                return constraint;
            }).join(", ");
            sql += ", " + foreignKeysSql;
        }
        var primaryColumns = table.columns.filter(function (column) { return column.isPrimary; });
        if (primaryColumns.length > 0) {
            var primaryKeyName = this.connection.namingStrategy.primaryKeyName(table, primaryColumns.map(function (column) { return column.name; }));
            var columnNames = primaryColumns.map(function (column) { return "\"" + column.name + "\""; }).join(", ");
            sql += ", CONSTRAINT \"" + primaryKeyName + "\" PRIMARY KEY (" + columnNames + ")";
        }
        sql += ")";
        table.columns
            .filter(function (it) { return it.comment; })
            .forEach(function (it) { return sql += "; COMMENT ON COLUMN " + _this.escapePath(table) + ".\"" + it.name + "\" IS " + _this.escapeComment(it.comment); });
        return new Query_1.Query(sql);
    };
    /**
     * Loads Postgres version.
     */
    PostgresQueryRunner.prototype.getVersion = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var result;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.query("SHOW SERVER_VERSION")];
                    case 1:
                        result = _a.sent();
                        return [2 /*return*/, result[0]["server_version"]];
                }
            });
        });
    };
    /**
     * Builds drop table sql.
     */
    PostgresQueryRunner.prototype.dropTableSql = function (tableOrPath) {
        return new Query_1.Query("DROP TABLE " + this.escapePath(tableOrPath));
    };
    PostgresQueryRunner.prototype.createViewSql = function (view) {
        var materializedClause = view.materialized ? "MATERIALIZED " : "";
        var viewName = this.escapePath(view);
        if (typeof view.expression === "string") {
            return new Query_1.Query("CREATE " + materializedClause + "VIEW " + viewName + " AS " + view.expression);
        }
        else {
            return new Query_1.Query("CREATE " + materializedClause + "VIEW " + viewName + " AS " + view.expression(this.connection).getQuery());
        }
    };
    PostgresQueryRunner.prototype.insertViewDefinitionSql = function (view) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var currentSchema, _a, schema, name, type, expression, _b, query, parameters;
            return (0, tslib_1.__generator)(this, function (_c) {
                switch (_c.label) {
                    case 0: return [4 /*yield*/, this.getCurrentSchema()];
                    case 1:
                        currentSchema = _c.sent();
                        _a = this.driver.parseTableName(view), schema = _a.schema, name = _a.tableName;
                        if (!schema) {
                            schema = currentSchema;
                        }
                        type = view.materialized ? "MATERIALIZED_VIEW" : "VIEW";
                        expression = typeof view.expression === "string" ? view.expression.trim() : view.expression(this.connection).getQuery();
                        _b = (0, tslib_1.__read)(this.connection.createQueryBuilder()
                            .insert()
                            .into(this.getTypeormMetadataTableName())
                            .values({ type: type, schema: schema, name: name, value: expression })
                            .getQueryAndParameters(), 2), query = _b[0], parameters = _b[1];
                        return [2 /*return*/, new Query_1.Query(query, parameters)];
                }
            });
        });
    };
    /**
     * Builds drop view sql.
     */
    PostgresQueryRunner.prototype.dropViewSql = function (view) {
        var materializedClause = view.materialized ? "MATERIALIZED " : "";
        return new Query_1.Query("DROP " + materializedClause + "VIEW " + this.escapePath(view));
    };
    /**
     * Builds remove view sql.
     */
    PostgresQueryRunner.prototype.deleteViewDefinitionSql = function (view) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var currentSchema, _a, schema, name, type, qb, _b, query, parameters;
            return (0, tslib_1.__generator)(this, function (_c) {
                switch (_c.label) {
                    case 0: return [4 /*yield*/, this.getCurrentSchema()];
                    case 1:
                        currentSchema = _c.sent();
                        _a = this.driver.parseTableName(view), schema = _a.schema, name = _a.tableName;
                        if (!schema) {
                            schema = currentSchema;
                        }
                        type = view.materialized ? "MATERIALIZED_VIEW" : "VIEW";
                        qb = this.connection.createQueryBuilder();
                        _b = (0, tslib_1.__read)(qb.delete()
                            .from(this.getTypeormMetadataTableName())
                            .where(qb.escape("type") + " = :type", { type: type })
                            .andWhere(qb.escape("schema") + " = :schema", { schema: schema })
                            .andWhere(qb.escape("name") + " = :name", { name: name })
                            .getQueryAndParameters(), 2), query = _b[0], parameters = _b[1];
                        return [2 /*return*/, new Query_1.Query(query, parameters)];
                }
            });
        });
    };
    /**
     * Drops ENUM type from given schemas.
     */
    PostgresQueryRunner.prototype.dropEnumTypes = function (schemaNames) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var selectDropsQuery, dropQueries;
            var _this = this;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        selectDropsQuery = "SELECT 'DROP TYPE IF EXISTS \"' || n.nspname || '\".\"' || t.typname || '\" CASCADE;' as \"query\" FROM \"pg_type\" \"t\" " +
                            "INNER JOIN \"pg_enum\" \"e\" ON \"e\".\"enumtypid\" = \"t\".\"oid\" " +
                            "INNER JOIN \"pg_namespace\" \"n\" ON \"n\".\"oid\" = \"t\".\"typnamespace\" " +
                            ("WHERE \"n\".\"nspname\" IN (" + schemaNames + ") GROUP BY \"n\".\"nspname\", \"t\".\"typname\"");
                        return [4 /*yield*/, this.query(selectDropsQuery)];
                    case 1:
                        dropQueries = _a.sent();
                        return [4 /*yield*/, Promise.all(dropQueries.map(function (q) { return _this.query(q["query"]); }))];
                    case 2:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Checks if enum with the given name exist in the database.
     */
    PostgresQueryRunner.prototype.hasEnumType = function (table, column) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var schema, enumName, sql, result;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        schema = this.driver.parseTableName(table).schema;
                        if (!!schema) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.getCurrentSchema()];
                    case 1:
                        schema = _a.sent();
                        _a.label = 2;
                    case 2:
                        enumName = this.buildEnumName(table, column, false, true);
                        sql = "SELECT \"n\".\"nspname\", \"t\".\"typname\" FROM \"pg_type\" \"t\" " +
                            "INNER JOIN \"pg_namespace\" \"n\" ON \"n\".\"oid\" = \"t\".\"typnamespace\" " +
                            ("WHERE \"n\".\"nspname\" = '" + schema + "' AND \"t\".\"typname\" = '" + enumName + "'");
                        return [4 /*yield*/, this.query(sql)];
                    case 3:
                        result = _a.sent();
                        return [2 /*return*/, result.length ? true : false];
                }
            });
        });
    };
    /**
     * Builds create ENUM type sql.
     */
    PostgresQueryRunner.prototype.createEnumTypeSql = function (table, column, enumName) {
        if (!enumName)
            enumName = this.buildEnumName(table, column);
        var enumValues = column.enum.map(function (value) { return "'" + value.replace("'", "''") + "'"; }).join(", ");
        return new Query_1.Query("CREATE TYPE " + enumName + " AS ENUM(" + enumValues + ")");
    };
    /**
     * Builds create ENUM type sql.
     */
    PostgresQueryRunner.prototype.dropEnumTypeSql = function (table, column, enumName) {
        if (!enumName)
            enumName = this.buildEnumName(table, column);
        return new Query_1.Query("DROP TYPE " + enumName);
    };
    /**
     * Builds create index sql.
     */
    PostgresQueryRunner.prototype.createIndexSql = function (table, index) {
        var columns = index.columnNames.map(function (columnName) { return "\"" + columnName + "\""; }).join(", ");
        return new Query_1.Query("CREATE " + (index.isUnique ? "UNIQUE " : "") + "INDEX \"" + index.name + "\" ON " + this.escapePath(table) + " " + (index.isSpatial ? "USING GiST " : "") + "(" + columns + ") " + (index.where ? "WHERE " + index.where : ""));
    };
    /**
     * Builds drop index sql.
     */
    PostgresQueryRunner.prototype.dropIndexSql = function (table, indexOrName) {
        var indexName = indexOrName instanceof TableIndex_1.TableIndex ? indexOrName.name : indexOrName;
        var schema = this.driver.parseTableName(table).schema;
        return schema ? new Query_1.Query("DROP INDEX \"" + schema + "\".\"" + indexName + "\"") : new Query_1.Query("DROP INDEX \"" + indexName + "\"");
    };
    /**
     * Builds create primary key sql.
     */
    PostgresQueryRunner.prototype.createPrimaryKeySql = function (table, columnNames) {
        var primaryKeyName = this.connection.namingStrategy.primaryKeyName(table, columnNames);
        var columnNamesString = columnNames.map(function (columnName) { return "\"" + columnName + "\""; }).join(", ");
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + primaryKeyName + "\" PRIMARY KEY (" + columnNamesString + ")");
    };
    /**
     * Builds drop primary key sql.
     */
    PostgresQueryRunner.prototype.dropPrimaryKeySql = function (table) {
        var columnNames = table.primaryColumns.map(function (column) { return column.name; });
        var primaryKeyName = this.connection.namingStrategy.primaryKeyName(table, columnNames);
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + primaryKeyName + "\"");
    };
    /**
     * Builds create unique constraint sql.
     */
    PostgresQueryRunner.prototype.createUniqueConstraintSql = function (table, uniqueConstraint) {
        var columnNames = uniqueConstraint.columnNames.map(function (column) { return "\"" + column + "\""; }).join(", ");
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + uniqueConstraint.name + "\" UNIQUE (" + columnNames + ")");
    };
    /**
     * Builds drop unique constraint sql.
     */
    PostgresQueryRunner.prototype.dropUniqueConstraintSql = function (table, uniqueOrName) {
        var uniqueName = uniqueOrName instanceof TableUnique_1.TableUnique ? uniqueOrName.name : uniqueOrName;
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + uniqueName + "\"");
    };
    /**
     * Builds create check constraint sql.
     */
    PostgresQueryRunner.prototype.createCheckConstraintSql = function (table, checkConstraint) {
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + checkConstraint.name + "\" CHECK (" + checkConstraint.expression + ")");
    };
    /**
     * Builds drop check constraint sql.
     */
    PostgresQueryRunner.prototype.dropCheckConstraintSql = function (table, checkOrName) {
        var checkName = checkOrName instanceof TableCheck_1.TableCheck ? checkOrName.name : checkOrName;
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + checkName + "\"");
    };
    /**
     * Builds create exclusion constraint sql.
     */
    PostgresQueryRunner.prototype.createExclusionConstraintSql = function (table, exclusionConstraint) {
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + exclusionConstraint.name + "\" EXCLUDE " + exclusionConstraint.expression);
    };
    /**
     * Builds drop exclusion constraint sql.
     */
    PostgresQueryRunner.prototype.dropExclusionConstraintSql = function (table, exclusionOrName) {
        var exclusionName = exclusionOrName instanceof TableExclusion_1.TableExclusion ? exclusionOrName.name : exclusionOrName;
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + exclusionName + "\"");
    };
    /**
     * Builds create foreign key sql.
     */
    PostgresQueryRunner.prototype.createForeignKeySql = function (table, foreignKey) {
        var columnNames = foreignKey.columnNames.map(function (column) { return "\"" + column + "\""; }).join(", ");
        var referencedColumnNames = foreignKey.referencedColumnNames.map(function (column) { return "\"" + column + "\""; }).join(",");
        var sql = "ALTER TABLE " + this.escapePath(table) + " ADD CONSTRAINT \"" + foreignKey.name + "\" FOREIGN KEY (" + columnNames + ") " +
            ("REFERENCES " + this.escapePath(this.getTablePath(foreignKey)) + "(" + referencedColumnNames + ")");
        if (foreignKey.onDelete)
            sql += " ON DELETE " + foreignKey.onDelete;
        if (foreignKey.onUpdate)
            sql += " ON UPDATE " + foreignKey.onUpdate;
        if (foreignKey.deferrable)
            sql += " DEFERRABLE " + foreignKey.deferrable;
        return new Query_1.Query(sql);
    };
    /**
     * Builds drop foreign key sql.
     */
    PostgresQueryRunner.prototype.dropForeignKeySql = function (table, foreignKeyOrName) {
        var foreignKeyName = foreignKeyOrName instanceof TableForeignKey_1.TableForeignKey ? foreignKeyOrName.name : foreignKeyOrName;
        return new Query_1.Query("ALTER TABLE " + this.escapePath(table) + " DROP CONSTRAINT \"" + foreignKeyName + "\"");
    };
    /**
     * Builds sequence name from given table and column.
     */
    PostgresQueryRunner.prototype.buildSequenceName = function (table, columnOrName) {
        var tableName = this.driver.parseTableName(table).tableName;
        var columnName = columnOrName instanceof TableColumn_1.TableColumn ? columnOrName.name : columnOrName;
        var seqName = tableName + "_" + columnName + "_seq";
        if (seqName.length > this.connection.driver.maxAliasLength) {
            // note doesn't yet handle corner cases where .length differs from number of UTF-8 bytes
            seqName = tableName.substring(0, 29) + "_" + columnName.substring(0, Math.max(29, 63 - (table.name.length) - 5)) + "_seq";
        }
        return seqName;
    };
    PostgresQueryRunner.prototype.buildSequencePath = function (table, columnOrName) {
        var schema = this.driver.parseTableName(table).schema;
        return schema ? schema + "." + this.buildSequenceName(table, columnOrName) : this.buildSequenceName(table, columnOrName);
    };
    /**
     * Builds ENUM type name from given table and column.
     */
    PostgresQueryRunner.prototype.buildEnumName = function (table, column, withSchema, disableEscape, toOld) {
        if (withSchema === void 0) { withSchema = true; }
        var _a = this.driver.parseTableName(table), schema = _a.schema, tableName = _a.tableName;
        var enumName = column.enumName ? column.enumName : tableName + "_" + column.name.toLowerCase() + "_enum";
        if (schema && withSchema)
            enumName = schema + "." + enumName;
        if (toOld)
            enumName = enumName + "_old";
        return enumName.split(".").map(function (i) {
            return disableEscape ? i : "\"" + i + "\"";
        }).join(".");
    };
    PostgresQueryRunner.prototype.getUserDefinedTypeName = function (table, column) {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var _a, schema, name, result, udtName;
            return (0, tslib_1.__generator)(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        _a = this.driver.parseTableName(table), schema = _a.schema, name = _a.tableName;
                        if (!!schema) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.getCurrentSchema()];
                    case 1:
                        schema = _b.sent();
                        _b.label = 2;
                    case 2: return [4 /*yield*/, this.query("SELECT \"udt_schema\", \"udt_name\" " +
                            ("FROM \"information_schema\".\"columns\" WHERE \"table_schema\" = '" + schema + "' AND \"table_name\" = '" + name + "' AND \"column_name\"='" + column.name + "'"))];
                    case 3:
                        result = _b.sent();
                        udtName = result[0]["udt_name"];
                        if (udtName.indexOf("_") === 0) {
                            udtName = udtName.substr(1, udtName.length);
                        }
                        return [2 /*return*/, {
                                schema: result[0]["udt_schema"],
                                name: udtName
                            }];
                }
            });
        });
    };
    /**
     * Escapes a given comment so it's safe to include in a query.
     */
    PostgresQueryRunner.prototype.escapeComment = function (comment) {
        if (!comment || comment.length === 0) {
            return "NULL";
        }
        comment = comment
            .replace(/'/g, "''")
            .replace(/\u0000/g, ""); // Null bytes aren't allowed in comments
        return "'" + comment + "'";
    };
    /**
     * Escapes given table or view path.
     */
    PostgresQueryRunner.prototype.escapePath = function (target) {
        var _a = this.driver.parseTableName(target), schema = _a.schema, tableName = _a.tableName;
        if (schema && schema !== this.driver.searchSchema) {
            return "\"" + schema + "\".\"" + tableName + "\"";
        }
        return "\"" + tableName + "\"";
    };
    /**
     * Builds a query for create column.
     */
    PostgresQueryRunner.prototype.buildCreateColumnSql = function (table, column) {
        var c = "\"" + column.name + "\"";
        if (column.isGenerated === true && column.generationStrategy !== "uuid") {
            if (column.type === "integer" || column.type === "int" || column.type === "int4")
                c += " SERIAL";
            if (column.type === "smallint" || column.type === "int2")
                c += " SMALLSERIAL";
            if (column.type === "bigint" || column.type === "int8")
                c += " BIGSERIAL";
        }
        if (column.type === "enum" || column.type === "simple-enum") {
            c += " " + this.buildEnumName(table, column);
            if (column.isArray)
                c += " array";
        }
        else if (!column.isGenerated || column.type === "uuid") {
            c += " " + this.connection.driver.createFullType(column);
        }
        if (column.charset)
            c += " CHARACTER SET \"" + column.charset + "\"";
        if (column.collation)
            c += " COLLATE \"" + column.collation + "\"";
        if (column.isNullable !== true)
            c += " NOT NULL";
        if (column.default !== undefined && column.default !== null)
            c += " DEFAULT " + column.default;
        if (column.isGenerated && column.generationStrategy === "uuid" && !column.default)
            c += " DEFAULT " + this.driver.uuidGenerator;
        return c;
    };
    /**
     * Checks if the PostgreSQL server has support for partitioned tables
     */
    PostgresQueryRunner.prototype.hasSupportForPartitionedTables = function () {
        return (0, tslib_1.__awaiter)(this, void 0, void 0, function () {
            var result;
            return (0, tslib_1.__generator)(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.query("SELECT TRUE FROM information_schema.columns WHERE table_name = 'pg_class' and column_name = 'relispartition'")];
                    case 1:
                        result = _a.sent();
                        return [2 /*return*/, result.length ? true : false];
                }
            });
        });
    };
    return PostgresQueryRunner;
}(BaseQueryRunner_1.BaseQueryRunner));
exports.PostgresQueryRunner = PostgresQueryRunner;

//# sourceMappingURL=PostgresQueryRunner.js.map
