import { IDiagnosticLogger, IPlugin, IDistributedTraceContext } from "@microsoft/applicationinsights-core-js";
import { ICorrelationConfig } from "./Interfaces/ICorrelationConfig";
import { ITelemetryTrace } from "./Interfaces/Context/ITelemetryTrace";
export declare function isInternalApplicationInsightsEndpoint(endpointUrl: string): boolean;
export interface IUtil {
    NotSpecified: string;
    createDomEvent: (eventName: string) => Event;
    disableStorage: () => void;
    /**
     *  Checks if endpoint URL is application insights internal injestion service URL.
     *
     *  @param endpointUrl Endpoint URL to check.
     *  @returns {boolean} True if if endpoint URL is application insights internal injestion service URL.
     */
    isInternalApplicationInsightsEndpoint: (endpointUrl: string) => boolean;
    /**
     *  Check if the browser supports local storage.
     *
     *  @returns {boolean} True if local storage is supported.
     */
    canUseLocalStorage: () => boolean;
    /**
     *  Get an object from the browser's local storage
     *
     *  @param {string} name - the name of the object to get from storage
     *  @returns {string} The contents of the storage object with the given name. Null if storage is not supported.
     */
    getStorage: (logger: IDiagnosticLogger, name: string) => string;
    /**
     *  Set the contents of an object in the browser's local storage
     *
     *  @param {string} name - the name of the object to set in storage
     *  @param {string} data - the contents of the object to set in storage
     *  @returns {boolean} True if the storage object could be written.
     */
    setStorage: (logger: IDiagnosticLogger, name: string, data: string) => boolean;
    /**
     *  Remove an object from the browser's local storage
     *
     *  @param {string} name - the name of the object to remove from storage
     *  @returns {boolean} True if the storage object could be removed.
     */
    removeStorage: (logger: IDiagnosticLogger, name: string) => boolean;
    /**
     *  Check if the browser supports session storage.
     *
     *  @returns {boolean} True if session storage is supported.
     */
    canUseSessionStorage: () => boolean;
    /**
     *  Gets the list of session storage keys
     *
     *  @returns {string[]} List of session storage keys
     */
    getSessionStorageKeys: () => string[];
    /**
     *  Get an object from the browser's session storage
     *
     *  @param {string} name - the name of the object to get from storage
     *  @returns {string} The contents of the storage object with the given name. Null if storage is not supported.
     */
    getSessionStorage: (logger: IDiagnosticLogger, name: string) => string;
    /**
     *  Set the contents of an object in the browser's session storage
     *
     *  @param {string} name - the name of the object to set in storage
     *  @param {string} data - the contents of the object to set in storage
     *  @returns {boolean} True if the storage object could be written.
     */
    setSessionStorage: (logger: IDiagnosticLogger, name: string, data: string) => boolean;
    /**
     *  Remove an object from the browser's session storage
     *
     *  @param {string} name - the name of the object to remove from storage
     *  @returns {boolean} True if the storage object could be removed.
     */
    removeSessionStorage: (logger: IDiagnosticLogger, name: string) => boolean;
    /**
     * @deprecated - Use the core.getCookieMgr().disable()
     * Force the SDK not to store and read any data from cookies.
     */
    disableCookies: () => void;
    /**
     * @deprecated - Use the core.getCookieMgr().isEnabled()
     * Helper method to tell if document.cookie object is available and whether it can be used.
     */
    canUseCookies: (logger: IDiagnosticLogger) => any;
    disallowsSameSiteNone: (userAgent: string) => boolean;
    /**
     * @deprecated - Use the core.getCookieMgr().set()
     * helper method to set userId and sessionId cookie
     */
    setCookie: (logger: IDiagnosticLogger, name: string, value: string, domain?: string) => void;
    stringToBoolOrDefault: (str: any, defaultValue?: boolean) => boolean;
    /**
     * @deprecated - Use the core.getCookieMgr().get()
     * helper method to access userId and sessionId cookie
     */
    getCookie: (logger: IDiagnosticLogger, name: string) => string;
    /**
     * @deprecated - Use the core.getCookieMgr().del()
     * Deletes a cookie by setting it's expiration time in the past.
     * @param name - The name of the cookie to delete.
     */
    deleteCookie: (logger: IDiagnosticLogger, name: string) => void;
    /**
     * helper method to trim strings (IE8 does not implement String.prototype.trim)
     */
    trim: (str: any) => string;
    /**
     * generate random id string
     */
    newId: () => string;
    /**
     * generate a random 32bit number (-0x80000000..0x7FFFFFFF).
     */
    random32: () => number;
    /**
     * generate W3C trace id
     */
    generateW3CId: () => string;
    /**
     * Check if an object is of type Array
     */
    isArray: (obj: any) => boolean;
    /**
     * Check if an object is of type Error
     */
    isError: (obj: any) => obj is Error;
    /**
     * Check if an object is of type Date
     */
    isDate: (obj: any) => obj is Date;
    toISOStringForIE8: (date: Date) => string;
    /**
     * Gets IE version returning the document emulation mode if we are running on IE, or null otherwise
     */
    getIEVersion: (userAgentStr?: string) => number;
    /**
     * Convert ms to c# time span format
     */
    msToTimeSpan: (totalms: number) => string;
    /**
     * Checks if error has no meaningful data inside. Ususally such errors are received by window.onerror when error
     * happens in a script from other domain (cross origin, CORS).
     */
    isCrossOriginError: (message: string | Event, url: string, lineNumber: number, columnNumber: number, error: Error) => boolean;
    /**
     * Returns string representation of an object suitable for diagnostics logging.
     */
    dump: (object: any) => string;
    /**
     * Returns the name of object if it's an Error. Otherwise, returns empty string.
     */
    getExceptionName: (object: any) => string;
    /**
     * Adds an event handler for the specified event to the window
     * @param eventName {string} - The name of the event
     * @param callback {any} - The callback function that needs to be executed for the given event
     * @return {boolean} - true if the handler was successfully added
     */
    addEventHandler: (obj: any, eventNameWithoutOn: string, handlerRef: any, useCapture: boolean) => boolean;
    /**
     * Tells if a browser supports a Beacon API
     */
    IsBeaconApiSupported: () => boolean;
    getExtension: (extensions: IPlugin[], identifier: string) => IPlugin | null;
}
export declare const Util: IUtil;
export interface IUrlHelper {
    parseUrl: (url: string) => HTMLAnchorElement;
    getAbsoluteUrl: (url: string) => string;
    getPathName: (url: string) => string;
    getCompleteUrl: (method: string, absoluteUrl: string) => string;
    parseHost: (url: string, inclPort?: boolean) => string;
    /**
     * Get the full host from the url, optionally including the port
     */
    parseFullHost: (url: string, inclPort?: boolean) => string;
}
export declare const UrlHelper: IUrlHelper;
export interface ICorrelationIdHelper {
    correlationIdPrefix: string;
    /**
     * Checks if a request url is not on a excluded domain list and if it is safe to add correlation headers.
     * Headers are always included if the current domain matches the request domain. If they do not match (CORS),
     * they are regex-ed across correlationHeaderDomains and correlationHeaderExcludedDomains to determine if headers are included.
     * Some environments don't give information on currentHost via window.location.host (e.g. Cordova). In these cases, the user must
     * manually supply domains to include correlation headers on. Else, no headers will be included at all.
     */
    canIncludeCorrelationHeader(config: ICorrelationConfig, requestUrl: string, currentHost?: string): boolean;
    /**
     * Combines target appId and target role name from response header.
     */
    getCorrelationContext(responseHeader: string): string | undefined;
    /**
     * Gets key from correlation response header
     */
    getCorrelationContextValue(responseHeader: string, key: string): string | undefined;
}
export declare const CorrelationIdHelper: ICorrelationIdHelper;
export declare function AjaxHelperParseDependencyPath(logger: IDiagnosticLogger, absoluteUrl: string, method: string, commandName: string): {
    target: string;
    name: string;
    data: string;
};
export declare function dateTimeUtilsNow(): number;
export declare function dateTimeUtilsDuration(start: number, end: number): number;
export interface IDateTimeUtils {
    /**
     * Get the number of milliseconds since 1970/01/01 in local timezone
     */
    Now: () => number;
    /**
     * Gets duration between two timestamps
     */
    GetDuration: (start: number, end: number) => number;
}
/**
 * A utility class that helps getting time related parameters
 */
export declare const DateTimeUtils: IDateTimeUtils;
/**
 * Creates a IDistributedTraceContext from an optional telemetryTrace
 * @param telemetryTrace - The telemetryTrace instance that is being wrapped
 * @param parentCtx - An optional parent distributed trace instance, almost always undefined as this scenario is only used in the case of multiple property handlers.
 * @returns A new IDistributedTraceContext instance that is backed by the telemetryTrace or temporary object
 */
export declare function createDistributedTraceContextFromTrace(telemetryTrace?: ITelemetryTrace, parentCtx?: IDistributedTraceContext): IDistributedTraceContext;
