(() => {
  var __require = /* @__PURE__ */ ((x) => typeof require !== "undefined" ? require : typeof Proxy !== "undefined" ? new Proxy(x, {
    get: (a, b) => (typeof require !== "undefined" ? require : a)[b]
  }) : x)(function(x) {
    if (typeof require !== "undefined")
      return require.apply(this, arguments);
    throw new Error('Dynamic require of "' + x + '" is not supported');
  });

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/utils/error.ts
  var createError = (msg) => new Error(`[gitlab-vscode] ${msg}`);

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/utils/getConfigFromDOM.ts
  var CONFIG_EL_ID = "gl-config-json";
  var getConfigFromDOM = () => {
    const el = document.getElementById(CONFIG_EL_ID);
    if (!el) {
      throw createError(`Could not find element for config in document (${CONFIG_EL_ID}).`);
    }
    const json = el.dataset.settings;
    if (!json) {
      throw createError(`Could not find "data-settings" in config element (${CONFIG_EL_ID}).`);
    }
    return JSON.parse(json);
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/gitlab-api-client/src/gitlab_client.ts
  var buildUrl = (baseUrl, ...parts) => {
    const relativePath = parts.map(encodeURIComponent).join("/");
    return new URL(relativePath, baseUrl).toString();
  };
  var withParams = (baseUrl, params) => {
    const paramEntries = Object.entries(params);
    if (!paramEntries.length) {
      return baseUrl;
    }
    const url = new URL(baseUrl);
    paramEntries.forEach(([key, value]) => {
      url.searchParams.append(key, value);
    });
    return url.toString();
  };
  var GitLabClient = class {
    constructor(config) {
      this._baseUrl = config.baseUrl;
      this._authToken = config.authToken;
    }
    fetchProjectBranch(projectId, branchName) {
      const url = this._buildApiUrl("projects", projectId, "repository", "branches", branchName);
      return this._fetchJson(url);
    }
    fetchTree(projectId, ref) {
      const url = this._buildApiUrl("projects", projectId, "repository", "tree");
      const params = {
        ref,
        recursive: "true",
        pagination: "none"
      };
      return this._fetchJson(url, params);
    }
    async fetchFileRaw(projectId, ref, path) {
      const url = this._buildApiUrl("projects", projectId, "repository", "files", path, "raw");
      const response = await this._fetch(url, { ref });
      return response.arrayBuffer();
    }
    _buildApiUrl(...parts) {
      return buildUrl(this._baseUrl, "api", "v4", ...parts);
    }
    _fetch(url, params = {}) {
      return fetch(withParams(url, params), {
        method: "GET",
        headers: this._getHeaders()
      });
    }
    async _fetchJson(url, params = {}) {
      const response = await this._fetch(url, params);
      const result = await response.json();
      return result;
    }
    _getHeaders() {
      if (!this._authToken) {
        return {};
      }
      return {
        "PRIVATE-TOKEN": this._authToken
      };
    }
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-mediator-commands/src/start.ts
  var handleStartCommand = async (config, client) => {
    const branch = await client.fetchProjectBranch(config.projectPath, config.ref);
    const tree = await client.fetchTree(config.projectPath, branch.commit.id);
    const blobs = tree.filter((item) => item.type === "blob");
    return {
      branch,
      files: blobs,
      repoRoot: config.repoRoot
    };
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-mediator-commands/src/commands.ts
  var COMMAND_START = "gitlab-vscode.start";
  var COMMAND_FETCH_FILE_RAW = "gitlab-vscode.fetch-file-raw";
  var COMMAND_READY = "gitlab-vscode.ready";
  var createCommands = (config, bufferWrapper) => {
    const client = new GitLabClient({ baseUrl: config.gitlabUrl, authToken: config.gitlabToken });
    return [
      {
        id: COMMAND_START,
        handler: () => handleStartCommand(config, client)
      },
      {
        id: COMMAND_FETCH_FILE_RAW,
        handler: async (ref, path) => {
          const buffer = await client.fetchFileRaw(config.projectPath, ref, path);
          const byteArr = new Uint8Array(buffer);
          return bufferWrapper(byteArr);
        }
      },
      {
        id: COMMAND_READY,
        handler: () => {
          window.postMessage({ key: "ready" });
        }
      }
    ];
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/utils/createBufferWrapper.ts
  var createBufferWrapper = (bufferModule) => {
    return (actual) => bufferModule.VSBuffer.wrap(actual);
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/utils/getRepoRoot.ts
  var getRepoRoot = (projectPath) => {
    const slashIdx = projectPath.lastIndexOf("/");
    if (slashIdx < 0) {
      return projectPath;
    }
    return projectPath.slice(slashIdx + 1);
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/start.ts
  var start = (config) => {
    define("gitlab-vscode/main", [
      "vs/workbench/workbench.web.main",
      "vs/base/common/buffer",
      "vs/platform/log/common/log"
    ], (workbench, bufferModule, commonLogModule) => {
      const repoRoot = getRepoRoot(config.projectPath);
      const fullConfig = {
        ...config,
        repoRoot
      };
      workbench.create(document.body, {
        enabledExtensions: ["gitlab.gitlab-web-ide"],
        developmentOptions: {
          logLevel: commonLogModule.parseLogLevel("trace")
        },
        homeIndicator: {
          href: "https://gitlab.com",
          icon: "code",
          title: "GitLab"
        },
        welcomeBanner: void 0,
        commands: createCommands(fullConfig, createBufferWrapper(bufferModule)),
        defaultLayout: {
          views: [],
          editors: [],
          force: true
        },
        settingsSyncOptions: {
          enabled: false
        },
        productConfiguration: {
          extensionsGallery: {
            serviceUrl: "https://open-vsx.org/vscode/gallery",
            itemUrl: "https://open-vsx.org/vscode/item",
            resourceUrlTemplate: "https://open-vsx.org/vscode/asset/{publisher}/{name}/{version}/Microsoft.VisualStudio.Code.WebResources/{path}",
            controlUrl: "",
            recommendationsUrl: ""
          }
        },
        workspaceProvider: {
          workspace: { folderUri: workbench.URI.parse(`gitlab-vscode-ide:///${repoRoot}`) },
          trusted: true,
          async open() {
            return false;
          }
        }
      });
    });
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/utils/insertScript.ts
  var insertScript = async (src, nonce = "") => {
    const script = document.createElement("script");
    script.src = src;
    script.nonce = nonce;
    return new Promise((resolve, reject) => {
      script.onerror = reject;
      script.onload = () => resolve(script);
      document.head.appendChild(script);
    });
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/utils/insertMeta.ts
  var insertMeta = (id, jsonSetting) => {
    const meta = document.createElement("meta");
    meta.id = id;
    meta.setAttribute("data-settings", JSON.stringify(jsonSetting));
    document.head.appendChild(meta);
    return meta;
  };

  // pnp:/builds/gitlab-org/gitlab-web-ide/packages/vscode-bootstrap/src/main.ts
  var SCRIPT_VSCODE_LOADER = "vscode/out/vs/loader.js";
  var SCRIPT_VSCODE_WEB_PACKAGE_PATHS = "vscode/out/vs/webPackagePaths.js";
  var SCRIPT_VSCODE_WORKBENCH_NLS = "vscode/out/vs/workbench/workbench.web.main.nls.js";
  var SCRIPT_VSCODE_WORKBENCH = "vscode/out/vs/workbench/workbench.web.main.js";
  var setupBuiltinExtensionMeta = async (config) => {
    const extensionPackageJSONUrl = `${config.baseUrl}/vscode/extensions/gitlab-web-ide/package.json`;
    const rawJson = await fetch(extensionPackageJSONUrl).then((x) => x.text());
    const packageJSON = JSON.parse(rawJson);
    insertMeta("gitlab-builtin-vscode-extensions", [
      {
        extensionPath: "gitlab-web-ide",
        packageJSON
      }
    ]);
  };
  var setupNavigatorKeyboard = () => {
    if (navigator.keyboard) {
      Object.assign(navigator.keyboard, {
        getLayoutMap: () => Promise.resolve(/* @__PURE__ */ new Map())
      });
    }
  };
  var setupAMDRequire = (config) => {
    const vscodeUrl = `${config.baseUrl}/vscode`;
    Object.keys(self.webPackagePaths).map(function(key) {
      self.webPackagePaths[key] = `${vscodeUrl}/bundled_node_modules/${key}/${self.webPackagePaths[key]}`;
    });
    __require.config({
      baseUrl: `${vscodeUrl}/out`,
      recordStats: true,
      trustedTypesPolicy: window.trustedTypes?.createPolicy("amdLoader", {
        createScriptURL(value) {
          if (value.startsWith(vscodeUrl)) {
            return value;
          }
          throw new Error(`Invalid script url: ${value}`);
        }
      }),
      paths: self.webPackagePaths
    });
    [
      "vs/workbench/contrib/welcome/gettingStarted/browser/gettingStarted.contribution",
      "vs/workbench/contrib/welcome/page/browser/welcomePage.contribution"
    ].forEach((path) => {
      define(path, () => {
      });
    });
  };
  var main = async () => {
    const config = getConfigFromDOM();
    await setupBuiltinExtensionMeta(config);
    setupNavigatorKeyboard();
    await Promise.all([
      insertScript(`${config.baseUrl}/${SCRIPT_VSCODE_LOADER}`, config.nonce),
      insertScript(`${config.baseUrl}/${SCRIPT_VSCODE_WEB_PACKAGE_PATHS}`, config.nonce)
    ]);
    setupAMDRequire(config);
    await Promise.all([
      insertScript(`${config.baseUrl}/${SCRIPT_VSCODE_WORKBENCH_NLS}`, config.nonce),
      insertScript(`${config.baseUrl}/${SCRIPT_VSCODE_WORKBENCH}`, config.nonce)
    ]);
    start(config);
  };
  main().catch((e) => {
    console.error(e);
    alert("An unexpected error occurred! Please open the developer console for details.");
  });
})();
