"use strict";
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerPPLLanguage = void 0;
const tslib_1 = require("tslib");
const monaco_1 = require("../monaco");
const constants_1 = require("./constants");
const worker_store_1 = require("../worker_store");
const worker_proxy_service_1 = require("./worker_proxy_service");
const ppl_language_analyzer_1 = require("./ppl_language_analyzer");
// @ts-ignore
const ppl_documentation_1 = require("./ppl_documentation");
// @ts-ignore
const ppl_editor_worker_js_1 = tslib_1.__importDefault(require("!!raw-loader!../../target/public/ppl.editor.worker.js"));
const PPL_LANGUAGE_ID = constants_1.ID;
const OWNER = 'PPL_WORKER';
// Register ppl worker to the worker map first
(0, worker_store_1.registerWorker)(constants_1.ID, ppl_editor_worker_js_1.default);
// PPL worker proxy service for worker-based syntax highlighting
const pplWorkerProxyService = new worker_proxy_service_1.PPLWorkerProxyService();
// PPL analyzer for synchronous tokenization (lazy initialization)
let pplAnalyzer;
/**
 * Map PPL Language Analyzer tokens to Monaco editor token classes
 * Based on ANTLR-generated token types from OpenSearchPPLLexer
 */
const mapPPLTokenToMonacoTokenType = (tokenType) => {
    const type = tokenType.toUpperCase();
    // Use optimized Set lookups from constants
    for (const [monacoType, tokenSet] of Object.entries(constants_1.PPL_TOKEN_SETS)) {
        if (tokenSet.has(type)) {
            return monacoType;
        }
    }
    // Default case
    return 'identifier';
};
/**
 * Create Monaco language configuration for PPL
 */
const createPPLLanguageConfiguration = () => ({
    comments: {
        lineComment: '//',
        blockComment: ['/*', '*/'],
    },
    brackets: [
        ['{', '}'],
        ['[', ']'],
        ['(', ')'],
    ],
    autoClosingPairs: [
        { open: '{', close: '}' },
        { open: '[', close: ']' },
        { open: '(', close: ')' },
        { open: '"', close: '"', notIn: ['string'] },
        { open: "'", close: "'", notIn: ['string', 'comment'] },
    ],
    surroundingPairs: [
        { open: '{', close: '}' },
        { open: '[', close: ']' },
        { open: '(', close: ')' },
        { open: '"', close: '"' },
        { open: "'", close: "'" },
    ],
});
/**
 * Set up synchronous tokenization for PPL
 */
const setupPPLTokenization = () => {
    monaco_1.monaco.languages.setTokensProvider(PPL_LANGUAGE_ID, {
        getInitialState: () => {
            const state = {
                clone: () => state,
                equals: () => true,
            };
            return state;
        },
        tokenize: (line, state) => {
            // Use PPL Language Analyzer for accurate tokenization
            const tokens = [];
            try {
                // Only process if line contains potential PPL content
                if (line.trim()) {
                    // Lazy initialize the PPL analyzer only when needed
                    if (!pplAnalyzer) {
                        pplAnalyzer = (0, ppl_language_analyzer_1.getPPLLanguageAnalyzer)();
                    }
                    const pplTokens = pplAnalyzer.tokenize(line);
                    for (const pplToken of pplTokens) {
                        const tokenType = mapPPLTokenToMonacoTokenType(pplToken.type);
                        tokens.push({
                            startIndex: pplToken.startIndex,
                            scopes: tokenType,
                        });
                    }
                }
            }
            catch (error) {
                // If ANTLR fails, return empty tokens
            }
            return {
                tokens,
                endState: state,
            };
        },
    });
};
/**
 * Set up syntax highlighting using PPL worker
 */
const setupPPLSyntaxHighlighting = () => {
    // Set up the worker
    pplWorkerProxyService.setup(ppl_editor_worker_js_1.default);
    const allDisposables = [];
    const modelHandlers = new Map();
    const processSyntaxHighlighting = async (model) => {
        // Only process if the model is still set to PPL language
        if (model.getLanguageId() !== PPL_LANGUAGE_ID) {
            // Clear any existing PPL markers if language changed
            monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
            return;
        }
        try {
            const content = model.getValue();
            // Get validation result from worker
            const validationResult = await pplWorkerProxyService.validate(content);
            if (validationResult.errors.length > 0) {
                // Convert errors to Monaco markers
                const markers = validationResult.errors.map((error) => {
                    // Handle different error types with safe property access
                    const startLineNumber = error.startLineNumber || error.line || 1;
                    const endLineNumber = error.endLineNumber || error.endLine || startLineNumber;
                    const startColumn = error.startColumn || error.column || 1;
                    const endColumn = error.endColumn || startColumn + 1;
                    const safeStartLine = Math.max(1, startLineNumber);
                    const safeEndLine = Math.max(safeStartLine, endLineNumber);
                    const safeStartColumn = Math.max(1, startColumn);
                    const safeEndColumn = Math.max(safeStartColumn, endColumn);
                    const docLink = (0, ppl_documentation_1.getPPLDocumentationLink)(error.message);
                    return {
                        severity: monaco_1.monaco.MarkerSeverity.Error,
                        message: error.message,
                        startLineNumber: safeStartLine,
                        startColumn: safeStartColumn,
                        endLineNumber: safeEndLine,
                        endColumn: safeEndColumn,
                        // Add error code for better categorization
                        code: {
                            value: 'View Documentation',
                            target: monaco_1.monaco.Uri.parse(docLink.url),
                        },
                    };
                });
                monaco_1.monaco.editor.setModelMarkers(model, OWNER, markers);
            }
            else {
                // Clear markers if no errors
                monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
            }
        }
        catch (error) {
            // Silent error handling - continue without worker-based highlighting
        }
    };
    const addModelHandlers = (model) => {
        if (model.getLanguageId() === PPL_LANGUAGE_ID) {
            const disposables = [];
            // Set up content change listener for syntax highlighting
            disposables.push(model.onDidChangeContent(async () => {
                await processSyntaxHighlighting(model);
            }));
            // Listen to language changes to clean up PPL-specific handling
            disposables.push(model.onDidChangeLanguage(() => {
                const currentLanguage = model.getLanguageId();
                if (currentLanguage !== PPL_LANGUAGE_ID) {
                    // Language changed away from PPL, clean up
                    monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
                    removeModelHandlers(model);
                }
            }));
            // Store handlers for this model
            modelHandlers.set(model, disposables);
            // Process initial syntax highlighting
            processSyntaxHighlighting(model);
        }
    };
    const removeModelHandlers = (model) => {
        const disposables = modelHandlers.get(model);
        if (disposables) {
            disposables.forEach((d) => d.dispose());
            modelHandlers.delete(model);
        }
    };
    const onModelAdd = (model) => {
        addModelHandlers(model);
    };
    const onModelRemove = (model) => {
        removeModelHandlers(model);
        // Clear any PPL markers when model is removed
        monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
    };
    // Handle newly created models
    allDisposables.push(monaco_1.monaco.editor.onDidCreateModel(onModelAdd));
    // Handle model disposal
    allDisposables.push(monaco_1.monaco.editor.onWillDisposeModel(onModelRemove));
    // Handle existing models
    const existingModels = monaco_1.monaco.editor.getModels();
    existingModels.forEach(onModelAdd);
    return () => {
        // Clean up all model handlers
        modelHandlers.forEach((disposables, model) => {
            disposables.forEach((d) => d.dispose());
            monaco_1.monaco.editor.setModelMarkers(model, OWNER, []);
        });
        modelHandlers.clear();
        // Clean up global disposables
        allDisposables.forEach((d) => d.dispose());
        pplWorkerProxyService.stop();
    };
};
/**
 * Register PPL language support with Monaco Editor
 */
const registerPPLLanguage = () => {
    // Register the PPL language
    monaco_1.monaco.languages.register({
        id: PPL_LANGUAGE_ID,
        extensions: ['.ppl'],
        aliases: ['PPL', 'ppl', 'Piped Processing Language'],
        mimetypes: ['application/ppl', 'text/ppl'],
    });
    // Set language configuration
    monaco_1.monaco.languages.setLanguageConfiguration(PPL_LANGUAGE_ID, createPPLLanguageConfiguration());
    // Set up synchronous tokenization
    setupPPLTokenization();
    // Set up syntax highlighting with worker
    const disposeSyntaxHighlighting = setupPPLSyntaxHighlighting();
    return {
        dispose: () => {
            disposeSyntaxHighlighting();
        },
    };
};
exports.registerPPLLanguage = registerPPLLanguage;
// Auto-register PPL language support
(0, exports.registerPPLLanguage)();
//# sourceMappingURL=language.js.map